#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

#
# This file contains perl subroutines used for data input and output.
# These subroutine are for use with flat file databases.
#

package webcalng_io;
use strict;
use Fcntl ':flock';

#
# Get a hash of the fields for admin preferences.
#
sub get_admin_preferences {
	my ($admin_preferences_file,$calendar_preferences_file,%data);

	$admin_preferences_file = $::db_dir . "/preferences";
	open FILE, "<$admin_preferences_file" or webcalng_subs::hard_error("Could not read $admin_preferences_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $admin_preferences_file: $!\n");
	while (<FILE>) {
		next if /^#/;
		next if /^\s*$/;
		chomp;
		my ($key,$information)                         = split /=/, $_, 2;
		my ($value,$options,$category,$usermod,$order) = split /;/, $information, 5;
		$data{$key}{'value'}    = $value || "";
		$data{$key}{'options'}  = $options;
		$data{$key}{'category'} = $category;
		$data{$key}{'usermod'}  = $usermod;
		$data{$key}{'order'}    = $order;
	}
	close FILE;
	
	return \%data;
}

#
# Get a hash of the fields for user preferences.
#
sub get_calendar_preferences {
	my ($calendar) = (@_);
	my ($calendar_preferences_file,%data);

	$calendar_preferences_file = $::db_dir . "/" . $calendar . "/preferences";

	if (-f "$calendar_preferences_file") {
		open FILE, "<$calendar_preferences_file" or webcalng_subs::hard_error("Could not read $calendar_preferences_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $calendar_preferences_file: $!\n");
		while (<FILE>) {
			next if /^#/;
			next if /^\s*$/;
			chomp;
			my ($key,$value) = split /=/, $_, 2;
			$data{$key}      = $value || "";
		}
		close FILE;
	}
	
	return \%data;
}

#
# Subroutine to write changes to the admin preference file.  The value
# and/or the usermod functionality can change for any given preference key.
# 
sub save_admin_preferences {
	my ($dataref) = (@_);
	my ($preferences_file,@data);

	$preferences_file = $::db_dir . "/preferences";
	open FILE, "+>>$preferences_file" or webcalng_subs::hard_error("Could not open $preferences_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $preferences_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		if (/^\s+$/) {
			push(@data,$_);
			next;
		}
		if (/^#/) {
			push(@data,$_);
			next;
		}
		my ($key,$information) = split /=/, $_, 2;
		if (defined $dataref->{$key}{'value'}) {
			chomp $information;
			my ($value,$options,$category,$usermod,$order) = split /;/, $information, 5;
			$dataref->{$key}{'value'} =~ s/;//g;
			my $line = "${key}=$dataref->{$key}{'value'};${options};${category};$dataref->{$key}{'usermod'};${order}\n";
			push(@data,$line);
		} else {
			push(@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Subroutine to write changes to a calendar preference file.  Users
# can only update the value for a preference key in their calendar.
# 
sub save_calendar_preferences {
	my ($dataref) = (@_);
	my ($preferences_file);

	$preferences_file = $::db_dir . "/" . $::calendar . "/preferences";
	open FILE, ">$preferences_file" or webcalng_subs::hard_error("Could not open $preferences_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $preferences_file: $!\n");
	for my $key (keys %::preferences) {
		if (defined $dataref->{$key}{'value'}) {
			print FILE "${key}=$dataref->{$key}{'value'}\n" if ($::preferences{$key}{'usermod'});
		} else {
			print FILE "${key}=$::preferences{$key}{'value'}\n";
		}
	}
	close FILE;

	return 1;
}

#
# Get a list of available calendars to display.  If they are
# searching for a calendar, send back a list of all calendars
# that match - or all available calendars if there are not matches.
#
sub get_calendar_list {
	my ($dir,@dir,@list,@all);
	opendir DIR, $::db_dir or webcalng_subs::hard_error("Could not open $::db_dir for reading: $!");
	@dir = readdir DIR;
	closedir DIR;
	for (@dir) {
		next if /^\./;
		my $path = $::db_dir . "/" . $_;
		if ($::calendar) {
			push(@list,$_) if ((-d $path) && (/$::calendar/i));
			return ($_) if ($_ eq $::calendar);
		} else {
			push(@list,$_) if (-d $path);
		}
		push(@all,$_) if (-d $path);
	}
	if ($#list == -1) {
		return @all;	
	} else {
		return @list;	
	}
}

#
# Find out who owns and has read and/or write access to a calendar.
#
sub get_calendar_permissions {
	my ($calendar) = (@_);
	my ($owner_file,$username,$type,$type_file,%data);
	return \%data unless ($calendar);

	# Find calendar owner.
	$owner_file = $::db_dir . "/" . $calendar . "/owner";
	open FILE, "<$owner_file" or webcalng_subs::hard_error("Could not open $owner_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $owner_file: $!\n");
	chomp($username = <FILE>);
	close FILE;
	$data{'owner'} = $username;

	# Find type of calendar.
	$type_file = $::db_dir . "/" . $calendar . "/type";
	open FILE, "<$type_file" or webcalng_subs::hard_error("Could not open $type_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $type_file: $!\n");
	chomp($type = <FILE>);
	close FILE;
	$data{'type'} = $type;

	# Create regexp of users who have read/write access, including the calendar owner.
	# Format of regexp is: user1|user2|user3|user4
	if ($username) {
		$data{'read'}  = $username . " " . $::preferences{'read_access'}{'value'};	
		$data{'write'} = $username . " " . $::preferences{'write_access'}{'value'};	
		$data{'read'}  =~ s/\s+/|/g;
		$data{'write'} =~ s/\s+/|/g;
		$data{'read'}  =~ s/\,/|/g;
		$data{'write'} =~ s/\,/|/g;
		$data{'read'}  =~ s/\|$//;
		$data{'write'} =~ s/\|$//;
		$data{'read'}  =~ s/^\|//;
		$data{'write'} =~ s/^\|//;
	} else {
		$data{'read'}  = "";
		$data{'write'} = "";
	}

	return \%data;
}

#
# See if a given calendar name already exists.  This should return 0 if the calendar does
# not exist, and 1 if it does.
#
sub calendar_exists {
	my ($calendar) = (@_);
	my ($calendar_exists,$calendar_directory);
	$calendar_exists    = 0;
	$calendar_directory = $::db_dir . "/" . $calendar;
	$calendar_exists    = 1 if (-d "$calendar_directory");
	return $calendar_exists;
}

#
# Create a new calendar.
#
sub create_calendar {
	my ($calendar,$type,$username) = (@_);
	my ($calendar_directory,$type_file,$owner_file,$items_file,$exceptions_file);

	$calendar_directory = $::db_dir . "/" . $calendar;
	$type_file          = $::db_dir . "/" . $calendar . "/type";
	$owner_file         = $::db_dir . "/" . $calendar . "/owner";
	$items_file         = $::db_dir . "/" . $calendar . "/items";
	$exceptions_file    = $::db_dir . "/" . $calendar . "/exceptions";

	mkdir $calendar_directory, 0700 or webcalng_subs::hard_error("Could not create $calendar_directory: $!");

	open FILE, ">$type_file" or webcalng_subs::hard_error("Could not open $type_file for writing: $!");
	print FILE "$type\n";
	close FILE;

	open FILE, ">$owner_file" or webcalng_subs::hard_error("Could not open $owner_file for writing: $!");
	print FILE "$username\n";
	close FILE;

	open FILE, ">$items_file" or webcalng_subs::hard_error("Could not open $items_file for writing: $!");
	close FILE;

	open FILE, ">$exceptions_file" or webcalng_subs::hard_error("Could not open $exceptions_file for writing: $!");
	close FILE;

	return 1;
}

#
# Remove all evidence of a calendar's existence!
#
sub remove_calendar {
	my ($calendar) = (@_);
	my ($calendar_directory,@files);

	return 1 if ((! defined $calendar) || ($calendar eq ""));
	$calendar_directory = $::db_dir . "/" . $calendar;

	# Sanity check.
	if (! -d $calendar_directory) {
		webcalng_subs::hard_error("Calendar $::calendar did not exist.");
	}
	
	# Blow it away.
	opendir DIR, $calendar_directory or webcalng_subs::hard_error("Could not open $calendar_directory for reading: $!");
	@files = readdir DIR;
	closedir DIR;
	for (@files) {
		my $file = "$calendar_directory/$_";
		next unless (-f "$file");
		unlink "$file" or webcalng_subs::hard_error("Could not remove file $file: $!");
	}
	rmdir $calendar_directory or webcalng_subs::hard_error("Could not remove directory $calendar_directory: $!");

	return 1;
}

#
# Update the owner and/or type of a calendar.
#
sub update_calinfo {
	my ($calendar,$owner,$type) = (@_);
	my ($type_file,$owner_file);

	$type_file  = $::db_dir . "/" . $calendar . "/type";
	$owner_file = $::db_dir . "/" . $calendar . "/owner";

	open FILE, ">$type_file" or webcalng_subs::hard_error("Could not open $type_file for writing: $!");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock $type_file: $!\n");
	print FILE "$type\n";
	close FILE;

	open FILE, ">$owner_file" or webcalng_subs::hard_error("Could not open $owner_file for writing: $!");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock $owner_file: $!\n");
	print FILE "$owner\n";
	close FILE;

	return 1;
}

#
# Change the name of a calendar.
#
sub update_calname {
	my ($oldcal,$cal,$type,$owner) = (@_);
	my ($oldcalendar_directory,$newcalendar_directory,@files,$oldfile,$newfile);
	
	# Create the new calendar name, copy all the information in, then delete the old calendar.
	create_calendar($cal,$type,$owner);
	$oldcalendar_directory = $::db_dir . "/" . $oldcal;
	$newcalendar_directory = $::db_dir . "/" . $cal;
	opendir DIR, $oldcalendar_directory or webcalng_subs::hard_error("Could not open $oldcalendar_directory for reading: $!");
	@files = readdir DIR;
	closedir DIR;
	for (@files) {
		$oldfile = "$oldcalendar_directory/$_";
		next unless (-f "$oldfile");
		$newfile = "$newcalendar_directory/$_";
		open OLDFILE, "<$oldfile" or webcalng_subs::hard_error("Could not open $oldfile: $!\n");
		flock OLDFILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $oldfile: $!\n");
		open NEWFILE, ">$newfile" or webcalng_subs::hard_error("Could not open $newfile: $!\n");
		flock NEWFILE, LOCK_EX or webcalng_subs::hard_error("Could not lock $newfile: $!\n");
		while (<OLDFILE>) {
			print NEWFILE;
		}
		close NEWFILE;
		close OLDFILE;
	}
	remove_calendar($oldcal);

	return 1;
}

#
# Get the info for a given addressid, or all addressids.
#
sub get_address {
	my ($addressid) = (@_);
	my ($address_file,%data);

	# Open the address file, and get the required data.
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	if (-f "$address_file") {
		open FILE, "<$address_file" or webcalng_subs::hard_error("Could not open $address_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $address_file: $!\n");
		while (<FILE>) {
			chomp;
			my ($tmpaddressid,$displayname,$emailaddress,$phonenumber,$phonenumber2) = (split /;/, $_);
			if ($addressid) {
				if ($tmpaddressid eq $addressid) {
					$data{'displayname'}  = $displayname  || "";
					$data{'emailaddress'} = $emailaddress || "";
					$data{'phonenumber'}  = $phonenumber  || "";
					$data{'phonenumber2'} = $phonenumber2 || "";
					last;
				}
			} else {
				$data{$tmpaddressid}{'displayname'}  = $displayname  || "";
				$data{$tmpaddressid}{'emailaddress'} = $emailaddress || "";
				$data{$tmpaddressid}{'phonenumber'}  = $phonenumber  || "";
				$data{$tmpaddressid}{'phonenumber2'} = $phonenumber2 || "";
			}
		}
		close FILE;
	}

	return \%data;
}

#
# Determine if a given address already exists.
#
sub address_exists {
	my ($displayname) = (@_);
	my ($exists,$address_file);

	# Open the address file, and see if we find the given displayname.
	$exists = 0;
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	if (-f "$address_file") {
		open FILE, "<$address_file" or webcalng_subs::hard_error("Could not open $address_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $address_file: $!\n");
		while (<FILE>) {
			my $tmpdisplayname = (split /;/, $_)[1];
			$exists = 1, last if ($tmpdisplayname eq $displayname);
		}
		close FILE;
	}

	return $exists;
}

#
# Get the next available addressid.
#
sub get_next_addressid {
	my ($address_file,$addressid);

	# Open the address file, and file out what the last entry is.
	$addressid    = 0;
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	if (-f "$address_file") {
		open FILE, "<$address_file" or webcalng_subs::hard_error("Could not open $address_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $address_file: $!\n");
		while (<FILE>) {
			$addressid = (split /;/, $_)[0];
		}
		close FILE;
	}
	$addressid++;

	return $addressid;
}

#
# Add a new address record to the address file.
#
sub addaddress_io {
	my ($dataref) = (@_);
	my ($addressid,$address_file,$line,$key);

	# Get next ID.
	$addressid = get_next_addressid();

	# Clean up the data.
	for $key (keys %$dataref) {
		$dataref->{$key} =~ s/;//g;
	}

	# Create the new line to be added.
	$line = join ';', $addressid, $dataref->{'displayname'}, $dataref->{'emailaddress'}, $dataref->{'phonenumber'}, $dataref->{'phonenumber2'};

	# Add the new line.
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	open FILE, ">>$address_file" or webcalng_subs::hard_error("Could not write to $address_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $address_file: $!\n");
	print FILE "$line\n";
	close FILE;

	return 1; 
}

#
# Update an address record.
#
sub editaddress_io {
	my ($dataref) = (@_);
	my ($address_file,$line,@data,$key);

	# Clean up the data.
	for $key (keys %$dataref) {
		$dataref->{$key} =~ s/;//g;
	}

	# Create the new line to be added.
	$line = join ';', $dataref->{'addressid'}, $dataref->{'displayname'}, $dataref->{'emailaddress'}, $dataref->{'phonenumber'}, $dataref->{'phonenumber2'};

	# Add the new line.
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	open FILE, "+>>$address_file" or webcalng_subs::hard_error("Could not write $address_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $address_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my $tmpaddressid = (split /;/, $_)[0];
		if ($tmpaddressid eq $dataref->{'addressid'}) {
			push(@data,"$line\n");
		} else {
			push(@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Delete an address record.
#
sub deladdress_io {
	my ($addressid) = (@_);
	my ($address_file,@data);

	# Delete the record.
	$address_file = $::db_dir . "/" . $::calendar . "/addresses";
	open FILE, "+>>$address_file" or webcalng_subs::hard_error("Could not write $address_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $address_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my $tmpaddressid = (split /;/, $_)[0];
		push(@data,$_) unless ($tmpaddressid eq $addressid);
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Get the info for a given task id, or all taskids.
#
sub get_task {
	my ($taskid) = (@_);
	my ($task_file,%data);

	# Open the task file, and get the required data.
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	if (-f "$task_file") {
		open FILE, "<$task_file" or webcalng_subs::hard_error("Could not open $task_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $task_file: $!\n");
		while (<FILE>) {
			chomp;
			my ($tmptaskid,$taskdata,$taskdone) = (split /;/, $_);
			if ($taskid) {
				if ($tmptaskid eq $taskid) {
					$data{'taskdata'}  = $taskdata;
					$data{'taskdone'}  = $taskdone;
					last;
				}
			} else {
				$data{$tmptaskid}{'taskdata'}  = $taskdata;
				$data{$tmptaskid}{'taskdone'}  = $taskdone;
			}
		}
		close FILE;
	}

	return \%data;
}

#
# Get the next available taskid.
#
sub get_next_taskid {
	my ($task_file,$taskid);

	# Open the address file, and file out what the last entry is.
	$taskid    = 0;
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	if (-f "$task_file") {
		open FILE, "<$task_file" or webcalng_subs::hard_error("Could not open $task_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock $task_file: $!\n");
		while (<FILE>) {
			$taskid = (split /;/, $_)[0];
		}
		close FILE;
	}
	$taskid++;

	return $taskid;
}

#
# Add a new task to the task file.
#
sub addtask_io {
	my ($dataref) = (@_);
	my ($task_file,$line,$key);

	# Clean up the data.
	for $key (keys %$dataref) {
		$dataref->{$key} =~ s/;//g;
	}
	
	# Get next taskid.
	$dataref->{'taskid'} = get_next_taskid();

	# Create the new line to be added.
	$line = join ';', $dataref->{'taskid'}, $dataref->{'taskdata'}, $dataref->{'taskdone'};

	# Add the new line.
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	open FILE, ">>$task_file" or webcalng_subs::hard_error("Could not write to $task_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $task_file: $!\n");
	print FILE "$line\n";
	close FILE;

	return 1;
}

#
# Update an task record.
#
sub edittask_io {
	my ($dataref) = (@_);
	my ($task_file,$line,@data,$key);

	# Clean up the data.
	for $key (keys %$dataref) {
		$dataref->{$key} =~ s/;//g;
	}
	# Create the new line to be added.
	$line = join ';', $dataref->{'taskid'}, $dataref->{'taskdata'}, $dataref->{'taskdone'};

	# Add the new line.
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	open FILE, "+>>$task_file" or webcalng_subs::hard_error("Could not write $task_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $task_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my $tmptaskid = (split /;/, $_)[0];
		if ($tmptaskid eq $dataref->{'taskid'}) {
			push(@data,"$line\n");
		} else {
			push(@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Mark a task complete.
#
sub marktask_io {
	my ($taskid) = (@_);
	my ($task_file,@data);

	# Update the given taskid.
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	open FILE, "+>>$task_file" or webcalng_subs::hard_error("Could not write $task_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $task_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my $tmptaskid = (split /;/, $_)[0];
		if ($tmptaskid eq $taskid) {
			chomp;
			my ($line);
			my ($taskid,$taskdata,$taskdone) = split /;/, $_;
			$line  = join ';', $taskid, $taskdata, '1';
			$line .= "\n";
			push (@data,$line);
		} else {
			push (@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Delete a task.
#
sub deltask_io {
	my ($taskid) = (@_);
	my ($task_file,@data);

	# Delete the record.
	$task_file = $::db_dir . "/" . $::calendar . "/tasks";
	open FILE, "+>>$task_file" or webcalng_subs::hard_error("Could not write $task_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $task_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my $tmptaskid = (split /;/, $_)[0];
		push(@data,$_) unless ($tmptaskid eq $taskid);
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# Add a new calendar item.
#
sub additem_io {
	my ($dataref,$itemid,$exceptionid,$exception,$calendar) = (@_);
	my ($items_file,$exceptions_file,$line,$key);

	# Open the filehandles for the calendar items and exceptions, and lock the files
	# throughout this subroutine.
	$items_file      = $::db_dir . "/" . $calendar . "/items";
	$exceptions_file = $::db_dir . "/" . $calendar . "/exceptions";
	open IFILE, "+>>$items_file" or webcalng_subs::hard_error("Could not write to $items_file: $!\n");
	flock IFILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
	open EFILE, "+>>$exceptions_file" or webcalng_subs::hard_error("Could not write to $exceptions_file: $!\n");
	flock EFILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");

	# Get next itemid.  The exceptionid is always 0 for new items, or items that are being
	# completely replaced in an edit sequence.
	if (! $itemid) {
		$itemid      = get_next_id(*IFILE);
		$exceptionid = 0;
	} else {
		if ($exception) {
			$exceptionid = get_next_id(*EFILE) if (! $exceptionid);
		} else {
			$exceptionid = 0;
		}
	}

	# Clean up the data.
	for $key (keys %$dataref) {
		next if (! $dataref->{$key});
		$dataref->{$key} =~ s/^\|/ |/;
		$dataref->{$key} =~ s/\|$/| /;
		$dataref->{$key} =~ s/\|\|\|//g;
	}


	# Add the new line.
	if ($exceptionid) {
		$line = join '|||', $exceptionid, $itemid, $dataref->{'modified'}, $::username, $dataref->{'description'}, $dataref->{'exceptiondate'}, $dataref->{'starttime'}, $dataref->{'endtime'}, $dataref->{'visibility'}, $dataref->{'hyperlink'}, $dataref->{'notes'};
		seek EFILE, 0, 2 or webcalng_subs::hard_error("Could not fast forward exceptions file: $!");
		print EFILE "$line\n";
		close EFILE;
	} else {
		$line = join '|||', $itemid, $dataref->{'modified'}, $::username, $dataref->{'description'}, $dataref->{'dateregexp'}, $dataref->{'startdate'}, $dataref->{'enddate'}, $dataref->{'starttime'}, $dataref->{'endtime'}, $dataref->{'visibility'}, $dataref->{'repeat'}, $dataref->{'repeatdays'}, $dataref->{'repeatend'}, $dataref->{'remindwho'}, $dataref->{'remindwhen'}, $dataref->{'remindersent'}, $dataref->{'meetingid'},$dataref->{'hyperlink'}, $dataref->{'notes'};
		seek IFILE, 0, 2 or webcalng_subs::hard_error("Could not fast forward items file: $!");
		print IFILE "$line\n";
		close IFILE;
	}

	return 1;
}

#
# Get the next available itemid for an item or an exception. 
#
sub get_next_id {
	my ($file) = (@_);
	my ($id,$tmpid);

	$id = 0;
	seek $file, 0, 0 or webcalng_subs::hard_error("Could not rewind file: $!");
	while (<$file>) {
		$tmpid = (split /\|\|\|/, $_)[0];
		$id    = $tmpid if ($tmpid > $id);
	}
	$id++;
	
	return $id;
}

#
# Delete an item.
#
sub delitem_io {
	my ($itemid,$exceptionid,$calendar) = (@_);
	my ($items_file,$exceptions_file,@data);

	# If there is an exceptionid, just remove that id from the exceptions file.  If there
	# is no exceptionid, remove all instances of this id from both items and exceptions file.
	$items_file      = $::db_dir . "/" . $calendar . "/items";
	$exceptions_file = $::db_dir . "/" . $calendar . "/exceptions";
	return 1 if (! -f "$items_file");
	if ($exceptionid) {
		# Remove exception.
		open FILE, "+>>$exceptions_file" or webcalng_subs::hard_error("Could not write $exceptions_file: $!\n");
		flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");
		seek FILE, 0, 0;
		while (<FILE>) {
			my ($tmpexceptionid) = (split /\|\|\|/, $_)[0];
			push(@data,$_) unless ($tmpexceptionid == $exceptionid);
		}
		seek FILE, 0, 0;
		truncate FILE, 0;
		print FILE @data;
		close FILE;
	} else {
		# Remove item from items file.	
		open FILE, "+>>$items_file" or webcalng_subs::hard_error("Could not write $items_file: $!\n");
		flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
		seek FILE, 0, 0;
		while (<FILE>) {
			my ($tmpitemid) = (split /\|\|\|/, $_)[0];
			push(@data,$_) unless ($tmpitemid == $itemid);
		}
		seek FILE, 0, 0;
		truncate FILE, 0;
		print FILE @data;
		close FILE;

		# Remove all exceptions that relate to this item.
		@data = ();
		open EFILE, "+>>$exceptions_file" or webcalng_subs::hard_error("Could not write $exceptions_file: $!\n");
		flock EFILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");
		seek EFILE, 0, 0;
		while (<EFILE>) {
			my ($tmpitemid) = (split /\|\|\|/, $_)[1];
			push(@data,$_) unless ($tmpitemid == $itemid);
		}
		seek EFILE, 0, 0;
		truncate EFILE, 0;
		print EFILE @data;
		close EFILE;
	}

	return 1;
}

#
# Create a data structure of all the items for a group of calendars during the
# given time period.
#
sub get_items {
	my ($startmark,$endmark,$searchtext,$targetitemid,$immediate,@calendars) = (@_);
	my ($calendar,%items,%exceptions,$items_file,$exceptions_file);

	# Loop through the data files, grabbing items that are in the time period of interest.
	for $calendar (@calendars) {

		# First get the items.
		$items_file = $::db_dir . "/" . $calendar . "/items";
		next if (! -f "$items_file");
		open FILE, "<$items_file" or webcalng_subs::hard_error("Could not open $items_file: $!\n");
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
		while (<FILE>) {
			chomp;
			my ($itemid,@fields) = split /\|\|\|/, $_;
			next unless ((! $immediate) || (($fields[13]) && (($fields[13] =~ /^1,/) || ($fields[13] =~ /^1$/))));
			my ($startdate,$enddate) = ($fields[4],$fields[5]);
			if ((($startdate >= $startmark) && ($startdate < $endmark)) || (($startdate <= $startmark) && ($enddate >= $startmark))) {
				next if (($searchtext) && ($fields[2] !~ /$searchtext/));
				next if (($targetitemid) && ($targetitemid ne $itemid));
				$items{$calendar}{$itemid}{'modified'}      = $fields[0]  || "";
				$items{$calendar}{$itemid}{'user'}          = $fields[1]  || "";
				$items{$calendar}{$itemid}{'description'}   = $fields[2]  || "";
				$items{$calendar}{$itemid}{'dateregexp'}    = $fields[3]  || "";
				$items{$calendar}{$itemid}{'startdate'}     = $fields[4]  || "";
				$items{$calendar}{$itemid}{'enddate'}       = $fields[5]  || "";
				$items{$calendar}{$itemid}{'starttime'}     = $fields[6]  || "";
				$items{$calendar}{$itemid}{'endtime'}       = $fields[7]  || "";
				$items{$calendar}{$itemid}{'visibility'}    = $fields[8];
				$items{$calendar}{$itemid}{'repeat'}        = $fields[9] || "";
				$items{$calendar}{$itemid}{'repeatdays'}    = $fields[10] || "";
				$items{$calendar}{$itemid}{'repeatend'}     = $fields[11] || "";
				$items{$calendar}{$itemid}{'remindwho'}     = $fields[12];
				$items{$calendar}{$itemid}{'remindwhen'}    = $fields[13] || "";
				$items{$calendar}{$itemid}{'remindersent'}  = $fields[14];
				$items{$calendar}{$itemid}{'meetingid'}     = $fields[15];
				$items{$calendar}{$itemid}{'hyperlink'}     = $fields[16] || "";
				$items{$calendar}{$itemid}{'notes'}         = $fields[17] || "";
				$items{$calendar}{$itemid}{'exceptiondate'} = "";
				if ($calendar eq $::calendar) {
					$items{$calendar}{$itemid}{'origcalendar'} = 0;
				} else {
					$items{$calendar}{$itemid}{'origcalendar'} = $calendar;
				}
			}
		}
		close FILE;

		# Now get the exceptions for all the items that matched the search.
		$exceptions_file = $::db_dir . "/" . $calendar . "/exceptions";
		next if (! -f "$exceptions_file");
		open EFILE, "<$exceptions_file" or webcalng_subs::hard_error("Could not open $exceptions_file: $!\n");
		flock EFILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");
		while (<EFILE>) {
			chomp;
			my ($exceptionid,$itemid,@fields)  = split /\|\|\|/, $_;
			next unless ($items{$calendar}{$itemid}{'description'});
			$exceptions{$calendar}{$itemid}{$exceptionid}{'modified'}      = $fields[0]  || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'user'}          = $fields[1]  || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'description'}   = $fields[2]  || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'dateregexp'}    = $items{$calendar}{$itemid}{'dateregexp'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'startdate'}     = $items{$calendar}{$itemid}{'startdate'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'enddate'}       = $items{$calendar}{$itemid}{'enddate'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'exceptiondate'} = $fields[3];
			$exceptions{$calendar}{$itemid}{$exceptionid}{'starttime'}     = $fields[4]  || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'endtime'}       = $fields[5]  || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'visibility'}    = $fields[6];
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeat'}        = $items{$calendar}{$itemid}{'repeat'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeatdays'}    = $items{$calendar}{$itemid}{'repeatdays'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'repeatend'}     = $items{$calendar}{$itemid}{'repeatend'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindwho'}     = $items{$calendar}{$itemid}{'remindwho'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindwhen'}    = $items{$calendar}{$itemid}{'remindwhen'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'remindersent'}  = $items{$calendar}{$itemid}{'remindersent'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'meetingid'}     = $items{$calendar}{$itemid}{'meetingid'};
			$exceptions{$calendar}{$itemid}{$exceptionid}{'hyperlink'}     = $fields[7] || "";
			$exceptions{$calendar}{$itemid}{$exceptionid}{'notes'}         = $fields[8] || "";
			if ($calendar eq $::calendar) {
				$exceptions{$calendar}{$itemid}{$exceptionid}{'origcalendar'} = 0;
			} else {
				$exceptions{$calendar}{$itemid}{$exceptionid}{'origcalendar'} = $calendar;
			}
		}
		close EFILE;

	}

	return (\%items,\%exceptions);
}

#
# Create a data structure of a single given item.
#
sub get_single_item {
	my ($itemid,$exceptionid) = (@_);
	my (%item,$items_file,$exceptions_file);

	# First, get the item from the items file.
	$items_file = $::db_dir . "/" . $::calendar . "/items";
	open FILE, "<$items_file" or webcalng_subs::hard_error("Could not open $items_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
	while (<FILE>) {
		chomp;
		my ($tmpitemid,@fields) = split /\|\|\|/, $_;
		if ($itemid == $tmpitemid) {
			$item{'modified'}      = $fields[0]  || "";
			$item{'user'}          = $fields[1]  || "";
			$item{'description'}   = $fields[2]  || "";
			$item{'dateregexp'}    = $fields[3]  || "";
			$item{'startdate'}     = $fields[4]  || "";
			$item{'enddate'}       = $fields[5]  || "";
			$item{'starttime'}     = $fields[6]  || "";
			$item{'endtime'}       = $fields[7]  || "";
			$item{'visibility'}    = $fields[8];
			$item{'repeat'}        = $fields[9] || "0";
			$item{'repeatdays'}    = $fields[10] || "";
			$item{'repeatend'}     = $fields[11] || "";
			$item{'remindwho'}     = $fields[12];
			$item{'remindwhen'}    = $fields[13] || "";
			$item{'remindersent'}  = $fields[14];
			$item{'meetingid'}     = $fields[15];
			$item{'hyperlink'}     = $fields[16] || "";
			$item{'notes'}         = $fields[17] || "";
			$item{'exceptiondate'} = "";
			last;
		}
	}
	close FILE;

	# If this is an exception item, fill in that information.
	if ($exceptionid) {
		$exceptions_file = $::db_dir . "/" . $::calendar . "/exceptions";
		open EFILE, "<$exceptions_file" or webcalng_subs::hard_error("Could not open $exceptions_file: $!\n");
		flock EFILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");
		while (<EFILE>) {
			chomp;
			my ($tmpexceptionid,$tmpitemid,@fields) = split /\|\|\|/, $_;
			if (($exceptionid == $tmpexceptionid) && ($itemid == $tmpitemid)) {
				$item{'modified'}      = $fields[0]  || "";
				$item{'user'}          = $fields[1]  || "";
				$item{'description'}   = $fields[2]  || "";
				$item{'exceptiondate'} = $fields[3];
				$item{'starttime'}     = $fields[4]  || "";
				$item{'endtime'}       = $fields[5]  || "";
				$item{'visibility'}    = $fields[6];
				$item{'hyperlink'}     = $fields[7] || "";
				$item{'notes'}         = $fields[8] || "";
				last;
			}
		}
		close EFILE;
	}

	return \%item;
}

#
# Get a hash of calendar names which the current calendar has access to import.
# Also, get the email address and auto_add_event information since we are already
# looking at the preferences file.
#
sub get_import_permissions {
	my ($calendar) = (@_);
	my (@dirs,$dir,$type,@export_list,$type_file,$preferences_file,%import,$email,$auto_add_event);
	
	# Loop through all the calendars.  Open and Public calendars get added right away.  Then,
	# if the calendar export permissions of a Private calendar list this calendar, add it as well.
	opendir DIR, $::db_dir or webcalng_subs::hard_error("Could not read $::db_dir: $!");
	@dirs = readdir DIR;
	closedir DIR;
	for $dir (@dirs) {
		# Setup file paths.
		$type_file        = $::db_dir . "/" . $dir . "/type";
		$preferences_file = $::db_dir . "/" . $dir . "/preferences";

		# If the type file is not there, continue.
		next unless (-f "$type_file");

		# Look at the type file first.  The open and public types are automatically fair game.
		$type = "";
		open FILE, $type_file or webcalng_subs::hard_error("Could not open $type_file: $!");	
		chomp($type = <FILE>);
		close FILE;	
		$import{$dir}{'import'}++ if (($type eq "open") || ($type eq "public"));

		# If the prefernces file exists, read it for export and meeting information.
		next unless (-f "$preferences_file");
		@export_list    = ();
		$email          = 0;
		$auto_add_event = 0;
		open FILE, $preferences_file or webcalng_subs::hard_error("Could not open $preferences_file: $!");	
		flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $preferences_file: $!\n");
		while (<FILE>) {
			chomp;
			if (/^export_calendars=(.+)/) {
				my $list     = $1;
				@export_list = split /,/, $list;
			}
			$email          = $1 if /^email=(.+)/;
			$auto_add_event = 1 if /^auto_add_event=1/;
		}
		close FILE;
		if (($type eq "open") || ($type eq "public")) {
			$import{$dir}{'auto'}  = $auto_add_event;
			$import{$dir}{'email'} = $email;
		} else {
			for (@export_list) {
				if ($_ eq $calendar) {
					$import{$dir}{'import'}++;
					$import{$dir}{'auto'}  = $auto_add_event;
					$import{$dir}{'email'} = $email;
					last;
				}
			}
		}
	}

	return \%import;
}

#
# Get information about all attendees of a meeting.
#
sub get_meeting_data {
	my ($meetingid,$calendar) = (@_);
	my ($meeting_file,%data);
	
	$meeting_file = $::db_dir . "/" . $calendar . "/meetings";
	open FILE, "<$meeting_file" or webcalng_subs::hard_error("Could not open $meeting_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $meeting_file: $!\n");
	while (<FILE>) {
		chomp;
		my ($tmpmeetingid,$id,@fields) = (split /\|\|\|/, $_);
		if ($tmpmeetingid eq $meetingid) {
			$data{$id}{'password'} = $fields[0] || "";
			$data{$id}{'response'} = $fields[1] || 0;
			$data{$id}{'note'}     = $fields[2] || "";
		}
	}
	close FILE;

	return \%data;
}

#
# This subroutine just determines what itemid a given meetingid is
# associated with.
#
sub get_itemid_for_meetingid {
	my ($meetingid,$calendar) = (@_);
	my ($itemid,$items_file);

	$itemid     = 0;
	$items_file = $::db_dir . "/" . $calendar . "/items";
	open FILE, "<$items_file" or webcalng_subs::hard_error("Could not open $items_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
	while (<FILE>) {
		chomp;
		my ($tmpitemid,$tmpmeetingid) = (split /\|\|\|/, $_)[0,16];
		$itemid = $tmpitemid, last if ($tmpmeetingid eq $meetingid);
	}
	close FILE;

	return $itemid;
}

#
# Subroutine to create a meeting in the meetings table.
#
sub create_meeting_io {
	my ($dataref) = (@_);
	my ($meeting_file,$meetingid,$id,$line);

	$meeting_file = $::db_dir . "/" . $::calendar . "/meetings";
	open FILE, ">>$meeting_file" or webcalng_subs::hard_error("Could not write $meeting_file: $!\n");
	for $meetingid (keys %$dataref) {
		for $id (keys %{ $dataref->{$meetingid} }) {
			$dataref->{$meetingid}{$id}{'response'} =~ s/\|//g;
			$dataref->{$meetingid}{$id}{'note'}     =~ s/^\|/ |/;
			$dataref->{$meetingid}{$id}{'note'}     =~ s/\|$/| /;
			$dataref->{$meetingid}{$id}{'note'}     =~ s/\|\|\|//g;
			$line = join '|||',$meetingid,$id,$dataref->{$meetingid}{$id}{'password'},$dataref->{$meetingid}{$id}{'response'},$dataref->{$meetingid}{$id}{'note'};
			print FILE "$line\n";
		}
	}
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $meeting_file: $!\n");
	close FILE;

	return 1;
}

#
# Remove a item from a calendar that was added due to a meeting, since the meeting is now cancelled.
#
sub remove_meetingitem {
	my ($calendar,$meetingid) = (@_);
	my ($items_file,$exceptions_file,@data,$itemid);

	$items_file = $::db_dir . "/" . $calendar . "/items";
	open FILE, "+>>$items_file" or webcalng_subs::hard_error("Could not write $items_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $items_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		my ($tmpitemid,$tmpmeetingid) = (split /\|\|\|/, $_)[0,16];
		push (@data,$_)  unless ($tmpmeetingid eq $meetingid); 
		$itemid = $tmpitemid if ($tmpmeetingid eq $meetingid);
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	$exceptions_file = $::db_dir . "/" . $calendar . "/exceptions";
	open EFILE, "+>>$exceptions_file" or webcalng_subs::hard_error("Could not write $exceptions_file: $!\n");
	flock EFILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $exceptions_file: $!\n");
	seek EFILE, 0, 0;
	while (<EFILE>) {
		my ($tmpitemid) = (split /\|\|\|/, $_)[1];
		push (@data,$_) unless ($tmpitemid eq $itemid); 
	}
	seek EFILE, 0, 0;
	truncate EFILE, 0;
	print EFILE @data;
	close EFILE;
	
	return 1;
}

#
# Remove all entries from the meetings table for a given meetingid.
#
sub remove_meeting_io {
	my ($meetingid,$targetid) = (@_);
	my ($meeting_file,@data);

	$meeting_file    = $::db_dir . "/" . $::calendar . "/meetings";
	open FILE, "+>>$meeting_file" or webcalng_subs::hard_error("Could not write $meeting_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $meeting_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		if ($targetid) {
			my ($tmpmeetingid,$tmpid) = (split /\|\|\|/, $_)[0,1];
			push (@data,$_) unless (($tmpmeetingid eq $meetingid) && ($targetid eq $tmpid)); 
		} else {
			my ($tmpmeetingid) = (split /\|\|\|/, $_)[0];
			push (@data,$_) unless ($tmpmeetingid eq $meetingid); 
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# This subroutine marks off a reminder as being sent.  This is only called from
# the webcalng_remind.pl script.
#
sub mark_remindersent {
	my ($calendar,$itemid,$x,$today) = (@_);
	my ($items_file,@data);

	$items_file = $::db_dir . "/" . $calendar . "/items";
	open FILE, "+>>$items_file" or die "Could not open $items_file: $!\n";
	flock FILE, LOCK_EX or die "Could not lock file $items_file: $!\n";
	seek FILE, 0, 0;
	while (<FILE>) {
		my ($tmpitemid) = (split /\|\|\|/, $_)[0];
		if ($itemid == $tmpitemid) {
			my (@fields,@remindersent,$newremindersent,$line);
			@fields           = split /\|\|\|/, $_;
			@remindersent     = split /,/, $fields[15];
			$remindersent[$x] = $today;
			$newremindersent  = join ',', @remindersent;	
			$fields[15]       = $newremindersent;
			$line             = join '|||', @fields;
			push(@data,$line);
		} else {
			push(@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# This clears the remindersent flag for a given reminder, or all of them.
#
sub clear_remindersent {
	my ($calendar,$itemid,$x) = (@_);
	my ($items_file,@data);

	$items_file = $::db_dir . "/" . $calendar . "/items";
	open FILE, "+>>$items_file" or die "Could not open $items_file: $!\n";
	flock FILE, LOCK_EX or die "Could not lock file $items_file: $!\n";
	seek FILE, 0, 0;
	while (<FILE>) {
		my ($tmpitemid) = (split /\|\|\|/, $_)[0];
		if ($itemid == $tmpitemid) {
			my (@fields,@remindersent,$newremindersent,$line);
			@fields           = split /\|\|\|/, $_;
			@remindersent     = split /,/, $fields[15];
			$remindersent[$x] = 0;
			$newremindersent  = join ',', @remindersent;	
			$fields[15]       = $newremindersent;
			$line             = join '|||', @fields;
			push(@data,$line);
		} else {
			push(@data,$_);
		}
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

#
# This subroutine schedules a one time message to be sent when the reminder script runs next.
#
sub schedule_message {
	my ($email,$message) = (@_);
	my ($message_file,$index,$next_index);

	$message_file = $::db_dir . "/messagequeue";
	open FILE, "+>>$message_file" or webcalng_subs::hard_error("Could not write to $message_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $message_file: $!\n");
	seek FILE, 0, 0;
	$next_index = 1;
	while (<FILE>) {
		chomp;
		($index)    = (split /\s+/)[0];
		$next_index = $index if ($index > $next_index);
	}
	$next_index++;
	print FILE "$next_index $email $message\n";
	close FILE;
	
	return 1;
}

#
# This subroutine reads entries from the message queue, and them removes them as they are on their way out.
#
sub read_messagequeue {
	my ($message_file,%messages,$index,$id,$text,$key,$counter,@indexes);

	$counter = 0;
	$message_file = $::db_dir . "/messagequeue";
	open FILE, "<$message_file" or webcalng_subs::hard_error("Could not read $message_file: $!\n");
	flock FILE, LOCK_SH or webcalng_subs::hard_error("Could not lock file $message_file: $!\n");
	seek FILE, 0, 0;
	while (<FILE>) {
		chomp;
		($index,$id,$text) = split /\s+/, $_, 3;
		$key = $counter . ";" . $id;
		$messages{$key} = $text;
		$counter++;
		push(@indexes,$index);
	}
	close FILE;
	
	return (\%messages,@indexes);
}

#
# Remove a set of indexes from the messagequeue.
#
sub empty_messagequeue {
	my (@indexes) = (@_);
	my ($message_file,$index,$tmpindex,@data,$delete);

	$message_file = $::db_dir . "/messagequeue";
	open FILE, "+>>$message_file" or webcalng_subs::hard_error("Could not write to $message_file: $!\n");
	flock FILE, LOCK_EX or webcalng_subs::hard_error("Could not lock file $message_file: $!\n");
	seek FILE, 0, 0;
	@data = ();
	while (<FILE>) {
		($tmpindex) = (split /\s+/, $_)[0];
		$delete     = 0;
		for $index (@indexes) {
			$delete++, last if ($index == $tmpindex);
		}
		push(@data,$_) unless ($delete);
	}
	seek FILE, 0, 0;
	truncate FILE, 0;
	print FILE @data;
	close FILE;

	return 1;
}

1;
