#!/usr/bin/perl -w

#
# The webcalng install script.  You should call this using "perl install.pl".
#

use strict;

#
# Global variables.
#
my (%config,$date,$root,$ftp);

#
# Check command line.  The ftp option is used to prepare the application files
# for transfer over to a different hosting server.
#
$ftp = 0;
$ftp = 1 if (($ARGV[0]) && ($ARGV[0] eq "-ftp"));

#
# Open logfile.
#
open LOG, ">install.log" or die "Error creating file install.log: $!\n";
$date = localtime;
print LOG "Starting install: $date\n";
print_and_log("\n");
print_and_log("Welcome to the webcalng install script!\n\n");

#
# Print warning if doing ftp install.
#
if ($ftp) {
	print_and_log("You have chosen to setup webcalng for a ftp install.\n");
	print_and_log("Answer all questions as if you were on the target web server.\n");
	print_and_log("The webcalng application will be setup in the ftp directory.\n\n");
}

# 
# Call subroutines to do the install steps.
#
get_perl_path_and_ostype();
set_defaults();
read_cache();
see_if_they_are_root();
get_config();
check_perl_modules();
continue_install();
install_files();
install_cron();
install_service();
setup_database();
set_permissions();
install_complete();
save_cache();

#
# Close logfile.
#
print_and_log("\n");
close LOG;

# Find out what the path to Perl is on this server.
#
sub get_perl_path_and_ostype {
	my ($perlv,$typeguess);

	# Get the path to perl.
	$config{'perl'} = "/usr/bin/perl";
	$config{'perl'} = "C:\\perl\\bin\\perl" if (($^O =~ /win/i) && ($^O !~ /darwin/i));
	$config{'perl'} = ask("Enter the path to Perl.", $config{'perl'}, undef);

	# Make sure they gave a correct path to perl and it works.
	$perlv=`$config{'perl'} -v`;	
	log_and_die("Error running $config{'perl'}.  Check the path to perl and try again.\n") if ($?);

	# Find out the OS type we are running on.  Mac OS X reports as darwin, which
	# is closer to unix than windows for the install defaults.
	if (($perlv =~ /win/mi) && ($perlv !~ /darwin/i)) {
		$config{'ostype'} = "win";
	} else {
		$config{'ostype'} = "unix";
	}

	return 1;
}

#
# Setup config defaults.
#
sub set_defaults {
	if ($config{'ostype'} eq "unix") {
		$config{'httpd_user'}      = "apache";
		$config{'http_host'}       = "";
		$config{'install'}         = "/var/www/cgi-bin/webcalng";
		$config{'url'}             = "/cgi-bin/webcalng/webcalng.pl";
		$config{'auth_type'}       = "session";
		$config{'mailhost'}        = "localhost";
		$config{'db_dir'}          = "/var/webcalng";
		$config{'db_type'}         = "flat";
		$config{'dbd'}             = "";
		$config{'db'}              = "webcalng";
		$config{'db_user'}         = "webcalng";
		$config{'db_pass'}         = "webcalng";
		$config{'db_host'}         = "localhost";
		$config{'db_table_prefix'} = "webcalng";
		$config{'oracle_home'}     = "";
		$config{'tnsname'}         = "";
		$config{'dsn'}             = "";
		$config{'check_modules'}   = "CGI,Fcntl,HTML::Template,Time::Local,Net::SMTP";
		$config{'setup_cron'}      = "yes";
		$config{'service'}         = "";
	} else {
		$config{'httpd_user'}      = "";
		$config{'http_host'}       = "";
		$config{'install'}         = "C:\\inetpub\\scripts\\webcalng";
		$config{'url'}             = "/scripts/webcalng/webcalng.pl";
		$config{'auth_type'}       = "session";
		$config{'mailhost'}        = "localhost";
		$config{'db_dir'}          = "C:\\webcalng";
		$config{'db_type'}         = "flat";
		$config{'dbd'}             = "";
		$config{'db'}              = "webcalng";
		$config{'db_user'}         = "webcalng";
		$config{'db_pass'}         = "webcalng";
		$config{'db_host'}         = "";
		$config{'db_table_prefix'} = "webcalng";
		$config{'oracle_home'}     = "";
		$config{'tnsname'}         = "";
		$config{'dsn'}             = "";
		$config{'check_modules'}   = "CGI,Fcntl,HTML::Template,Time::Local,Net::SMTP";
		$config{'setup_cron'}      = "";
		$config{'service'}         = "yes";
	}
}

#
# Read in the install cache if it exists.
#
sub read_cache {
	if (-f "install.cache") {
		open OLDCACHE, "<install.cache" or log_and_die("Error reading cache: $!\n");
		while (<OLDCACHE>) {
			chomp;
			my ($key,$value) = split /=/, $_;
			next if ($key eq "check_modules");
			$config{$key} = $value;
		}
		close OLDCACHE;
	}

	return 1;
}

#
# See if they are installing as root.
#
sub see_if_they_are_root {
	my (@continue,$continue,$ans);

	return unless (($config{'ostype'} eq "unix") && (! $ftp));
	@continue = qw(yes no);
	$continue = "yes,no";
	if ($< == 0) {
		$root = 1;
	} else {
		$root = 0;
		$ans = ask("You are not running the install program as root!  Continue? ($continue)","no",@continue);
		if ($ans eq "yes") {
			print_and_log("Permissions will NOT be set securely since you are not root!\n\n");
		} else {
			log_and_die("Exiting install program by user request.\n");
		}
	}

	return 1;
}

#
# Get the configuration choices from the user.
#
sub get_config {
	my (@dbtypes,$dbtypes,@authtypes,$authtypes,@cron,$cron);

	# Get general server information.
	if (($config{'ostype'} eq "unix") && ($root)) {
		$config{'httpd_user'} = ask("Enter the uid or username that your webserver is runnnig as.", $config{'httpd_user'},undef);
	} else {
		$config{'httpd_user'} = "";
	}
	$config{'http_host'}      = ask("Enter the fully qualified hostname.domainname of your webserver.", $config{'http_host'},undef);
	$config{'install'}        = ask("Enter the full path to where you want webcalng installed.", $config{'install'},undef);
	$config{'install'}        =~ s/\\/\//g;  # In case they enter windows style paths.
	$config{'url'}            = ask("Enter the URL that clients will use to access webcalng, not including\nthe webserver hostname.", $config{'url'},undef);
	$config{'private_url'}    = $config{'url'};
	$config{'private_url'}    =~ s/\/webcalng.pl//;
	$config{'private_url'}   .= "/private/webcalng.pl";
	if ($config{'ostype'} eq "unix") {
		@authtypes                = qw (session htaccess);
		$authtypes                = "session,htaccess";
		$config{'auth_type'}      = ask("Enter the type of authentication you would like to use ($authtypes).", $config{'auth_type'},@authtypes);
	} else {
		# We only support session based authentication from windows servers for now.
		$config{'auth_type'} = "session";
	}
	$config{'check_modules'} .= ",Digest::MD5" if ($config{'auth_type'} eq "session");
	$config{'mailhost'}       = ask("Enter the hostname of a mailhost that the reminder script can use to relay\nemail through.", $config{'mailhost'},undef);
	if (! $ftp) {
		@cron = qw(yes no);
		$cron = "yes,no";
		if ($config{'ostype'} eq "unix") {
			$config{'setup_cron'} = ask("Setup the cron job needed to use webcalng reminders and notices ($cron).", $config{'setup_cron'},@cron);
			$config{'service'}    = "";
		} else {
			$config{'service'}    = ask("Setup the Windows Service for webcalng reminders and notices ($cron).", $config{'service'},@cron);
			$config{'setup_cron'} = "";
		}
	}

	# Get database related information.
	if ($config{'ostype'} eq "unix") {
		@dbtypes = qw(flat mysql postgresql oracle);
		$dbtypes = "flat,mysql,postgresql,oracle";
	} else {
		@dbtypes = qw(flat mssql oracle);
		$dbtypes = "flat,mssql,oracle";
	}
	$config{'db_type'} = ask("Enter the database type that you will use for webcalng storage\n($dbtypes).", $config{'db_type'},@dbtypes);
	if ($config{'db_type'} eq "flat") {

		# Set configuration for flat file database.
		if ($ftp) {
			print_and_log("Setting db_dir to $config{'install'}/data due to ftp install.\n");
			$config{'db_dir'} = $config{'install'} . "/data";
		} else {
			$config{'db_dir'} = ask("Enter the path to a directory that can be used for webcalng storage.\nThis directory should not be inside any directory that could be\naccessed by your webserver.", $config{'db_dir'},undef);
			$config{'db_dir'} =~ s/\\/\//g;  # In case they enter windows style paths.
		}
		$config{'db'} = $config{'db_user'} = $config{'db_pass'} = $config{'db_host'} = $config{'db_table_prefix'} = "";

	} else {

		# Get more configuration information for relational database storage.
		if ($ftp) {
			print_and_log("Setting db_dir to $config{'install'}/data due to ftp install.\n\n");
			$config{'db_dir'} = $config{'install'} . "/data";
		} else {
			$config{'db_dir'}          = ask("Enter the path to a directory that can be used for webcalng configuration files.\nThis directory should not be inside any directory that could be accessed by your webserver.", $config{'db_dir'}, undef);
			$config{'db_dir'}          =~ s/\\/\//g;  # In case they enter windows style paths.
		}
		$config{'db_user'}         = ask("Enter the name of the database user for webcalng connections.", $config{'db_user'}, undef);
		$config{'db_pass'}         = ask("Enter the password for the database user for webcalng connections.", $config{'db_pass'}, undef);
		$config{'db_table_prefix'} = ask("Enter a prefix to use for webcalng tables in the database.", $config{'db_table_prefix'}, undef);
		$config{'check_modules'} .= ",DBI";

		# Database specific section.
		if ($config{'ostype'} eq "unix") {

			if ($config{'db_type'} eq "mysql") {
				$config{'db'}             = ask("Enter the name of the database to use for webcalng.", $config{'db'}, undef);
				$config{'db_host'}        = ask("Enter the host which your database runs on.", $config{'db_host'}, undef);
				$config{'dbd'}            = "mysql";
				$config{'check_modules'} .= ",DBD::mysql";
			}

 			if ($config{'db_type'} eq "postgresql") {
				$config{'db'}             = ask("Enter the name of the database to use for webcalng.", $config{'db'}, undef);
				$config{'db_host'}        = ask("Enter the host which your database runs on.", $config{'db_host'}, undef);
				$config{'dbd'}            = "Pg";
				$config{'check_modules'} .= ",DBD::Pg";
			}

			if ($config{'db_type'} eq "oracle") {
				$config{'oracle_home'}    = ask("Enter the path to your ORACLE_HOME directory.", $config{'oracle_home'}, undef);
				$config{'tnsname'}        = ask("Enter the TNSNAME to use for the connection to the Oracle database.", $config{'tnsname'}, undef);
				$config{'db_host'}        = "";
				$config{'dbd'}            = "Oracle";
				$config{'check_modules'} .= ",DBD::Oracle";
			}

		} else {

			print_and_log("You will use ODBC to connect to your database.\nSee the webcalng administrators guide for instructions on setting\nup the ODBC datasource.\n");
			$config{'dsn'}            = ask("Enter the DSN to use for the connection to the database.", $config{'dsn'}, undef);
			$config{'dbd'}            = "ODBC";
			$config{'check_modules'} .= ",DBD::ODBC";
		
		}

	}

	return 1;
}

#
# This subroutine make sure they have all the required Perl modules installed.
#
sub check_perl_modules {
	my (@required_modules,$module,$result,$dummy,$string,$error,$missing);

	@required_modules = split /,/, $config{'check_modules'};

	# If they are doing a ftp install, do not check the modules in this host
	# that the install script is running under.  Just tell them what they will
	# need on the hosting server.
	if ($ftp) {
		print_and_log("Skipping module check since we are in ftp install mode.\n");
		print_and_log("The following modules will be required in the target web server.\n");
		for $module (@required_modules) {
			print_and_log("    $module\n");
		}
		ask("", "return", undef);
		return 1;
	}

	# If we get here, this is not a ftp install, so see if all the
	# required modules are installed.
	$error   = 0;
	$missing = "";
	print_and_log("\nChecking Perl modules:\n\n");
	for $module (@required_modules) {
		$string = sprintf "%40s", "Checking for $module ... ";
		print_and_log($string);
		$dummy = `$config{'perl'} -e "use $module"`;
		if ($?) {
			$error    = 1;
			$missing .= "$module ";
			print_and_log("ERROR: Module $module not found.\n");
		} else {
			print_and_log("OK.\n");
		}
	}	
	print_and_log("\n");

	# Exit the install script if any modules were missing.
	if ($error) {
		print_and_log("\n");
		print_and_log("ERROR: Missing Perl modules: $missing\n");
		if ($config{'ostype'} eq "unix") {
			print_and_log("See http://search.cpan.org to download and install any missing Perl modules.\n");
		} else {
			print_and_log("See the following link for information on installing Perl modules:\n");
			print_and_log("    http://aspn.activestate.com/ASPN/Downloads/ActivePerl/PPM/\n");
		}
		print_and_log("\n");
		exit 1;
	}

	return 1;
}

#
# See if they want to continue with the install.
#
sub continue_install {
	my (@showkeys,$key,@respond,$continue,$string);

	print_and_log("\nReady to install with the following configuration:\n\n");
	@showkeys = qw (httpd_user http_host install url auth_type mailhost setup_cron service db_dir db_type dbd db db_user db_pass db_host db_table_prefix oracle_home tnsname);
	for $key (@showkeys) {
		next unless ($config{$key});
		$string = sprintf "%20s %-50s\n", $key, $config{$key};
		print_and_log($string);
	}
	$string = sprintf "%20s %-50s\n", "ftp_install", $ftp;
	print_and_log($string);
	@respond  = qw (yes no);
	$continue = ask("\nContinue with install using these values (yes,no).","yes",@respond);
	log_and_die("Exiting install script by user request.\n") if ($continue eq "no");

	return 1;
}

#
# Install the webcalng files.
#
sub install_files {
	my ($path,@files,$file,@dirs,$dir);

	print_and_log("\nCreating directories and installing files\n\n");

	# Create directories as needed.
	if ($ftp) {
		mkdir "ftp", 0755 or log_and_die("Error creating ftp directory\n") if (! -d "ftp");
		open FILE, ">ftp/README" or log_and_die("Error writing ftp/README: $!\n");
		print FILE "Everything in this directory should be installed in:\n\n";
		print FILE "$config{'install'}\n\n";
		print FILE "on your webserver $config{'http_host'}\n";
		close FILE;
	} else {
		$path = $config{'install'};	
		$path =~ s/(.*\/).*/$1/;
		$path =~ s/\/$// unless ($path eq "/");
		log_and_die("ERROR: The directory $path must exist.\n") if (! -d $path);
	}
	if ((! $ftp) && (! -d $config{'install'})) {
		print_and_log("    $config{'install'}\n");
		mkdir $config{'install'}, 0755 or log_and_die("Error creating $config{'install'}\n");
	}
	if ($ftp) {
		if (! -d "ftp/language") {
			print_and_log("    $config{'install'}/language\n");
			mkdir "ftp/language", 0755 or log_and_die("Error creating ftp/language\n");
		}
	} else {
		if (! -d "$config{'install'}/language") {
			print_and_log("    $config{'install'}/language\n");
			mkdir "$config{'install'}/language", 0755 or log_and_die("Error creating $config{'install'}/language\n");
		}
	}
	if ($ftp) {
		if (! -d "ftp/themes") {
			print_and_log("    $config{'install'}/themes\n");
			mkdir "ftp/themes", 0755 or log_and_die("Error creating ftp/themes\n");
		}
	} else {
		if (! -d "$config{'install'}/themes") {
			print_and_log("    $config{'install'}/themes\n");
			mkdir "$config{'install'}/themes", 0755 or log_and_die("Error creating $config{'install'}/themes\n");
		}
	}
	if ($ftp) {
		if (! -d "ftp/themes/webcalng.Basic") {
			print_and_log("    $config{'install'}/themes/webcalng.Basic\n");
			mkdir "ftp/themes/webcalng.Basic", 0755 or log_and_die("Error creating ftp/themes/webcalng.Basic\n");
		}
		if (! -d "ftp/themes/webcalng.NoBorders") {
			print_and_log("    $config{'install'}/themes/webcalng.NoBorders\n");
			mkdir "ftp/themes/webcalng.NoBorders", 0755 or log_and_die("Error creating ftp/themes/webcalng.NoBorders\n");
		}
		if (! -d "ftp/themes/webcalng.BigBorders") {
			print_and_log("    $config{'install'}/themes/webcalng.BigBorders\n");
			mkdir "ftp/themes/webcalng.BigBorders", 0755 or log_and_die("Error creating ftp/themes/webcalng.BigBorders\n");
		}
	} else {
		if (! -d "$config{'install'}/themes/webcalng.Basic") {
			print_and_log("    $config{'install'}/themes/webcalng.Basic\n");
			mkdir "$config{'install'}/themes/webcalng.Basic", 0755 or log_and_die("Error creating $config{'install'}/themes/webcalng.Basic\n");
		}
		if (! -d "$config{'install'}/themes/webcalng.NoBorders") {
			print_and_log("    $config{'install'}/themes/webcalng.NoBorders\n");
			mkdir "$config{'install'}/themes/webcalng.NoBorders", 0755 or log_and_die("Error creating $config{'install'}/themes/webcalng.NoBorders\n");
		}
		if (! -d "$config{'install'}/themes/webcalng.BigBorders") {
			print_and_log("    $config{'install'}/themes/webcalng.BigBorders\n");
			mkdir "$config{'install'}/themes/webcalng.BigBorders", 0755 or log_and_die("Error creating $config{'install'}/themes/webcalng.BigBorders\n");
		}
	}
	if ($config{'auth_type'} eq "session") {
		$config{'use_cookies'} = 1;
	} else {
		$config{'use_cookies'} = 0;
	}
	if (! $config{'use_cookies'}) {
		if ($ftp) {
			if (! -d "ftp/private") {
				print_and_log("    $config{'install'}/private\n");
				mkdir "ftp/private", 0755 or log_and_die("Error creating ftp/private\n");
			}
		} else {
			if (! -d "$config{'install'}/private") {
				print_and_log("    $config{'install'}/private\n");
				mkdir "$config{'install'}/private", 0755 or log_and_die("Error creating $config{'install'}/private\n");
			}
		}
	}
	if ($ftp) {
		if (! -d "ftp/data") {
			print_and_log("    $config{'install'}/db_dir\n");
			mkdir "ftp/data", 0755 or log_and_die("Error creating ftp/data\n");
		}
	} else {
		$path = $config{'db_dir'};	
		$path =~ s/(.*\/).*/$1/;
		$path =~ s/\/$// unless ($path eq "/");
		if (! -d $path) {
			log_and_die("ERROR: The directory $path must exist.\n");
		}
		if (! -d $config{'db_dir'}) {
			print_and_log("    $config{'install'}/db_dir\n");
			mkdir $config{'db_dir'}, 0755 or log_and_die("Error creating $config{'db_dir'}\n");
		}
	}
	if (! $ftp) {
		if (! -d "util") {
			print_and_log("   util\n");
			mkdir "util", 0755 or log_and_die("Error creating util\n");
		}
	}

	# Copy in the webcalng source code.
	@files = qw(webcalng.pl webcalng_remind.pl webcalng_subs.pm);
	for $file (@files) {
		print_and_log("    $config{'install'}/$file\n");
		open READ, "<src/webcalng/$file" or log_and_die("Error reading $file: $!\n");
		if ($ftp) {
			open WRITE, ">ftp/$file" or log_and_die("Error writing $file: $!\n");
		} else {
			open WRITE, ">$config{'install'}/$file" or log_and_die("Error writing $file: $!\n");
		}
		while (<READ>) {
			s/__perl__/#!$config{'perl'}/;
			s/__install__/$config{'install'}/;
			s/__db_dir__/$config{'db_dir'}/;
			# Do not set the PATH variable for windows if using an odbc driver.
			next if ((/ENV\{'PATH'\}/) && ($config{'ostype'} ne "unix") && ($config{'db_type'} ne "flat"));
			print WRITE;
		}
		close WRITE;
		close READ;
	}
	if ($config{'use_cookies'}) {
		$dir  = "session";
	} else {
		$dir  = "htaccess";
	}
	$file = "webcalng_auth.pm";
	print_and_log("    $config{'install'}/$file\n");
	open READ, "<src/$dir/$file" or log_and_die("Error reading $file: $!\n");
	if ($ftp) {
		open WRITE, ">ftp/$file" or log_and_die("Error writing $file: $!\n");
	} else {
		open WRITE, ">$config{'install'}/$file" or log_and_die("Error writing $file: $!\n");
	}
	while (<READ>) {
		print WRITE;
	}
	close WRITE;
	close READ;
	if ($config{'db_type'} eq "flat") {
		@files = qw(webcalng_io.pm webcalng_auth_io.pm);
		for $file (@files) {
			print_and_log("    $config{'install'}/$file\n");
			open READ, "<src/flat/$file" or log_and_die("Error reading $file: $!\n");
			if ($ftp) {
				open WRITE, ">ftp/$file" or log_and_die("Error writing $file: $!\n");
			} else {
				open WRITE, ">$config{'install'}/$file" or log_and_die("Error writing $file: $!\n");
			}
			while (<READ>) {
				print WRITE;
			}
			close WRITE;
			close READ;
		}
	} else {
		@files = qw(webcalng_io.pm webcalng_auth_io.pm webcalng_rdbms.pm);
		for $file (@files) {
			print_and_log("    $config{'install'}/$file\n");
			open READ, "<src/rdbms/$file" or log_and_die("Error reading $file: $!\n");
			if ($ftp) {
				open WRITE, ">ftp/$file" or log_and_die("Error writing $file: $!\n");
			} else {
				open WRITE, ">$config{'install'}/$file" or log_and_die("Error writing $file: $!\n");
			}
			while (<READ>) {
				print WRITE;
			}
			close WRITE;
			close READ;
		}
	}
	if (! $config{'use_cookies'}) {
		print_and_log("    $config{'install'}/private/.htaccess\n");
		open READ, "<src/htaccess/.htaccess" or log_and_die("Error reading .htaccess: $!\n");
		if ($ftp) {
			open WRITE, ">ftp/private/.htaccess" or log_and_die("Error writing .htaccess: $!\n");
		} else {
			open WRITE, ">$config{'install'}/private/.htaccess" or log_and_die("Error writing .htaccess: $!\n");
		}
		while (<READ>) {
			s/__db_dir__/$config{'db_dir'}/;
			print WRITE;
		}
		close WRITE;
		close READ;
		print_and_log("    $config{'install'}/private/webcalng.pl\n");
		open READ, "<src/webcalng/webcalng.pl" or log_and_die("Error reading webcalng.pl: $!\n");
		if ($ftp) {
			open WRITE, ">ftp/private/webcalng.pl" or log_and_die("Error writing private/webcalng.pl: $!\n");
		} else {
			open WRITE, ">$config{'install'}/private/webcalng.pl" or log_and_die("Error writing private/webcalng.pl: $!\n");
		}
		while (<READ>) {
			s/__perl__/#!$config{'perl'}/;
			s/__install__/$config{'install'}/;
			s/__db_dir__/$config{'db_dir'}/;
			print WRITE;
		}
		close WRITE;
		close READ;
	}
	
	# Copy in the language files and themes files.
	@files = qw(webcalng.english webcalng.spanish webcalng.hungarian webcalng.german webcalng.swedish webcalng.danish);
	for $file (@files) {
		print_and_log("    $config{'install'}/language/$file\n");
		open READ, "<src/language/$file" or log_and_die("Error reading $file: $!\n");
		if ($ftp) {
			open WRITE, ">ftp/language/$file" or log_and_die("Error writing $file: $!\n");
		} else {
			open WRITE, ">$config{'install'}/language/$file" or log_and_die("Error writing $file: $!\n");
		}
		while (<READ>) {
			print WRITE;
		}
		close WRITE;
		close READ;
	}
	@dirs = qw(webcalng.Basic webcalng.NoBorders webcalng.BigBorders);
	for $dir (@dirs) {
		opendir DIR, "src/themes/$dir" or log_and_die("Error reading directory $dir: $!\n");
		@files = readdir DIR or log_and_die("Error reading files in $dir: $!\n");
		closedir DIR;
		for $file (@files) {
			next unless (($file =~ /\.tmpl/) || ($file =~ /\.html/));
			print_and_log("    $config{'install'}/themes/$dir/$file\n");
			open READ, "<src/themes/$dir/$file" or log_and_die("Error reading $dir/$file: $!\n");
			if ($ftp) {
				open WRITE, ">ftp/themes/$dir/$file" or log_and_die("Error writing $dir/$file: $!\n");
			} else {
				open WRITE, ">$config{'install'}/themes/$dir/$file" or log_and_die("Error writing $dir/$file: $!\n");
			}
			while (<READ>) {
				print WRITE;
			}
			close WRITE;
			close READ;
		}
	}

	# Copy in things that belong in the db_dir.
	if (-f "$config{'db_dir'}/webcalng.conf") {
		# Make a backup of the webcalng.conf file if already there.
		open BACKUPR, "<$config{'db_dir'}/webcalng.conf" or log_and_die("Error reading webcalng.conf for backup: $!");
		my $time = time;
		my $backupfile = $config{'db_dir'} . "/webcalng.conf." . $time;
		open BACKUPW, ">$backupfile" or log_and_die("Error writing webcalng.conf backup file: $!");
		while(<BACKUPR>) {
			print BACKUPW;
		}
		close BACKUPR;
		close BACKUPW;
	}
	print_and_log("    $config{'db_dir'}/webcalng.conf\n");
	open READ, "<src/conf/webcalng.conf" or log_and_die("Error reading webcalng.conf: $!\n");
	if ($ftp) {
		open WRITE, ">ftp/data/webcalng.conf" or log_and_die("Error writing webcalng.conf: $!\n");
	} else {
		open WRITE, ">$config{'db_dir'}/webcalng.conf" or log_and_die("Error writing webcalng.conf: $!\n");
	}
	while (<READ>) {
		next if ((/PRIVATE/) && ($config{'use_cookies'}));
		s/__use_cookies__/$config{'use_cookies'}/;
		s/__db_dir__/$config{'db_dir'}/;
		s/__db_type__/$config{'db_type'}/;
		s/__db__/$config{'db'}/;
		s/__dbd__/$config{'dbd'}/;
		s/__db_user__/$config{'db_user'}/;
		s/__db_pass__/$config{'db_pass'}/;
		s/__db_host__/$config{'db_host'}/;
		s/__db_table_prefix__/$config{'db_table_prefix'}/;
		s/__oracle_home__/$config{'oracle_home'}/;
		s/__tnsname__/$config{'tnsname'}/;
		s/__dsn__/$config{'dsn'}/;
		s/__url__/$config{'url'}/;
		s/__private_url__/$config{'private_url'}/;
		s/__http_host__/$config{'http_host'}/;
		s/__mailhost__/$config{'mailhost'}/;
		print WRITE;
	}
	close WRITE;
	close READ;
	if ($config{'db_type'} eq "flat") {
		@files = qw(preferences passwords);
		# Create an empty sessions file if one does not exist.
		if (! -f "$config{'db_dir'}/sessions") {
			print_and_log("    $config{'db_dir'}/sessions\n");
			if ($ftp) {
				open WRITE, ">ftp/data/sessions" or log_and_die("Error writing sessions: $!\n");
			} else {
				open WRITE, ">$config{'db_dir'}/sessions" or log_and_die("Error writing sessions: $!\n");
			}
			close WRITE;
		}
		# Create an empty messagequeue file if one does not exist.
		if (! -f "$config{'db_dir'}/messagequeue") {
			print_and_log("    $config{'db_dir'}/messagequeue\n");
			if ($ftp) {
				open WRITE, ">ftp/data/messagequeue" or log_and_die("Error writing messagequeue: $!\n");
			} else {
				open WRITE, ">$config{'db_dir'}/messagequeue" or log_and_die("Error writing messagequeue: $!\n");
			}
			close WRITE;
		}
	} else {
		if (! $config{'use_cookies'}) {
			@files = qw (passwords);
		} else {
			@files = ();
		}
	}
	for $file (@files) {
		next if (-f "$config{'db_dir'}/$file");
		print_and_log("    $config{'db_dir'}/$file\n");
		open READ, "<src/flat/$file" or log_and_die("Error reading $file: $!\n");
		if ($ftp) {
			open WRITE, ">ftp/data/$file" or log_and_die("Error writing $file: $!\n");
		} else {
			open WRITE, ">$config{'db_dir'}/$file" or log_and_die("Error writing $file: $!\n");
		}
		while (<READ>) {
			print WRITE;
		}
		close WRITE;
		close READ;
	}

	# Create scripts in the util directory.
	if (! $ftp) {
		$file = "webcal3_to_webcalng.pl";
		print_and_log("    util/$file\n");
		open READ, "<src/util/$file" or log_and_die("Error reading $file: $!\n");
		open WRITE, ">util/$file" or log_and_die("Error writing $file: $!\n");
		while (<READ>) {
			s/__perl__/#!$config{'perl'}/;
			s/__install__/$config{'install'}/;
			s/__db_dir__/$config{'db_dir'}/;
			print WRITE;
		}
		close WRITE;
		close READ;
	}

	print_and_log("\nFile installation done.\n\n");

	return 1;
}

#
# Install a cron job for webcalng_remind.pl if they asked for it.
#
sub install_cron {
	my ($crontabl,$found,@continue,$continue,$addcron);

	if (($config{'setup_cron'} eq "yes") && ($config{'ostype'} eq "unix") && (! $ftp)) {
		$found = 0;
		print_and_log("\nSetting up cron job for webcalng_remind.pl\n\n");
		if (! -d "crontab") {
			mkdir "crontab", 0755 or log_and_die("Error creating crontab directory: $!\n");
		}
		system("crontab -l > crontab/crontab.prewebcalngsetup");
		open FILE, "<crontab/crontab.prewebcalngsetup" or log_and_die("Error reading crontab file: $!\n");
		open NEW, ">crontab/crontab.webcalngsetupnew" or log_and_die("Error writing new crontab file: $!\n");
		while (<FILE>) {
			$found = 1 if /webcalng_remind.pl/;
			print NEW;
		}
		close FILE;
		print NEW "#\n# Send webcalng notices.  Added by webcalng install $date\n#\n";
		print NEW "0,5,10,15,20,25,30,35,40,45,50,55 * * * * $config{'install'}/webcalng_remind.pl >> $config{'db_dir'}/webcalng_remind.log 2>&1\n";
		close NEW;
		if ($found) {
			@continue = qw(yes no);
			$continue = "yes,no";
			$addcron  = "no";
			$addcron  = ask("It appears that the webcalng reminder script is already scheduled in cron.\nInstall another entry ($continue)","no",@continue);
			return 1 if ($addcron eq "no");
		} 
		system("crontab crontab/crontab.webcalngsetupnew");
		print_and_log("\nCron setup complete.\n\n");
	}

	return 1;
}

#
# Install reminder script as windows service.
#
sub install_service {
	my ($instsrv,$srvany,$perl,$path,$regedit,$return);

	if (($config{'service'} eq "yes") && ($config{'ostype'} ne "unix") && (! $ftp)) {
		# Setup variables.
		$instsrv = "C:\\WINNT\\instsrv.exe";
		$srvany  = "C:\\WINNT\\srvany.exe";
		$perl    = $config{'perl'};
		$perl    =~ s/\//\\/g;
		$perl    =~ s/\\/\\\\/g;
		$path    = $config{'install'};
		$path    =~ s/\//\\/g;
		$path    =~ s/\\/\\\\/g;
		$path   .= "\\\\webcalng_remind.pl";

		print_and_log("\nSetting up Windows Service for webcalng_remind.pl\n\n");
		
		# Find instsrv and srvany and install the service.
		$instsrv = ask("Enter the path to instsrv.exe",$instsrv,undef);
		$instsrv =~ s/\//\\/g;
		if (! -f "$instsrv") {
			print_and_log("\nError: Could not find instsrv.exe at $instsrv.\n\n");
			return 1;
		}
		$srvany = ask("Enter the path to srvany.exe",$srvany,undef);	
		$srvany =~ s/\//\\/g;
		if (! -f "$srvany") {
			print_and_log("\nError: Could not find srvany.exe at $srvany.\n\n");
			return 1;
		}
		print_and_log("Installing webcalng_reminders service.\n");
		$return = `$instsrv webcalng_reminders $srvany`;
		if ($return !~ /success/i) {
			print_and_log("\nError installing service: $return\n");
			return 1;
		}

		# Make registry changes.
		if (! -d "service") {
			mkdir "service", 0755 or log_and_die("Error creating service directory: $!\n");
		}
		open FILE, ">service/regedit.reg" or log_and_die("Could not open regedit.reg for writing: $!\n");
		print FILE "REGEDIT4\n\n";
		print FILE "    [HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\webcalng_reminders\\Parameters]\n";
		print FILE "    \"Application\"=\"$perl\"\n";
		print FILE "    \"AppParameters\"=\"$path -loop\"\n";
		close FILE;
		print_and_log("Making registry changes for webcalng_remind servce.\n");
		system ("regedit /s service\\regedit.reg");
		if ($?) {
			print_and_log("\nError setting up registry keys!\n");
			print_and_log("\nSee the webcalng administrators guide on setting up the registry keys manually.\n");
		} else {
			print_and_log("\nWindows Service setup complete.\nStart the webcalng_reminders service when installation is complete.\n\n");
		}

	}

	return 1;
}

#
# Setup the database with initial information, if needed.
#
sub setup_database {
	my (@continue,$continue,$sql,$create);

	# See if they even want to configure the database.
	return 1 if ($config{'db_type'} eq "flat");
	@continue = qw (yes no);
	$continue = "yes,no";
	$sql = ask("Create SQL scripts used for webcalng database installation ($continue).","yes",@continue);
	if ($sql eq "yes") {
		$sql = 1;
	} else {
		$sql = 0;
	}
	return 1 unless ($sql);
	if (($config{'ostype'} eq "unix") && (! $ftp)) {
		$create = ask("After creating SQL scripts, run them to configure database ($continue).","yes",@continue);
		if ($create eq "yes") {
			$create = 1;
		} else {
			$create = 0;
		}
	} else {
		$create = 0;
	}

	# Create the sql scripts, and if needed, run them on the database to setup webcalng.
	if ($config{'db_type'} eq "mysql") {
		print_and_log("Creating mysql scripts.\n");
		open SKEL, "<src/mysql/mysql_setup.skel" or log_and_die("Error reading mysql_setup.skel: $!\n");
		open SQL, ">src/mysql/mysql_setup.sql" or log_and_die("Error writing mysql_setup.sql: $!\n");
		while (<SKEL>) {
			s/__DB_TABLE_PREFIX__/$config{'db_table_prefix'}/g;
			s/__DB_NAME__/$config{'db'}/g;
			s/__DB_USER__/$config{'db_user'}/g;
			s/__DB_HOST__/$config{'db_host'}/g;
			s/__DB_PASS__/$config{'db_pass'}/g;
			print SQL;
		}
		close SQL;
		close SKEL;
		if ($create) {
			print_and_log("Creating mysql database.\n");
			my ($mysqlpath,$mysqluser,$mysqlpass);
			$mysqlpath = `which mysql`;
			chomp $mysqlpath;
			$mysqlpath = "" if ($?);
			$mysqlpath = ask("Enter the path to the mysql client.",$mysqlpath,undef);
			$mysqluser = ask("Enter the root user for mysql.","root",undef);
			$mysqlpass = ask("Enter the password for the mysql root user.","",undef);
			system("$mysqlpath -h $config{'db_host'} -u $mysqluser -p${mysqlpass} < src/mysql/mysql_setup.sql > src/mysql/mysql_setup.log 2>&1");
			print_and_log("The mysql setup appeared to encounter errors!\n") if ($?);
			print_and_log("The mysql setup is complete. See src/mysql/mysql_setup.log for details.\n");
		} else {
			print_and_log("You can use src/mysql/mysql_setup.sql to configure mysql for webcalng.\n");
		}
	} elsif ($config{'db_type'} eq "postgresql") {
		print_and_log("Creating postgresql scripts.\n");
		open SKEL, "<src/postgresql/postgresql_setup.skel" or log_and_die("Error reading postgresql_setup.skel: $!\n");
		open SQL, ">src/postgresql/postgresql_setup.sql" or log_and_die("Error writing postgresql_setup.sql: $!\n");
		while (<SKEL>) {
			s/__DB_TABLE_PREFIX__/$config{'db_table_prefix'}/g;
			s/__DB_NAME__/$config{'db'}/g;
			s/__DB_USER__/$config{'db_user'}/g;
			s/__DB_PASS__/$config{'db_pass'}/g;
			print SQL;
		}
		close SQL;
		close SKEL;
		if (($create) && ($root)) {
			my ($pguser,,$path);
			$path = `pwd`;
			chomp $path;
			$pguser = ask("Enter the name of the postgresql unix account.","postgres",undef);
			print_and_log("Creating postgresql database.  You will be prompted for the postgres\ndatabase account password if one is set.\n");
			system("su -l $pguser -c \"createdb $config{'db'}\" > ${path}/src/postgresql/postgresql_setup.log 2>&1");
			print_and_log("You may be prompted for the postgres database account password again.\n");
			system("su -l $pguser -c \"psql -f ${path}/src/postgresql/postgresql_setup.sql $config{'db'}\" > ${path}/src/postgresql/postgresql_setup.log 2>&1");
			print_and_log("The postgresql setup appeared to encounter errors!\n") if ($?);
			print_and_log("The postgresql setup is complete. See src/postgresql/postgresql_setup.log for details.\n");
		} else {
			print_and_log("You can use src/postgresql/postgresql_setup.sql to configure postgresql for webcalng.\n");
		}
	} elsif ($config{'db_type'} eq "oracle") {
		print_and_log("Creating oracle scripts.\n");
		my ($tablespace,$datafile,$path);
		$tablespace     = ask("Enter the name of a new tablespace to use for webcalng.","webcalng",undef);
		$path           = "/oradata/" . $config{'tnsname'} . "/" . $tablespace . "01.dbf";
		$datafile       = ask("Enter the full path of the datafile to use for this tablespace.",$path,undef);
		open SKEL, "<src/oracle/oracle_setup1.skel" or log_and_die("Error reading oracle_setup1.skel: $!\n");
		open SKEL2, "<src/oracle/oracle_setup2.skel" or log_and_die("Error reading oracle_setup2.skel: $!\n");
		open SQL, ">src/oracle/oracle_setup1.sql" or log_and_die("Error writing oracle_setup1.sql: $!\n");
		open SQL2, ">src/oracle/oracle_setup2.sql" or log_and_die("Error writing oracle_setup2.sql: $!\n");
		while (<SKEL>) {
			s/__DB_TABLE_PREFIX__/$config{'db_table_prefix'}/g;
			s/__DB_NAME__/$config{'db'}/g;
			s/__DB_USER__/$config{'db_user'}/g;
			s/__DB_PASS__/$config{'db_pass'}/g;
			s/__DB_TABLESPACE__/$tablespace/g;
			s/__DB_DATAFILE__/$datafile/g;
			print SQL;
		}
		while (<SKEL2>) {
			s/__DB_TABLE_PREFIX__/$config{'db_table_prefix'}/g;
			print SQL2;
		}
		close SQL;
		close SQL2;
		close SKEL;
		close SKEL2;
		if (($create) && ($root)) {
			my ($orauser,$orasystempass,$path);
			$path = `pwd`;
			chomp $path;
			$orauser       = "oracle";
			$orauser       = ask("Enter the name of the oracle unix account.",$orauser,undef);
			$orasystempass = "manager";
			$orasystempass = ask("Enter the password for the oracle system account.",$orasystempass,undef);
			print_and_log("Setting up tablespace.\n");
			system("su -l $orauser -c \"sqlplus system/${orasystempass}\@$config{'tnsname'} < ${path}/src/oracle/oracle_setup1.sql\" > ${path}/src/oracle/oracle_setup.log 2>&1");
			print_and_log("The oracle setup appeared to encounter errors!\n") if ($?);
			print_and_log("Setting up webcalng in tablespace.\n");
			system("su -l $orauser -c \"sqlplus $config{'db_user'}/$config{'db_pass'}\@$config{'tnsname'} < ${path}/src/oracle/oracle_setup2.sql\" >> ${path}/src/oracle/oracle_setup.log 2>&1");
			print_and_log("The oracle setup appeared to encounter errors!\n") if ($?);
			print_and_log("The oracle setup is complete. See src/oracle/oracle_setup.log for details.\n");
		} else {
			print_and_log("You can use src/oracle/oracle_setup1.sql and src/oracle/oracle_setup2.sql\nto configure oracle for webcalng.\n");
		}
	} elsif ($config{'db_type'} eq "mssql") {
		print_and_log("Creating mssql scripts.\n");
		open SKEL, "<src/mssql/mssql_setup1.skel" or log_and_die("Error reading mssql_setup1.skel: $!\n");
		open SQL, ">src/mssql/mssql_setup1.sql" or log_and_die("Error writing mssql_setup1.sql: $!\n");
		while (<SKEL>) {
			s/__DB_NAME__/$config{'db'}/g;
			s/__DB_USER__/$config{'db_user'}/g;
			s/__DB_PASS__/$config{'db_pass'}/g;
			next if /^#/;
			print SQL;
		}
		close SQL;
		close SKEL;
		open SKEL, "<src/mssql/mssql_setup2.skel" or log_and_die("Error reading mssql_setup2.skel: $!\n");
		open SQL, ">src/mssql/mssql_setup2.sql" or log_and_die("Error writing mssql_setup2.sql: $!\n");
		while (<SKEL>) {
			s/__DB_TABLE_PREFIX__/$config{'db_table_prefix'}/g;
			next if /^#/;
			print SQL;
		}
		close SQL;
		close SKEL;
		print_and_log("You can use src/mssql/mssql_setup1.sql and src/mssql/mssql_setup2.sql\nto configure MS SQL Server for webcalng.\n");
	} else {
		log_and_die("Invalid database type: $config{'db_type'}\n");
	}

	return 1;
}

#
# Set up file permissions as needed.
#
sub set_permissions {
	print_and_log("\nSetting permissions.\n\n");

	# The db_dir should be owned by the webserver owner, and mode 700;
	if ($config{'ostype'} eq "unix") {
		if ((! $config{'db_dir'}) || ($config{'db_dir'} eq "/")) {
			print_and_log("Weird db_dir setting.  Skipping permission settings.\n")
		} elsif (($root) && (! $ftp)) {
			print_and_log("    $config{'db_dir'}\n");
			system("chown -R $config{'httpd_user'} $config{'db_dir'}");
			system("chmod 700 $config{'db_dir'}");
			if (-f "$config{'db_dir'}/passwords") {
				print_and_log("    $config{'db_dir'}/passwords\n");
				system("chmod 600 $config{'db_dir'}/passwords");
			}
			if (-f "$config{'db_dir'}/sessions") {
				print_and_log("    $config{'db_dir'}/sessions\n");
				system("chmod 600 $config{'db_dir'}/sessions");
			}
			if (-f "$config{'db_dir'}/messagequeue") {
				print_and_log("    $config{'db_dir'}/messagequeue\n");
				system("chmod 600 $config{'db_dir'}/messagequeue");
			}
			print_and_log("    $config{'db_dir'}/webcalng.conf\n");
			system("chmod 600 $config{'db_dir'}/webcalng.conf");
			print_and_log("    $config{'install'}/webcalng.pl\n");
			system("chmod 755 $config{'install'}/webcalng.pl");
			if (-f "$config{'install'}/private/webcalng.pl") {
				print_and_log("    $config{'install'}/private/webcalng.pl\n");
				system("chmod 755 $config{'install'}/private/webcalng.pl")
			}
			print_and_log("    $config{'install'}/webcalng_remind.pl\n");
			system("chmod 755 $config{'install'}/webcalng_remind.pl");
			print_and_log("    util/webcal3_to_webcalng.pl\n");
			system("chmod 755 util/webcal3_to_webcalng.pl");
		} else {
			if ($ftp) {
				$config{'db_dir'}  = "ftp/data";
				$config{'install'} = "ftp";
			}
			if ($config{'db_type'} eq "flat") {
				print_and_log("    $config{'db_dir'}\n");
				system("chmod 777 $config{'db_dir'}");
			}
			if (-f "$config{'db_dir'}/passwords") {
				print_and_log("    $config{'db_dir'}/passwords\n");
				system("chmod 666 $config{'db_dir'}/passwords");
			}
			if (-f "$config{'db_dir'}/sessions") {
				print_and_log("    $config{'db_dir'}/sessions\n");
				system("chmod 666 $config{'db_dir'}/sessions");
			}
			if (-f "$config{'db_dir'}/preferences") {
				print_and_log("    $config{'db_dir'}/preferences\n");
				system("chmod 666 $config{'db_dir'}/preferences");
			}
			if (-f "$config{'db_dir'}/messagequeue") {
				print_and_log("    $config{'db_dir'}/messagequeue\n");
				system("chmod 666 $config{'db_dir'}/messagequeue");
			}
			print_and_log("    $config{'db_dir'}/webcalng.conf\n");
			system("chmod 644 $config{'db_dir'}/webcalng.conf");
			print_and_log("    $config{'install'}/webcalng.pl\n");
			system("chmod 755 $config{'install'}/webcalng.pl");
			if (-f "$config{'install'}/private/webcalng.pl") {
				print_and_log("    $config{'install'}/private/webcalng.pl\n");
				system("chmod 755 $config{'install'}/private/webcalng.pl")
			}
			print_and_log("    $config{'install'}/webcalng_remind.pl\n");
			system("chmod 755 $config{'install'}/webcalng_remind.pl");
		}
	} else {
		print_and_log("See the webcalng system administrators guide for instructions on\nsecuring the webcalng data directory.\n");
	}

	print_and_log("\nPermissions setting complete.\n\n");

	return 1;
}

#
# Just display a message saying that the install is complete.
#
sub install_complete {
	print_and_log("Congratulations, the webcalng install is complete!\n");
	print_and_log("A full log of this installation can be found in the install.log file.\n");
	if ($config{'ostype'} ne "unix") {
		if ($config{'db_type'} ne "flat") {
			print_and_log("\nTo configure your database for use with webcalng, see the instructions\nin the webcalng administrators guide.\n\n");
		}
	}
}

#
# Close the cache file.
#
sub save_cache {
	my ($key);

	open NEWCACHE, ">install.cache" or log_and_die("Error creating install.cache: $!\n");
	for $key (keys %config) {
		next if ($key eq "check_modules");
		print NEWCACHE "$key=$config{$key}\n";
	}
	close NEWCACHE;

	return 1;
}

#
#
# Ask a question of the user, and get an answer.  The questions, default answer, and
# optionally a list of valid responses are passed into this subroutine.  If a list of
# valid responses is given, the subroutine will loop until it gets an answer that matches
# one of those valid responses.
#
sub ask {
	my ($q,$ans,@valid) = (@_);
	my ($newans,$valid,$check,$x,$origans);

	# If they pass in an array of valid choice, then we will make sure they give us a valid answer.
	if (! $valid[0]) {
		$valid = 1;
		$check = 0;
	} else {
		$valid = 0;
		$check = 1;
	}

	# Get their reponse.
	$x       = 0;
	$origans = $ans;
	while ((! $x) || (! $valid)) {
		print_and_log("$q\n");
		print_and_log("[$ans]: ");
		chomp($newans=<STDIN>);
		if ($q =~ /password/i) {
			print LOG "********\n";
		} else {
			print LOG "$newans\n";
		}
		print_and_log("\n");
		$ans = $newans if ($newans);
		if ($check) {
			for (@valid) {
				$valid = 1 if (($ans) && ($ans eq $_));
			}
		} else {
 			if ((! defined $ans) || ($ans eq "")) {
				$valid = 0;
			} else {
				$valid = 1;
			}
		}
		if (! $valid) {
			print_and_log("\nInvalid answer: $ans\n\n\n");
			$ans = $origans;
		}
		$x++;
	}

	return $ans;
}

#
# Subroutine to print messages to the screen, as well as log the output to a file.
#
sub print_and_log {
	my ($line) = (@_);
	
	print LOG $line;
	print $line;
	
	return 1;
}

#
# Print a message, and then die.
#
sub log_and_die {
	my ($line) = (@_);
	print_and_log($line);
	save_cache();
	exit 1;
}
