VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "VarStack"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
Attribute VB_Ext_KEY = "SavedWithClassBuilder" ,"Yes"
Attribute VB_Ext_KEY = "Top_Level" ,"Yes"
'**
' A stack object used to store and retrieve named values with or without
' hierarchical inheritance.
'
' @describe
'   This object is designed to allow the storage and retrieval of
'   named values from a stack.  Each layer of the stack references
'   a "Variables" collection of "Variable" objects.  Named values
'   are stored to and retrieved from the stack using one of two main API's.
'   <blockquote>
'   <ul>
'   <li>The LocalValue properties store and retrieve values only
'   from the top of stack.
'   <li> The Value properties offer the same functionality while
'   allowing variables to be inherited from any level within the stack.
'   <li> The GetValue and SetValue methods allow the user to supply parameters
'   to mimic either of the above API's and to create ReadOnly variables.
'   </ul>
'   </blockquote>
'
' @author       Andrew Friedl
' @copyright    1997.03.10, BlackBox Software & Consulting
'*
Option Explicit

Private mVars() As Variables

'**
' Retrieve a reference to any layer of the stack.
'
' @describe
'   By default this property returns a reference of the Variables collection that
'   is "top of stack".  Using the optional Index parameter allows the retrieval
'   of any layer in the stack.
'
' @param    An optional parameter that may be used to specify a specific layer.
' @returns  A Variables collection.
'*
Public Property Get Peek(Optional Index As Variant) As Variables
    '
    ' Get the top of stack context
    '
    If IsMissing(Index) Then
        Set Peek = mVars(UBound(mVars))
    Else
        Set Peek = mVars(CInt(Index))
    End If
End Property

'**
' The number of Variables collection stored on the stack.
'
' @returns  An integer.
'*
Public Property Get Count() As Integer
    Count = UBound(mVars)
End Property

'**
' Pushes a new Variables collection to the top of the stack.
'
' @describe
'   Thus function places a new Variables collection on the top
'   of the stack.  Optionally, a Variables collection may be passed as a
'   parameter.  If one is supplied it is pushed to the top of stack, if not,
'   a new Variables collection is created and placed on the stack.
'
' @param
'   Vars A Variables compatible object to be pushed to the top
'   of the stack. (optional)
'
' @returns
'   The Variables object that was placed on the top of stack.
'*
Public Function Push(Optional Vars) As Variables
    ReDim Preserve mVars(UBound(mVars) + 1)
    
    If IsMissing(Vars) Then
        Set mVars(UBound(mVars)) = New Variables
    Else
        If Vars Is Nothing Then
            Set mVars(UBound(mVars)) = New Variables
        Else
            Set mVars(UBound(mVars)) = Vars
        End If
    End If
    Set Push = mVars(UBound(mVars))
End Function

'**
' Removes the top of stack Variables object from the stack.
'
' @describe
'   Thus function removes the current Variables object from the top of the stack.
'   An exception may be thrown if an attempt is made to pop the lowest (the zero)
'   level of the stack.
'
' @returns  The Variables object removed from the stack.
'*
Public Function Pop() As Variables
    Set Pop = mVars(UBound(mVars))
    If UBound(mVars) > 0 Then
        ReDim Preserve mVars(UBound(mVars) - 1)
    End If
End Function

'**
' Retrieves a named value from the stack.
'
' @describe
'   Thus function retrieves a named value from variable stack. By default,
'   the value will be sought at all lower levels if it is not located at the top of
'   stack.
'
' @param    Name The name of the value to be retrieved.
' @param    Value A reference to the Variant where the value should be stored.
' @param    LocalOnly An optional value used to indicate that the variable lookup
'   is to be performed only at the highest level of the stack.
'
' @returns      True if the value is found, False otherwise.
'*
Public Function GetValue(Name As String, Value As Variant, Optional LocalOnly) As Boolean
    Dim Var As Variable
    If Search(Name, Var, LocalOnly) Then
        Call Var.GetValue(Value)
        GetValue = True
    Else
        GetValue = False
    End If
End Function

'**
' Stores a named value within the stack.
'
' @describe
'   Thus function stores a named value to the variable stack.
'   By default, the value will be sought at all lower levels if the value is
'   not found at the top of stack.
'
' @param    Name        The name of the value to be retrieved.
' @param    Value       A reference to the Variant where the value should be stored.
' @param    ReadOnly    An optional value used to indicate that the variable should
'           be a readonly value. This parameter only applies when the variable is
'           initially created.
'
' @param    LocalOnly   An optional value used to indicate that the variable is
'   is to be set only at the highest level of the stack and not inheritance is to be
'   used.
'
' @returns  True if the value is set, False otherwise.
'*
Public Function SetValue(Name As String, Value As Variant, Optional ReadOnly As Variant, Optional LocalOnly As Variant) As Boolean
    Dim Var As Variable, IsLocal As Boolean
    
    If Search(Name, Var, LocalOnly) Then
        SetValue = Var.SetValue(Value)
    Else
        SetValue = Peek().SetValue(Name, Value, ReadOnly)
    End If
End Function

Private Sub Class_Initialize()
    ReDim mVars(0)
    ' root level variables (globals or ???)
    Set mVars(0) = New Variables
End Sub

'**
' Retrieves a value from the stack.
'
' @describe
'   Thus property retrieves a named value from the variable stack.
'   If the value does not exist at the top of stack level, it will be sought at
'   a lower level until all levels have been exhausted.  If the value cannot
'   be located an exception will be thrown.
'
' @returns  A variant that is the current value of the named variable.
'*
Public Property Get Value(Name As String) As Variant
    Dim Var As Variable
    If SearchScope(Name, Var) Then
        If Var.IsAnObject Then
            Set Value = Var.Value
        Else
            Value = Var.Value
        End If
    Else
        Err.Raise vbObjectError + 1, "VarStack", Name & "is unknown"
    End If
End Property

'**
' Stores a value to a named variable within the stack.
'
' @describe
'   Thus property stores an object reference to the variable stack.
'   If the value does not exist at the top of stack level, it will be sought at
'   a lower level until all levels have been exhausted.  If the value cannot
'   be located it will be created at the top of stack level.
'
' @param    Name    The name of the value to be stored.
' @param    Value   The value to be stored.
'*
Public Property Let Value(Name As String, Value As Variant)
    Dim Var As Variable
    If SearchScope(Name, Var) Then
        Var.SetValue Value
    Else
        mVars(UBound(mVars)).SetValue Name, Value
    End If
End Property

'**
' Stores an object reference to a named variable within the stack.
'
' @describe
'   Thus property stores an object reference to the variable stack.
'   If the value does not exist at the top of stack level, it will be sought at
'   a lower level until all levels have been exhausted.  If the value cannot
'   be located it will be created at the top of stack level.
'
' @param    Name    The name of the value to be stored.
' @param    Value   The value to be stored.
'*
Public Property Set Value(Name As String, Value As Variant)
    Dim Var As Variable
    If SearchScope(Name, Var) Then
        Var.SetValue Value
    Else
        mVars(UBound(mVars)).SetValue Name, Value
    End If
End Property

'**
' Retrieves a named value from the top of stack.
'
' @describe
'   Thus function retrieves a named value from the top of stack.
'   If the value does not exist an exception will be thrown.
'
' @param    Name    The name of the value to be retrieved.
'
' @returns  A variant that is the current value of the named variable.
'*
Public Property Get LocalValue(Name As String) As Variant
    Dim Val As Variant
    If GetValue(Name, Val, True) Then
        If IsObject(Val) Then
            Set LocalValue = Val
        Else
            LocalValue = Val
        End If
    Else
        Err.Raise vbObjectError + 1, "VarStack", "unknown value '" & Name & "'"
    End If
End Property

'**
' Stores a non-object value to a variable at the top of stack.
'
' @describe
'   Thus property stores a value to the variable stack. If the value does not
'   exist at the top of stack level, it will be created at that level.
'
' @param    Name    The name of the value to be stored.
' @param    Value   The value to be stored.
'*
Public Property Let LocalValue(Name As String, Value As Variant)
    Call SetValue(Name, Value, , True)
End Property

'**
' Stores an object reference to a variable at the top of stack.
'
' @describe
'   Thus property stores an object reference to the variable stack.
'   If the value does not exist at the top of stack level, it will be created
'   at that level.
'
' @param    Name    The name of the value to be stored.
' @param    Value   The value to be stored.
'*
Public Property Set LocalValue(Name As String, Value As Variant)
    Call SetValue(Name, Value, , True)
End Property

'**
' Creates an alias to an existing named value.
'
' @describe
'   Thus function creates a named variable that is an alias to an
'   existing variable.  Reads and writes may still occur to the aliased variable
'   as well as the new alias.  Additionally, traces may be added to or removed from
'   each variable independently of the other, however, traces to the alias will not
'   occur if the original variable is modified.  All aliased variables are created
'   at the current top of stack.
'
' @param    NewName     The name of the new alias variable.
' @param    Name        The name of the variable to be aliased.
' @param    ReadOnly    (optional) Flag indicating the new Alias to be readonly.
' @param    LocalOnly   (optional) Flag indicating the search method for the variable
'   to be aliased.
'
' @returns  True if the variable was aliased, False otherwise.
'*
Public Function AliasValue(NewName As String, Name As String, Optional ReadOnly As Variant, Optional LocalOnly As Variant) As Boolean
    Dim Existing As Variant
    Dim Value As Variable

    ' see if the named value exists
    If Search(Name, Value) Then
        AliasValue = mVars(UBound(mVars)).SetValue(NewName, Value, ReadOnly)
        Exit Function
    Else
        AliasValue = False
    End If
End Function

'**
' Searches the stack for a variable.
'
' @describe
'   Thus function searches the variable stack for a variable. By default, the
'   entire stack is searched, unless otherwise modified by optional parameters.
'   If the variable is located, the parameter Var will be set to a reference
'   to the variable.
'
' @param    Name        The name of the variable to be located.
' @param    Var         The reference to be set if the variable is located.
' @param    LocalOnly   (optional) Flag indicating the search method for the variable.
'
' @returns  True if the variable is found, False otherwise.
'*
Public Function Search(Name As String, Var As Variable, Optional LocalOnly As Variant) As Boolean
    If SearchLocal(Name, Var) Then
        Search = True
    Else
        ' if confined to local then fail
        If OptParmBool(False, LocalOnly) Then
            Search = False
        Else
            ' if not found in hierarchy then fail
            If SearchScope(Name, Var) Then
                Search = True
            Else
                Search = False
            End If
        End If
    End If
End Function

'**
' Searches the stack for a named variable object only at the local level.
'
' @describe
'   Thus function searches the top of stack for a named variable.
'   If the variable is located, the parameter Value will be set to a reference
'   to the variable.
'
' @param    Name        The name of the variable to be located.
' @param    Var         The reference to be set if the variable is located.
'
' @returns  True if the variable is found, False otherwise.
'*
Public Function SearchLocal(Name As String, Var As Variable) As Boolean
    On Error GoTo NotFound
    SearchLocal = mVars(UBound(mVars)).GetVariable(Name, Var)
    Exit Function
NotFound:
    SearchLocal = False
End Function

'**
' Searches the entire stack for a named variable object.
'
' @describe
'   Thus function searches the entire variable stack for a named variable.
'   The stack is searched from the current top of stack downward.  Searching terminates
'   when the first occurance of the named variable is found, or when the bottom of
'   of the stack is reached without a successful match.
'
' @param    Name        The name of the variable to be located.
' @param    Value       the reference to be set if the variable is located.
'
' @returns  True if the variable is found, False otherwise.
'*
Public Function SearchScope(Name As String, Value As Variable) As Boolean
    Dim Index As Integer
    For Index = UBound(mVars) - 1 To 0 Step -1
        If mVars(Index).GetVariable(Name, Value) Then
            SearchScope = True
            Exit Function
        End If
    Next Index
End Function
