/*
  wfNameC.c - Windows Filename Completion.

  Jason Hood, 10 to 23 August and 1 & 2 September, 2000.

  Windows Filename Completion (wfNameC) is a utility for keyboard users. It
  allows filenames to be completed by pressing a key. Two keys are available.
  One will cycle through all names; the other will display a list. These keys
  can be used interchangeably (the list can be displayed during a cycle; the
  cycle will continue after a list selection).

  Many thanks to Alex Vallat, who supplied me with his Visual Basic code for
  Universal AutoComplete (UAC) and for his comments.

  v1.10 - 13, 22 to 25 January, 2003:
    use the system link color (COLOR_HOTLIGHT/HotTrackingColor);
    complete only up to the cursor position, removing the terminating quote
     recognition;
    replace slashes with backslashes;
    complete filenames within the text, not just from the beginning (to allow
     filenames as parameters, eg: "notepad c:\compeleted\name");
    work with multi-line edit controls (that aren't too big);
    use a dynamic text buffer;
    set the current directory to the initial completion path;
    fixed a bug with a blank path (matching ':' from a previous completion).
*/


#include <windows.h>
#include <shellapi.h>
#include "wfNameC.h"


#define KEY "Software\\Adoxa\\wfNameC"
#define RUN "Software\\Microsoft\\Windows\\CurrentVersion\\Run"

#define WM_TRAYICON WM_USER

#ifndef IDC_LINK
#define IDC_LINK MAKEINTRESOURCE( 32649 )  // Why isn't this already defined?
#endif
#ifndef COLOR_HOTLIGHT
#define LINKCOLOR GetSysColor( 26 )	  // RGB( 0, 0, 255 ) for Win95?
#else
#define LINKCOLOR GetSysColor( COLOR_HOTLIGHT )
#endif


#define MIN_LINES      1
#define MAX_LINES     20
#define DEFAULT_LINES  5
#define ITEM_MARGIN    2	// Pixels between the listbox edge and string

#define MAX_TEXT       65000	// Don't read text bigger than this


HINSTANCE g_hinst;
HWND	  g_hwnd;

struct
{
  WORD cycle;			// Cycle hotkey
  WORD list;			// List hotkey
  UINT start;			// Start with windows
  UINT hide;			// Hide tray icon
  UINT imm;			// Cycle immediately
  UINT lines;			// Maximum number of lines in the list
}
settings, old_settings;

BOOL installed = FALSE;

ATOM cycleAtom, listAtom;	// Hotkey identifiers

struct sl
{
  char* s;			// String
  int	l;			// Length of above
};

struct
{
  struct sl* list;		// Array of filenames
  int  count;			// Number of above
  char partial[MAX_PATH];	// The string common to all matches
  int  common;			// Length of above
  int  offset;			// Offset into the array to start displaying
}
files;

struct
{
  HWND	wnd;			// Window containing filename
  char* text;			// Text from above
  int	size;			// Space allocated for above
  char* start;			// Pointer into above for the name portion
  char* end;			// Pointer into above for the end of the name
  char	prev[MAX_PATH]; 	// The previous path
  char* name;			// Pointer into above for the name portion
  int	cycle;			// Index into the array for the cycle
  DWORD line;			// The line number (in a multi-line control)
  int	all;			// The entire control is a single filename
}
FileName;

int item_height, item_width;	// Listbox and character dimensions


void Complete( ATOM );
BOOL CALLBACK ListDlg( HWND, UINT, WPARAM, LPARAM );

int  init_settings( void );
void store_settings( void );
int  process_settings( void );
UINT RunEntry( UINT );

LRESULT CALLBACK WindowProc( HWND, UINT, WPARAM, LPARAM );

HWND WINAPI   HotKeyControl( HWND, int );
BOOL CALLBACK SettingsDlg( HWND, UINT, WPARAM, LPARAM );
BOOL CALLBACK AboutDlg( HWND, UINT, WPARAM, LPARAM );

void TrayAddIcon( HWND );
void TrayRemoveIcon( HWND );
void TrayProcess( HWND,	LPARAM );


// Case-insensitive sort function to sort the filenames.
int sort_files( const void* a, const void* b )
{
  char* f1 = ((struct sl*)a)->s;
  char* f2 = ((struct sl*)b)->s;
  int	c;

  // I don't use lstrcmpi or CompareString, because they place nonalphanumeric
  // before alphanumeric, whereas I want a normal "ASCII" sequence, to make
  // the listbox selection easier.
  // Two filenames in the one directory will always be different,
  // so there is no need to test for the lengths.
  while ((c = CharLower( (char*)*f1 ) - CharLower( (char*)*f2 )) == 0)
    ++f1, ++f2;

  return c;
}


// Find files based on the specification in path. input is used to determine
// the path for the Open/Save common dialogs.
// Returns the number of files found.
int find_files( char* path, HWND input )
{
  HANDLE find;
  WIN32_FIND_DATA ffblk;
  int	 size;
  char*  name;
  HANDLE map;
  VOID*  mem;
  int	 i;

  if (files.list != NULL)
  {
    free( files.list );
    files.list = NULL;
  }

  // If we're in an Explorer-style common Open/Save dialog,
  // determine it's path and change to it.
  map = CreateFileMapping( (HANDLE)0xffffffff, NULL, PAGE_READWRITE,
			   0, MAX_PATH, NULL );
  mem = MapViewOfFile( map, FILE_MAP_WRITE, 0, 0, 0 );
  if (CommDlg_OpenSave_GetFolderPath( GetParent( input ), mem, MAX_PATH ) > 0)
  {
    SetCurrentDirectory( mem );
    FileName.all = 1;
  }
  else FileName.all = 0;
  UnmapViewOfFile( mem );
  CloseHandle( map );

  // Scan all the files first to determine memory requirements.
  files.count = size = 0;
  find = FindFirstFile( path, &ffblk );
  if (find != INVALID_HANDLE_VALUE)
  {
    do
    {
      if (ffblk.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
      {
	if (strcmp( ffblk.cFileName, "."  ) == 0 ||
	    strcmp( ffblk.cFileName, ".." ) == 0)
	  continue;
	++size; 	// One more for the appended backslash
      }
      size += sizeof(struct sl) + strlen( ffblk.cFileName ) + 1;
      ++files.count;
    }
    while (FindNextFile( find, &ffblk ));
    FindClose( find );

    files.list = malloc( size );
    name = (char*)(files.list + files.count);
    files.count = 0;
    find = FindFirstFile( path, &ffblk );
    do
    {
      if (strcmp( ffblk.cFileName, "."  ) == 0 ||
	  strcmp( ffblk.cFileName, ".." ) == 0)
	continue;

      files.list[files.count].l = size = strlen( ffblk.cFileName );
      files.list[files.count].s = memcpy( name, ffblk.cFileName, ++size );

      // Find the common portion.
      if (files.count == 0)
      {
	files.common = size - 1;
	memcpy( files.partial, name, size );
	CharLower( files.partial );
      }
      else
      {
	for (i = 0; i < files.common; ++i)
	{
	  if (files.partial[i] != (char)CharLower( (char*)name[i] ))
	  {
	    files.partial[files.common = i] = 0;
	    break;
	  }
	}
      }
      name += size;
      if (ffblk.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
      {
	name[-1] = '\\';
	*name++	= 0;
	++files.list[files.count].l;
      }
      ++files.count;
    }
    while (FindNextFile( find, &ffblk ));
    qsort( files.list, files.count, sizeof(struct sl), sort_files );
  }
  return files.count;
}


// Replace the filename portion of the window's text with name and
// leave the cursor at its end. The previous path is also updated.
void copy_name( char* name )
{
  int len, rep;

  len = strlen( name );
  memcpy( FileName.name, name, len+1 );
  if (FileName.end == NULL)
    ++len;
  else
  {
    rep = len - (FileName.end - FileName.start);
    if (rep != 0)
    {
      memmove( FileName.end + rep, FileName.end, strlen( FileName.end ) + 1 );
      FileName.end += rep;
    }
  }
  memcpy( FileName.start, name, len );
  SendMessage( FileName.wnd, WM_SETTEXT, 0, (LPARAM)FileName.text );
  len += FileName.start - FileName.text - FileName.line;
  SendMessage( FileName.wnd, EM_SETSEL, (WPARAM)len, (LPARAM)len );
}


// Perform the filename completion, depending on the key pressed.
void Complete( ATOM method )
{
  static char  wild[MAX_PATH];	// Stores the path that was entered
  static char* prefix;		// Points to the above or what is common
  static int   list_dir;	// List a directory from the list display
  static int   old_quote;	// The previous path's quote status
  char	path[MAX_PATH]; 	// Same as wild, but with the appended '*'
  char* start;
  char* begin;
  DWORD my_thread, path_thread;
  int	pos, len, quote;
  char* p;

  my_thread   = GetCurrentThreadId();
  path_thread = GetWindowThreadProcessId( GetForegroundWindow(), NULL );

  if (my_thread == path_thread) // Already in the list
  {
    list_dir = (method == listAtom);
    PostMessage( GetForegroundWindow(), WM_COMMAND, IDOK, 0 );
    return;
  }
  list_dir = FALSE;

  if (!AttachThreadInput( my_thread, path_thread, TRUE ))
  {
    MessageBox( NULL, "Unable to receive input.", "wfNameC", MB_ICONWARNING);
    return;
  }

  FileName.wnd = GetFocus();
  pos = -1;
  SendMessage( FileName.wnd, EM_GETSEL, (WPARAM)NULL, (LPARAM)&pos );
  len = SendMessage( FileName.wnd, WM_GETTEXTLENGTH, 0, 0 );
  if (pos == -1 || len > MAX_TEXT)
  {
  no_good:
    AttachThreadInput( my_thread, path_thread, FALSE );
    return;
  }
  len += MAX_PATH + 2;
  if (len > FileName.size)
  {
    p = realloc( FileName.text, len );
    if (!p) goto no_good;
    FileName.start += p - FileName.text;
    FileName.text = p;
    FileName.size = len;
  }
  SendMessage( FileName.wnd, WM_GETTEXT, len, (LPARAM)FileName.text );

  // It appears Edit controls count the CR, but RichEdit controls do not,
  // which causes a discrepancy between the position and the text.
  FileName.line = SendMessage( FileName.wnd, EM_LINEFROMCHAR, pos, 0 );
  if (FileName.line)
  {
    len = SendMessage( FileName.wnd, EM_LINEINDEX, 1, 0 );
    if (FileName.text[len] != '\n')
      FileName.line = 0;
  }
  FileName.end = FileName.text + pos + FileName.line;

  // Find the beginning of the line. Treat tabs as a line beginning, too,
  // since they are invalid in filenames.
  begin = FileName.end;
  while (--begin >= FileName.text && *begin != '\n' && *begin != '\t')
    ; // do nothing
  ++begin;

  // Determine the start of the filename. If we're in an Open/Save common
  // dialog, the filename is the entire text; otherwise it begins either
  // at the first unterminated quote or the space before the cursor.
  p = start = begin;
  quote = 0;
  if (FileName.all)
  {
    if (*start == '\"')
    {
      quote = 1;
      ++start;
    }
  }
  else
  {
    while (p < FileName.end)
    {
      if (*p == '\"')
      {
	quote = !quote;
	start = p + 1;
      }
      ++p;
    }
    if (!quote && start != FileName.end)
    {
      // This is overly complicated, to try to get around unquoted spaces.
      // If it's not the root directory or a (supposed) drive specifier,
      // check if the previous space is.
      p = NULL;
      start = FileName.end;
    try_next_space:
      while (start > begin && start[-1] != ' ')
	--start;
      if (!(start[0] == '\\' || start[0] == '/' || start[1] == ':'))
      {
	if (start > begin)
	{
	  if (p == NULL) p = start;
	  --start;
	  goto try_next_space;
	}
	else if (p) start = p;
      }
    }
  }

  // Replace slashes with backslashes.
  for (p = start; p < FileName.end; ++p)
    if (*p == '/') *p = '\\';

  if (*FileName.end == '\0')
  {
    FileName.end = NULL;
    strcpy( path, start );
  }
  else
  {
    len = FileName.end - start;
    memcpy( path, start, len );
    path[len] = 0;
  }

  if (lstrcmpi( path, FileName.prev ) == 0)
  {
    // Adjust the start position if a quote has been added or removed.
	 if ( quote && !old_quote) ++FileName.start;
    else if (!quote &&	old_quote) --FileName.start;

    if (files.count > 1)
    {
  complete_now:
      if (method == cycleAtom)
      {
	if (++FileName.cycle == files.count)
	{
	  copy_name( prefix );
	  FileName.cycle = -1;
	}
	else
	  copy_name( files.list[FileName.cycle].s );
      }
      else
      {
	FileName.cycle = DialogBox( g_hinst, MAKEINTRESOURCE( DLG_LIST ),
				    FileName.wnd, ListDlg );
	if (FileName.cycle >= 0)
	{
	  copy_name( files.list[FileName.cycle].s );
	  // Force a new completion if a directory was selected.
	  if (files.list[FileName.cycle].s[files.list[FileName.cycle].l-1]
		== '\\')
	  {
	    FileName.prev[0] = '\"';
	    FileName.prev[1] = 0;
	    if (list_dir) PostMessage( g_hwnd, WM_HOTKEY, listAtom, 0 );
	  }
	}
	else copy_name( prefix );
      }
    }
  }
  else
  {
    strcpy( FileName.prev, path );
    // Find the filename portion of the path.
    FileName.name = strrchr( FileName.prev, '\\' );
    if (FileName.name == NULL)
    {
      FileName.name = FileName.prev;
      if (FileName.name[0] && FileName.name[1] == ':') FileName.name += 2;
    }
    else
    {
      *FileName.name = '\0';
      SetCurrentDirectory( FileName.prev );
      *FileName.name++ = '\\';
    }
    // Set the filename portion within the text.
    FileName.start = start + (FileName.name - FileName.prev);

    FileName.cycle = -1;

    if ((prefix = strpbrk( path, "*?" )) == NULL)
      strcat( path, "*" );      // Append the wildcard for searching

    if (!find_files( path, FileName.wnd ))
      MessageBeep( MB_OK );
    else if (files.count == 1)
    {
      copy_name( files.list[0].s );
      // If only a directory matched, start a new completion.
      if (files.list[0].s[files.list[0].l-1] == '\\')
      {
	FileName.prev[0] = '\"';
	FileName.prev[1] = 0;
      }
    }
    else
    {
      if (prefix != NULL || method == listAtom || settings.imm == BST_CHECKED)
      {
	prefix = strcpy( wild, FileName.name );
	goto complete_now;
      }
      copy_name( prefix = files.partial );
    }
  }
  old_quote = quote;

  AttachThreadInput( my_thread, path_thread, FALSE );
}


// Search the filenames from index for a match to prefix, which is pos chars.
// Returns the index of the matching name, or LB_ERR if none found.
// Assumes a sorted list.
int find_prefix( char* prefix, int pos, int index )
{
  int  j;
  char c;

  for (; index < files.count; ++index)
  {
    if (files.list[index].l < pos) continue;	// Name is too small
    for (j = files.common; j < pos; ++j)	// Start after the common part
    {
      c = (char)CharLower( (char*)files.list[index].s[j] );
      if (c < prefix[j]) break; 	// Too small, try next name
      if (c > prefix[j]) return LB_ERR; // Too large, no match
    }
    if (j == pos) break;		// All characters matched
  }
  return (index == files.count) ? LB_ERR : index;
}


// Find all names matching the prefix and adjust the listbox appropriately.
// Returns 0 for a match, or LB_ERR for none.
int match_prefix( char* prefix, int pos, int index, HWND list )
{
  int  matches = 0;
  RECT r;

  if (pos == files.common)	// Nothing extra to match, so reset
  {
    matches = files.count;
    files.offset = 0;
  }
  else
  {
    while ((index = find_prefix( prefix, pos, index )) != LB_ERR)
    {
      if (++matches == 1) files.offset = index;
      ++index;
    }
    if (matches == 0) return LB_ERR;
  }

  copy_name( prefix );
  GetWindowRect( GetParent( list ), &r );
  SetWindowPos( GetParent( list ), HWND_TOP, 0, 0, r.right - r.left,
		((matches >= settings.lines) ? settings.lines : matches) *
		  item_height + GetSystemMetrics( SM_CYBORDER ) * 2,
		SWP_NOMOVE );
  // No need to resize the listbox itself.
  SendMessage( list, LB_SETCOUNT,  matches, 0 );
  SendMessage( list, LB_SETCURSEL, 0, 0 );

  return 0;
}


// Display the list of filenames. Set the width the same as the window, or to
// the length of the longest name if the window is too small.
BOOL CALLBACK ListDlg( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  static int  closed;		// Stop LBN_KILLFOCUS from returning -1
  static char name[MAX_PATH];	// Remember additional characters
  static int  pos;		// Length of above
  HWND list;
  int  width, height;
  RECT r;
  LPDRAWITEMSTRUCT di;
  COLORREF oldbk, oldtxt;
  int  c, i;

  switch (msg)
  {
    case WM_MEASUREITEM:
      ((LPMEASUREITEMSTRUCT)lParam)->itemHeight = item_height;
    break;

    case WM_INITDIALOG:
      if (!item_height) return 0;	// First time to determine listbox size
      closed = 0;

      list = GetDlgItem( hwnd, IDL_LIST );
      height = item_height * ((files.count >= settings.lines) ?
			      settings.lines : files.count);
      GetWindowRect( FileName.wnd, &r );
      width = r.right - r.left; 	// Width of the host window
      c = 0;				// Determine the longest filename
      for (i = 0; i < files.count; ++i)
	if (files.list[i].l > c) c = files.list[i].l;
      i = GetSystemMetrics( SM_CXBORDER ) * 2;
      c = ITEM_MARGIN + c * item_width + i;	// Approximate pixel width
      if (files.count > settings.lines)
	c += GetSystemMetrics( SM_CXVSCROLL );	// Include the scrollbar
      if (width < c) width = c;
      // Make adjustments if it is too long for the screen.
      if (r.left + width > GetSystemMetrics( SM_CXSCREEN ))
      {
	if (width != c)
	{				// Try reducing it to the screen edge
	  width = GetSystemMetrics( SM_CXSCREEN ) - r.left;
	  if (width < c) width = c;	// Nope, too small
	}
	r.left = GetSystemMetrics( SM_CXSCREEN ) - width;
	if (r.left < 0) r.left = 0;
      }
      c = GetSystemMetrics( SM_CYBORDER ) * 2;
      height += c;
      // Drop it down from the bottom of the window,
      // but if it goes off-screen pop it up from the top.
      if (r.bottom + height > GetSystemMetrics( SM_CYSCREEN ))
      {
	r.bottom = r.top - height;
	if (r.bottom < 0) r.bottom = 0;
      }
      SetWindowPos( hwnd, HWND_TOP, r.left, r.bottom, width, height, 0 );
      MoveWindow( list, 0, 0, width - i, height - c, FALSE );
      pos = files.common;
      memcpy( name, files.partial, pos+1 );
      // Let the edit window display its cursor.
      SendMessage( FileName.wnd, WM_SETFOCUS, (WPARAM)NULL, 0 );
      match_prefix( name, pos, 0, list );
    break;

    case WM_DRAWITEM:
      di = (LPDRAWITEMSTRUCT)lParam;
      i  = (int)di->itemID;
      if (i >= 0 && !(di->itemAction & ODA_FOCUS))
      {
	if (di->itemState & ODS_SELECTED)
	{
	  oldbk	 = SetBkColor(	 di->hDC, GetSysColor( COLOR_HIGHLIGHT ) );
	  oldtxt = SetTextColor( di->hDC, GetSysColor( COLOR_HIGHLIGHTTEXT ) );
	  FillRect( di->hDC, &di->rcItem, (HBRUSH)(COLOR_HIGHLIGHT+1) );
	}
	else
	  FillRect( di->hDC, &di->rcItem, (HBRUSH)(COLOR_WINDOW+1) );
	i += files.offset;
	TextOut( di->hDC, di->rcItem.left + ITEM_MARGIN, di->rcItem.top,
		 files.list[i].s, files.list[i].l );
	if (di->itemState & ODS_SELECTED)
	{
	  SetBkColor(	di->hDC, oldbk );
	  SetTextColor( di->hDC, oldtxt );
	}
      }
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDL_LIST:
	  switch (HIWORD( wParam ))
	  {
	    case LBN_DBLCLK:
	      ++closed;
	      EndDialog( hwnd, files.offset + SendMessage( (HWND)lParam,
						LB_GETCURSEL, 0, 0 ) );
	    break;

	    case LBN_KILLFOCUS:
	      if (!closed) EndDialog( hwnd, -1 );
	    break;

	    default: return 0;
	  }
	break;

	case IDOK:
	  ++closed;
          EndDialog( hwnd, files.offset + SendDlgItemMessage( hwnd, IDL_LIST,
                                            LB_GETCURSEL, 0, 0 ) );
	break;

	case IDCANCEL:
	  ++closed;
	  EndDialog( hwnd, -1 );
	break;

	default: return 0;
      }
    break;

    case WM_CHARTOITEM:
      c = LOWORD( wParam );
      c = (char)CharLower( (char*)c );
      name[pos++] = c;
      name[pos]   = 0;
      i = match_prefix( name, pos, files.offset, (HWND)lParam );
      if (i == LB_ERR) name[--pos] = 0;
      return -2;
    break;

    case WM_VKEYTOITEM:
      if (LOWORD( wParam ) == VK_BACK)
      {
	if (pos > files.common)
	{
	  name[--pos] = 0;
	  match_prefix( name, pos, 0, (HWND)lParam );
	}
	else
	{
	  ++closed;
	  EndDialog( hwnd, -1 );
	}
	return -2;
      }
      return -1;
    break;

    default: return 0;
  }
  return 1;
}


// Create the main window and initialize the settings.
// If it's already running, bring up the settings dialog.
int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrev,
		    LPSTR lpCmdLine, int nCmdShow )
{
  WNDCLASS wc;
  HWND hwnd;
  MSG  msg;
  HWND list;
  HDC  ldc;
  TEXTMETRIC tm;

  hwnd = FindWindow( "wfNameC", "Windows Filename Completion" );
  if (hwnd)
  {
    PostMessage( hwnd, WM_COMMAND, IDM_SETTINGS, 0 );
    SetForegroundWindow( hwnd );
    return 0;
  }

  wc.style	   = 0;
  wc.lpfnWndProc   = WindowProc;
  wc.cbClsExtra    = 0;
  wc.cbWndExtra    = 0;
  wc.hInstance	   = hInst;
  wc.hIcon	   = LoadIcon( hInst, MAKEINTRESOURCE( IDI_ICON	) );
  wc.hCursor	   = LoadCursor( NULL, IDC_ARROW );
  wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
  wc.lpszMenuName  = NULL;
  wc.lpszClassName = "wfNameC";
  if (!RegisterClass( &wc )) return 0;

  hwnd = CreateWindow( "wfNameC", "Windows Filename Completion",
		       0, 0, 0, 0, 0, NULL, NULL, hInst, NULL );
  if (!hwnd) return 0;

  g_hinst   = hInst;
  g_hwnd    = hwnd;
  cycleAtom = GlobalAddAtom( "wfNameC_Cycle" );
  listAtom  = GlobalAddAtom( "wfNameC_List" );

  InitCommonControls();

  if (!init_settings())
  {
    GlobalDeleteAtom( cycleAtom );
    GlobalDeleteAtom( listAtom );
    return 0;
  }

  // Ensure the completion starts the first time around.
  FileName.prev[0] = '\"';      // Quote is an invalid filename character

  // Determine the height of items in the listbox and average character width.
  list = CreateDialog( hInst, MAKEINTRESOURCE( DLG_LIST ), hwnd, ListDlg );
  ldc  = GetDC( list );
  GetTextMetrics( ldc, &tm );
  ReleaseDC( list, ldc );
  DestroyWindow( list );
  item_height = tm.tmHeight;
  item_width  = tm.tmAveCharWidth;

  installed = TRUE;

  while (GetMessage( &msg, NULL, 0, 0 ))
  {
    TranslateMessage( &msg );
    DispatchMessage( &msg );
  }
  return msg.wParam;
}


// Read the settings from the registry, or bring up the dialog if this is
// the first time it's been run.
// Returns TRUE if OK was selected, FALSE if Exit or Cancel.
int init_settings( void )
{
  HKEY	key;
  DWORD exist;
  BOOL	rc = TRUE;

  settings.cycle = MAKEWORD( VkKeyScan('`'), HOTKEYF_CONTROL );
  settings.list  = MAKEWORD( VkKeyScan('`'), HOTKEYF_CONTROL|HOTKEYF_SHIFT );
  settings.start = BST_CHECKED;
  settings.hide  = BST_UNCHECKED;
  settings.imm	 = BST_UNCHECKED;
  settings.lines = DEFAULT_LINES;

  RegCreateKeyEx( HKEY_CURRENT_USER, KEY, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &key, &exist );

  if (exist == REG_CREATED_NEW_KEY)
  {
    if (DialogBox( g_hinst, MAKEINTRESOURCE( DLG_SETTINGS ),
		   g_hwnd, SettingsDlg ) != 1)
      rc = FALSE;
  }
  else
  {
    exist = sizeof(settings.cycle);
    RegQueryValueEx( key, "Cycle", NULL, NULL, &settings.cycle, &exist );
    exist = sizeof(settings.list);
    RegQueryValueEx( key, "List", NULL, NULL, &settings.list, &exist );
    exist = sizeof(settings.hide);
    RegQueryValueEx( key, "Hide", NULL, NULL, &settings.hide, &exist );
    exist = sizeof(settings.imm);
    RegQueryValueEx( key, "Immediate", NULL, NULL, &settings.imm, &exist );
    exist = sizeof(settings.lines);
    RegQueryValueEx( key, "Lines", NULL, NULL, &settings.lines, &exist );
    settings.start = RunEntry( -1 );
    if (!process_settings())
      if (DialogBox( g_hinst, MAKEINTRESOURCE( DLG_SETTINGS ),
		     g_hwnd, SettingsDlg ) != 1)
	rc = FALSE;
  }

  RegCloseKey( key );
  return rc;
}


// Write the settings into the registry.
void store_settings( void )
{
  HKEY key;
  DWORD exist;

  RegCreateKeyEx( HKEY_CURRENT_USER, KEY, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &key, &exist );

  RegSetValueEx( key, "Cycle", 0, REG_BINARY,
		 &settings.cycle, sizeof(settings.cycle) );
  RegSetValueEx( key, "List", 0, REG_BINARY,
		 &settings.list, sizeof(settings.list) );
  RegSetValueEx( key, "Hide", 0, REG_BINARY,
		 &settings.hide, sizeof(settings.hide) );
  RegSetValueEx( key, "Immediate", 0, REG_BINARY,
		 &settings.imm, sizeof(settings.imm) );
  RegSetValueEx( key, "Lines", 0, REG_BINARY,
		 &settings.lines, sizeof(settings.lines) );
  RunEntry( settings.start );

  RegCloseKey( key );
}


// Remove the old hotkey and create the new.
// Returns 0 if the hotkey could not be registered, non-zero otherwise.
int set_hotkey( ATOM id, UINT key )
{
  UINT shift, vk;
  BYTE hkf;

  if (installed) UnregisterHotKey( g_hwnd, id );

  // Translate the hotkey control format into the RegisterHotKey format.
  vk  = LOBYTE( key );
  hkf = HIBYTE( key );
  shift = 0;
  if (hkf & HOTKEYF_CONTROL) shift |= MOD_CONTROL;
  if (hkf & HOTKEYF_SHIFT)   shift |= MOD_SHIFT;
  if (hkf & HOTKEYF_ALT)     shift |= MOD_ALT;
  //if (hkf & HOTKEYF_WIN)   shift |= MOD_WIN;

  return RegisterHotKey( g_hwnd, id, shift, vk );
}


// Ensure the settings are valid and setup the hotkeys.
int process_settings( void )
{
  int  rc = 0;
  char buf[256];

  if (settings.cycle != old_settings.cycle || !installed)
    if (!set_hotkey( cycleAtom, settings.cycle ))
      rc = 1;

  if (settings.list != old_settings.list || !installed)
    if (!set_hotkey( listAtom, settings.list ))
      rc |= 2;

  if (settings.hide != old_settings.hide || !installed)
  {
    if (settings.hide == BST_UNCHECKED) TrayAddIcon(	g_hwnd );
    else if (installed) 		TrayRemoveIcon( g_hwnd );
  }

  if (rc)
  {
    if (rc == 3)
      strcpy( buf, "Both cycle and list hotkeys are already defined." );
    else
      wsprintf( buf, "The %s hotkey is already defined.",
		     (rc == 1) ? "cycle" : "list" );
    MessageBox( g_hwnd, buf, "wfNameC", MB_OK | MB_ICONWARNING );
  }

  if (settings.lines > MAX_LINES)
    settings.lines = MAX_LINES;
  else if (settings.lines < MIN_LINES)
    settings.lines = MIN_LINES;

  return !rc;
}


// Retrieve, set or remove the User Run entry.
UINT RunEntry( UINT type )
{
  HKEY	run;
  DWORD len;
  UINT	rc = 0;
  char	path[MAX_PATH+2];

  RegCreateKeyEx( HKEY_CURRENT_USER, RUN, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &run, &len );

  switch (type)
  {
    case BST_CHECKED:
      len = GetModuleFileName( NULL, path+1, sizeof(path)-2 );
      path[0] = path[len+1] = '\"';
      path[len+2] = 0;
      RegSetValueEx( run, "wfNameC", 0, REG_SZ, path, len+2 );
    break;

    case BST_UNCHECKED:
      RegDeleteValue( run, "wfNameC" );
    break;

    default:
      rc = (RegQueryValueEx( run, "wfNameC", NULL, NULL, NULL, NULL ) ==
	      ERROR_SUCCESS) ? BST_CHECKED : BST_UNCHECKED;
    break;
  }

  RegCloseKey( run );
  return rc;
}


// Process the hotkey and tray messages.
LRESULT CALLBACK WindowProc( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  switch (msg)
  {
    case WM_HOTKEY:
      Complete( wParam );
    break;

    case WM_TRAYICON:
      TrayProcess( hwnd, lParam );
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDM_ABOUT:
	  DialogBox( g_hinst, MAKEINTRESOURCE( DLG_ABOUT ), hwnd, AboutDlg );
	break;

	case IDM_SETTINGS:
	  if (DialogBox( g_hinst, MAKEINTRESOURCE( DLG_SETTINGS ), hwnd,
			 SettingsDlg ) != -1)
	    break;
	  // else fall through

	case IDM_EXIT:
	  PostMessage( hwnd, WM_DESTROY, 0, 0 );
	break;

	default: return DefWindowProc( hwnd, msg, wParam, lParam );
      }
    break;

    case WM_DESTROY:
      if (settings.hide == BST_UNCHECKED) TrayRemoveIcon( hwnd );
      UnregisterHotKey( g_hwnd, cycleAtom );
      UnregisterHotKey( g_hwnd, listAtom );
      GlobalDeleteAtom( cycleAtom );
      GlobalDeleteAtom( listAtom );
      PostQuitMessage( 0 );
    break;

    default: return DefWindowProc( hwnd, msg, wParam, lParam );
  }
  return TRUE;
}


// Add the hotkey control to the settings dialog.
// Returns the handle of the new window.
HWND WINAPI HotKeyControl( HWND hDlg, int hotkey )
{
  int	w = (hotkey == IDH_CYCLE);
  RECT	r;
  HWND	hkWnd;
  HFONT hkFont;

  r.left   = 38;
  r.right  = 100;
  r.top    = (w) ? 16 : 30;
  r.bottom = r.top + 12;
  MapDialogRect( hDlg, &r );

  hkWnd = CreateWindowEx( 0, HOTKEY_CLASS, "",
			  WS_CHILD | WS_VISIBLE | WS_TABSTOP,
			  r.left, r.top, r.right - r.left, r.bottom - r.top,
			  hDlg, (HMENU)hotkey, g_hinst, NULL );
  // Set the hotkey control font to the same as the dialog font.
  hkFont = (HFONT)SendMessage( hDlg, WM_GETFONT, 0, 0 );
  SendMessage( hkWnd, WM_SETFONT, (WPARAM)hkFont, MAKELPARAM( FALSE, 0 ) );
  // Add control to plain and shift combinations.
  SendMessage( hkWnd, HKM_SETRULES, (WPARAM)(HKCOMB_NONE | HKCOMB_S),
				    MAKELPARAM( HOTKEYF_CONTROL, 0 ) );
  SendMessage( hkWnd, HKM_SETHOTKEY, (w) ? settings.cycle :
					   settings.list, 0 );

  return hkWnd;
}


// Display and update the settings.
BOOL CALLBACK SettingsDlg( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  static HWND cycle, list, updown;
  HICON icon;
  UINT	lines;

  switch (msg)
  {
    case WM_INITDIALOG:
      icon = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
      SetClassLong( hwnd, GCL_HICON, (LONG)icon );

      cycle  = HotKeyControl( hwnd, IDH_CYCLE );
      list   = HotKeyControl( hwnd, IDH_LIST );
      updown = CreateUpDownControl( UDS_ALIGNRIGHT | UDS_ARROWKEYS |
				    UDS_SETBUDDYINT | WS_CHILD | WS_BORDER |
				    WS_VISIBLE, 0, 0, 0, 0, hwnd, IDE_SPIN,
				    g_hinst, GetDlgItem( hwnd, IDE_LINES ),
				    MAX_LINES, MIN_LINES, settings.lines );
      CheckDlgButton( hwnd, IDC_START, settings.start );
      CheckDlgButton( hwnd, IDC_HIDE,  settings.hide );
      CheckDlgButton( hwnd, IDC_IMM,   settings.imm );
      SendDlgItemMessage( hwnd, IDE_LINES, EM_SETLIMITTEXT, 2, 0 );
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDE_LINES:
	  if (HIWORD( wParam ) == EN_KILLFOCUS)
	  {
	    lines = GetDlgItemInt( hwnd, IDE_LINES, NULL, FALSE );
	    if (lines > MAX_LINES)
	      SetDlgItemInt( hwnd, IDE_LINES, MAX_LINES, FALSE );
	    else if (lines < MIN_LINES)
	      SetDlgItemInt( hwnd, IDE_LINES, MIN_LINES, FALSE );
	  }
	  else return 0;
	break;

	case IDOK:
	  old_settings	 = settings;
	  settings.cycle = SendMessage( cycle, HKM_GETHOTKEY, 0, 0 );
	  settings.list  = SendMessage( list,  HKM_GETHOTKEY, 0, 0 );
	  settings.start = IsDlgButtonChecked( hwnd, IDC_START );
	  settings.hide  = IsDlgButtonChecked( hwnd, IDC_HIDE );
	  settings.imm	 = IsDlgButtonChecked( hwnd, IDC_IMM );
	  settings.lines = GetDlgItemInt( hwnd, IDE_LINES, NULL, FALSE );
	  store_settings();
	  if (process_settings())
	    EndDialog( hwnd, 1 );
	break;

	case IDCANCEL:
	  EndDialog( hwnd, 0 );
	break;

	case ID_ABOUT:
	  DialogBox( g_hinst, MAKEINTRESOURCE( DLG_ABOUT ), hwnd, AboutDlg );
	break;

	case ID_EXIT:
	  // Remove the Run entry, assuming an uninstall.
	  if (IsDlgButtonChecked( hwnd, IDC_START ) == BST_UNCHECKED)
	    RunEntry( BST_UNCHECKED );
	  EndDialog( hwnd, -1 );
	break;

	default: return 0;
      }
      break;

    case WM_CLOSE:
      EndDialog( hwnd, 0 );
    break;

    default: return 0;
  }
  return 1;
}


// Display author, version and copyright info.
// It can also send me an e-mail or go to my homepage.
BOOL CALLBACK AboutDlg( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  HICON   icon;
  HCURSOR hand;
  HWND	  link;
  LONG	  id;

  switch (msg)
  {
    case WM_INITDIALOG:
      icon = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
      SetClassLong( hwnd, GCL_HICON, (LONG)icon );

      hand = LoadCursor( NULL, IDC_LINK	);
      link = GetDlgItem( hwnd, ID_AUTHOR );
      SetClassLong( link, GCL_HCURSOR, (LONG)hand );
      //link = GetDlgItem( hwnd, ID_URL );
      //SetClassLong( link, GCL_HCURSOR, (LONG)hand );
    break;

    case WM_CTLCOLORSTATIC:
      id = GetWindowLong( (HWND)lParam, GWL_ID );
      if (id == ID_AUTHOR || id == ID_URL)
      {
	SetTextColor( (HDC)wParam, LINKCOLOR );
	SetBkMode( (HDC)wParam, TRANSPARENT );
	return (BOOL)GetSysColorBrush( COLOR_3DFACE );
      }
      return 0;
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDOK: EndDialog( hwnd, 0 ); break;

	case ID_AUTHOR:
	case ID_URL:
	  if (HIWORD( wParam ) == STN_CLICKED)
	    ShellExecute( hwnd, NULL, (LOWORD( wParam ) == ID_AUTHOR) ?
			  "mailto:jadoxa@yahoo.com.au?Subject=wfNameC "PVERSION
			  : PURL, NULL, ".", 0 );
	break;

	default: return 0;
      }
    break;

    case WM_CLOSE:
      EndDialog( hwnd, 0 );
    break;

    default: return 0;
  }
  return 1;
}


// Add wfNameC to the system tray.
void TrayAddIcon( HWND hwnd )
{
  NOTIFYICONDATA tray;

  tray.cbSize = sizeof(tray);
  tray.hWnd   = hwnd;
  tray.uID    = IDI_TRAY;
  tray.uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP;
  tray.uCallbackMessage = WM_TRAYICON;
  tray.hIcon  = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
  strcpy( tray.szTip, "Windows Filename Completion" );

  Shell_NotifyIcon( NIM_ADD, &tray );
}


// Remove wfNameC from the system tray.
void TrayRemoveIcon( HWND hwnd )
{
  NOTIFYICONDATA tray;

  tray.cbSize = sizeof(tray);
  tray.hWnd   = hwnd;
  tray.uID    = IDI_TRAY;
  Shell_NotifyIcon( NIM_DELETE, &tray );
}


// Process wfNameC's system tray icon.
void TrayProcess( HWND hwnd, LPARAM lParam )
{
  UINT	msg = (UINT)lParam;
  HMENU menu, popup;
  POINT pos;

  switch (msg)
  {
    case WM_LBUTTONUP:
      PostMessage( hwnd, WM_COMMAND, IDM_SETTINGS, 0 );
      SetForegroundWindow( hwnd );
    break;

    case WM_RBUTTONUP:
      menu  = LoadMenu( g_hinst, MAKEINTRESOURCE( MNU_TRAY ) );
      popup = GetSubMenu( menu, 0 );
      GetCursorPos( &pos );
      TrackPopupMenuEx( popup, TPM_CENTERALIGN | TPM_LEFTBUTTON,
			pos.x, pos.y, hwnd, NULL );
      DestroyMenu( menu );
    break;
  }
}
