/*  Tty  Generate tide output in flat ASCII.
    XTide original source code date: 1997-04-26
    Last modified 1997-11-11 by Mike Hopper for WXTide32

    Copyright (C) 1997  David Flater.
    Also starring:  Jef Poskanzer, Mike Hopper

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Changes by Mike Hopper for WXTide32:
    Changed do_tty_header to print locator's station name
         also increased size of time buffer to handle NT's LONG TZ names
    Changed calander tide routines to reverse arrays for line rather
         than character transfers.  Much faster.
    Changed list_tides beyond recognition:
         1. Made entry number of DAYS, not TIDES.
         2. Added line breaks with day-of-week (I never remember the date :)
         3. Added phase of moon since I'm a sailor.
         4. 2/9/98 Corrected day of week from next_ht to next_ht_adj.
*/

#include "everythi.h"
extern double IDX_lat, IDX_lon;

/**********************************************************************/

/* Print out header lines */
static void
do_tty_header ()
{
//  puts (location);
  printf ("%s\n", custom_name); //mgh
  printf ("Units are %s, ", units);
  if (tadjust)
    printf ("timezone is UTC%s", seconds2hhmm (tadjust));
  else if (calendar || graphmode || skinny) {
    if (utc)
      printf ("timezone is UTC");
    else {
//      char buf[64];
//      strftime (buf, 64, "%Z", tmtime (faketime));
//      printf ("\nInitial Time Zone is %s", buf);
      tmtime (faketime);
      printf ("initial timezone is %s", tz_get_name());
    }
  }
  printf ("\n");
  if (mark && !iscurrent)
    printf ("Mark level:  %f\n", marklev);
}

#define MAXTIDESPERDAY 25
//#define DAYWIDTH 7
#define DAYWIDTH 11
static char **week = NULL;

/* Set one tide in the calendar week being constructed */
static void
do_one_calendar_tide (int dayofweek, int tidenumber, int event_type)
{
  int x;
  char desc, temp[DAYWIDTH];
//  double adjust = 0.0, mult = 1.0;
  win_assert (tidenumber < MAXTIDESPERDAY - 1);  /* Leave room for dates */
  win_assert (dayofweek < 7);
  if (iscurrent) {
    switch (event_type) {
    case 1:
      desc = 'E';
      break;
    case 2:
      desc = 'F';
      break;
    case 4:
    case 8:
      desc = 'S';
      break;
    default:
      win_assert (0);
    }
  } else {
    switch (event_type) {
    case 1:
      desc = 'L';
//      adjust = ltleveloff;
//      mult   = llevelmult;
      break;
    case 2:
      desc = 'H';
//      adjust = htleveloff;
//      mult   = hlevelmult;
      break;
    case 4:
      desc = 'F';
      break;
    case 8:
      desc = 'R';
      break;
    default:
      win_assert (0);
    }
  }
  week[1+tidenumber][dayofweek*DAYWIDTH] = desc;
  for (x=0;x<(int)(strlen(next_ht_text));x++)
    week[1+tidenumber][dayofweek*DAYWIDTH+x+1] = next_ht_text[x];
  /* This will die horribly if the amplitude is too big. */
  sprintf (temp, "% 5.1f", time2asecondary (next_ht_adj));
  for (x=0;x<(int)(strlen(temp));x++)
    week[1+tidenumber][dayofweek*DAYWIDTH+x+5] = temp[x];
}

/* Do one day for a calendar.  next_ht is expected to be set to the first
   tide event of the day.  Stop is expected to be the midnight that is the
   end of the day.  event_type is expected to be maintained for next_ht. */
static void
do_calendar_day (time_t stop, int dayofweek, int *maxline, int *event_type)
{
  int tidenumber = 0, x, moon;
  time_t moon_time;
  char date_str[DAYWIDTH+1];

  while (next_ht_adj < stop) {

    /* Add date stamp from first tide (hope we get one) */
    if (tidenumber == 0) {
      strcpy(date_str,"   ");
      moon_time = prev_day(next_ht_adj);
      next_moon_phase(&moon_time, -1);
      if (moon_phase(&moon_time, &moon, next_ht_adj)) {
        if      (moon == 0) strcpy(date_str,"New ");
        else if (moon == 1) strcpy(date_str,"FQtr ");
        else if (moon == 2) strcpy(date_str,"Full ");
        else if (moon == 3) strcpy(date_str,"LQtr ");
      }
      strcat(date_str, next_ht_date);
      for (x=0;x<(int)(strlen(date_str));x++)
        week[0][dayofweek*DAYWIDTH+x+0] = date_str[x];
    }

    /* Refer to list_tides to understand why this is like this. */
    if ((*event_type & 4) && (*event_type & 1))
      do_one_calendar_tide (dayofweek, tidenumber++, 4);
    if ((*event_type & 8) && (*event_type & 2))
      do_one_calendar_tide (dayofweek, tidenumber++, 8);
    if (*event_type & 1)
      do_one_calendar_tide (dayofweek, tidenumber++, 1);
    if (*event_type & 2)
      do_one_calendar_tide (dayofweek, tidenumber++, 2);
    if ((*event_type & 4) && !(*event_type & 1))
      do_one_calendar_tide (dayofweek, tidenumber++, 4);
    if ((*event_type & 8) && !(*event_type & 2))
      do_one_calendar_tide (dayofweek, tidenumber++, 8);

    if (1+(tidenumber-1) > *maxline)
      *maxline = 1+(tidenumber-1);

    *event_type = update_high_tide ();
  }
}

/* Do one week for a calendar.  next_ht is expected to be set to the first
   tide event of the week.  event_type is expected to be maintained for
   next_ht.  start is expected to begin as the midnight that starts the
   week, and is updated to the following week by this routine. */
static void
do_calendar_week (int *event_type, time_t *start)
{
  int x, y, maxline = 0, dayofweek;
  for (x=0;x<DAYWIDTH*7;x++)
//    for (y=0;y<MAXTIDESPERDAY*2;y++)
    for (y=0;y<MAXTIDESPERDAY;y++)
      week[y][x] = ' ';
  for (dayofweek=0;dayofweek<7;dayofweek++) {
    *start = increment_day (*start);
    /* This "start" becomes the "stop" for do_calendar_day */
    do_calendar_day (*start, dayofweek, &maxline, event_type);
  }
  for (y=0;y<=maxline;y++) {
    for (x=0;x<DAYWIDTH*7;x++)
//      putchar (week[y][x]);
      week[y][DAYWIDTH*7-1] = '\0';
    printf ("%s\n", week[y]);
  }
}

/* Do a calendar. */
void
do_calendar ()
{
  time_t start = sunday_month (faketime), midmonth_t;
  int month = tmtime(faketime)->tm_mon, looper;
  int event_type, howmany = num_months;
  float level, highest = -1000, lowest = 1000;
  char moyr[64];

  if (!week) {
    /* Allocate storage */
    win_assert (week = (char **) malloc (MAXTIDESPERDAY* sizeof (char *)));
    for (looper=0;looper<MAXTIDESPERDAY;looper++) {
     win_assert (week[looper] = (char *) malloc (DAYWIDTH*7*sizeof(char)));
    }
  }

  /* Clumsy calibration step, init next_ht and event_type */
  win_assert (start > DAYSECONDS*1.5);
  next_ht = start - DAYSECONDS*1.5;
  prev_ht_adj = next_ht_adj = 0;
  while (next_ht_adj < start)
    event_type = update_high_tide ();

  while (next_ht_adj < start + 31*DAYSECONDS) {
    level = time2asecondary (next_ht_adj);
    if (highest < level) highest = level;
    if (lowest  > level) lowest  = level;
    event_type = update_high_tide ();
  }

  do_tty_header ();
  midmonth_t = start+DAYSECONDS*15;
  strftime(moyr, sizeof(moyr),"%B %Y", gmtime(&midmonth_t));
  printf("%s low is %1.1f%s, high is %1.1f%s, range is %1.1f%s.\n",
    moyr, lowest, units_abbrv, highest, units_abbrv, (highest-lowest), units_abbrv);
  printf("Historical low is %1.1lf%s, high is %1.1lf%s, range is %1.1lf%s.\n",
    absmin, units_abbrv, absmax, units_abbrv, (absmax-absmin), units_abbrv);

  do {
    printf ("\n  Sunday     Monday    Tuesday   Wednesday   Thursday   Friday     Saturday\n");

    next_ht = start - DAYSECONDS*1.5;
    prev_ht_adj = next_ht_adj = 0;
    while (next_ht_adj < start)
      event_type = update_high_tide ();

    do {
      printf ("\n");
      do_calendar_week (&event_type, &start);
    } while (tmtime(start)->tm_mon != (month+1)%12);

    month = tmtime(start)->tm_mon;
    start = sunday_month (start);
    if (howmany > 1)
      printf("\n");
  }
  while (--howmany > 0);

  /* Added by mgh to clean up */
  for (looper=0;looper<MAXTIDESPERDAY;looper++)
    free (week[looper]);
  free(week);
  week = NULL;
}

/* List several days of high and low tides to standard output.  This
got ugly because of the need to make things come out in the right order
with the enhanced mark code. */
/* mgh changed "number of tides" to "number of days", added moon phase */
void
list_tides ()
{
  int a, event_type, moon;
  char *high, *low, *rise, *fall, prev_ht_date[80], *s_moon_phase;
  char daynm[64];
  time_t moon_time;
  next_ht = prev_day(faketime); // mgh: Start with today's first tide
  if (iscurrent) {
    high =        "  Max Flood";
    low =         "    Max Ebb";
    rise = fall = "Slack Water";
  } else {
    high =        "  High Tide";
    low =         "   Low Tide";
    rise =        "     Rising";
    fall =        "    Falling";
  }
  do_tty_header ();

  if (text < 1)
    text = 21;
  prev_ht_date[0] = '\0';
  a = text+1;
  moon_time = next_ht;
  next_moon_phase(&moon_time, -1);
  while (a > 0) {
    event_type = update_high_tide ();
    if (strcmp(next_ht_date, prev_ht_date)) { // Added mgh for DAY breaks and moon
      if (--a) {
        strftime (daynm, 64, "%A", tmtime (next_ht_adj)); // mgh 2/9/98 was (next_ht)
        s_moon_phase = "";
        if (moon_phase(&moon_time, &moon, next_ht_adj)) {
          if      (moon == 0) s_moon_phase = "New Moon";
          else if (moon == 1) s_moon_phase = "First Quarter Moon";
          else if (moon == 2) s_moon_phase = "Full Moon";
          else if (moon == 3) s_moon_phase = "Last Quarter Moon";
        }
        printf("\n%s %s   %s\n", daynm, next_ht_date, s_moon_phase);

        if (sun_moon) {
           char s_sun[80], s_moon[80];
           s_sunrise_set(s_sun, next_ht_adj);
           printf("%s\n", s_sun);

           s_moonrise_set(s_moon, next_ht_adj);
           printf("%s\n", s_moon);
        }
      }
      strcpy(prev_ht_date, next_ht_date);
    }
    if (a) {
       if ((event_type & 4) && (event_type & 1)) {
         printf ("%s:  %s % 6.2f\n", fall, next_ht_text,
         time2asecondary (next_ht_adj));
       }
       if ((event_type & 8) && (event_type & 2)) {
         printf ("%s:  %s % 6.2f\n", rise, next_ht_text,
         time2asecondary (next_ht_adj));
       }
       if (event_type & 1) {
         printf ("%s:  %s ", low, next_ht_text);
         printf ("% 6.2f\n", time2asecondary (next_ht_adj));
       }
       if (event_type & 2) {
         printf ("%s:  %s ", high, next_ht_text);
         printf ("% 6.2f\n", time2asecondary (next_ht_adj));
       }
       if ((event_type & 4) && !(event_type & 1)) {
         printf ("%s:  %s % 6.2f\n", fall, next_ht_text,
         time2asecondary (next_ht_adj));
       }
       if ((event_type & 8) && !(event_type & 2)) {
         printf ("%s:  %s % 6.2f\n", rise, next_ht_text,
         time2asecondary (next_ht_adj));
       }
    }
  }
  printf("\n");
}

/* List incremental tides */
void
do_incremental ()
{
  int a, incr;
  char timetx[20],datetx[20], prev_date[20];
  struct tm *t;
  time_t incr_time = prev_day(faketime); // mgh: Start with midnight

  do_tty_header ();

  if (incremental_tides > 0)
    incr = incremental_tides;
  else
    incr = increment_step;

  a = num_days+1;
  while (a > 0) {
    t = tmtime( incr_time );
    do_timestamp(timetx, t );
    do_datestamp(datetx, t );
    if (strcmp(datetx, prev_date)) {
      if (--a) {
        strcpy(prev_date, datetx);
        printf("\n");
      }
    }
    else datetx[0] = '\0';
    if (a)
      printf("% 6.2f %s %s\n", time2asecondary(incr_time), timetx, datetx);
    incr_time += incr * 60;
  }
}


/**********************************************************************/


/* Banner mode */
#define BANNERWIDTH 68
#define BANNERSTART 11
void
do_banner ()
{
  int a, b, c, markchar, middlechar, mllwchar, event_type;
  struct tm *t;
  time_t curtime;
  char bdate[6], btime[5], buf[80], tbuf[80];
  if (text < 1)
    text = 21;
  curtime = next_ht = faketime;
  do_tty_header();
  /* Initialize the amplitude. */
  happy_new_year (yearoftimet (faketime));
  if (mark) {
    markchar = (int)((double)BANNERWIDTH * (((marklev - fakedatum)
            / fakeamplitude) * 0.5 + 0.5)) + BANNERSTART;
    if (markchar < BANNERSTART || markchar > 78)
      mark = 0;
  }
  /* Needed if mark turned off */
  else if (iscurrent) {
    markchar = (int)((double)BANNERWIDTH * (((-fakedatum) / fakeamplitude)
           * 0.5 + 0.5)) + BANNERSTART;
    win_assert (markchar >= BANNERSTART && markchar <= 78);
  }
  if (middle)
    middlechar = BANNERWIDTH / 2 + BANNERSTART;
  if (mllw) {
    mllwchar =  (int)((double)BANNERWIDTH * (((-fakedatum) / fakeamplitude)
           * 0.5 + 0.5)) + BANNERSTART;
    if (mllwchar < BANNERSTART || mllwchar > 78)
      mllw = 0;
  }
  buf[79] = '\0';
  event_type = update_high_tide ();
  for (a=0;a<text;a++) {
    t = tmtime (curtime);
    strftime (btime, 5, "%H%M", t);
    strftime (bdate, 6, "%m-%d", t);
    b = (int)((double)BANNERWIDTH * (time2secondary(curtime) * 0.5 + 0.5))
      + BANNERSTART;
    if (b < BANNERSTART)
      b = BANNERSTART;
    if (b > 78)
      b = 78;
    sprintf (buf, "%s %s ", bdate, btime);
    for (c=BANNERSTART;c<79;c++)
      buf[c] = ' ';
    if (linegraph)
      buf[b] = '*';
    else {
      if (iscurrent) {
        buf[markchar] = '|'; /* Needed if -mark turned off */
        if (b < markchar)
          for (c=b;c<markchar;c++)
            buf[c] = '*';
        else
          for (c=markchar+1;c<=b;c++)
            buf[c] = '*';
      } else {
        for (c=BANNERSTART;c<=b;c++)
          buf[c] = '*';
      }
    }
    if (mark)
      buf[markchar] = '|';
    if (middle)
      buf[middlechar] = '|';
    if (mllw)
      buf[mllwchar] = '|';
    if (curtime+(tstep>>1) >= next_ht_adj) {
      if (event_type & 1) {
         sprintf (tbuf, "%s %0.2f", next_ht_text,
            ltleveloff + llevelmult * time2atide (next_ht));
      }
      else if (event_type & 2) {
         sprintf (tbuf, "%s %0.2f", next_ht_text,
            htleveloff + hlevelmult * time2atide (next_ht));
      }
      else {
        sprintf (tbuf, "%s %0.2f", next_ht_text, time2atide (next_ht));
      }
      strncpy (buf+BANNERSTART+1, tbuf, strlen (tbuf));
      while (curtime+(tstep>>1) >= next_ht_adj)
        event_type = update_high_tide ();
    }
    puts (buf);
    curtime += tstep;
  }
}

/* Stats */
void
do_stats (time_t time1, time_t time2)
{
  time_t start, end, maximum, minimum, looper;
  double mean = 0.0, mintide, maxtide;
  long progress;
  if (time1 < time2) {
    start = time1;
    end = time2;
  } else {
    start = time2;
    end = time1;
  }
  if (end != start)
    progress = (long)(end-start) / 20;
  else
    progress = 1;
  for (looper=start;looper<=end;looper+=60) {
    double ct = time2atide (looper);
    if (looper == start) {
      maximum = minimum = looper;
      mintide = maxtide = ct;
    } else {
      if (ct < mintide) {
        minimum = looper;
        mintide = ct;
      }
      if (ct > maxtide) {
        maximum = looper;
        maxtide = ct;
      }
    }
    mean += ct;
    if ((((long)(looper - start)) % progress) < 60) {
      printf ("%ld%% done\r", (((long)(looper - start)) / progress) * 5);
      fflush (stdout);
    }
  }
  mean /= (((double)(end-start))/60.0) + 1.0;
  printf ("\n\n");
  do_tty_header();
  printf ("Stats from %s",
    do_long_timestamp (tmtime (start)));
  printf (" to %s\n",
    do_long_timestamp (tmtime (end)));
  if (iscurrent) {
    printf ("Maximum ebb current was %f at %s\n", mintide,
      do_long_timestamp (tmtime (minimum)));
    printf ("Maximum flood current was %f at %s\n", maxtide,
      do_long_timestamp (tmtime (maximum)));
    printf ("Mean permanent current was %f\n", mean);
  } else {
    printf ("Minimum tide level was %f at %s\n", mintide,
      do_long_timestamp (tmtime (minimum)));
    printf ("Maximum tide level was %f at %s\n", maxtide,
      do_long_timestamp (tmtime (maximum)));
    printf ("Mean tide level was %f\n", mean);
  }
}

/* Raw output. */
void
do_raw (time_t time1, time_t time2, int step)
{
  time_t start, end;
  /* char abbrvtime[13]; */
  win_assert (step > 0);
  if (time1 < time2) {
    start = time1;
    end = time2;
  } else {
    start = time2;
    end = time1;
  }
  for (; start<=end; start+=(time_t)step) {
    /* strftime (abbrvtime, 13, "%Y%m%d%H%M", tmtime (start)); */
    /* printf ("%s %f\n", abbrvtime, time2atide (start)); */
    printf ("%lu %f\n", (unsigned long)start, time2asecondary (start));
  }
}


/**********************************************************************/


/* ASCII graph mode was done by Jef Poskanzer. */
/* Generate an ASCII graph of the tides. */
#define TEXTHEIGHT 14
#define TTEXTHEIGHT 10
static char textmap[TEXTWIDTH*TEXTHEIGHT];
static void t_setchr (int x, int y, char chr) {
  if (x < 0 || y < 0 || x >= TEXTWIDTH || y >= TEXTHEIGHT)
    return;
  textmap[y*TEXTWIDTH+x] = chr;
}
static char t_getchr (int x, int y) {
  if (x < 0 || y < 0 || x >= TEXTWIDTH || y >= TEXTHEIGHT)
    return '\0';
  return textmap[y*TEXTWIDTH+x];
}
static double t_tide2wl (double tide) {
  return ((double)TTEXTHEIGHT) * (-tide) * 0.5 + (double)TTEXTHEIGHT * 0.5;
}
static double t_wl2tide (int y) {
  return -2.0*(y - TTEXTHEIGHT * 0.5) / ((double)TTEXTHEIGHT);
}
static void t_putstr (int x, int y, char *s) {
  int a;
  for (a=0;a<(int)(strlen(s));a++)
    t_setchr (x+a, y, s[a]);
}
static void t_center_text (int x, int y, char *text)
{
  int l = strlen (text);
  t_putstr (x - l / 2, y, text);
}
static void t_write_tide (int slope, int x)
{
  if (slope) {
    t_center_text (x, TTEXTHEIGHT, next_ht_date);
    t_center_text (x, TTEXTHEIGHT+1, next_ht_text);
  } else {
    t_center_text (x, TTEXTHEIGHT+2, next_ht_date);
    t_center_text (x, TTEXTHEIGHT+3, next_ht_text);
  }
}
void
tide2ascii (void)
{
  int x, y, event_type, midwl;
  time_t start = faketime, this_time = time(NULL), finish;

  do_tty_header();

  /* Set the background. */
  for (x=0;x<TEXTWIDTH;x++)
    for (y=0;y<TEXTHEIGHT;y++)
      t_setchr (x, y, ' ');

  /* Initialize the amplitude. */
  happy_new_year (yearoftimet (faketime));

  /* Draw day separators. */
  if (lines) {
    time_t day;
    for (day=prev_day(start);day<start+TEXTWIDTH*tstep;
                             day=increment_day(day)) {
      x = (int)((day - start) / tstep + 0.5);
      for (y=0;y<TEXTHEIGHT;y++)
        t_setchr (x, y, '|');
    }
  }

  /* Draw the tides. */
  midwl = (int)(t_tide2wl (-fakedatum / fakeamplitude) + 0.5);
  for (x=0;x<TEXTWIDTH;x++) {
    int yy = (int)(t_tide2wl (time2secondary (faketime)) + 0.5);
    if (linegraph) {
      t_setchr (x, yy, '*');
    } else {
      if (iscurrent) {
        t_setchr (x, midwl, '-'); /* Needed when -mark turned off */
        if (yy < midwl)
          for (y=midwl-1;y>=yy;y--)
            t_setchr (x, y, '*');
        else
          for (y=midwl+1;y<=yy;y++)
            t_setchr (x, y, '*');
      } else {
        for (y=TTEXTHEIGHT-1;y>=yy;y--)
          t_setchr (x, y, '*');
      }
    }
    faketime += tstep;
  }

  /* Extra lines */
  if (mark) {
    y = (int) (t_tide2wl ((marklev - fakedatum) / fakeamplitude) + 0.5);
    for (x=0;x<TEXTWIDTH;x++)
      if (t_getchr (x, y) == '*' || iscurrent)
        t_setchr (x, y, '-');
  }
  if (middle) {
    y = (int) (t_tide2wl (0.0) + 0.5);
    for (x=0;x<TEXTWIDTH;x++)
      if (t_getchr (x, y) == '*')
        t_setchr (x, y, '-');
  }
  if (mllw) {
    y = (int) (t_tide2wl (-fakedatum / fakeamplitude) + 0.5);
    for (x=0;x<TEXTWIDTH;x++)
      if (t_getchr (x, y) == '*')
        t_setchr (x, y, '-');
  }

  if (now) {
    /* X marks the spot */
    x = (int)((this_time - start) / tstep);
    y = (int) (t_tide2wl (time2secondary (this_time)) + 0.5);
    t_setchr (x, y, '+');
  }

  /* Scrawl timestamps */
  next_ht = start - MAX(abs(httimeoff),abs(lttimeoff));
  finish = start + TEXTWIDTH*tstep + MAX(abs(httimeoff),abs(lttimeoff));
  event_type = update_high_tide ();
  while (next_ht < finish) {
    t_write_tide ((event_type&2), (int)((next_ht_adj - start) / tstep + 0.5));
    event_type = update_high_tide ();
  }

  /* Write output. */
  for (y=TTEXTHEIGHT;y<TTEXTHEIGHT+2;y++) {
    printf("[");
    for (x=0;x<TEXTWIDTH;x++)
      putchar(t_getchr(x,y));
    printf("]\n");
  }
  for (y=0;y<TTEXTHEIGHT;y++) {
    double temp;
    printf("[");
    for (x=0;x<TEXTWIDTH;x++)
      putchar(t_getchr(x,y));
    temp = t_wl2tide(y) * fakeamplitude + fakedatum;
    /* Chop off minus sign for super-low tides to save space */
    if (temp <= -10.0)
      temp = -temp;
    printf("]%0.1f\n", temp);
  }
  for (y=TTEXTHEIGHT+2;y<TTEXTHEIGHT+4;y++) {
    printf("[");
    for (x=0;x<TEXTWIDTH;x++)
      putchar(t_getchr(x,y));
    printf("]\n");
  }
}