{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}


Unit pFIBDataSet;

interface
{$I FIBPlus.inc}

uses
 {$IFDEF MSWINDOWS}
   Windows, Messages, SysUtils, Classes,
   Controls, Forms, Dialogs, StdFuncs,
   DB,ibase,IB_Intf, ib_externals,fib,FIBMiscellaneous,  pFIBDataBase,
   FIBDataBase,FIBDataSet,  FIBQuery,   pFIBQuery,DSContainer,pFIBLists,
   {$IFDEF D6+}
     FMTBcd,Variants,
   {$ENDIF}
   {$IFDEF FOR_ALL} pFIBProps {$ELSE}sbFIBProps {$ENDIF};
 {$ENDIF}
 {$IFDEF LINUX}
   Types, SysUtils, Classes, QControls, QForms, QDialogs, StdFuncs,
   DB,ibase, IB_Intf, IB_Externals, fib, FIBMiscellaneous, pFIBDataBase,
   FIBDataBase, FIBDataSet, FIBQuery, pFIBLists,
   pFIBQuery, DSContainer, FMTBcd, Variants,{$IFDEF FOR_ALL} pFIBProps {$ELSE}sbFIBProps {$ENDIF};
 {$ENDIF}

type



  TpSQLKind      =(skModify, skInsert, skDelete,skRefresh);
  TUpdateKinds  = set of TUpdateKind;


  TLockStatus = (lsSuccess,lsDeadLock,lsNotExist,lsMultiply,lsUnknownError);
  TLockErrorEvent=
   procedure(DataSet: TDataSet;LockError:TLockStatus;
    var ErrorMessage :string ;var Action: TDataAction) of object;

  TCachRefreshKind=(frkEdit,frkInsert);
  TOnGetSQLTextProc = procedure (DataSet:TFIBDataSet; var SQLText:string) of object;

  TFilteredCacheInfo =
  record
   AllRecords      :integer;
   FilteredRecords :integer;
   NonVisibleRecords:TSortedList;
  end;

  TpFIBDataSet = class(TFIBDataSet)
  private
   vQryRecordCount:TFIBQuery;
   vLockQry       :TFIBQuery;
   FDataSet_ID:integer;
   FDefaultFormats:TFormatFields;
   FOnLockError   :TLockErrorEvent;
   FAllowedUpdateKinds:TUpdateKinds;
   vUserOnPostError:TDataSetErrorEvent;
   vUserOnDeleteError:TDataSetErrorEvent;
// lists of UpdateObjects
   vUpdates:TList;
   vDeletes:TList;
   vInserts:TList;
   FFilteredCacheInfo:TFilteredCacheInfo;
   FContainer:TDataSetsContainer;
   FReceiveEvents:Tstrings;
   FOnUserEvent:TUserEvent;

   FAutoUpdateOptions:TAutoUpdateOptions;
   FHasUncommitedChanges:boolean;

   FAllRecordCount:integer;
   FDescription   :string;
   FOnAskRecordCount :TOnGetSQLTextProc;
   FFNFields :TStringList; 
   procedure SetReceiveEvents(Value:Tstrings);
   procedure SetContainer(Value:TDataSetsContainer);
   //Property access procedures
   function  GetSelectSQL:TStrings;
   procedure SetSelectSQL(Value: TStrings);


   // UpdateObjects support
   function  ListForUO(KindUpdate:TUpdateKind):TList;
   procedure SynchroOrdersUO(List:TList);

   //
   procedure PrepareQuery(KindQuery:TUpdateKind);
   procedure SetDataSet_ID(Value:Integer);
   function  GetFieldForTable(const Relation:string):TField;
   function  WillGenerateSQLs:boolean;
   function  GetFIBVersion : string;
   procedure SetFIBVersion(const vs:string);
  protected
    function  GetVisibleRecno  : Integer;
    procedure SetVisibleRecno( Value :Integer);
    function  GetRecNo: Integer; override;
    procedure SetRecNo(Value: Integer); override;
    procedure ShiftBuffer(const FromRecNo,Distance:integer); override;
    procedure SetPrepareOptions(Value:TpPrepareOptions); override;
    procedure DoOnPostError(DataSet: TDataSet; E: DB.EDatabaseError; var Action: TDataAction); override;
    procedure DoOnDeleteError(DataSet: TDataSet; E: DB.EDatabaseError; var Action: TDataAction);

    {$IFDEF D5+}
//     IProviderSupport
  protected
      FParams:  TParams;

      procedure PSSetCommandText(const CommandText: string); override;
      function  PSGetTableName: string; override;
      function  PSGetKeyFields: string; override;     
      function  PSGetQuoteChar: string; override;
      function  PSGetUpdateException(E: Exception; Prev: EUpdateError): EUpdateError; override;
      function  PSInTransaction: Boolean; override;
      function  PSIsSQLBased: Boolean; override;
      function  PSIsSQLSupported: Boolean; override;
      procedure PSStartTransaction; override;
      procedure PSEndTransaction(Commit: Boolean); override;

      procedure PSReset; override;
      function  PSUpdateRecord(UpdateKind: TUpdateKind; Delta: TDataSet): Boolean; override;
      function  PSExecuteStatement(const ASQL: string; AParams: TParams;
       ResultSet: Pointer = nil): Integer; override;
      procedure PSExecute; override;
      function  PSGetParams: TParams; override;
      procedure PSSetParams(AParams: TParams); override;

    {$ENDIF}


    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    function  GetCanModify: Boolean; override;
    function  RecordCountFromSrv:integer;
    function  IsVisible(Buffer: PChar): Boolean; override;
    procedure InternalPostRecord(Qry: TFIBQuery; Buff: Pointer);override;
    procedure InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);override;

    procedure  AddedFilterRecord(DataSet: TDataSet; var Accept: Boolean); virtual; //abstract

    procedure  SetFiltered(Value: Boolean); override;
    procedure  InternalDoBeforeOpen; override;
    procedure  DoBeforeOpen;    override;
    procedure  DoAfterOpen;     override;
    procedure  DoAfterClose;    override;
    procedure  DoBeforeInsert;  override;
    procedure  DoAfterInsert;   override;
    procedure  DoBeforeEdit;    override;
    procedure  DoBeforePost;    override;
    procedure  DoAfterPost;     override;
    procedure  DoBeforeCancel;  override;
    procedure  DoAfterCancel;   override;
    procedure  DoBeforeDelete;  override;
    procedure  DoOnNewRecord;   override;
    procedure  DoAfterDelete;   override;
    procedure  DoAfterEdit;     override;
    procedure  DoAfterScroll;   override;
    procedure  DoBeforeClose;   override;
    procedure  DoBeforeScroll;  override;
    procedure  DoOnCalcFields;  override;
    {$IFDEF D5+}
    procedure DoBeforeRefresh;  override;
    procedure DoAfterRefresh;  override;
    {$ENDIF}


    function   GetRecordCount: Integer; override;
    procedure  UpdateFieldsProps; virtual;
   {$IFNDEF D5+}
     function BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean; override;
     function CurrToBCD(const Curr: Currency; BCD: Pointer; Precision,
      Decimals: Integer): Boolean; override;
   {$ENDIF}


    procedure DoTransactionEnding(Sender:TObject);
    procedure DoTransactionEnded(Sender:TObject);
    procedure ClearModifFlags(Kind:byte);
    procedure CloseProtect;
    function  RaiseLockError(LockError:TLockStatus;ExceptMessage:string):TDataAction;

  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure MoveRecord(OldRecno,NewRecno:integer); override;
    function  VisibleRecordCount: Integer;     
    procedure Prepare;override;
    function  CanEdit  : Boolean; override;
    function  CanInsert: Boolean; override;
    function  CanDelete: Boolean; override;
    function  IsSequenced: Boolean; override;        // Scroll bar


    function  ExistActiveUO(KindUpdate:TUpdateKind):boolean;
     // Exist active Update objects

    function  AddUpdateObject(Value:TpFIBUpdateObject):integer;
    procedure RemoveUpdateObject(Value:TpFIBUpdateObject);
    procedure RefreshClientFields;


    function  ParamByName(const ParamName:string): TFIBXSQLVAR;
    function  FindParam (const ParamName:string): TFIBXSQLVAR;
    
    function  RecordStatus(RecNumber:integer):TUpdateStatus;
    procedure CloneRecord(SrcRecord:integer; IgnoreFields:array of const);
    procedure CloneCurRecord(IgnoreFields:array of const);
    // Cached Routine
    procedure CommitUpdToCach; // Clear CU buffers
    procedure ApplyUpdToBase;  // Send Updates to Base
    procedure CancelUpdates; override;
    procedure ApplyUpdates; 

   // SaveLoad Buffer functions

    procedure SaveToStream(Stream:TStream; SeekBegin :boolean );
    procedure LoadFromStream(Stream:TStream; SeekBegin :boolean );

    procedure SaveToFile(const FileName:string);
    procedure LoadFromFile(const FileName:string);

   // End SaveLoad

    function  LockRecord(RaiseErr:boolean):TLockStatus;
    function  FN(const FieldName:string):TField; //FindField
    function  FBN(const FieldName:string):TField; //FieldByName    

    function  ParamNameCount(const aParamName:string):integer;
    procedure ExecUpdateObjects(KindUpdate:TUpdateKind;Buff: Pointer;
     aExecuteOrder:TFIBOrderExecUO);

    procedure DoUserEvent(Sender:Tobject; const UDE:string;var Info:string); dynamic;
    procedure OpenWP(ParamValues: array of Variant);

    procedure AutoGenerateSQLText(ForState:TDataSetState);

    function  GenerateSQLText
     (const TableName,KeyFieldNames:string;SK:TpSQLKind):string;
    function  GenerateSQLTextWA
     (const TableName:string;SK:TpSQLKind):string; // Where All
    procedure GenerateSQLs;

//AutoUpdate operations
    function  KeyField:TField;
    function  SqlTextGenID:string;
    procedure IncGenerator;


    function  PrimaryKeyFields(const TableName:string):string;
    function  AllFields:string;

    procedure CacheModify(
     aFields: array of integer;Values: array of Variant;
     KindModify : byte
    ) ;

    procedure CacheEdit(aFields: array of integer;Values: array of Variant)  ;
    procedure CacheAppend(aFields: array of integer;Values: array of Variant);

    procedure CacheInsert(aFields: array of integer;Values: array of Variant);


   {$IFDEF CAN_OVERLOAD}
    {$IFNDEF VER120}
    overload;
    procedure CacheInsert(Value:Variant;DoRefresh:boolean =false);    overload;
    {$ENDIF}
   {$ENDIF}
    procedure CacheRefresh(FromDataSet:TDataSet;Kind:TCachRefreshKind
     ;FieldMap:Tstrings
    );
    procedure CacheRefreshByArrMap(
     FromDataSet:TDataSet;Kind:TCachRefreshKind;
     const SourceFields,DestFields:array of String
    );
    function  RecordFieldAsFloat(Field:TField;RecNumber:integer):Double;

 public

    property  HasUncommitedChanges:boolean read FHasUncommitedChanges write FHasUncommitedChanges;
    property  AllRecordCount:integer read FAllRecordCount;
    property VisibleRecno:integer read GetVisibleRecno write SetVisibleRecno;
 published

//Added properties
    property  Filtered;
    property  OnFilterRecord;

    property DeleteSQL: TStrings read GetDeleteSQL write  SetDeleteSQL;
    property UpdateSQL: TStrings read GetUpdateSQL write  SetUpdateSQL;
    property InsertSQL: TStrings read GetInsertSQL write  SetInsertSQL;
    property RefreshSQL:TStrings read GetRefreshSQL write SetRefreshSQL;
    property SelectSQL: TStrings read GetSelectSQL write  SetSelectSQL;


    property  DefaultFormats:TFormatFields read FDefaultFormats write FDefaultFormats;
    property OnPostError:TDataSetErrorEvent read vUserOnPostError   write vUserOnPostError;
    property OnDeleteError:TDataSetErrorEvent read vUserOnDeleteError   write vUserOnDeleteError;
    property OnLockError:TLockErrorEvent read FOnLockError write FOnLockError;
    property AllowedUpdateKinds:TUpdateKinds  read FAllowedUpdateKinds write FAllowedUpdateKinds
     default [ukModify,ukInsert,ukDelete];
    property OnUserEvent:TUserEvent   read FOnUserEvent write FOnUserEvent;
    property Container:TDataSetsContainer read  FContainer Write SetContainer;
    property ReceiveEvents:Tstrings read  FReceiveEvents write    SetReceiveEvents;
    property AutoUpdateOptions:TAutoUpdateOptions  read FAutoUpdateOptions write FAutoUpdateOptions;
    property DataSet_ID:integer read FDataSet_ID write SetDataSet_ID default 0;
    property Description:string read FDescription write FDescription;
    property OnAskRecordCount :TOnGetSQLTextProc read FOnAskRecordCount  write FOnAskRecordCount ;
    property About   :string read GetFIBVersion write SetFIBVersion stored false;
  end;


function FieldInArray(Field:TField; Arr: array of const):boolean;



implementation

uses
{$IFNDEF NO_MONITOR}
 FIBSQLMonitor,
{$ENDIF}

StrUtil,DBConsts,SqlTxtRtns,FIBConsts,pFIBDataInfo,Commonib,pFIBFieldsDescr,pFIBCacheQueries;

const
  SQLKindNames: array[TpSQLKind] of String = (
   'UpdateSQL',
   'InsertSQL',
   'DeleteSQL',
   'RefreshSQL'
  );

function UseFormat(const sFormat:string):string;
var pD:integer;
begin
 pD:=Pos('.',sFormat);
 if pD=0 then Result:=sFormat
 else
  Result:=Copy(sFormat,1,Pred(pd))+'.'
end;

//
constructor TpFIBDataSet.Create(AOwner: TComponent);
begin
 inherited Create(AOwner);
 FAllowedUpdateKinds:=[ukModify,ukInsert,ukDelete];
 FDefaultFormats:=TFormatFields.Create;
 inherited OnPostError  :=DoOnPostError;
 inherited OnDeleteError:=DoOnDeleteError;

 vUpdates:=TList.Create;
 vDeletes:=TList.Create;
 vInserts:=TList.Create;
 FFilteredCacheInfo.NonVisibleRecords:=TSortedList.Create;
 FFilteredCacheInfo.AllRecords       :=-1;

 FReceiveEvents     :=TStringList.Create;
 FAutoUpdateOptions :=TAutoUpdateOptions.Create(Self);

 FBase.OnTransactionEnding := DoTransactionEnding;
 FBase.OnTransactionEnded  := DoTransactionEnded;
 FHasUncommitedChanges     := false;
 FDataSet_ID               := 0;

 FAllRecordCount:=0;
 vQryRecordCount:=TFIBQuery.Create(Self);
 vQryRecordCount.ParamCheck:=true;
 vLockQry       :=TFIBQuery.Create(Self);
 vLockQry.ParamCheck:=true;
 {$IFDEF D5+}
  FParams := nil;
 {$ENDIF}

 FFNFields :=TStringList.Create;
 with FFNFields do
 begin
   Sorted:=true;
   Duplicates:=dupIgnore;
 end;  
end;

destructor TpFIBDataSet.Destroy;
begin
 FDefaultFormats.Free; vUpdates.Free; vDeletes.Free;
 vInserts.Free;        FReceiveEvents.Free;
 FFilteredCacheInfo.NonVisibleRecords.Free;
 
 FAutoUpdateOptions.Free;
 {$IFDEF D5+}
  if Assigned(FParams) then
    FParams.Free;
 {$ENDIF}
 FFNFields.Free;
 inherited Destroy;
end;


procedure TpFIBDataSet.Notification(AComponent: TComponent; Operation: TOperation); //override;
var i:integer;
begin
 inherited Notification(AComponent,Operation);
 case Operation of
  opRemove:  if (AComponent=FContainer)  then FContainer:=nil
             else
             if (AComponent is TField) and not (csDestroying in ComponentState) then
             begin
               i:=FFNFields.IndexOfObject(AComponent);
               if i<>-1 then FFNFields.Delete(i);
             end;
 end;
end;

function  TpFIBDataSet.GetVisibleRecno  : Integer;
var
 R:integer;
begin
  R:=inherited GetRecno;
  FFilteredCacheInfo.NonVisibleRecords.Find(R,Result);
  Result:=R-Result;
end;

procedure TpFIBDataSet.SetVisibleRecno(Value :Integer);
var
  R,R1:integer;
  Diff,i:integer;
begin
 if (FFilteredCacheInfo.NonVisibleRecords.Count=0) or
   (FFilteredCacheInfo.NonVisibleRecords[0]>Value)
 then
  inherited SetRecno(Value)
 else
 begin
   R:=FFilteredCacheInfo.NonVisibleRecords[0];
   R1:=R-1;
   for i:=1 to Pred(FFilteredCacheInfo.NonVisibleRecords.Count) do
   begin
    Diff:=FFilteredCacheInfo.NonVisibleRecords[i]-R-1;
    if Diff<>0 then
    begin
      if R1+Diff>Value then Diff:=Value-R1;
      Inc(R1,Diff);
      if R1=Value then
      begin
       R:=FFilteredCacheInfo.NonVisibleRecords[i-1]+Diff;
       Break
      end
    end;
    R:=FFilteredCacheInfo.NonVisibleRecords[i];
   end;
   if R1<Value then
   begin
    R:=FFilteredCacheInfo.NonVisibleRecords.LastItem+Value-R1
   end;
   inherited SetRecno(R)
 end;
end;

function  TpFIBDataSet.GetRecNo: Integer;
begin
 if poVisibleRecno in Options then
  Result := VisibleRecNo
 else
  Result := inherited GetRecNo
end;

procedure TpFIBDataSet.SetRecNo(Value: Integer);
begin
 if  poVisibleRecno in Options then
  SetVisibleRecno( Value)
 else
  inherited SetRecNo(Value)
end;

procedure TpFIBDataSet.MoveRecord(OldRecno,NewRecno:integer);
var Buff,BuffRec:PChar;
    i:integer;
    frI,toI:integer;
    Sign:ShortInt;
    IsCurRecno:boolean;
begin
  IsCurRecno:=Recno=OldRecno;
  if NewRecno<1 then NewRecno:=1;
  if NewRecno>FRecordCount then
  begin
    FetchAll;
    if NewRecno>FRecordCount then
     NewRecno:=FRecordCount;
  end;
  Dec(OldRecno);Dec(NewRecno);
  if OldRecno=NewRecno then Exit;
  if OldRecno>NewRecno then
  begin
   frI:=Pred(OldRecno);
   toI:=Pred(NewRecno);
   Sign:=1;
  end
  else
  begin
   frI:=Succ(OldRecno);
   toI:=Succ(NewRecno);
   Sign:=-1;
  end;
  Buff      :=AllocRecordBuffer;
  BuffRec   :=AllocRecordBuffer;
  ReadRecordCache(OldRecno, BuffRec, False);
 try
  i:=frI;
  while i<>toI do
  begin
   ReadRecordCache(i, Buff, False);
   PRecordData(Buff)^.rdRecordNumber := PRecordData(Buff)^.rdRecordNumber+Sign;
   WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
   i:=i-Sign
  end;
  FFilteredCacheInfo.NonVisibleRecords.IncValuesDiapazon(OldRecno+1,NewRecno+1 ,Sign);
  PRecordData(BuffRec)^.rdRecordNumber :=NewRecno;
  WriteRecordCache(PRecordData(BuffRec)^.rdRecordNumber, BuffRec);
  SynchronizeBuffers(true,False);
 finally
   FreeRecordBuffer(Buff);
   FreeRecordBuffer(BuffRec);
   if IsCurRecno then Recno:=NewRecno+1
   else Recno:=Recno+Sign
 end;
end;

function  TpFIBDataSet.VisibleRecordCount: Integer;
var
  Buff:PChar;
  i   :integer;
  OldCurrentRec:integer;
  begin
  if not Filtered then
   Result:=RecordCount
  else
  begin
   Result:=0;
   Buff   :=AllocRecordBuffer;
   OldCurrentRec:=FCurrentRecord;
   try
     if FFilteredCacheInfo.AllRecords<>FRecordCount then
     begin
       for i:=0 to Pred(FRecordCount) do
       begin
        FCurrentRecord:=i;
        ReadRecordCache(FCurrentRecord, Buff, False);
        if  IsVisible(Buff) then Inc(Result)
       end;
      FFilteredCacheInfo.AllRecords:=FRecordCount;
      FFilteredCacheInfo.FilteredRecords:=Result
     end
     else
     begin
      Result:=FFilteredCacheInfo.FilteredRecords     
     end
   finally
     FCurrentRecord:=OldCurrentRec;
     FreeRecordBuffer(Buff);
   end;
  end;
end;

procedure TpFIBDataSet.ShiftBuffer(const FromRecNo,Distance:integer);
begin
 inherited ShiftBuffer(FromRecNo,Distance);
 FFilteredCacheInfo.NonVisibleRecords.IncValues(FromRecNo ,Distance)
end; 

procedure TpFIBDataSet.DoTransactionEnding(Sender:TObject);
begin
 FHasUncommitedChanges:=FHasUncommitedChanges and
  Assigned(UpdateTransaction) and
  (UpdateTransaction.State in [tsDoRollBack,tsDoRollBackRetaining]);
 inherited DoTransactionEnding(Sender);
end;

procedure TpFIBDataSet.DoTransactionEnded(Sender:TObject);
begin
 inherited DoTransactionEnded(Sender);
 if Assigned(UpdateTransaction) then
 if UpdateTransaction.State in  [tsDoCommitRetaining,tsDoRollBackRetaining]
 then
  if  (poProtectedEdit in Options) and not CachedUpdates
   and UpdateTransaction.IsReadCommitedTransaction
  then
    CloseProtect;  
end;

function TpFIBDataSet.GetCanModify: Boolean; //override;
begin
 Result := CanEdit or CanInsert or  CanDelete
end;

function TpFIBDataSet.CanEdit: Boolean; //override;
begin
 Result:=((inherited CanEdit or ExistActiveUO(ukModify)
          or WillGenerateSQLs
          or(CachedUpdates and Assigned(OnUpdateRecord))

          )
          and  (ukModify in FAllowedUpdateKinds)
         )
 or vInCachRefresh
end;

function TpFIBDataSet.CanInsert: Boolean; //override;
begin
 Result:=((inherited CanInsert or ExistActiveUO(ukInsert)
          or WillGenerateSQLs
          or (CachedUpdates and Assigned(OnUpdateRecord))
          )
          and (ukInsert in FAllowedUpdateKinds)
         )
  or vInCachRefresh
end;

function TpFIBDataSet.CanDelete: Boolean; //override;
begin
 Result:=(inherited CanDelete or ExistActiveUO(ukDelete) 
          or WillGenerateSQLs
          or (CachedUpdates and Assigned(OnUpdateRecord))
          )
   and (ukDelete in FAllowedUpdateKinds)
end;


function  TpFIBDataSet.IsSequenced: Boolean; //override;        // Scroll bar
begin
 Result:=inherited IsSequenced;
 if not Result then
  if not Filtered or not (poVisibleRecno in Options) then
   Result:=FAllRecordCount<>0
  else
   Result:=AllFetched
end;


// UpdateObjects support
function TpFIBDataSet.ListForUO(KindUpdate:TUpdateKind):TList;
begin
 Result:=nil;
 case KindUpdate of
  ukModify: Result:=vUpdates;
  ukInsert: Result:=vInserts;
  ukDelete: Result:=vDeletes;
 end;
end;

function  TpFIBDataSet.ExistActiveUO(KindUpdate:TUpdateKind):boolean;
var List:TList;
    i,lc:integer;

begin
  List:=ListForUO(KindUpdate);
  Result:=false;
  if List=nil then Exit;
  lc:=Pred(List.Count) ;
  for i:=0 to lc do
  begin
    Result:=TpFIBUpdateObject(List[i]).Active;
    if Result then Exit;
  end; 
end;

procedure TpFIBDataSet.SynchroOrdersUO(List:TList);
var i,lc:integer;
begin
 lc:=Pred(List.Count);
 with List do
 for i:=0 to lc do
 begin
   TpFIBUpdateObject(List[i]).ChangeOrderInList(i);
 end;
end;

function   TpFIBDataSet.AddUpdateObject(Value:TpFIBUpdateObject):integer;
var List:TList;
    OldPos:integer;
begin
 Result:=-1;
 if Value=nil then Exit;
 List:=ListForUO(Value.KindUpdate);
 if List=nil then Exit;
 with List do
 begin
  OldPos:=IndexOf(Value);
  if OldPos=-1 then
  begin
   if Value.OrderInList<Count then
    Insert(Value.OrderInList,Value)
   else
    Add(Value);
  end
  else
  begin
   if Value.OrderInList<Count then
     Move(OldPos,Value.OrderInList)
   else
     Move(OldPos,Pred(Count))
  end;
 end;
 SynchroOrdersUO(List);
 Result:= List.IndexOf(Value)
end;

procedure  TpFIBDataSet.RemoveUpdateObject(Value:TpFIBUpdateObject);
var List:TList;
begin
 if Value=nil then Exit;
 List:=ListForUO(Value.KindUpdate);
 if List<>nil then
 begin
  List.Remove(Value);
  SynchroOrdersUO(List);
 end;
end;

/// Execute UpdateObjects

{$WARNINGS OFF}
procedure TpFIBDataSet.ExecUpdateObjects(KindUpdate:TUpdateKind;Buff: Pointer;
 aExecuteOrder:TFIBOrderExecUO
);
var List:TList;
    i:integer;
begin
 List:=ListForUO(KindUpdate);
 if List.Count>0 then
 begin
   UpdateBlobInfo(Buff,true,False);
   for i:=0 to Pred(List.Count) do
    with TpFIBUpdateObject(List[i]) do
     if Active and (ExecuteOrder=aExecuteOrder) and (not EmptyStrings(SQL)) then
       begin
         SetQueryParams(TpFIBUpdateObject(List[i]),Buff);
         ExecQuery;
       end;
 end;
end;
{$WARNINGS ON}
///

procedure TpFIBDataSet.SetDataSet_ID(Value:Integer);
begin
 if FDataSet_Id=Value then Exit; 
 CheckDataSetClosed(' change DataSetID ');
 FDataSet_Id:=Value;
 FPrepared:=false;
 if Value>0 then
   PrepareOptions:=PrepareOptions+[psApplyRepositary]
end;

procedure TpFIBDataSet.PrepareQuery(KindQuery:TUpdateKind);
var Qry:TFIBQuery;
begin
 case KindQuery of
  ukModify: Qry:=QUpdate;
  ukInsert: Qry:=QInsert;
 else
//  ukDelete:
  Qry:=QDelete;
 end;
 try
  if not EmptyStrings(Qry.SQL) and (not Qry.Prepared)
  and (Qry.SQL[0]<>'No Action')
  then
  begin
   AutoStartUpdateTransaction;  
   Qry.Prepare;
  end;
 except
  on E: Exception do
   if (E is EFIBInterbaseError) and (EFIBInterbaseError(E).sqlcode = -551) and
    not (csDesigning in ComponentState) then
    begin
     FAllowedUpdateKinds:=FAllowedUpdateKinds -[KindQuery];
     Abort;
    end
   else  Raise;
 end;
end;

{$WARNINGS OFF}
procedure TpFIBDataSet.Prepare;//override;
var
  iCurScreenState: Integer;
begin
 if FSQLScreenCursor <> crDefault then
 begin
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FSQLScreenCursor;
 end;
  try
   FBase.CheckDatabase;
   StartTransaction;
   FBase.CheckTransaction;
   if (psApplyRepositary in PrepareOptions) and  (DataSet_ID>0) then
   begin
    ListDataSetInfo.LoadDataSetInfo(Self);
    if Conditions.Applied then  Conditions.Apply;
   end;

      if not EmptyStrings(FQSelect.SQL) then
      begin
        if not FQSelect.Open then
         FQSelect.Prepare;
        if csDesigning in ComponentState then
        begin
         PrepareQuery(ukModify);
         PrepareQuery(ukInsert);
         PrepareQuery(ukDelete);
        end;
        FPrepared := True;
        InternalInitFieldDefs;
      end
      else
        FIBError(feEmptyQuery, ['Prepare '+CmpFullName(Self)]);
  finally
     if FSQLScreenCursor <> crDefault then
      Screen.Cursor := iCurScreenState;
  end;
end;
{$WARNINGS ON}

function   TpFIBDataSet.GetRecordCount: Integer; 
begin
  Result:=inherited GetRecordCount;
  if Filtered and (poVisibleRecno in Options)  then
    Result:=VisibleRecordCount
  else
   if Result<FAllRecordCount-FDeletedRecords then Result:=FAllRecordCount-FDeletedRecords
end;

procedure TpFIBDataSet.InternalPostRecord(Qry: TFIBQuery; Buff: Pointer);//override;
begin
  if vInCachRefresh then Exit;
  if Qry.SQL.Count>0 then
   if Qry.SQL[0]='No Action' then Exit;
  AutoStartUpdateTransaction; 
  UpdateBlobInfo(Buff,true,False);
  if Qry=QInsert then
   ExecUpdateObjects(ukInsert,Buff,oeBeforeDefault)
  else
   ExecUpdateObjects(ukModify,Buff,oeBeforeDefault);

  CheckDataSetOpen(' continue post ') ;

  if not EmptyStrings(Qry.SQL) then
  begin
   SetQueryParams(Qry, Buff);
   if Qry.Open then Qry.Close;
   Qry.ExecQuery;
   if  Qry.SQLType in  [SQLSelect,SQLExecProcedure] then
   begin
    InternalRefreshRow(Qry,Buff);
   end;
  end;
  if Qry=QInsert then
   ExecUpdateObjects(ukInsert,Buff,oeAfterDefault)
  else
   ExecUpdateObjects(ukModify,Buff,oeAfterDefault);

  PRecordData(Buff)^.rdUpdateStatus := usUnmodified;
  PRecordData(Buff)^.rdCachedUpdateStatus := cusUnmodified;
  SetModified(False);
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
  if not FCachedUpdates then
  begin
   CopyRecordBuffer(Buff, FOldBuffer);  
   AutoCommitUpdateTransaction;
  end;

  if not EmptyStrings(FQRefresh.SQL) and (poRefreshAfterPost in Options) then
  begin
   {$IFDEF VER130}
    DoBeforeRefresh;
   {$ENDIF}
    InternalRefreshRow(FQRefresh,Buff);
   {$IFDEF VER130}
    DoAfterRefresh;
   {$ENDIF}
  end;
  FHasUncommitedChanges:=Transaction.State=tsActive;
end;

procedure TpFIBDataSet.InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);//override;
begin
  AutoStartUpdateTransaction;
  ExecUpdateObjects(ukDelete,Buff,oeBeforeDefault);
  if not EmptyStrings(Qry.SQL) then
  begin
   SetQueryParams(Qry, Buff);
   Qry.ExecQuery;
  end;
  ExecUpdateObjects(ukDelete,Buff,oeAfterDefault);
  with PRecordData(Buff)^ do
  begin
    rdUpdateStatus := usDeleted;
    rdCachedUpdateStatus := cusUnmodified;
  end;
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
  if not FCachedUpdates then
   AutoCommitUpdateTransaction;
  FHasUncommitedChanges:=Transaction.State=tsActive;
end;



procedure TpFIBDataSet.DoOnDeleteError
 (DataSet: TDataSet; E: DB.EDatabaseError; var Action: TDataAction);
begin
 if FContainer<>nil then FContainer.DataSetError(DataSet,deOnDeleteError,E,Action);
 if Assigned(vUserOnDeleteError) then vUserOnDeleteError(DataSet,E,Action);
end;

procedure TpFIBDataSet.DoOnPostError
 (DataSet: TDataSet; E: DB.EDatabaseError; var Action: TDataAction);
begin
 inherited;
 if FContainer<>nil then FContainer.DataSetError(DataSet,deOnPostError,E,Action);
 if Assigned(vUserOnPostError) then vUserOnPostError(DataSet,E,Action);
end;

procedure  TpFIBDataSet.DoBeforePost; //override;
begin
  if (State=dsInsert) and
   (FAutoUpdateOptions.WhenGetGenID=wgBeforePost) then IncGenerator;
  if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforePost);
  inherited;
  if not CachedUpdates then
   AutoGenerateSQLText(State);
end;


procedure  TpFIBDataSet.DoAfterPost; //override;
var ActBuff:PRecordData;
begin
  if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterPost);
  if  (poProtectedEdit in Options) and not CachedUpdates then
  begin
   ActBuff:=PRecordData(GetActiveBuf);
   if ActBuff<>nil then
   begin
    with ActBuff^ do  rdUpdateStatus:=usModified; //   CommitRetaining
    SynchronizeBuffers(false,true)
   end
  end;
  inherited;
  if AutoCommit and CachedUpdates then
  begin
   ApplyUpdToBase;
   AutoCommitUpdateTransaction;
   CommitUpdToCach
  end;
  if (dcForceMasterRefresh in FDetailConditions) then  RefreshMasterDS;
end;



procedure  TpFIBDataSet.DoBeforeCancel;//override;
begin
 inherited;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeCancel);
end;

procedure  TpFIBDataSet.DoAfterCancel; //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterCancel); 
 inherited;
end;

procedure  TpFIBDataSet.DoBeforeDelete;//override;
begin
 if  not CanDelete then Abort;
 if not CachedUpdates
   and not (FAutoUpdateOptions.UpdateOnlyModifiedFields)
   and not vInCachRefresh
 then
 begin
  PrepareQuery(ukDelete);
  if  not CanDelete then Abort;
 end;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeDelete);
 inherited;
end;

procedure  TpFIBDataSet.DoBeforeInsert; //override;
begin
 PRecordData(FOldBuffer).rdRecordNumber:=-1;
 ForceEndWaitMaster;
 if not CanInsert then Abort;
 if not CachedUpdates and not (FAutoUpdateOptions.UpdateOnlyModifiedFields)
   and not vInCachRefresh
 then
  PrepareQuery(ukInsert);
 if not CanInsert then Abort;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeInsert);
 if Assigned(BeforeInsert) then BeforeInsert(Self);
end;

procedure  TpFIBDataSet.DoAfterInsert;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterInsert);
 inherited;
end;

procedure  TpFIBDataSet.DoAfterEdit;      //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterEdit);
 inherited
end;

procedure TpFIBDataSet.DoAfterDelete; //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterDelete);
 if AutoCommit and CachedUpdates then
 begin
   ApplyUpdToBase;
   AutoCommitUpdateTransaction;
   CommitUpdToCach
 end;
 inherited
end;

procedure  TpFIBDataSet.DoAfterScroll;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterScroll);
 inherited
end;

procedure TpFIBDataSet.DoBeforeClose;
begin
 if DefaultFields then FFNFields.Clear;
 {$IFDEF D5+}
  if DefaultFields and Assigned(FFilterParser) then
   FFilterParser.ResetFields;
 {$ENDIF}

 if DisableCOCount>0 then Exit;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeClose);
 inherited
end;

{$IFDEF D5+}
procedure TpFIBDataSet.DoBeforeRefresh;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeRefresh);
 inherited;
end;

procedure TpFIBDataSet.DoAfterRefresh;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterRefresh);
 inherited;
end;
{$ENDIF}

procedure TpFIBDataSet.DoOnCalcFields;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deOnCalcFields);
 inherited
end;

procedure TpFIBDataSet.DoBeforeScroll;  
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeScroll);
 inherited
end;

procedure  TpFIBDataSet.DoOnNewRecord; //override;
var i:integer;
    de:string;
    vDifferenceTime:double;
begin
 if (FAutoUpdateOptions.WhenGetGenID=wgOnNewRecord) then IncGenerator
 else
  if (FAutoUpdateOptions.WhenGetGenID=wgBeforePost) and (KeyField<>nil)
   then
  KeyField.Required:=false;
 if DataBase is TpFIBDataBase then
  vDifferenceTime:=TpFIBDataBase(DataBase).DifferenceTime
 else
  vDifferenceTime:=0;
 for i:=0 to Pred(FieldCount) do
  with Fields[i] do  begin
   if (Fields[i]=KeyField) and
    AutoUpdateOptions.SelectGenId then Continue;
   de:=UpperCase(Trim(DefaultExpression));
   if  (Fields[i] is TFIBStringField) and
       TFIBStringField(Fields[i]).DefaultValueEmptyString
   then
     asString:=''
   else
   if de<>'' then
    if StringInArray(de,['USER','CURRENT_USER']) then
    begin
      asString:=DataBase.DBParamByDPB[isc_dpb_user_name]
    end
    else
    if StringInArray(de,['ROLE','CURRENT_ROLE']) then
    begin
      asString:=DataBase.DBParamByDPB[isc_dpb_sql_role_name]
    end
    else
    if StringInArray(de,['NOW','CURRENT_TIME','CURRENT_TIMESTAMP']) then
     asDateTime:=Now-vDifferenceTime
    else
    if StringInArray(de,['TODAY','CURRENT_DATE']) then
     asDateTime:=Trunc(Now-vDifferenceTime)
    else
    if (de='TOMORROW') then
     asDateTime:=Trunc(Now-vDifferenceTime)+1
    else
    if (de='YESTERDAY') then
     asDateTime:=Trunc(Now-vDifferenceTime)-1
    else
    if de<>'NULL' then
     asString:=DefaultExpression;
  end;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deOnNewRecord);
 inherited DoOnNewRecord
end;

function  TpFIBDataSet.WillGenerateSQLs:boolean;
begin
 with FAutoUpdateOptions do
   Result :=UpdateOnlyModifiedFields and
    AutoReWriteSqls and CanChangeSQLs and
    (KeyFields<>'') and (UpdateTableName<>'') ;
end;


procedure TpFIBDataSet.SaveToStream(Stream:TStream;SeekBegin :boolean);
var i,fc,fs:integer;
    Version:integer;
begin
  with Stream do
  begin
   if SeekBegin then
    Seek(0,soFromBeginning);
   //
   Version:=3;
   WriteBuffer(Version,SizeOf(Integer));
   fc:=FieldCount;
   WriteBuffer(fc,SizeOf(Integer));
   for i :=0  to Pred(FieldCount) do
   begin
    fs:=Fields[i].DataSize;
    WriteBuffer(fs,SizeOf(Integer));
    WriteBuffer(Fields[i].DataType,SizeOf(TFieldType));
   end;

   vFieldDescrList.SaveToStream(Stream);
   WriteBuffer(FCacheSize,SizeOf(Integer));
   WriteBuffer(FRecordCount,SizeOf(Integer));
   WriteBuffer(FDeletedRecords,SizeOf(Integer));   
   WriteBuffer(FBufferCache^,CacheSize);
  end;
end;

procedure TpFIBDataSet.LoadFromStream(Stream:TStream; SeekBegin :boolean );
var b:PChar;
    fc   :integer;
    i,fs :integer;
    Version :integer;
    ft   :TFieldType;
procedure RaizeErrStream;
begin
 raise Exception.Create(
  Format(SFIBErrorUnableStreamLoad, [CmpFullName(Self)])
 );
end;

begin
  with Stream do
  begin
   if SeekBegin then  Seek(0,soFromBeginning);
    ReadBuffer(Version,SizeOf(Integer));
    ReadBuffer(fc,SizeOf(Integer));
    if fc<>FieldCount then RaizeErrStream;   
    for i :=0  to Pred(FieldCount) do
    begin
      ReadBuffer(fs,SizeOf(Integer));
      if fs<>Fields[i].DataSize then RaizeErrStream;
      ReadBuffer(ft,SizeOf(TFieldType));
      if ft<>Fields[i].DataType then RaizeErrStream;
    end;
   try
    vFieldDescrList.LoadFromStream(Stream);
    ReadBuffer(FCacheSize,SizeOf(Integer));
    ReadBuffer(FRecordCount  ,SizeOf(Integer));
    if Version>2 then
     ReadBuffer(FDeletedRecords,SizeOf(Integer));
    GetMem(b,FCacheSize);
    ReadBuffer(b^,FCacheSize);
   except
    Close;
    raise;
   end
  end;
  FreeMem(FBufferCache);
  FBufferCache:=b ;
  SynchronizeBuffers(true,false);
end;

procedure TpFIBDataSet.SaveToFile(const FileName:string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
   SaveToStream(Stream,True);
  finally
   Stream.Free;
  end;
end;

procedure TpFIBDataSet.LoadFromFile(const FileName:string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
   First;
   LoadFromStream(Stream,True);
  finally
   Stream.Free;
  end;
end;

function
 TpFIBDataSet.RaiseLockError(LockError:TLockStatus;ExceptMessage:string):TDataAction;
begin
 Result:=daFail;
 if Assigned(FOnLockError) then FOnLockError(Self,LockError,ExceptMessage,Result);
 case Result of
  daFail : raise Exception.Create(ExceptMessage);
  daAbort: Abort
 end
end;

procedure TpFIBDataSet.DoBeforeEdit; 
var oldUM:boolean;
begin
  if not CanEdit then Abort;
  if not CachedUpdates
   and not (FAutoUpdateOptions.UpdateOnlyModifiedFields)
   and not vInCachRefresh
  then
  begin
   PrepareQuery(ukModify);
   if not CanEdit then Abort;
  end;

  if  (poProtectedEdit in Options) then
   with AutoUpdateOptions do
   if UpdateStatus = usUnModified then
   begin
    if EmptyStrings(RefreshSQL) and
    (AutoReWriteSqls or UpdateOnlyModifiedFields)
    then
     QRefresh.SQL.Text:=GenerateSQLText(UpdateTableName,KeyFields,skRefresh);
    InternalRefresh;
    if EmptyStrings(UpdateSQL) and
    (AutoReWriteSqls or UpdateOnlyModifiedFields)
    then
    begin
     oldUM:=UpdateOnlyModifiedFields;
     UpdateOnlyModifiedFields:=false;
     QUpdate.SQL.Text:=GenerateSQLText(UpdateTableName,KeyFields,skModify);
     UpdateOnlyModifiedFields:=oldUM;
    end;
    LockRecord(true);
   end;
  if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeEdit);
  inherited;
end;

procedure  TpFIBDataSet.InternalDoBeforeOpen;
begin
 if psAskRecordCount in PrepareOptions then
 begin
  FAllRecordCount:=RecordCountFromSrv
 end
 else
  FAllRecordCount:=0;
end;

procedure TpFIBDataSet.DoBeforeOpen;
begin
 StartTransaction;
 FFilteredCacheInfo.NonVisibleRecords.Clear;
 if DisableCOCount>0 then Exit;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeOpen);
 inherited DoBeforeOpen;
end;


procedure  TpFIBDataSet.DoAfterClose;// override;
begin
 if DisableCOCount>0 then Exit;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterClose);
 inherited DoAfterClose;
end;

function  TpFIBDataSet.GetSelectSQL:TStrings;
begin                             
 Result:=inherited SelectSQL
end;

procedure TpFIBDataSet.SetSelectSQL(Value: TStrings);
begin
 if Active and (csDesigning in ComponentState) then Close;
 inherited SelectSQL:=Value
end;

procedure TpFIBDataSet.SetPrepareOptions(Value:TpPrepareOptions);
var NeedUpdateFieldsProps:boolean;
begin
 NeedUpdateFieldsProps:=(psApplyRepositary in Value-FPrepareOptions);
 inherited;
 if NeedUpdateFieldsProps then UpdateFieldsProps
end;


function  TpFIBDataSet.ParamNameCount(const aParamName:string):integer;
var i:integer;
begin
 Result:=0;
 with Params do
 begin
   for i := Pred(Count) downto 0   do
    if Params[i].Name=aParamName then Inc(Result)
 end;  
end;

function  TpFIBDataSet.FindParam (const ParamName:string): TFIBXSQLVAR;
begin
  StartTransaction;
  Result:=Params.ByName[ParamName];
end;

function  TpFIBDataSet.ParamByName(const ParamName:string): TFIBXSQLVAR;
begin
  Result:=Params.ByName[ParamName];
  if Result=nil then
   raise Exception.Create(
     Format(SFIBErrorParamNotExist, [ParamName, CmpFullName(Self)])
   );
end;


procedure TpFIBDataSet.ApplyUpdates;
var
  CurBookmark: String;
  Buffer: PRecordData;
  CurUpdateTypes: TFIBUpdateRecordTypes;
  UpdateAction: TFIBUpdateAction;
  UpdateKind: TUpdateKind;
  bRecordsSkipped: Boolean;
  i,rc:integer;
begin
  if FRecordCount=0 then Exit;
  if State in [dsEdit, dsInsert] then Post;
  FBase.CheckDatabase;
  FBase.CheckTransaction;
  DisableControls;
  CurBookmark := Bookmark;
  CurUpdateTypes := FUpdateRecordTypes;
  FUpdateRecordTypes := [cusModified, cusInserted, cusDeleted];
  FInApplyUpdates :=true;
  try
    First;
    bRecordsSkipped := False;
    i:=1; rc:=FRecordCount;
    while (i<=rc)  do
    begin
      Inc(i);
      Buffer := PRecordData(GetActiveBuf);
      case Buffer^.rdCachedUpdateStatus of
        cusModified:
          UpdateKind := ukModify;
        cusInserted:
          UpdateKind := ukInsert;
      else
          UpdateKind := ukDelete;
      end;

      (*
       * Do the OnUpdateRecord stuff, if necessary
       *)
      if (Assigned(FOnUpdateRecord)) then
      begin
        UpdateAction := uaFail;
        FOnUpdateRecord(Self, UpdateKind, UpdateAction);
      end
      else
        UpdateAction := uaApply;
        
      case UpdateAction of
        uaFail:
          FIBError(feUserAbort, [nil]);
        uaAbort:
          SysUtils.Abort;
        uaApplied:
        begin
          Buffer^.rdCachedUpdateStatus := cusUnmodified;
          Buffer^.rdUpdateStatus       := usUnmodified;
          WriteRecordCache(Buffer^.rdRecordNumber, PChar(Buffer));
          Dec(FCountUpdatesPending);
        end;
        uaSkip:
          bRecordsSkipped := True;
      end;

      (*
       * Now, do the update, if the user allows it
       *)
      while (UpdateAction in [uaApply, uaRetry]) do
      begin
        try
          case Buffer^.rdCachedUpdateStatus of
            cusModified:
            begin
              AutoGenerateSQLText(dsEdit);
              InternalPostRecord(FQUpdate, Buffer);
            end;
            cusInserted:
            begin
             AutoGenerateSQLText(dsInsert);
             InternalPostRecord(FQInsert, Buffer);
            end;
            cusDeleted:
              InternalDeleteRecord(FQDelete, Buffer);
          end;
          UpdateAction := uaApplied;
          Dec(FCountUpdatesPending);
        except
          (*
           * If there is an exception, then allow the user
           * to intervene (decide what to do about it).
           *)
          on E: EFIBError do
          begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            case UpdateAction of
              uaFail: raise;
              uaAbort: raise EAbort(E.Message);
              uaSkip:  bRecordsSkipped := True;
            end;
          end;
        end;
      end;
      Next;
    end;
    FUpdatesPending := bRecordsSkipped;
    if not FUpdatesPending then
     AutoCommitUpdateTransaction;
  finally
    FInApplyUpdates    :=False;
    FUpdateRecordTypes := CurUpdateTypes;
    Bookmark           := CurBookmark;
    EnableControls;
  end;
end;

procedure TpFIBDataSet.CancelUpdates;
var i:integer;
begin
   if State in [dsEdit, dsInsert] then  Post;
   if not FUpdatesPending or not FCachedUpdates then Exit;
   for i:=0 to Pred(FRecordCount) do
    InternalRevertRecord(i,false);
   FCountUpdatesPending:=0;
   FUpdatesPending:=false;
   RefreshFilters;
end;

procedure TpFIBDataSet.ClearModifFlags(Kind:byte);
var i   :integer;
    Buff:Pchar;
begin
// Commit Updates
 Buff:=AllocRecordBuffer;
 try
   for i:=0 to Pred(FRecordCount) do
   begin
    ReadRecordCache(i, Buff, False);
    with PRecordData(Buff)^ do
    begin
     case Kind of
      0: if rdCachedUpdateStatus in [cusUnmodified,cusUnInserted] then Continue;
      1: if (rdUpdateStatus in [usUnModified,usDeleted]) then Continue;
     end; //case
     if not((Kind=0) and (rdCachedUpdateStatus=cusDeleted)) then
      rdUpdateStatus:=usUnModified;
     rdCachedUpdateStatus:=cusUnmodified;      
     WriteRecordCache(i, Buff);
     SaveOldBuffer(Buff)
    end;
   end;
   FUpdatesPending:=false;
   FCountUpdatesPending:=0;
 finally
  FreeRecordBuffer(Buff);
 end;
 SynchronizeBuffers(false,false)
end;

procedure TpFIBDataSet.CloseProtect;
begin
 if Active then ClearModifFlags(1)
end;

//
procedure TpFIBDataSet.CommitUpdToCach;
begin
 SynchronizeBuffers(false,false);
 ClearModifFlags(0);
end;

procedure TpFIBDataSet.ApplyUpdToBase;
var i   :integer;
    Buff,OldBuff:Pchar;
    UpdateKind: TUpdateKind;
    UpdateAction: TFIBUpdateAction;
    cus:TCachedUpdateStatus;
    us :TUpdateStatus;
begin
 if State in [dsEdit, dsInsert] then  Post;
 StartTransaction;
 Buff   :=AllocRecordBuffer;
 OldBuff:=AllocRecordBuffer;
 try
   for i:=0 to Pred(FRecordCount) do
   begin
    ReadRecordCache(i, Buff, False);
    with PRecordData(Buff)^ do
    begin
     if rdCachedUpdateStatus in [cusUnmodified,cusUnInserted] then
      Continue;
     us :=rdUpdateStatus;
     cus:=rdCachedUpdateStatus;
     FUpdatesPending:=true;
     case rdCachedUpdateStatus of
        cusModified:
          UpdateKind := ukModify;
        cusInserted:
          UpdateKind := ukInsert;
        else
          UpdateKind := ukDelete;
     end;
     try
       vTypeDispositionField:=dfRRecNumber;
       vInspectRecno     :=i;
       if (Assigned(FOnUpdateRecord)) then
       begin
        UpdateAction := uaFail;
        FOnUpdateRecord(Self, UpdateKind, UpdateAction);
       end
       else
        UpdateAction := uaApply;
        vTypeDispositionField:=dfNormal;
      Except
       on E: EFIBError do
       begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            vTypeDispositionField:=dfNormal;
            case UpdateAction of
              uaFail : raise;
              uaAbort: raise EAbort(E.Message);
            end;
        case UpdateAction of
        uaFail:
          FIBError(feUserAbort, [nil]);
        uaAbort:
          raise EAbort.Create(
            Format(SFIBErrorAbortUpdates, [iifStr(Self.Owner<>nil,Self.Owner.Name+'.',''), Self.Name])
          );
        end;
       end;
      end;

      while (UpdateAction in [uaApply, uaRetry]) do
      begin
        try
          vTypeDispositionField:=dfRRecNumber;
          vInspectRecno     :=i;          
          case rdCachedUpdateStatus of
            cusModified:
             if CanEdit then
             begin
               AutoGenerateSQLText(dsEdit);
               ReadRecordCache(i, OldBuff, True);
               InternalPostRecord(FQUpdate, Buff);
               WriteCache(
                   FOldBufferCache,
                   PRecordData(Buff)^.rdSavedOffset,
                   FILE_BEGIN,
                   OldBuff
                  );
             end;
            cusInserted:
             if CanInsert then
             begin
              AutoGenerateSQLText(dsInsert);
              InternalPostRecord(FQInsert, Buff);
             end;
            cusDeleted:
             if CanDelete then
              InternalDeleteRecord(FQDelete, Buff);
          end;
//        Restore CU status
          rdUpdateStatus:=us;
          rdCachedUpdateStatus:=cus;
          WriteRecordCache(i, Buff);
{         ^^^^^^^^^^         }
          UpdateAction := uaApplied;
        except
          (*
           * If there is an exception, then allow the user
           * to intervene (decide what to do about it).
           *)
          on E: EFIBError do
          begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            case UpdateAction of
              uaFail: raise;
              uaAbort: raise EAbort(E.Message);
            end;
          end;
        end;
      end;
    end;
   end;
 finally
  FreeRecordBuffer(Buff);
  FreeRecordBuffer(OldBuff);
  vTypeDispositionField:=dfNormal;
 end;
// SynhronizeBuffers
end;




function  TpFIBDataSet.RecordStatus(RecNumber:integer):TUpdateStatus;
var Buff:Pchar;
begin
 Buff :=AllocRecordBuffer;
 try
  ReadRecordCache(RecNumber, Buff, False);
  Result:=PRecordData(Buff)^.rdUpdateStatus;
 finally
  FreeRecordBuffer(Buff);
 end
end;

procedure TpFIBDataSet.UpdateFieldsProps;
var i:integer;
    scale:Short;
    vFiAlias,vFi:TpFIBFieldInfo;
    RelTable,RelField:string;
    ZeroStr:string;
begin
  if FInClone then Exit;    
// Format Fields routine
   for i:=0 to Pred(FieldCount) do
   begin
    if poAutoFormatFields in FOptions then
    begin
     case Fields[i].DataType of
       ftDate:
       with TDateField(Fields[i]) do
       begin
        if DisplayFormat='' then
         DisplayFormat:=FDefaultFormats.DisplayFormatDate;
       end;
       ftTime:
       with TTimeField(Fields[i]) do
       begin
        if DisplayFormat='' then
         DisplayFormat:=FDefaultFormats.DisplayFormatTime;
       end;
       ftDateTime:
       with TDateTimeField(Fields[i]) do
       begin
        if DisplayFormat='' then
         DisplayFormat:=FDefaultFormats.DateTimeDisplayFormat;
       end;
       ftSmallint,ftInteger,ftFloat:
       with TNumericField(Fields[i]) do
       begin
         scale  :=GetFieldScale(TNumericField(Fields[i]));
         if scale<0 then
         begin
           ZeroStr:=MakeStr('0',-scale);
           if DisplayFormat='' then
            DisplayFormat:=
             UseFormat(FDefaultFormats.NumericDisplayFormat)+ZeroStr;
           if EditFormat=''    then
            EditFormat   :=
             UseFormat(FDefaultFormats.NumericEditFormat)   +ZeroStr;
         end;
       end;
       ftBCD	:
       with TBCDField(Fields[i]) do
       if Size>0 then         
       begin
        ZeroStr:=MakeStr('0',Size);
        if DisplayFormat='' then
         DisplayFormat:=
           UseFormat(FDefaultFormats.NumericDisplayFormat)+ZeroStr;
        if EditFormat=''    then
         EditFormat   :=
            UseFormat(FDefaultFormats.NumericEditFormat)+ZeroStr;
       end;
     end;
    end; //end Format

    if PrepareOptions*
       [pfSetRequiredFields,pfSetReadOnlyFields,pfImportDefaultValues,
        psUseBooleanField,psApplyRepositary]<>[]
    then
    begin
     RelTable:='ALIAS';
     RelField:=Fields[i].FieldName;
     vFiAlias:=ListTableInfo.GetFieldInfo(DataBase,RelTable,RelField);
     RelTable:= GetRelationTableName(Fields[i]);
     RelField:= GetRelationFieldName(Fields[i]);
      if  not Fields[i].ReadOnly
          and(pfSetReadOnlyFields in PrepareOptions)
          and(Fields[i].FieldKind=fkData)
      then
      begin
       Fields[i].ReadOnly :=
        (EasyFormatIdentifier(Database.SQLDialect, RelTable, True) <> QInsert.ModifyTable) and
        (EasyFormatIdentifier(Database.SQLDialect, RelTable, True) <> QUpdate.ModifyTable)
      end;
      if ((RelField='') or (RelTable='')) and  (vFiAlias=nil) then Continue;
      vFi:=ListTableInfo.GetFieldInfo(DataBase,RelTable,RelField);

      if (vFi=nil) then vFi:=vFiAlias;
      if (vFi=nil) then Continue;

       if pfImportDefaultValues in PrepareOptions then
        if (Fields[i] is TNumericField) then
        begin
        // Be sure to handle '1.0' with different DecSep!        
           if (DecimalSeparator <> '.') then
            Fields[i].DefaultExpression:=ReplaceStr(vFi.DefaultValue,'.', DecimalSeparator)
           else
            Fields[i].DefaultExpression:=vFi.DefaultValue
        end
        else
        if (Fields[i] is TDateTimeField)
         and not StringInArray(vFi.DefaultValue,
          ['NULL','NOW','CURRENT_TIME','CURRENT_TIMESTAMP','TODAY','CURRENT_DATE',
           'TOMORROW','YESTERDAY'
          ]
         )
        then
          with Fields[i] do
          case DataType of
             ftDate:DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,1);
             ftTime:DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,2);
          else
             DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,0)
          end
        else
        begin
          Fields[i].DefaultExpression:=vFi.DefaultValue;
          if Fields[i] is TFIBStringField then
             TFIBStringField(Fields[i]).DefaultValueEmptyString:=vFi.DefaultValueEmptyString
        end;


     if (vFiAlias<>nil) then vFi:=vFiAlias;


     if Fields[i].FieldKind = fkData then
     begin
       if not Fields[i].ReadOnly and (pfSetReadOnlyFields in PrepareOptions) then
       begin
        Fields[i].ReadOnly:= vFi.IsComputed;
       end;
       if pfSetRequiredFields in PrepareOptions then
        Fields[i].Required:=
          not  QSelect[Fields[i].FieldName].IsNullable
          and IsBlank(vFi.DefaultValue)    and not vFi.IsTriggered;
     end;
     
     if psApplyRepositary in PrepareOptions then
     // Info from FIB$FIELDS_INFO
      if vFi.WithAdditionalInfo then
      begin
       if vFi.DisplayWidth<>0 then Fields[i].DisplayWidth :=vFi.DisplayWidth;
       if vFi.DisplayLabel<>'' then Fields[i].DisplayLabel:=vFi.DisplayLabel;
       Fields[i].Visible:=vFi.Visible;
       if (vFi.DisplayFormat<>'') then
        if (Fields[i] is TNumericField)  then
          TNumericField(Fields[i]).DisplayFormat:=vFi.DisplayFormat
        else
        if (Fields[i] is TDateTimeField)  then
          TDateTimeField(Fields[i]).DisplayFormat:=vFi.DisplayFormat;
       if (vFi.EditFormat<>'') then
        if (Fields[i] is TNumericField)  then
          TNumericField(Fields[i]).EditFormat:=vFi.EditFormat;
      end;
    end;
   end;
end;

procedure  TpFIBDataSet.DoAfterOpen;
begin
 FHasUncommitedChanges:=false;
 if vSelectSQLTextChanged and  FAutoUpdateOptions.AutoReWriteSqls then
  if  not FAutoUpdateOptions.UpdateOnlyModifiedFields  then
   GenerateSQLs
  else
  with FAutoUpdateOptions do
  begin
    RefreshSQL.Text:=
     GenerateSQLText(UpdateTableName,KeyFields,skRefresh );
    DeleteSQL.Text:=
     GenerateSQLText(UpdateTableName,KeyFields,skDelete );
  end;

 UpdateFieldsProps;
 FUpdatesPending:=false;

 if psGetOrderInfo in PrepareOptions then
   PrepareAdditionalInfo;
 //
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterOpen);
 inherited;
end;

// Filter works


procedure TpFIBDataSet.AddedFilterRecord(DataSet: TDataSet; var Accept: Boolean);
begin
 // abstract method
end;

function TpFIBDataSet.IsVisible(Buffer: PChar): Boolean;
var
  OldState : TDataSetState;
  FR :integer;           
begin
 Result:= inherited IsVisible(Buffer);
 FR :=PRecordData(Buffer)^.rdRecordNumber + 1;
 if not Result then
//  if vNonVisibleRecords.IndexOf(FR)=-1  then
   FFilteredCacheInfo.NonVisibleRecords.Add(FR);
 if FInApplyUpdates then Exit;
 OldState := State;
 if IsFiltered and Result then
 try
  SetTempState(dsFilter);
  Result:= FCurrentRecord>-1;
  if Result and Filtered then
  begin
   {$IFDEF D5+}
   if Assigned(FFilterParser) then  Result := FFilterParser.BooleanResult;
   {$ENDIF}
   if Result then
    if Assigned(OnFilterRecord) then OnFilterRecord(Self,Result);
  end;
  if Result then AddedFilterRecord(Self,Result);
  if Result then Inc(vFilteredRecCounts);
  if not Result then
    FFilteredCacheInfo.NonVisibleRecords.Add(FR)
  else
   FFilteredCacheInfo.NonVisibleRecords.Remove(FR)
 finally
  RestoreState(OldState);
 end  ;
end;

function TpFIBDataSet.RecordCountFromSrv:integer;
var
    i:integer;
    s:string;
begin
 Result:=0;
 if (QSelect.Transaction=nil) or  (not QSelect.Transaction.Active) then
  Exit;
 with vQryRecordCount do
 try
  Close;
  if QSelect.Database<>Database then
   Database   :=QSelect.Database;
  if QSelect.Transaction<>Transaction then
   Transaction:=QSelect.Transaction;
   s:='';
   if Assigned(FOnAskRecordCount ) then  FOnAskRecordCount (Self,s);
   if s='' then
    s:=CountSelect(QSelect.SQL.Text);
   if (s+#13#10)<>SQL.Text then  SQL.Text :=s;
   if SQL.Text='' then Exit;
   for i:=0 to Pred(Params.Count) do
    Params[i].Value:=QSelect.Params.ByName[Params[i].Name].Value;
   Prepare;
   for i:=0 to Pred(OnlySrvParams.Count) do
    Params.ByName[OnlySrvParams[i]].Value:=QSelect.Params.ByName[OnlySrvParams[i]].Value;
   try
    ExecQuery;
    Result:=Fields[0].asInteger
   except
    raise Exception.Create(
      Format(SFIBErrorUnableGetRecordCount, [CmpFullName(Self)])
    );
   end;
 finally
 end
end;

procedure TpFIBDataSet.RefreshClientFields;
var i:integer;
    FIBBuff:Pchar;
begin
 FIBBuff:=AllocRecordBuffer;
 try
  vTypeDispositionField:=dfRWRecNumber;
  for i:=0 to Pred(FRecordCount) do
  begin
   ReadRecordCache(i, FIBBuff, False);
   vInspectRecno:=i;
   CalculateFields(FIBBuff);
  end;
  SynchronizeBuffers(true,false);
 finally
  vTypeDispositionField:=dfNormal;
  FreeRecordBuffer(FIBBuff)
 end
end;             

procedure TpFIBDataSet.SetFiltered(Value: Boolean);
begin
 FFilteredCacheInfo.NonVisibleRecords.Clear;
 FFilteredCacheInfo.AllRecords     :=-1;
 {$IFDEF D5+}
 if Assigned(FFilterParser) and (FFilterParser.ExpressionText<>Filter) then
  Filter:=Filter;
 {$ENDIF} 
 inherited SetFiltered(Value);
 RefreshFilters
end;

function  TpFIBDataSet.GetFieldForTable(const Relation:string):TField;
var i:integer;
begin
 Result:=nil;
 for i:=0 to Pred(FieldCount) do
  if GetRelationTableName(Fields[i])=Relation then
  begin
    Result:=Fields[i];
    Exit;
  end;
end;

function  TpFIBDataSet.LockRecord(RaiseErr:boolean):TLockStatus;
var ExceptMessage:string;
    Retry:boolean;
    tf:Tfield;
    i:integer;
    UnknownErrorMsg,LockTxt:string;
begin
 if  EmptyStrings(QUpdate.SQL) then
  raise Exception.Create(SFIBErrorUnableLock);
 LockTxt:=GetModifyTable(QUpdate.NormalizedSQL,true);
 if (LockTxt<>'') then
 begin
  tf:=GetFieldForTable(LockTxt);
  if tf<>nil then
  begin
   LockTxt:=
    'UPDATE '+FormatIdentifier(Database.SQLDialect,LockTxt)+' SET '+
    FormatIdentifier(Database.SQLDialect,GetRelationFieldName(tf))+'= ?'+
     FormatIdentifier(Database.SQLDialect,tf.FieldName) +' WHERE '
   +QUpdate.WhereClause[1];
  end
  else
   LockTxt:=''                 
 end;
 if LockTxt<>'' then
 begin
  if vLockQry.SQL.Text<>LockTxt+#13#10 then
   vLockQry.SQL.Text:=LockTxt
 end
 else
  vLockQry.SQL.Assign(QUpdate.SQL);
  
 if vLockQry.Database<>Database then
  vLockQry.Database:=Database;
 if vLockQry.Transaction<>UpdateTransaction then
  vLockQry.Transaction:=UpdateTransaction;

 SetQueryParams(vLockQry,GetActiveBuf);
 if CachedUpdates then
  for i:=0 to Pred(vLockQry.Params.Count) do
  begin
   tf:=  FindField(vLockQry.Params[i].Name);
   if (tf<>nil) and not tf.IsBlob then
   if tf.isNull then
    vLockQry.Params[i].IsNull:=true
   else
   if tf is TDateTimeField then
    vLockQry.Params[i].AsDateTime:=tf.OldValue
   else
    vLockQry.Params[i].Value:=tf.OldValue
  end;
 Result:=lsUnknownError;
 Retry:=true;
 while Retry do
 begin
  try
   if (not UpdateTransaction.InTransaction)
    and (poStartTransaction in Options) then UpdateTransaction.StartTransaction;
   vLockQry.ExecQuery;
   if vLockQry.RowsAffected=1 then  Result:=lsSuccess
   else
   if vLockQry.RowsAffected=0 then  Result:=lsNotExist
   else
   Result:=lsMultiply;
  except
   On E: EFIBError do
   if E.SQLCode=-913 then Result:=lsDeadLock
   else
   begin
    UnknownErrorMsg := E.Message;   
    Result:=lsUnknownError
   end;
  end;
  if (Result<>lsSuccess) and RaiseErr then
  begin
   case Result of
     lsDeadLock: ExceptMessage   :=SEdDeadLockMess;
     lsNotExist: ExceptMessage   :=SEdNotExistRecord;
     lsMultiply: ExceptMessage   :=SEdMultiplyRecord;
     lsUnknownError:ExceptMessage:=UnknownErrorMsg;
   end;
   ExceptMessage:=SEdErrorPrefix+ExceptMessage;
   Retry:=RaiseLockError(Result,ExceptMessage)=daRetry
  end
  else
    Retry:=false
 end;
end;
//



// BCD routine
{$IFNDEF D5+}
type
   TBCDArr= array[0..7] of Byte;
   PBCDArr=^TBCDArr;

function TpFIBDataSet.BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean;//override;
begin
   try
    Curr:=Double(TBCDArr(BCD^));
    Result :=true;
   except
    Result :=false;
   end
end;

function TpFIBDataSet.CurrToBCD(const Curr: Currency; BCD: Pointer; Precision,
      Decimals: Integer): Boolean; //override;
var Dest:Double;
    i,st:integer;
begin
    try
      // Round rtns
     st:=1;
     for i:=1 to Decimals do st:=10*st;
     Dest:=System.Int(Curr)+Round(Frac(Curr)*st)/st;
//
     Move(Dest, PBCDArr(BCD)^, 8);
     Result :=true;
   except
    Result :=false;
   end
end;
{$ENDIF}






//  other routine stuff

procedure TpFIBDataSet.OpenWP(ParamValues: array of Variant);
begin
 if High(ParamValues)>-1 then
  FQSelect.SetParamValues(ParamValues);
 Open
end;

// Clone Records routine
function FieldInArray(Field:TField; Arr: array of const) :boolean;
var i:integer;
    CI:boolean;
begin
  CI:=true;
  if Field.DataSet is TFIBDataSet then
   if (TFIBDataSet(Field.DataSet).Database<>nil) then
   with TFIBDataSet(Field.DataSet).Database do
   begin
    CI:= (SQLDialect<3) or (UpperOldNames)
   end;
  Result:=false;
  for i:=Low(Arr) to High(Arr) do
   with Arr[i] do
   begin
     case VType of
      vtInteger:    Result:=Field.Index=VInteger;
      vtPChar:
        Result :=
         EquelNames(CI,Field.FieldName,vPChar);
      vtAnsiString:
        Result:=
         EquelNames(CI,Field.FieldName,string(VAnsiString));
      vtPointer:
       if TObject(VPointer) is TStrings then
        if CI then
         Result:= TStrings(VPointer).IndexOf(AnsiUpperCase(Field.FieldName))<>-1
        else
         Result:= TStrings(VPointer).IndexOf(Field.FieldName)<>-1
       else
       if TObject(VPointer) is TField then Result:=VPointer=Field;
      vtObject:
       if VObject is TStrings then
        if CI then
         Result:= TStrings(VObject).IndexOf(AnsiUpperCase(Field.FieldName))<>-1
        else
         Result:= TStrings(VObject).IndexOf(Field.FieldName)<>-1        
       else
        if VObject is TField then Result:=VObject=Field;
     end;
     if Result then Exit
   end;
end;

procedure TpFIBDataSet.CloneRecord(SrcRecord:integer; IgnoreFields:array of const);
var i,r:integer;
    ForceInsert:boolean;
begin
 if State=dsInsert then
  r:=FRecordCount+1
 else
  r:=FRecordCount;
 if (SrcRecord>=(r-FDeletedRecords)) or (SrcRecord<0) then Exit;
 ForceInsert:=State<>dsInsert;
 if ForceInsert then Insert
 else
  ForceInsert:=not Modified;
// if ForceInsert  then Inc(SrcRecord);
 for i:=0 to Pred(FieldCount) do
  if (Fields[i].FieldKind in [fkData]) and not Fields[i].IsBlob
   and not FieldInArray(Fields[i],IgnoreFields) then
   begin
    Fields[i].Value:=RecordFieldValue(Fields[i],SrcRecord);
    if ForceInsert  then begin
     Inc(SrcRecord);
     ForceInsert:=False;
    end;    
   end;
end;

procedure TpFIBDataSet.CloneCurRecord(IgnoreFields:array of const );
begin
 CloneRecord(Recno-1, IgnoreFields)
end;

// Wrappers

function  TpFIBDataSet.FN(const FieldName:string):TField; //FindField
var
   i :integer;
   fn:string;
begin
 if FFNFields.Find(FieldName,i) then
  Result:=TField(FFNFields.Objects[i])
 else
 begin
  if (Length(FieldName)>0) and (FieldName[1]='"') then
    fn:=Copy(FieldName,2,Length(FieldName)-2)
  else
   fn:=FieldName;
  Result:=FindField(fn);
  if Assigned(Result) then
   FFNFields.AddObject(FieldName,Result)
 end;
end;

function  TpFIBDataSet.FBN(const FieldName:string):TField; //FieldByName
begin
  Result:=FN(FieldName);
  {$IFDEF D5+}
   if Result = nil then DatabaseErrorFmt(SFieldNotFound, [FieldName], Self);
  {$ELSE}
   if Result = nil then DatabaseErrorFmt(SFieldNotFound, [FieldName]);
  {$ENDIF}
end;

// Containers stuff

procedure TpFIBDataSet.SetReceiveEvents(Value:Tstrings);
begin
 FReceiveEvents.Assign(Value)
end;

procedure TpFIBDataSet.SetContainer(Value:TDataSetsContainer);
begin
  if FContainer=Value then Exit;
  if FContainer<>nil then FContainer.RemoveDataSet(Self);
  FContainer:=Value;
  if FContainer<>nil then
  begin
   FContainer.FreeNotification(Self);
   FContainer.AddDataSet(Self);
  end;
end;

procedure TpFIBDataSet.DoUserEvent(Sender:Tobject; const UDE:string;var Info:string);
begin
 if UpperCase(UDE)='CLOSE' then  Close else
 if UpperCase(UDE)='OPEN' then   Open  else
 if FContainer<>nil then FContainer.UserEvent(Sender,Self,UDE,Info);
 if Assigned(FOnUserEvent) then FOnUserEvent(Sender,Self,UDE,Info)
end;

procedure TpFIBDataSet.AutoGenerateSQLText(ForState:TDataSetState);
var s:string;
begin
  with FAutoUpdateOptions do
    if UpdateOnlyModifiedFields and (KeyFields<>'')
     and (UpdateTableName<>'')
  then
   case ForState of
    dsEdit: begin
             s:=GenerateSQLText(UpdateTableName,KeyFields,skModify);
             if s<>UpdateSQL.Text then   UpdateSQL.Text :=s;
            end;
    dsInsert:begin
              s:=GenerateSQLText(UpdateTableName,KeyFields,skInsert);
              if s<>InsertSQL.Text then  InsertSQL.Text :=s;
             end;
   end;
end;

function  TpFIBDataSet.GenerateSQLText
     (const TableName,KeyFieldNames:string;SK:TpSQLKind):string;
 //    Updates
const
  Indent='     ';
var i:integer;
    pWhereClause,InsValuesStr:string;
    AcceptCount:integer;
    RealKeyFieldName,RealFieldName:string;
    FieldTableName :string;
    KeyFieldList:TList;
    FormatTableName:string;
    RelTableName   :string;
    vpFIBTableInfo:TpFIBTableInfo;
    vFi:TpFIBFieldInfo;

procedure  GetFieldList(List: TList; const FieldNames: string);
var
  Pos: Integer;
  Field: TField;
begin
  Pos := 1;
  while Pos <= Length(FieldNames) do
  begin
    Field := FN(ExtractFieldName(FieldNames, Pos));
    if Field<>nil then List.Add(Field);
  end;
end;

begin

KeyFieldList:=TList.Create;
try
 GetFieldList(KeyFieldList, KeyFieldNames);
 if KeyFieldList.Count=0 then Exit;

 FormatTableName := EasyFormatIdentifier(Database.SQLDialect, TableName,
  Database.EasyFormatsStr
 );
 vpFIBTableInfo:=ListTableInfo.GetTableInfo(Database,FormatTableName);
 if vpFIBTableInfo=nil then Exit;
 // Validate KeyFields
 pWhereClause:='';

 for I := Pred(KeyFieldList.Count) downto 0 do
 begin
  RelTableName :=
   EasyFormatIdentifier(Database.SQLDialect, GetRelationTableName(TField(KeyFieldList[i])),
    Database.EasyFormatsStr
   );

  if RelTableName <> FormatTableName then
  begin
    KeyFieldList.Delete(i);
    Continue;
  end;  
  RealKeyFieldName :=
   EasyFormatIdentifier(Database.SQLDialect,
    GetRelationFieldName(TField(KeyFieldList[i])),
    Database.EasyFormatsStr
  );
  vFi := vpFIBTableInfo.FieldInfo(RealKeyFieldName);
  if (vFi=nil) or not vFi.CanIncToWhereClause then
    KeyFieldList.Delete(i)
  else
   pWhereClause:=pWhereClause + iifStr(pWhereClause='','',' and ')+
    RelTableName + '.' + RealKeyFieldName + '=?' +
    EasyFormatIdentifier(
     Database.SQLDialect, 'OLD_' + TField(KeyFieldList[i]).FieldName,
     Database.EasyFormatsStr
    );
 end;
 if KeyFieldList.Count=0 then
 begin
    raise Exception.Create(
      Format(SFIBErrorGenerationError, [CmpFullName(Self), SQLKindNames[SK], TableName])
    )
 end;

 case SK of
  skModify: Result:='Update '+FormatTableName+' Set';
  skInsert: begin
             Result:='Insert into '+FormatTableName+'(';
             InsValuesStr:='values (';
            end;
  skDelete: begin
             Result:='Delete from '+FormatTableName+CLRF+
                     'where '+pWhereClause
             ;
             Exit;
            end;
  skRefresh:begin
             Result:=
              AddToWhereClause(SelectSQL.text,pWhereClause,true);
             Exit;
            end;
 end;

 AcceptCount:=0;
 for i:=0 to FieldCount-1 do
 begin
   if (Fields[i].FieldKind<>fkData) then Continue;
   FieldTableName:=
    EasyFormatIdentifier(Database.SQLDialect,
      GetRelationTableName(Fields[i]),
      Database.EasyFormatsStr
    );
   if (FieldTableName<>FormatTableName) then Continue;

   if SK<>skDelete then
   if FAutoUpdateOptions.UpdateOnlyModifiedFields then
   begin
    if not Fields[i].IsBlob  then
    case  SK of
     skModify:  if Fields[i].OldValue=Fields[i].Value then  Continue;
     skInsert:  if Fields[i].IsNull then Continue;
    end
    else
     if not BlobModified(Fields[i]) and (Fields[i].OldValue=Fields[i].Value) then     Continue;
   end;

   Inc(AcceptCount) ;
   RealFieldName:=
    EasyFormatIdentifier(Database.SQLDialect,
      GetRelationFieldName(Fields[i]),
      Database.EasyFormatsStr
    );

   vFi:=vpFIBTableInfo.FieldInfo(RealFieldName);
   if (vFi=nil) or (vFi.IsComputed )    then Continue;

   Result       :=Result+iifStr(AcceptCount=1,'',',')+CLRF;
   InsValuesStr :=InsValuesStr+iifStr(AcceptCount=1,'',',')+CLRF;
   case SK of
    skModify:Result:=Result+Indent+RealFieldName+'=?'+
              EasyFormatIdentifier(Database.SQLDialect,'NEW_'+Fields[i].FieldName,
               Database.EasyFormatsStr
              );
    skInsert:begin
              Result      :=Result+Indent+RealFieldName;
              InsValuesStr:=InsValuesStr+Indent+'?'+
               EasyFormatIdentifier(Database.SQLDialect,
                'NEW_'+Fields[i].FieldName,Database.EasyFormatsStr
              );
             end;
   end;
 end; // end for
 if AcceptCount=0 then
 begin
  Result:='No Action'; Exit;
 end;
 case SK of
    skModify:Result:=Result+CLRF+'where '+pWhereClause;
    skInsert:begin
              Result:=Result+CLRF+')'+CLRF+InsValuesStr+CLRF+')'
             end;
 end;
finally
 KeyFieldList.Free
end;

end;

function  TpFIBDataSet.GenerateSQLTextWA
     (const TableName:string;SK:TpSQLKind):string; // Where All
begin
 Result:=GenerateSQLText(TableName,AllFields,SK)
end;


//AutoUpdate operations
function  TpFIBDataSet.KeyField:TField;
begin
 with FAutoUpdateOptions do Result:=FindField(KeyFields)
end;

function  TpFIBDataSet.SqlTextGenID:string;
begin
 with FAutoUpdateOptions do
  Result:='SELECT GEN_ID('+
   FormatIdentifier(Database.SQLDialect, GeneratorName)+
    ',1) FROM  RDB$DATABASE';
end;

procedure TpFIBDataSet.IncGenerator;
var kf:TField;
begin
  with FAutoUpdateOptions do
   if WhenGetGenID=wgNever then Exit;
   if vInCachRefresh then Exit;     
   if not Assigned(DataBase) then
    FIBError(feDatabaseNotAssigned,[CmpFullName(Self)]);
   kf:=KeyField;
   if (kf=nil) or not (kf is TNumericField) then Exit;
   if kf.IsNull then
    begin
     if (kf is TFIBBCDField) and (TFIBBCDField(kf).Scale = 0) then
      TFIBBCDField(kf).asComp:=
       DataBase.Gen_Id(FAutoUpdateOptions.GeneratorName,1)
     else
      {$IFNDEF VER100}
       kf.AsInteger:=
        DataBase.Gen_Id(FAutoUpdateOptions.GeneratorName,1)
      {$ELSE}
       kf.AsInteger:=
        Trunc(DataBase.Gen_Id(FAutoUpdateOptions.GeneratorName,1))
      {$ENDIF}
    end;
end;

function  TpFIBDataSet.AllFields:string;
var
    i:integer;
begin
 Result:='';
 for i:=0  to Pred(FieldCount) do
  Result:=Result+iifStr(i>0,';','')+Fields[i].FieldName;
end;

function  TpFIBDataSet.PrimaryKeyFields(const TableName:string):string;
var PrimKeyFields:string;
    i,wc:integer;
    tf:TField;
begin
  Result:='';
  PrimKeyFields:=
   Trim(ListTableInfo.
    GetTableInfo(Database,TableName).PrimaryKeyFields
   );
  if PrimKeyFields='' then Exit;
  wc:=WordCount(PrimKeyFields,[';']);
  for i:=1 to wc do
  begin
    tf:=
     FieldByOrigin(TableName+'.'+
      ExtractWord(i,PrimKeyFields,[';'])
     );
    if tf=nil then
    begin
     Result:='';     Exit;
    end;
    Result:=Result+iifStr(i>1,';','')+tf.FieldName;
  end;
end;

procedure TpFIBDataSet.GenerateSQLs;
begin
 if FieldCount=0 then FieldDefs.Update;
 QDelete .OnSQLChanging:=nil;
 QInsert .OnSQLChanging:=nil;
 QUpdate .OnSQLChanging:=nil;
 QRefresh.OnSQLChanging:=nil;
 with FAutoUpdateOptions do
 try
  if IsBlank(KeyFields) then
   KeyFields:=PrimaryKeyFields(UpdateTableName);
   if IsBlank(KeyFields) then   KeyFields:=AllFields;
  if IsBlank(KeyFields) then Exit;
  if CanChangeSQLs or (EmptyStrings(DeleteSQL)) then
   DeleteSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skDelete);
  if CanChangeSQLs or (EmptyStrings(UpdateSQL)) then
   UpdateSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skModify);
  if CanChangeSQLs or (EmptyStrings(InsertSQL)) then
   InsertSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skInsert);
  if CanChangeSQLs or (EmptyStrings(RefreshSQL)) then
   RefreshSQL.Text:=GenerateSQLText(UpdateTableName,KeyFields,skRefresh);
 finally
  QDelete .OnSQLChanging:=SQLChanging;
  QInsert .OnSQLChanging:=SQLChanging;
  QUpdate .OnSQLChanging:=SQLChanging;
  QRefresh.OnSQLChanging:=SQLChanging;
 end;
end;

{$IFDEF CAN_OVERLOAD}
 {$IFNDEF VER120}
procedure TpFIBDataSet.CacheInsert(Value:Variant;DoRefresh:boolean =false);
begin
 CacheInsert([0],[Value]);
 if DoRefresh then  Refresh   
end;
 {$ENDIF}
{$ENDIF}


procedure TpFIBDataSet.CacheModify(
     aFields: array of integer;Values: array of Variant;
     KindModify : byte
) ;
var i:integer;
begin
 CheckBrowseMode;
 vInCachRefresh:=true;
 try
  case KindModify  of
   0: Edit;
   1: Insert;
   2: Append;
  end;
   for i:=0 to High(Values) do
    if i>High(aFields) then Break
    else
     Fields[aFields[i]].Value:=Values[i]
    ;
  Post;
 finally
  vInCachRefresh:=false;
 end;
end;


procedure TpFIBDataSet.CacheEdit(
     aFields: array of integer;Values: array of Variant
) ;
begin
 CacheModify( aFields,Values,0 );
end;


procedure TpFIBDataSet.CacheAppend(
     aFields: array of integer;Values: array of Variant
) ;
begin
 CacheModify( aFields,Values,2);
end;

procedure TpFIBDataSet.CacheInsert(
 aFields: array of integer;Values: array of Variant
);
begin
 CacheModify( aFields,Values,1);
end;

procedure TpFIBDataSet.CacheRefresh(FromDataSet:TDataSet;Kind:TCachRefreshKind
 ;FieldMap:Tstrings
);
var i:integer;
    fn1:Tfield;
    Buff: PChar;
    sfn:String;
    ForcedEdit:boolean;
    IsReadOnlyField:boolean;
begin
 vInCachRefresh:=not (State in [dsInsert,dsEdit]);
 try
  ForcedEdit:=not (State in [dsInsert,dsEdit]) ;
  if ForcedEdit then
   if Kind= frkInsert then Insert else Edit;
  for i:=0 to Pred(FromDataSet.FieldCount) do
  begin
   if (FieldMap<>nil) then
   begin
    sfn:=FieldMap.Values[FromDataSet.Fields[i].FieldName];
    if sfn='' then  sfn:=FromDataSet.Fields[i].FieldName;
   end
   else
    sfn:=FromDataSet.Fields[i].FieldName;
   fn1:=FN(sfn);
   if fn1<>nil then
   begin
    IsReadOnlyField:=fn1.ReadOnly;
    fn1.ReadOnly:=false;
    try
     fn1.Value:=FromDataSet.Fields[i].Value;
    finally
     fn1.ReadOnly:=IsReadOnlyField;
    end; 
   end;
  end;
  if ForcedEdit then Post;
  Buff := GetActiveBuf;
  with PRecordData(Buff)^ do
  begin
     rdCachedUpdateStatus:=cusUnmodified;
     rdUpdateStatus := usUnmodified;
     WriteRecordCache(rdRecordNumber, Buff);
     SaveOldBuffer(Buff)
  end;
 finally
  vInCachRefresh:=false
 end
end;

procedure TpFIBDataSet.CacheRefreshByArrMap(
           FromDataSet:TDataSet;Kind:TCachRefreshKind;
          const SourceFields,DestFields:array of String
 );
var ts:TStrings;
    i,m:integer;

begin
 ts:=TStringList.Create;
 m:=High(SourceFields);
 if High(DestFields)>m then m:=High(DestFields);
 with ts do
 try
  for i:=0 to m do  Values[SourceFields[i]]:=DestFields[i];
  CacheRefresh(FromDataSet,Kind, ts);
 finally
  Free
 end;
end;


function  TpFIBDataSet.RecordFieldAsFloat(Field:TField;RecNumber:integer):Double;
begin
 Result:=0;
 if (RecNumber>FRecordCount) then Exit;
 vInspectRecno:=RecNumber;
 try
  vTypeDispositionField:=dfRRecNumber;
  Result:=Field.AsFloat;
 finally
  vTypeDispositionField:=dfNormal
 end;
end;

{$IFDEF D5+}
//     IProviderSupport

function TpFIBDataSet.PSGetQuoteChar: string;
begin
  Result := '';
  if Assigned(Database) and (Database.SQLDialect = 3) then
    Result := '"'
end;

function TpFIBDataSet.PSGetTableName: string;
var ts:TStrings;
begin
 Result :='';
 ts :=TStringList.Create;
 try
  AllSQLTables(SelectSQL.Text,ts);
  if ts.Count=0 then Exit;
  if ts.IndexOf(AutoUpdateOptions.UpdateTableName)<>-1 then
   Result :=AutoUpdateOptions.UpdateTableName
  else
   Result :=ts[0]
 finally
  ts.Free;
 end;
end;

function  TpFIBDataSet.PSGetKeyFields: string;
begin
 Result := inherited PSGetKeyFields;
 if Result='' then
  Result := FAutoUpdateOptions.KeyFields;
end;

procedure TpFIBDataSet.PSSetCommandText(const CommandText: string);
begin
  if CommandText <> '' then
    SelectSQL.Text := CommandText;
end;

function TpFIBDataSet.PSGetUpdateException(E: Exception;
  Prev: EUpdateError): EUpdateError;
var ErrC:integer;
begin
  if E is EFIBError then
  begin
    if Prev <> nil then
     ErrC:=Prev.ErrorCode
    else
     ErrC:=0;
    with EFIBError(E) do
     Result :=EUpdateError.Create(Message, '', SQLCode, ErrC,E)
  end
  else
   Result := inherited PSGetUpdateException(E, Prev);
end;

function TpFIBDataSet.PSInTransaction: Boolean;
begin
  Result := Transaction.InTransaction;
end;

function TpFIBDataSet.PSIsSQLBased: Boolean;
begin
  Result := True;
end;

function TpFIBDataSet.PSIsSQLSupported: Boolean;
begin
  Result := True;
end;


procedure TpFIBDataSet.PSStartTransaction;
begin
  if not Assigned(Database) then
    FIBError(feDatabaseNotAssigned, [nil]);
  if not Assigned(Transaction) then
    FIBError(feTransactionNotAssigned, [nil]);
  if not Database.Connected then Database.Open;
  if not Transaction.InTransaction then Transaction.StartTransaction;
end;

procedure TpFIBDataSet.PSEndTransaction(Commit: Boolean);
begin
  if Assigned(Transaction) and Transaction.InTransaction then
   if Commit then
    Transaction.Commit
   else
    Transaction.RollBack
end;


type THackDS=class(TDataSet);

function TpFIBDataSet.PSUpdateRecord(UpdateKind: TUpdateKind;
  Delta: TDataSet): Boolean;


var UpdateAction :TFIBUpdateAction;
    j :integer;
    UpdQry:TFIBQuery;
    CurParam:TFIBXSQLVAR;
    ListUO:TList;
    NewBcd,OldBcd:TBcd;


procedure ExecUpdate;
var    i:integer;
    OldVal,NewVal:variant;


procedure SetParam;
var lValue: Comp;
    lScale: byte;
    pValue: PBcd;
begin
  if CurParam<>nil then
   if Delta.Fields[i].DataType<>ftBcd then
   begin
     if VarIsEmpty(NewVal) then
      CurParam.Value:=OldVal
     else
      CurParam.Value:=NewVal;
   end
   else
   begin
    // Avoid variants
     if VarIsEmpty(NewVal) then
     begin
      if VarIsNull(OldVal) then
      begin
       CurParam.IsNull:=true;
       Exit;
      end;
      pValue:=@OldBcd;
     end
     else
     begin
      if VarIsNull(NewVal) then
      begin
       CurParam.IsNull:=true;
       Exit;
      end;
      pValue:=@NewBcd;
     end;
     BCDToCompWithScale(pValue^, lValue,lScale);
     CurParam.asComp:=lValue;
     CurParam.Data.sqlscale:=-lScale;
   end;
end;

begin
    for i:=0 to Pred(Delta.FieldCount) do
    with Delta.Fields[i] do
    begin
      if (DataType<>ftBcd)  then
      begin
        OldVal:=OldValue;
        NewVal:=NewValue;
      end
      else
      begin
        try
         OldVal:=OldValue;
        except
         on E:EOverFlow do OldVal:=unAssigned
         // .       OldIsNull
         //    unAssigned
        end;
        try
         NewVal:=NewValue;
        except
         // .       NewIsNull
         //    unAssigned
         on E:EOverFlow do NewVal:=unAssigned
        end;

       if not VarIsNull(OldVal) then
         if  GetBCDFieldData(Delta.Fields[i],true,OldBcd) then
          OldVal:='';
       if not VarIsNull(NewVal) then
         if  GetBCDFieldData(Delta.Fields[i],false,NewBcd) then
          NewVal:='';
      end;

       CurParam :=UpdQry.FindParam(FieldName);
       SetParam;
       CurParam :=UpdQry.FindParam('NEW_'+FieldName);
       SetParam;
       CurParam :=UpdQry.FindParam('OLD_'+FieldName);
       NewVal   :=unAssigned;
       SetParam;
    end;
    UpdQry.ExecQuery;
end;

begin
  Result := False;
  if Assigned(OnUpdateRecord) then
  begin
    UpdateAction := uaFail;
    if Assigned(FOnUpdateRecord) then
    begin
      FOnUpdateRecord(Delta, UpdateKind, UpdateAction);
      Result := UpdateAction = uaApplied;
    end;
  end
  else
  begin
    Result:=false;
    ListUO:=ListForUO(UpdateKind);
    for j:=0 to Pred(ListUO.Count) do
    with TpFIBUpdateObject(ListUO[j]) do
    if Active and (ExecuteOrder=oeBeforeDefault) and (not EmptyStrings(SQL)) then
    begin
     UpdQry :=TpFIBUpdateObject(ListUO[j]);
     ExecUpdate;
     Result:=true;
    end;
    case UpdateKind of
     ukModify: UpdQry := QUpdate;
     ukInsert: UpdQry := QInsert;
    else
     UpdQry := QDelete;
    end;
    Result:=Result or (UpdQry.SQL.Count>0);
    if (not EmptyStrings(UpdQry.SQL)) then
     ExecUpdate;
    for j:=0 to Pred(ListUO.Count) do
    with TpFIBUpdateObject(ListUO[j]) do
    if Active and (ExecuteOrder=oeAfterDefault) and (not EmptyStrings(SQL)) then
    begin
     UpdQry :=TpFIBUpdateObject(ListUO[j]);
     ExecUpdate;
     Result:=true;
    end;
  end
end;

function TpFIBDataSet.PSExecuteStatement(const ASQL: string; AParams: TParams;
       ResultSet: Pointer = nil): Integer;
var
  vDataset: TpFIBDataSet;
  vQuery  : TpFIBQuery;
  i,j : Integer;
  sqlStr : string;

procedure AssignParams(qry:TFIBQuery);
var i:integer;
begin
  for i := 0 to AParams.Count - 1 do
   qry.Params[i].Value := AParams[i].Value;
end;

begin
  // Create SQLText
   sqlStr := ASQL;
   j:=0;
   repeat
    i:=Pos('?',sqlStr);
    if i>0 then
    begin
      sqlStr:=Copy(sqlStr,1,i-1)+':Param'+IntToStr(j)+Copy(sqlStr,i+1,MaxInt);
      Inc(j);
    end;
   until i=0;

  if ResultSet=nil then
  begin
   vQuery  :=GetQueryForUse(UpdateTransaction,sqlStr);
   try
     AssignParams(vQuery);
     vQuery.ExecQuery;
    if vQuery.SQLType = SQLSelect then
    begin
     while not vQuery.Eof do vQuery.Next;
     Result := vQuery.RecordCount;
    end
    else
     Result := vQuery.RowsAffected;
   finally
     FreeQueryForUse(vQuery);
   end;
  end
  else
  begin
    vDataset := TpFIBDataSet.Create(nil);
    try
      vDataset.Database := Database;
      vDataset.Transaction := Transaction;
      if not Transaction.InTransaction then
        Transaction.StartTransaction;

      vDataset.SelectSQL.Text := sqlStr;

     AssignParams(vDataset.QSelect);
     vDataset.Open;
     if vDataset.QSelect.SQLType = SQLSelect then  begin
       vDataset.FetchAll;
       Result := vDataset.RecordCount;
      end
      else
       Result := vDataset.QSelect.RowsAffected;
    finally
      if Assigned(ResultSet) then
        TDataSet(ResultSet^) := vDataset
      else
        vDataset.Free;
    end;
  end;


end;

procedure TpFIBDataSet.PSReset;
begin
  inherited PSReset;
  if Active then
  begin
    CloseOpen(false)
  end;
end;

procedure TpFIBDataSet.PSExecute;
begin
  QSelect.ExecQuery
end;

function  TpFIBDataSet.PSGetParams: TParams;
var   i: integer;
      CurParam:TParam;
      FldType: TFieldType;
begin
 if FParams=nil then
  FParams:=TParams.Create
 else
  FParams.Clear;
 Result :=FParams;
 for  i:= 0 to Pred(Params.Count) do
 begin
  case Params[i].SQLType of
    SQL_TEXT, SQL_VARYING : FldType:=ftString;
    SQL_DOUBLE, SQL_FLOAT : FldType:=ftFloat;
    SQL_SHORT             : FldType:=ftSmallint;
    SQL_LONG              : FldType:=ftInteger;
    SQL_INT64             : FldType:=ftFloat;       // may be BCD for D6+?
    SQL_TIMESTAMP         : FldType:=ftDateTime;
    SQL_TYPE_TIME         : FldType:=ftTime;
    SQL_TYPE_DATE         : FldType:=ftDate;
    SQL_BLOB,SQL_ARRAY    : FldType:=ftBlob;
  else
   FldType:=ftString;
  end;
  CurParam:=FParams.CreateParam(
              FldType, Params[i].Name, ptInputOutput
            );
  CurParam.Value:=Params[i].Value
 end;
end;

procedure TpFIBDataSet.PSSetParams(AParams: TParams);
var   i: integer;
      CurParam:TFIBXSQLVAR;
begin
 if Assigned(AParams) then
  for  i:= 0 to Pred(AParams.Count) do
  begin
    CurParam :=FindParam(AParams[i].Name);
    if CurParam<>nil then
     CurParam.Value:=AParams[i].Value;
  end;
end;

{$ENDIF}



function TpFIBDataSet.GetFIBVersion: string;
begin
 Result:=IntToStr(FIBPlusVersion)+'.'+ IntToStr(FIBPlusBuild)+'.'+
  IntToStr(FIBCustomBuild)
end;

procedure TpFIBDataSet.SetFIBVersion(const vs:string);
begin

end;

end.




