/* PhotoOrganizer - HTTPCourier 
 * Copyright (C) 1999-2001 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * $Id: HTTPCourier.java,v 1.9 2001/06/27 20:22:44 rogatkin Exp $
 */
package photoorganizer.courier;

import java.io.*;
import java.net.*;
import java.util.*;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.renderer.*;
import photoorganizer.formats.*;
import photoorganizer.Courier;

// TODO: remove calculation content length, HttpURLConnection does it

public class HTTPCourier implements Courier {
	public final static String SECNAME = "HTTPCourier";
	public final static String CRLF = "\r\n";
    final static String CONTENT_TYPE = "Content-Type";
    final static String CONTENT_LENGTH = "Content-Length";
    final static String SET_COOKIE = "Set-Cookie";

    final static String USER_AGENT="User-Agent";

    final static String CONTENT_DISP = "Content-Disposition: form-data; name=\"";
    final static String FILENAME = "\"; filename=\"";
    final static String CONTENT_ENCODING = "Content-Encoding: ";
    final static String CONTENT_TYPE_ = CONTENT_TYPE+": ";
    final static String MULTIPART = "multipart/form-data; boundary=";
    final static String POST_ENCODING = "application/x-www-form-urlencoded";
    final static String DEFAULT_CONTENTTYPE = "application/octet-stream";
    final static String SEP = "--";
    final static String COOKIE = "Cookie";
	final static String SECURE = "secure";
	

    final static String DEFAULT_AGENT= "Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt)";

    final static int AUTH_COOKIE = 1;
    final static int METHOD_GET  = 0;
    final static int METHOD_POST = 1;

    final static int BUFSIZE = 1024;
	final private static boolean debugHeaders = false;

    Controller controller;
    Serializer s;
	boolean manualMode;
	TwoPanesView mmView;
	String albumName;

	public HTTPCourier(Controller controller) {
		this(controller, null);
	}
	
	public HTTPCourier(Controller controller, String _albumName) {
		this.controller = controller;
		this.albumName = _albumName;
		if (this.albumName != null && this.albumName.length() == 0)
			this.albumName = null;
		s = controller.getSerializer();
		manualMode = Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTP_MANUAL_MODE), 0) != 0;
		if (manualMode) {
			mmView = TwoPanesView.createFramed(false, null, Controller.BTN_MSK_OK+Controller.BTN_MSK_CANCEL, null); 
			mmView.setSize(300, 500);
		}
	}

	public void deliver(StringBuffer buf, String destPath, String contentType, String encoding) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		// compute content length
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		PrintWriter osw = null;
		try {
			osw = new PrintWriter(new OutputStreamWriter(con.getOutputStream(),	encoding));
		} catch(Exception e) { // UnsupportedEncodingException, NullPointerException
			osw = new PrintWriter(con.getOutputStream()); // use default encoding
		}
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.print("\""+CRLF);
		osw.print(CRLF);
		osw.print(destPath+CRLF);
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(destPath); osw.print("\""+CRLF);
		if (contentType != null && contentType.length() > 0) {
			osw.print(CONTENT_TYPE_);
			osw.print(contentType+CRLF);
		}
		if (encoding != null && encoding.length() > 0) // do not use for a while 
			;
		osw.print(CRLF);
		osw.print(buf+CRLF);
		osw.print(SEP+boundary+SEP+CRLF);
		osw.close();
		saveCookies(con, cookieValues);
		reportConnectionStatus(con, "HTML", false, false);
		con.disconnect();
	}

	public void deliver(String srcPath, String destPath) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		// compute content length
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		String name = new File(srcPath).getName();
		if (destPath == null)
			destPath = "";
		if (destPath.length() > 0)
			if (destPath.charAt(destPath.length()-1) != '/' && destPath.charAt(destPath.length()-1) != '\\')
				destPath += '/';        
		destPath += name;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		// TODO: read constant parameters from configuration and add them as form parameters
		String contentType = null;
		try {
			contentType = URLConnection.getFileNameMap().getContentTypeFor(srcPath);
		} catch(Throwable t) {
			// JDK 1.1
		} 
		if (contentType == null || contentType.length() == 0) {			
			contentType = DEFAULT_CONTENTTYPE;
			System.err.println("Content type for "+srcPath+" not found, "+contentType+" will be used.");
		}
		File sf = new File(srcPath);
		OutputStream os;
		PrintWriter osw = new PrintWriter(os = con.getOutputStream());
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.print("\""+CRLF);
		osw.print(CRLF);
		osw.print(destPath+CRLF);
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(srcPath); osw.print("\""+CRLF);
		osw.print(CONTENT_TYPE_);
		osw.print(contentType+CRLF);
		osw.print(CRLF);
		if (osw.checkError())
			System.err.println("Error happened in output stream at writing form fields.");
		Controller.copyFile(sf, os);
		if (osw.checkError())
			System.err.println("Error happened in output stream at writing form data.");
		osw.print(CRLF);
		osw.print(SEP+boundary+SEP+CRLF);
		if (osw.checkError())
			System.err.println("Error happened in output stream at finishing.");
		osw.close();
		saveCookies(con, cookieValues);
		reportConnectionStatus(con, "IMAGE", false, false);
		con.disconnect();
	}
	
	
	public void checkForDestPath(String path) throws IOException {
	}
	
	public String deliver(AbstractFormat format, String destPath) throws IOException {
		HttpURLConnection con = getConnectedToPublish();
		String boundary = genBoundary();
		con.setRequestProperty(CONTENT_TYPE, MULTIPART+boundary);
		String newName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DEST_NAME);
		if (newName == null || newName.length() == 0)
			newName = WebPublishOptionsTab.UPL_DEST_NAME;
		AbstractInfo ii = format.getInfo();
		if (ii == null)
			return "";
		String name = FileNameFormat.makeValidPathName(new FileNameFormat(mask, true).format(format), format.getThumbnailType());
		if (destPath == null)
			destPath = "";
		if (destPath.length() > 0)
			if (destPath.charAt(destPath.length()-1) != '/' && destPath.charAt(destPath.length()-1) != '\\')
				destPath += '/';        
		destPath += name;
		String dataName = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_DATA_NAME);
		if (dataName == null || dataName.length() == 0)
			dataName = WebPublishOptionsTab.UPL_DATA_NAME;
		String contentType = "image/"+format.getThumbnailType();
		ByteArrayOutputStream os = new ByteArrayOutputStream(8192);
		if (ii instanceof AbstractImageInfo)
			((AbstractImageInfo)ii).saveThumbnailImage((BasicJpeg)format, os);
		else
			os.write(format.getThumbnailData(null/*get desired dimension from props*/));
		OutputStream os2;
		PrintWriter osw = new PrintWriter(os2 = con.getOutputStream());
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(newName); osw.print("\""+CRLF);
		osw.print(CRLF);
		osw.print(destPath+CRLF);
		osw.print(SEP+boundary+CRLF);
		osw.print(CONTENT_DISP);
		osw.print(dataName); osw.print(FILENAME); osw.print(name); osw.print("\""+CRLF);
		osw.print(CONTENT_TYPE_);
		osw.print(contentType+CRLF);
		osw.print(CRLF);
		if (osw.checkError())
			System.err.println("Error happened in output stream at writing form fields.");
		os2.write(os.toByteArray());
		if (osw.checkError())
			System.err.println("Error happened in output stream at writing thumbnail.");
		osw.print(CRLF);
		osw.print(SEP+boundary+SEP+CRLF);
		osw.close();
		os.close();
		saveCookies(con, cookieValues);
		reportConnectionStatus(con, "THUMBNAIL", false, false);
		con.disconnect();
		return destPath;
	}

	public void init() throws IOException {
		// do login if authentication requested
		authMode = Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTP_AUTH_SHC), 0);
		boolean isGet = Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINMETHOD), METHOD_GET) == METHOD_GET;
		if (authMode == AUTH_COOKIE) {
			cookieValues = new Hashtable(2);
			HttpURLConnection con = null;
			HttpURLConnection.setFollowRedirects(false);
			String query = URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGIN_NAME));
			query += '=';
			query += URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGIN));
			query += '&';
			query += URLEncoder.encode((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPPASSWORD_NAME));
			query += '=';
			String sp = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPPASSWORD);
			if (sp != null) {
				try {
					query += URLEncoder.encode(Controller.encryptXor(
						new String(BaseController.hexToBytes(sp), BaseController.ISO_8859_1)));
				} catch(UnsupportedEncodingException uee) {
					throw new IOException("Unsupported encoding in password decryption: "+uee);
				}
			}
			query += '&';
			// no URL encode for additional part, TODO: make a tokenizer and do encode
			query += (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPSTATICQUERY);
			if (isGet) {
				try {
					URL tu = new URL((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINURL)+'?'+query);
					con = (HttpURLConnection)tu.openConnection();					
				} catch(MalformedURLException mue) {
					throw new IOException("MalformedURLException at GET method: "+mue);
				}
				// set cookie if requested
				restoreCookies(con, cookieValues);
				con.setRequestProperty(USER_AGENT, DEFAULT_AGENT);
			} else { // POST
				try {
					con = (HttpURLConnection)new URL((String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPLOGINURL)).openConnection();
					con.setDoOutput(true);
					con.setUseCaches(false);
					con.setAllowUserInteraction(false);
					con.setRequestProperty(USER_AGENT, DEFAULT_AGENT);
					// set cookie if requested
					restoreCookies(con, cookieValues);
					// note: not necessary to set method, content type, and lenght
					// HttpURLConnection does it authomatically (
					// send parameters
					PrintWriter out = new PrintWriter(con.getOutputStream());
					out.print(query+CRLF);
					if (out.checkError())
						System.err.println("Error happened in output stream at POST query.");
					out.close();
				} catch(MalformedURLException mue) {
					throw new IOException("MalformedURLException at POST method: "+mue);
				}
			}
			if (con != null) {
				int respCode = con.getResponseCode();
				// save cookie
				saveCookies(con, cookieValues);
				if (respCode == HttpURLConnection.HTTP_OK) {
					reportConnectionStatus(con, "LOGIN", false, true);
				} else if (respCode>= 300 && respCode <=305) {
					String redirectPath = con.getHeaderField(Resources.HDR_LOCATION);
					reportConnectionStatus(con, "LOGIN", false, false);
					URL origUrl = con.getURL();
					con.disconnect();
					try {
						con = (HttpURLConnection)new URL(origUrl, redirectPath).openConnection();
						restoreCookies(con, cookieValues);
						con.setRequestProperty(USER_AGENT, DEFAULT_AGENT);
						saveCookies(con, cookieValues);
						reportConnectionStatus(con, "LOGIN-REDIRECT", false, false);
					} catch(MalformedURLException mue) {
						throw new IOException("MalformedURLException at redirect: "+mue);
					}
				} else
					reportConnectionStatus(con, "LOGIN", false, false);
				con.disconnect();
			}
		} // else auth not cookie
		String uploadUrl = s.arrayToString(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.UPL_SERVLET_URL));
		if (albumName != null && uploadUrl != null) {
			if (uploadUrl.indexOf('?') < 0)
				uploadUrl += "?";
			else if (uploadUrl.charAt(uploadUrl.length()-1) != '&')
				uploadUrl += "&";
			String name = s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPALBUMNAME).toString();
			if (name == null || name.length() == 0)
				name = s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.HTTPALBUMID).toString();
			if (name != null && name.length() > 0) {
				uploadUrl += name;
				uploadUrl += '=';
				uploadUrl += albumName;
			}
		}
		try {
			publisherURL = new URL(uploadUrl);
		} catch(MalformedURLException mue) {
			throw new IOException("Invalid upload URL format "+uploadUrl);
		} catch(NullPointerException npe) {
			  throw new IOException("Upload URL not specified "+uploadUrl);
		}

		mask = (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FILEMASK);
		if (mask == null || mask.length() == 0)
			mask = PhotoCollectionPanel.DEFTNMASK;
	}

	public void done() {

	}

	public boolean isLocal() {
		return false;
	}
	
	public boolean isContentIncluded() {
		return false;
	}

	private String genBoundary() {
		return Long.toHexString(new Random().nextLong());
	}
	
	private void reportConnectionStatus(HttpURLConnection _con, String _msg, boolean _cookies, boolean _content) throws IOException {
		int code = _con.getResponseCode();
		System.err.println("************* "+_msg+" *************");
		System.err.println("Request: "+_con.getURL());
		System.err.println("Server returned: "+ code + "/"+_con.getResponseMessage());
		if (code>= 300 && code <=305)
			System.err.println("Redirect requested to: "+ _con.getHeaderField(Resources.HDR_LOCATION));
		if(_cookies)
			System.err.println("Cookies: "+cookieValues);
		if(_content)
			System.err.println("Returned content: \n"+
			new String(Controller.readStreamInBuffer(_con.getInputStream())));
		if (manualMode) {
			InputStream in;
			mmView.readToUpper(in=_con.getInputStream());
			in.close();
			if (Resources.CMD_OK.equals(mmView.showModal()))
				manualMode = false;
		}	
	}

	private HttpURLConnection getConnectedToPublish() throws IOException {
		HttpURLConnection result = (HttpURLConnection)publisherURL.openConnection();
		result.setDoOutput(true);
		result.setDoInput(true);
		result.setUseCaches(false);
		result.setAllowUserInteraction(false); //
		result.setRequestProperty(USER_AGENT, DEFAULT_AGENT);
		if (authMode == AUTH_COOKIE)
			restoreCookies(result, cookieValues);
		return result;
	}
    
	static public void saveCookies( URLConnection _con, Map _cookieValues ) {
		for (int i=0; _con.getHeaderField(i)!=null; i++) {
			if (debugHeaders)
				System.err.println("Header "+i+"-->"+_con.getHeaderFieldKey(i)+':'+_con.getHeaderField(i));
			if (SET_COOKIE.equalsIgnoreCase(_con.getHeaderFieldKey(i))) {
				String cookie = _con.getHeaderField(i);
				int pos = cookie.indexOf(';');
				if (cookie.lastIndexOf(SECURE) > pos) // skip the cookie because not secure conn
					continue;
				// TODO: check for expiration and remove
				if (pos > 0) {
					cookie = cookie.substring(0, pos);
					pos = cookie.indexOf('=');
					if (pos > 0)
						_cookieValues.put(cookie.substring(0, pos), cookie);
				}
			}
		}
	}

	static public void restoreCookies( URLConnection _con, Map _cookieValues ) {
		String cookies = null;
		Iterator i = _cookieValues.entrySet().iterator();
		while(i.hasNext())
			if (cookies == null)
				cookies = (String)((Map.Entry)i.next()).getValue();
			else {
				cookies += "; ";
				cookies += (String)((Map.Entry)i.next()).getValue();
			}
		if (cookies != null)
			_con.setRequestProperty(COOKIE, cookies);
	}			
		
    URL publisherURL;
    String mask;
    int authMode;
    Map cookieValues;
}

class myPrintWriter extends PrintWriter {
	
	myPrintWriter(OutputStream _os) {
		super(_os);
	}
	
	public void print(String _string) {
		super.print(_string);
		System.err.print(_string);
	}

	public void println(char _char) {
		super.println(_char);
		System.err.println(_char);
	}

	public void println() {
		super.println();
		System.err.println();
	}
}