/*
 * 29/01/00		Initial version. mdm@techie.com
/*-----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------
 */



import java.util.Hashtable;
import java.util.Enumeration;



/**
 * The <code>AP_FactoryRegistry</code> class stores the factories
 * for all the audio device implementations available in the system.
 * <p>
 * Instances of this class are thread-safe.
 *
 * @since 0.0.8
 * @author Mat McGowan
 */

public class AP_FactoryRegistry extends AP_AudioDeviceFactory
{
	static private AP_FactoryRegistry instance = null;

	static synchronized public AP_FactoryRegistry systemRegistry()
	{
		if (instance==null)
		{
			instance = new AP_FactoryRegistry();
			instance.registerDefaultFactories();
		}
		return instance;
	}


	protected Hashtable factories = new Hashtable();

	/**
	 * Registers an <code>AP_AP_AudioDeviceFactory</code> instance
	 * with this registry.
	 */
	public void addFactory(AP_AudioDeviceFactory factory) {
		factories.put(factory.getClass(), factory);
	}

	public void removeFactoryType(Class cls) {
		factories.remove(cls);
	}

	public void removeFactory(AP_AudioDeviceFactory factory) {
		factories.remove(factory.getClass());
	}

	public AP_AudioDevice createAudioDevice() throws AD_JavaLayerException {
		AP_AudioDevice device = null;
		AP_AudioDeviceFactory[] factories = getFactoriesPriority();
		if (factories==null)
			throw new AD_JavaLayerException(this + ": no factories registered");
		AD_JavaLayerException lastEx = null;
		for (int i=0; (device==null) && (i<factories.length); i++) {
			try {
				device = factories[i].createAudioDevice();
			}
			catch (AD_JavaLayerException ex) {
				lastEx = ex;
			}
		}
		if (device==null && lastEx!=null) {
			throw new AD_JavaLayerException("Cannot create AP_AudioDevice", lastEx);
		}
		return device;
	}


	protected AP_AudioDeviceFactory[] getFactoriesPriority() {
		AP_AudioDeviceFactory[] fa = null;
		synchronized (factories) {
			int size = factories.size();
			if (size!=0) {
				fa = new AP_AudioDeviceFactory[size];
				int idx = 0;
				Enumeration e = factories.elements();
				while (e.hasMoreElements()) {
					AP_AudioDeviceFactory factory = (AP_AudioDeviceFactory)e.nextElement();
					fa[idx++] = factory;
				}
			}
		}
		return fa;
	}

	protected void registerDefaultFactories() {
		// FIXME!!!!!!  This should be automatic
		//
		// Use EightBitAP_AP_AudioDeviceFactory for browser play,
		// AP_JavaSoundAP_AP_AudioDeviceFactory for application or Netscape 6 play
            /*String version = System.getProperty("java.version");
            if(version.startsWith("1.1"))*/
      	      addFactory(new AP_EightBitAudioDeviceFactory());
            /*else
	     addFactory(new AP_JavaSoundAudioDeviceFactory());*/
	}
}
