//	<><><><><><><><><><><><><>  Plot3DF.h  <><><><><><><><><><><><><><> 
//
// -----------------------------------------------------------------
//
// Plot3DF is an abstract C++ class used to
// plot a 3D view of a function defined as f(x,y,z)=0
//
// Highlights:
//   1) Plots an arbitrary equation func(x,y,z)=0
//   2) Can also plot parametrically x(u,v), y(u,v), z(u,v)
//   3) Cross sections are cut through the object and drawn
//   4) Plot to any type of device;
//      MoveTo(), DrawTo() pure virtual methods must be
//      supplied by the derived class to do the actual plotting.
//   5) User selectable 3D view point.
//   6) Supports hidden line removal.
//   7) Provides for scaling.
//   8) Video resolution is settable.
//   9) Allows for user supplied (optional) call-back functions
//      that can be used to monitor plotting progress.
//  10) Has the ability to produce color gradient plots.
//      Virtual method ColorTo() is overridable.
//
// Limitations:
//   f(x,y,z) must produce a sign change in order to be plottable
//
// Author:
//   Clark Dailey
//   9037 Cambridge Road
//   Woodbury, MN 55125
//   ccdailey@imation.com
//   (651)-714-9036
//
// This source code may be modified or used as is,
// as long as credit is given to the author.
//
// -----------------------------------------------------------------

#ifndef	_PLOT3DF_H_	 	// avoid repeated including
#define	_PLOT3DF_H_	1	// has been included

#include "..\common\plot3d.h"

// Plotting modes
enum PlotMode
{
     PLOT_CROSS_SECTIONS=0,
     PLOT_PARAMETRIC=1
};

// Clipping modes
enum ClipMode
{
     CLIP_NONE=0,
     CLIP_USER=1,
	 CLIP_CUBE=2,
	 CLIP_SPHERE=3,
	 LAST_CLIP_VALUE=3 // update this when adding a new value
};


class Plot3DF : public Plot3D // abstract class - must be derived from to use
{
public:
	Plot3DF();	// constructor
	virtual ~Plot3DF();	// destructor

	// pure virtual functions must be defined by derived class
	virtual double Func(double x,double y,double z)=0;	// the mathematical plotting function func(x,y,z)=0

	// overridable clipping function; return: 0=clipped, 1=point is within drawable region
	virtual int Within(double x,double y,double z)
	{
		int	v;
		if (m_clip_mode == CLIP_NONE) return(1);
		if (m_clip_mode == CLIP_CUBE)
		{
			v = (x>=m_xmin && x<=m_xmax && y>=m_ymin && y<=m_ymax && z>=m_zmin && z<=m_zmax) ? 1 : 0;
			if (m_is_clip_inverted) v = !v;
			return(v);
		}
		if (m_clip_mode == CLIP_SPHERE)
		{
			double dx,dy,dz,r;
			dx = x - m_sphere_xcenter;
			dy = y - m_sphere_ycenter;
			dz = z - m_sphere_zcenter;
			r = sqrt(dx*dx+dy*dy+dz*dz);
			v = (r <= m_sphere_radius) ? 1 : 0;
			if (m_is_clip_inverted) v = !v;
			return(v);
		}
		// Note: derived class handles user defined clipping
		return(1);
	}

	// Parameteric form of the equation
	virtual double XParametric(double u,double v)=0;
	virtual double YParametric(double u,double v)=0;
	virtual double ZParametric(double u,double v)=0;

	// settings
	int SetParameters(unsigned nzplanes=50,unsigned remove_hidden_lines=1);

	// set spherical range
	int SetSphereRange(int UseSphereRange,double r=1.,double xc=0.,double yc=0.,double zc=0.)
	{
		int rc = m_use_sphere_range;
		m_use_sphere_range = UseSphereRange;
		m_sphere_radius  = fabs(r);
		m_sphere_xcenter = xc;
		m_sphere_ycenter = yc;
		m_sphere_zcenter = zc;
		return(rc); // return previous setting
	}

	// returns: 0=ok, 1=invalid value
	int SetParametric(int UseParametric,unsigned nulines,unsigned nvlines,double umin,double umax,double vmin,double vmax,unsigned nuincs=100,unsigned nvincs=100)
	{
		if (nulines<2 || nvlines<2 ||
			nuincs <2 || nvincs<2  ||
			umin>=umax || vmin>=vmax) return(1);
		m_plot_mode = (UseParametric) ? PLOT_PARAMETRIC : PLOT_CROSS_SECTIONS;
		m_nulines = nulines;
		m_nvlines = nvlines;
		m_nuincs  = nuincs;
		m_nvincs  = nvincs;
		m_umin = umin;
		m_umax = umax;
		m_vmin = vmin;
		m_vmax = vmax;
		return(0);
	}

	// set clipping mode; 0=ok, 1=invalid parameter
	int SetClipMode(ClipMode mode=CLIP_NONE,int IsInverted=0)
	{
		if (mode<CLIP_NONE || mode>LAST_CLIP_VALUE) return(1);
		m_clip_mode = mode;
		m_is_clip_inverted = IsInverted;
		return(0);
	}

	// returns: 0=no, 1=point is visible
	int IsVisible(double xp,double yp,double zp);

	// plotting
	int Plot();

	// call back routines
	virtual void cbBegZPlane(double zp) { }	// called at beginning of plotting of constant zp plane
	virtual void cbEndZPlane(double zp) { }	// called when plotting of zp plane is complete
	virtual void cbBegULine(double u) { }	// called at beginning of plotting of constant u line
	virtual void cbEndULine(double u) { }	// called when plotting of u line is complete
	virtual void cbBegVLine(double v) { }	// called at beginning of plotting of constant v line
	virtual void cbEndVLine(double v) { }	// called when plotting of v line is complete

	// debugging routines
	void Dump();

private:	// routines
	int  InitPlot(void);
	void CalcDeltas(void);
	int  AllocateMaskArrays(void);
	void DeleteMaskArrays(void);
	void ZeroMaskPointers(void);
	void PlotMask(unsigned iy);
	int PlotSectional(void);
	int PlotParametric(void);
	void DrawAxis(void);

	// hidden line removal matrix access functions
	int ClearHidden(int ix,int iy);
	int SetHidden(int ix,int iy);
	int GetHidden(int ix,int iy);

protected:	// user input data
	PlotMode	m_plot_mode;	// the plotting mode

	// number of constant zp planes
	unsigned m_nzplanes;

	// flags
	unsigned m_remove_hidden_lines;	// 0=no, 1=remove hidden lines

	// clipping
	ClipMode	m_clip_mode;	// the clipping mode
	int		m_is_clip_inverted;	// 0=no, 1=invert clipped region (inside or outside)

	// spherical range values
	int		m_use_sphere_range;	// 0=use cube range, else use spherical range
	double	m_sphere_radius;	// radius of sphere
	double	m_sphere_xcenter;	// x value of center
	double	m_sphere_ycenter;	// y value of center
	double	m_sphere_zcenter;	// z value of center

	// parametric parameters
	unsigned m_nulines, m_nvlines;	// number of u,v contour lines to draw
	unsigned m_nuincs,  m_nvincs;	// number of u,v increments per line
	int		m_draw_u, m_draw_v;		// 0=dont draw, 1=draw u/v line
	double	m_umin, m_umax;			// range of u parameter
	double	m_vmin, m_vmax;			// range of v parameter

protected:	// internal data
	// values used during plotting
	double m_dxp, m_dyp, m_dzp;

	// coordinate dimensions
	unsigned m_nXps, m_nYps, m_nZps;
	double   m_xstart[2],m_ystart[2], m_zstart[2];	// starting x,y,z points: 0=even,1=odd

	// masking point structure
	typedef struct mask_point_s
	{
		int within;	// 0=no, 1=within drawable area
		int visible;
		long color;
		double xp,yp,zp;
		double x, y, z;
		double f;	// function value
		int   sf;	// 0=(f<0), 0xffff=(f>=0)
	} MASK_POINT;

	// hidden line removal masking arrays
	MASK_POINT* m_pMask1;
	MASK_POINT* m_pMask2;
	unsigned m_mask_size;	// sizeof m_pMask1 and m_pMask2
};


#endif	/* if _PLOT3DF_H_ included */

/*	<><><><><><><><><><><><><>  Plot3DF.h  <><><><><><><><><><><><><><> */
