/*
 * This software is provided as is without any warranty of any kind.
 * You are free to copy and distribute it as you like in source or
 * binary form provided that this notice and author credits remain intact
 * in the source.  You are also free to alter, enhance, embellish, or 
 * otherwise change its content provided that the credits to the
 * original author remain, and that any changes are commented stating
 * the change made and the author of that change.  You may NOT profit in
 * any way from the sale or distribution of this software in its original
 * or altered form.
 */

/*
 * Program : extrfile.c
 * Author  : Jeffrey M. Metcalf
 * Date    : 14-May-98
 * Re      : extract a length 'l' size chunk from a file
 *           starting at offset 'o'.
 */

#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>


/*
 * Uncomment the following if compiling
 * with Cygnus B-19 GNU environment for Win32.
 */

#define WIN32

/*
 * Include file 'getopt.h' in Cygnus B-19 GNU environment.
 * May or may not be applicable in other GNU Win32
 * environments.
 */

#ifdef WIN32
#include "getoptux.c"
#endif



unsigned long fileinfo(char *);
char *bufoffread(char *, unsigned long, unsigned long);
void spitfilename(char *, char *, unsigned long);
void usage(void);

char *progname;             /* name of the executable that invokes us */

int
main(argc, argv)
	int argc;
	char *argv[];
{
	extern int optind;
	extern char *optarg;
	extern int errno;

	char *buf;                   /* buffer to contain the file slice */
	unsigned long offset;        /* the offset at which the slice starts */
	unsigned long len;           /* total length of the slice in bytes */
	unsigned long filesize;      /* size of the file in bytes */
	int ch;

	setlocale(LC_CTYPE, "");

	progname = *argv;

	while ((ch = getoptux(argc, argv, "o:l:?h")) != EOF)
		switch (ch) {
		case 'o':                  /* get offset to begin extract */
			offset = strtoul(optarg, NULL, 0);
			break;
		case 'l':                  /* get length of extract in bytes */
			len = strtoul(optarg, NULL, 0);
			break;
		case '?':
		case 'h':
		default:
			usage();
		}
	argc -= optind;
	argv += optind;

	if (argc != 2)
		usage();

	filesize = fileinfo(*argv);    /* get the size of the source file */

	if (offset > filesize) {       /* don't set offset too high */
		(void)fprintf(stderr,
			"%s: offset greater than filesize\n", progname);
		exit(1);
	}

	buf = bufoffread(*argv,offset,len); /* seek to offset; get file slice */

	argc--;
	argv++;

	spitfilename(*argv, buf, len);      /* output file slice */
	return errno;
}


unsigned long fileinfo(char *filen) {      /* function returns file size */

        struct stat stbuf;     /* inode data structure */
	unsigned long bs;

	if (stat(filen, &stbuf) == -1) {
		(void)fprintf(stderr,
			"%s: can't stat %s\n", progname, filen);
		exit(1);
	}
	if ((bs = (unsigned long)stbuf.st_size) == 0) {
	        (void)fprintf(stderr,
			"%s: empty file %s\n", progname, filen);
		exit(1);
	}

	return bs;
}


/*
 * the following function opens the file 'filen', seeks to offset 'os,
 * reads 'bs' bytes into a buffer, and returns a pointer to the buffer
 */
char *bufoffread(char *filen, unsigned long os, unsigned long bs) {

        char *bf;     /* pointer to buffer to allocate, fill, and return */
	FILE *fp;     /* file pointer */
#ifdef WIN32
        const char *mode="rb";   /* DOS text file standard cr-lf */
#else
        const char *mode="r";
#endif

	if ((fp = fopen(filen, mode)) == NULL) {
		(void)fprintf(stderr,
			"%s: can't open %s\n", progname, filen);
		exit(1);
	}
	if ((bf = (char *)malloc(bs)) == NULL) {
	        (void)fprintf(stderr,
			"%s: can't malloc buffer for %s\n", progname, filen);
		exit(1);
	}

	if (fseek(fp, os, SEEK_SET) == -1) {
		(void)fprintf(stderr,
			"%s: can't seek file %s\n", progname, filen);
		exit(1);
	}

	if (fread(bf,bs,1,fp) < 1) {
		(void)fprintf(stderr,
			"%s: buffer for %s smaller than expected\nperhaps offset 0x%x too high for bufsize %u\n",
				progname, filen, os, bs);
		exit(1);
	}
	if (fclose(fp) == EOF)
		(void)fprintf(stderr,
			"%s: error closing %s\n", progname, filen);

	return bf;
}


/*
 * the following function outputs the buffer 'outbuf' of size 'bs' bytes
 * to a file named 'outptfn'
 */
void spitfilename(char *outptfn, char *outbuf, unsigned long bs) {

	FILE *fp;
#ifdef WIN32
        const char *mode="wb";
#else
        const char *mode="w";
#endif

	if ((fp = fopen(outptfn,mode)) == NULL) {
	        (void)fprintf(stderr,
			"%s: can't create output file %s\n",progname, outptfn);
	}
	else if (fwrite(outbuf, bs, 1, fp) < 1) {
	        (void)fprintf(stderr,
			"%s: error writing %s\n", progname, outptfn);
	}
	if ((fp != NULL) && (fclose(fp) == EOF))
	        (void)fprintf(stderr,
			"%s: error closing %s\n", progname, outptfn);
}

void usage(void) {              /* print usage and exit with return value 1 */
	(void)fprintf(stderr,
		"Usage: %s -o offset -l length  file ofile\n", progname);
	exit(1);
}

