/*
 * An add-on utility for Duncan Murdoch's DOSLFNBK program
 * to create a long file name backup file from an NTFS
 * partition
 *
 * Emmet P. Gray              US Army, HQ III Corps & Fort Hood
 * graye@hood-emh3.army.mil   Attn: AFZF-PW-ENV
 *                            Directorate of Public Works
 *                            Environmental Division
 *                            Fort Hood, TX 76544-5057
 */

#undef DEBUG
#ifdef DEBUG
#pragma option -f-
#else
#pragma option -f- -R- -v-
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <dirent.h>
#include <fcntl.h>
#include <io.h>
#include <sys/stat.h>

#define NAME "ntlfnbk"
#define VERSION "v1.0"
#define DATED "3 May 97"
#define DEF_NAME "BACKUP.LFN"

int optind = 1;
char *optarg;
int subdir, all, prune, quiet, compat, files, dirs, out_fd;
char lpath[MAXPATH], spath[MAXPATH];

int
main(argc, argv)
int argc;
char *argv[];
{
   int c, getopt(int, char**, char*);
   char output[MAXPATH], start[MAXPATH];
   void usage(void), traverse(char*);

   subdir = 0;
   all = 0;
   prune = 0;
   quiet = 0;
   compat = 0;
   files = 0;
   dirs = 0;
   strcpy(output, DEF_NAME);

   while ((c = getopt(argc, argv, "sapqcf:")) != EOF) {
      switch(c) {
         case 's':      /* include subdirectories */
            subdir++;
            break;
         case 'a':      /* include all files */
            all++;
            break;
         case 'p':      /* prune the starting directory info */
            prune++;
            break;
         case 'q':      /* operate quietly */
            quiet++;
            break;
         case 'c':      /* compatible with DOSLFNBK versions < 2.0 */
            compat++;
            break;
         case 'f':      /* specify an output file */
            strcpy(output, optarg);
            break;
         default:
            usage();
            exit(1);
      }
   }

                        /* early sanity checking... */
   if ((argc - optind) != 1) {
      fprintf(stderr, "A starting directory is required\n");
      usage();
      exit(1);
   }

   if (access(argv[optind], 0)) {
      fprintf(stderr, "Can't find starting directory\n");
      usage();
      exit(1);
   }
                        /* does output file already exist? */
   if (!quiet) {
      if (!access(output, 0)) {
         int done;
         char ans[10];

         done = 0;
         while(!done) {
            printf("Overwrite \"%s\" [yes, No] : ", output);
            fgets(ans, 10, stdin);

            switch(toupper(ans[0])) {
               case 'Y':
                  done++;
                  break;
               case '\n':
               case 'N':
                  return(1);
               default:
                  fputc('\a', stderr);
            }
         }
      }
   }

   if ((out_fd = open(output, O_CREAT|O_TRUNC|O_BINARY|O_WRONLY,S_IREAD|S_IWRITE)) < 0) {
      fprintf(stderr, "Can't open \"%s\" for write\n", output);
      return(1);
   }
                        /* convert to absolute path */
   if ((argv[optind][1] != ':') && (argv[optind][0] != '\\')) {
      getcwd(start, MAXPATH);
      if (start[strlen(start)-1] != '\\')
         strcat(start, "\\");
      strcat(start, argv[optind]);
   }
   else
      strcpy(start, argv[optind]);

   if (prune) {
      prune = strlen(start);
      if (start[prune-1] == '\\')
         prune--;
   }
                        /* here we go! */
   traverse(start);

   printf("%79.79s\r", "");
   printf("Processed %d files and %d directories\n", files, dirs);
   close(out_fd);
   return(0);
}

/*
 * Traverse the directory tree
 */

void
traverse(dir)
char *dir;
{
   DIR *dirp;
   int attr, entries;
   void vfat_dir(char*);
   char *s, *file, *sfile, *scwd, len;
   struct dirent *entry;
   static char cwd[MAXPATH] = {'\0'};

                        /* intialize CWD (first time only) */
   if (cwd[0] == '\0')
      strcpy(cwd, dir);

   if ((dirp = opendir(dir)) == NULL)
      return;
   /*
    * We read the directory 3 times!!!  First to get the number of records
    * that will be in the output, then to produce the output, then finally
    * to descend the directory structure
    */
                        /* count the number of entries */
   entries = 0;
   while ((entry = readdir(dirp)) != NULL) {
      file = entry->d_name;

                        /* skip the "." and ".." directory entries */
      if (!strcmp(file, ".") || !strcmp(file, ".."))
         continue;
                        /* reconstruct the long path */
      strcpy(lpath, cwd);
      if (lpath[strlen(lpath)-1] != '\\')
         strcat(lpath, "\\");
      strcat(lpath, file);

                        /* generate the short path */
      GetShortPathName(lpath, spath, MAXPATH);
      if ((sfile = strrchr(spath, '\\')) != NULL)
         sfile++;
      else
         sfile = spath;

                        /* should we skip this file? */
      attr = GetFileAttributes(lpath);
      if (!all && !(attr & 0x7) && !strcmp(file, sfile))
         continue;

      if (!quiet)
         printf("%-79.79s\r", spath);

                        /* account for extra directory entries */
      if (strcmp(file, sfile))
         entries += ((strlen(file)-1) / 13) +2;
      else
         entries++;
   }
   rewinddir(dirp);
                        /* convert cwd to short name */
   GetShortPathName(cwd, spath, MAXPATH);
   if (spath[strlen(spath)-1] != '\\')
      strcat(spath, "\\");
   scwd = spath;
   if (prune)
      scwd += prune;
   else {
      if ((scwd = strchr(spath, ':')) != NULL)
         scwd++;
      else
         scwd = spath;
   }
   len = (char) strlen(scwd);

#ifdef DEBUG
   printf("len=%d, scwd='%s', entries=%d\n", len, scwd, entries);
#else
                        /* write the header info */
   write(out_fd, &len, sizeof(len));
   write(out_fd, scwd, len);
   if (compat) {
      short e;
      e = (short) entries;
      write(out_fd, &e, sizeof(e));
   }
   else
      write(out_fd, &entries, sizeof(entries));
#endif
   /*
    * Second time thru... now produce the output
    */
   while ((entry = readdir(dirp)) != NULL) {
      file = entry->d_name;

                        /* skip the "." and ".." directory entries */
      if (!strcmp(file, ".") || !strcmp(file, ".."))
         continue;
                        /* reconstruct the long path */
      strcpy(lpath, cwd);
      if (lpath[strlen(lpath)-1] != '\\')
         strcat(lpath, "\\");
      strcat(lpath, file);

                        /* generate the short path */
      GetShortPathName(lpath, spath, MAXPATH);
      if ((sfile = strrchr(spath, '\\')) != NULL)
         sfile++;
      else
         sfile = spath;

                        /* should we skip this file? */
      attr = GetFileAttributes(lpath);
      if (!all && !(attr & 0x7) && !strcmp(file, sfile))
         continue;
                        /* write the record */
      vfat_dir(spath);

      if (attr & FILE_ATTRIBUTE_DIRECTORY)
         dirs++;
      else
         files++;
   }
   rewinddir(dirp);

   if (!subdir) {
      closedir(dirp);
      return;
   }

   /*
    * Third time thru... now descend the directory tree
    */
   while ((entry = readdir(dirp)) != NULL) {
      file = entry->d_name;

                        /* skip the "." and ".." directory entries */
      if (!strcmp(file, ".") || !strcmp(file, ".."))
         continue;
                        /* reconstruct the long path */
      strcpy(lpath, cwd);
      if (lpath[strlen(lpath)-1] != '\\')
         strcat(lpath, "\\");
      strcat(lpath, file);

                        /* if a directory */
      attr = GetFileAttributes(lpath);
      if (attr & FILE_ATTRIBUTE_DIRECTORY) {

                        /* make this the Current Working Directory */
         GetShortPathName(lpath, cwd, MAXPATH);

                        /* recursive call to traverse() */
         traverse(cwd);
         continue;
      }
   }
   closedir(dirp);
   /*
    * Remove the last portion of the Current Working Directory.
    * Essentially the same as doing a "CD .."
    */
   if ((s = strrchr(cwd, '\\')) != NULL)
      *s = '\0';
   return;
}

/*
 * Reconstruct a MSDOS-style VFAT directory structure
 */

struct short_slot {           // Short 8.3 names
   unsigned char name[8];     // file name
   unsigned char ext[3];      // file extension
   unsigned char attr;        // attribute byte
   unsigned char lcase;       // Case for base and extension
   unsigned char ctime_ms;    // Creation time, milliseconds
   unsigned char ctime[2];    // Creation time
   unsigned char cdate[2];    // Creation date
   unsigned char adate[2];    // Last access date
   unsigned char reserved[2]; // reserved values (ignored)
   unsigned char mtime[2];    // time stamp
   unsigned char mdate[2];    // date stamp
   unsigned char start[2];    // starting cluster number
   unsigned char size[4];     // size of the file
};

struct long_slot {            // Up to 13 characters of a long name
   unsigned char id;          // sequence number for slot
   unsigned char name0[10];   // first 5 characters in name
   unsigned char attr;        // attribute byte
   unsigned char reserved;    // always 0
   unsigned char checksum;    // checksum for 8.3 alias
   unsigned char name1[12];   // 6 more characters in name
   unsigned char start[2];    // starting cluster number
   unsigned char name2[4];    // last 2 characters in name
};


void
vfat_dir(spath)
char *spath;
{
   char name[15], *ext, *sfile;
   unsigned short mtime, mdate, ctime, cdate, atime, adate;
   int i, len, ctime_ms;
   unsigned char attr;
   unsigned long size;
   FILETIME local, local1;
   struct short_slot sslot;
   struct long_slot lslot;
   WIN32_FIND_DATA fdata;
   HANDLE find;

   if ((find = FindFirstFile(spath, &fdata)) != INVALID_HANDLE_VALUE);
      FindClose(find);

   /*
    * Win95 does not fill in the AlternateName if the name is short, but
    * WinNT always does
    */
   if (strlen(fdata.cAlternateFileName))
      sfile = fdata.cAlternateFileName;
   else
      sfile = fdata.cFileName;

                        /* get length of long file name */
   len = 0;
   if (strcmp(fdata.cFileName, sfile))
      len = strlen(fdata.cFileName);

   strcpy(name, sfile);
   if ((ext = strchr(name, '.')) != NULL) {
      *ext = '\0';
      ext++;
   }
   else
      ext = "";
                        /* strip NT-specific attributes */
   attr = (unsigned char) (fdata.dwFileAttributes & 0x3f);

   /*
    * Convert the Win32 32bit file date/time stamp into local time
    * then into the DOS data format
    */
   FileTimeToLocalFileTime(&(fdata.ftLastWriteTime), &local);
   FileTimeToDosDateTime(&local, (LPWORD) &mdate, (LPWORD) &mtime);
   FileTimeToLocalFileTime(&(fdata.ftCreationTime), &local);
   FileTimeToDosDateTime(&local, (LPWORD) &cdate, (LPWORD) &ctime);
   FileTimeToLocalFileTime(&(fdata.ftLastAccessTime), &local);
   FileTimeToDosDateTime(&local, (LPWORD) &adate, (LPWORD) &atime);

                        /* work it backwards to get ctime_ms */
   DosDateTimeToFileTime(cdate, ctime, &local);
   LocalFileTimeToFileTime(&local, &local1);
   ctime_ms = ((fdata.ftCreationTime).dwLowDateTime - local1.dwLowDateTime) / 100000;

   size = (fdata.nFileSizeHigh * MAXDWORD) + fdata.nFileSizeLow;

                        /* build the long dir slot structure */
   if (len) {
      int first, id, j;
      unsigned char sum, lname[MAXPATH];

      sprintf((char *) lname, "%-8.8s%-3.3s", name, ext);
      sum = 0;
#pragma warn -sig
      for (i=0; i<11; i++)
         sum = (((sum & 1) << 7) | ((sum & 0xfe ) >> 1)) + lname[i];
#pragma warn .sig
                        /* pad the long name with 0xff */
      strcpy((char *) lname, fdata.cFileName);
      for (i=strlen((char *) lname)+1; i<MAXPATH; i++)
         lname[i] = 0xff;

      first = 0;
      i = ((len -1) / 13) * 13;
      for (; i >= 0; i-=13) {
         id = (i / 13) +1;
         if (!first) {
            first++;
            id |= 0x40;
         }
         lslot.id = (unsigned char) id;

                        /* simulate UNICODE */
         for (j=0; j<5; j++) {
            lslot.name0[j*2] = lname[i +j];
            if (lslot.name0[j*2] == 0xff)
               lslot.name0[(j*2)+1] = 0xff;
            else
               lslot.name0[(j*2)+1] = 0;
         }
         lslot.attr = 0x0f;
         lslot.reserved = 0;
         lslot.checksum = sum;

         for (j=0; j<6; j++) {
            lslot.name1[j*2] = lname[5 +i +j];
            if (lslot.name1[j*2] == 0xff)
               lslot.name1[(j*2)+1] = 0xff;
            else
               lslot.name1[(j*2)+1] = 0;
         }
         lslot.start[0] = 0;
         lslot.start[1] = 0;

         for (j=0; j<2; j++) {
            lslot.name2[j*2] = lname[11 +i +j];
            if (lslot.name2[j*2] == 0xff)
               lslot.name2[(j*2)+1] = 0xff;
            else
               lslot.name2[(j*2)+1] = 0;
         }
#ifdef DEBUG
      printf("id=%x, lname='%.13s'\n", id, &lname[i]);
#else
                        /* write the long dir slot structure */
      write(out_fd, &lslot, sizeof(lslot));
#endif
      }
   }
                        /* build the short dir slot structure */
#pragma warn -sig
   sprintf((char *) sslot.name, "%-8.8s", name);
   sprintf((char *) sslot.ext, "%-3.3s", ext);
   sslot.attr = attr;
   sslot.lcase = 0;
   sslot.ctime_ms = (unsigned char) ctime_ms;
   sslot.ctime[0] = ctime % 0x100;
   sslot.ctime[1] = ctime / 0x100;
   sslot.cdate[0] = cdate % 0x100;
   sslot.cdate[1] = cdate / 0x100;
   sslot.adate[0] = adate % 0x100;
   sslot.adate[1] = adate / 0x100;
   sslot.reserved[0] = 0;
   sslot.reserved[1] = 0;
   sslot.mtime[0] = mtime % 0x100;
   sslot.mtime[1] = mtime / 0x100;
   sslot.mdate[0] = mdate % 0x100;
   sslot.mdate[1] = mdate / 0x100;
   sslot.start[0] = 0;
   sslot.start[1] = 0;
   sslot.size[0] = ((size % 0x1000000L) % 0x10000L) % 0x100;
   sslot.size[1] = ((size % 0x1000000L) % 0x10000L) / 0x100;
   sslot.size[2] = (size % 0x1000000L) / 0x10000L;
   sslot.size[3] = size / 0x1000000L;
#pragma warn .sig

#ifdef DEBUG
   printf("name='%-8.8s%-3.3s', attr=%x, size=%ld\n", name, ext, attr, size);
#else
                        /* write the short dir slot structure */
   write(out_fd, &sslot, sizeof(sslot));
#endif
   return;
}

/*
 * The "usage" error message
 */

void
usage()
{
   fprintf(stderr, "%s, Version %s, Dated %s\n", NAME, VERSION, DATED);
   fprintf(stderr, "Usage: %s [-s] [-a] [-p] [-q] [-f output] directory\n", NAME);
   fprintf(stderr, "\t-s include subdirectories (including empty directories)\n");
   fprintf(stderr, "\t-a include all files\n");
   fprintf(stderr, "\t-p prune the starting directory info\n");
   fprintf(stderr, "\t-q operate quietly\n");
   fprintf(stderr, "\t-c make compatible with DOSLFNBK versions < 2.0\n");
   fprintf(stderr, "\t-f specify an output file (default is %s)\n", DEF_NAME);
   return;
}

/*
 * The Unix-style option parsing routine
 */

int
getopt(argc, argv, opts)
int argc;
char *argv[];
char *opts;
{
   static int sp = 1;
   int c;
   char *cp;

   if (sp == 1) {
      if (optind >= argc || argv[optind][0] != '-' || argv[optind][1] == '\0')
         return(EOF);
      else if (strcmp(argv[optind], "--") == NULL) {
         optind++;
         return(EOF);
      }
   }
   c = argv[optind][sp];
   if (c == ':' || (cp = strchr(opts, c)) == NULL) {
      fprintf(stderr, "Illegal option '%c'\n", c);
      if (argv[optind][++sp] == '\0') {
         optind++;
         sp = 1;
      }
      return('?');
   }
   if (*++cp == ':') {
      if (argv[optind][sp + 1] != '\0')
         optarg = &argv[optind++][sp + 1];
      else if (++optind >= argc) {
         fprintf(stderr, "Option '%c' requires an argument\n", c);
         sp = 1;
         return('?');
      }
      else
         optarg = argv[optind++];
      sp = 1;
   }
   else {
      if (argv[optind][++sp] == '\0') {
         sp = 1;
         optind++;
      }
      optarg = NULL;
   }
   return(c);
}
