// CalcEdit.cpp : implements the subclassed edit control for the calculator
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <locale.h>

#include "HexEdit.h"
#include "CalcEdit.h"
#include "CalcDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CCalcEdit

CCalcEdit::CCalcEdit()
{
    pp_ = NULL;

    // Work out how to display decimal numbers in this culture
    struct lconv *plconv = localeconv();
    if (strlen(plconv->thousands_sep) == 1)
    {
        dec_sep_char_ = *plconv->thousands_sep;
        dec_group_ = *plconv->grouping;
    }
    else
    {
        dec_sep_char_ = ',';
        dec_group_ = 3;
    }
}

CCalcEdit::~CCalcEdit()
{
}


BEGIN_MESSAGE_MAP(CCalcEdit, CEdit)
	//{{AFX_MSG_MAP(CCalcEdit)
	ON_WM_CHAR()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CCalcEdit::Put()
{
    ASSERT(pp_ != NULL);
    ASSERT(pp_->radix_ > 1 && pp_->radix_ <= 36);
    ASSERT(pp_->visible_);

    char buf[72];

    if (pp_->radix_ == 10)
    {
        signed char s8;
        signed short s16;
        signed long s32;

        switch (pp_->bits_)
        {
        case 8:
            s8 = (signed char)pp_->current_;
            itoa((int)s8, buf, pp_->radix_);
            break;
        case 16:
            s16 = (signed short)pp_->current_;
            itoa((int)s16, buf, pp_->radix_);
            break;
        case 32:
            s32 = (signed long)pp_->current_;
            _i64toa((__int64)s32, buf, pp_->radix_);
            break;
        case 64:
            _i64toa(pp_->current_, buf, pp_->radix_);
            break;
        default:
            ASSERT(0);
        }
    }
    else
        _ui64toa(pp_->current_ & pp_->mask_, buf, pp_->radix_);
    SetWindowText(buf);
    SetSel(strlen(buf), strlen(buf), FALSE);

    add_sep();
}

// Returns the value in the control converted according to the current radix
// If the value exceeds the size specified by bits then overflow_ is set.
__int64 CCalcEdit::value()
{
    ASSERT(pp_ != NULL);
    ASSERT(pp_->visible_);
    ASSERT(pp_->radix_ > 1 && pp_->radix_ <= 36);

    CString ss;                         // The string with the number
    GetWindowText(ss);

    __int64 newval = 0;
    BOOL none_yet = TRUE;		        // Have we seen any digits yet?
    BOOL neg = FALSE;                   // Have we seen a leading minus sign?

    pp_->overflow_ = FALSE;             // Default to no overflow

    for (const char *src = ss.GetBuffer(0); *src != '\0'; ++src)
    {
        if (pp_->radix_ == 10 && none_yet && *src == '-')
        {
            neg = TRUE;
            continue;
        }

        // Ignore anything else except valid digits
        unsigned int digval;
        if (isdigit(*src))
            digval = *src - '0';
        else if (isalpha(*src))
            digval = toupper(*src) - 'A' + 10;
        else
            continue;                   // Ignore separators (or any other garbage)

        if (digval >= pp_->radix_)
        {
            ASSERT(0);                  // How did this happen?
            continue;                   // Ignore invalid digits
        }

        none_yet = FALSE;               // We've now seen a valid digit

        if (newval > pp_->mask_ / pp_->radix_ ||
            (newval == pp_->mask_ / pp_->radix_ && 
             digval > pp_->mask_ % pp_->radix_) )
        {
            pp_->overflow_ = TRUE;
        }
        newval = newval * pp_->radix_ + digval;
    }

    if (neg)
        return -newval;
    else
        return newval;
}

#if 0
// This is no longer needed as pp_->current_ is updated continually
void CCalcEdit::Get()
{
    // Move value to correct bits of current value
    pp_->current_ = (pp_->current_ & ~pp_->mask_) | (value() & pp_->mask_);
}
#endif

void CCalcEdit::add_sep()
{
    ASSERT(pp_ != NULL);
    ASSERT(pp_->visible_);
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Work out how to group the digits
    int group;
    if (pp_->radix_ == 10)
        group = dec_group_;             // Use decimal grouping from locale
    else if (pp_->radix_ == 16)
        group = 4;                      // For hex make groups of 4 nybbles (2 bytes)
    else
        group = 8;                      // Group = whole number of bytes (1 byte for binary, 3 for octal)

    char sep_char = ' ';
    if (pp_->radix_ == 10) sep_char = dec_sep_char_;

    CString ss;                         // Current string
    GetWindowText(ss);
    int start, end;                     // Current selection (start==end if just caret)
    GetSel(start, end);

    const char *src;                    // Ptr into orig. string
    int newstart, newend;               // New caret position/selection
    newstart = newend = 0;              // In case of empty string

    char out[72];                       // Largest output is 64 bit binary (64 bits + 7 spaces + nul)

    size_t ii, jj;                      // Number of digits written/read
    char *dst = out;                    // Ptr to current output char
    int ndigits;                        // Numbers of chars that are part of number

    BOOL none_yet = TRUE;		        // Have we seen any non-zero digits yet?

    // Work out how many digits we have
    for (src = ss.GetBuffer(0), ndigits = 0; *src != '\0'; ++src)
    {
        unsigned int val;
        if (isdigit(*src))
            val = *src - '0';
        else if (isalpha(*src))
            val = toupper(*src) - 'A' + 10;
        else
            continue;

        // Check for valid digits according to current radix
        if (val < pp_->radix_)
            ++ndigits;
    }

    for (src = ss.GetBuffer(0), ii = jj = 0; /*forever*/; ++src, ++ii)
    {
        if (ii == start)
            newstart = dst - out;       // save new caret position
        if (ii == end)
            newend = dst - out;

        if (*src == '\0')
            break;

        // Copy -ve sign
        if (jj < ndigits && pp_->radix_ == 10 && none_yet && *src == '-')
        {
            *dst++ = '-';
            continue;
        }

        // Ignore anything else except valid digits
        unsigned int val;
        if (isdigit(*src))
            val = *src - '0';
        else if (isalpha(*src))
            val = toupper(*src) - 'A' + 10;
        else
            continue;

        if (val >= pp_->radix_)
        {
            ASSERT(0);                  // Digits invalid for this base should never have got in there!
            continue;
        }

        ++jj;

        // Ignore leading zeroes
        if (jj < ndigits && none_yet && *src == '0')
            continue;

        none_yet = FALSE;

        if (aa->hex_ucase_)
            *dst++ = toupper(*src);     // convert all to upper case
        else
            *dst++ = tolower(*src);     // convert all to lower case


        // If at end of group insert pad char
        if ((ndigits - jj) % group == 0)
            *dst++ = sep_char;
    }
    if (dst > out)
        dst--;                          // Forget last sep_char if added
    *dst = '\0';                        // Terminate string

    SetWindowText(out);
    SetSel(newstart, newend);
}

/////////////////////////////////////////////////////////////////////////////
// CCalcEdit message handlers

void CCalcEdit::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    ASSERT(pp_ != NULL);
    ASSERT(pp_->visible_);
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    char sep_char = ' ';
    if (pp_->radix_ == 10) sep_char = dec_sep_char_;

    if (nChar == sep_char)
    {
        // Just ignore separator chars (user has tendency to type them even though not required)
        return;
    }
    else if (nChar == '\b')
    {
        // If backspace deletes 1st digit of a group then also delete preceding separator
        // (If we don't do this the add_sep call below stops the backspace from working.)
        if (start > 1 && start == end && ss[start-1] == sep_char)
            SetSel(start-2, end);
    }
    else if (isalnum(nChar))
    {
        // Calculate digit value and beep if it is invalid for the current radix
        unsigned int val;
        if (isdigit(nChar))
            val = nChar - '0';
        else
            val = toupper(nChar) - 'A' + 10;

        if (val >= pp_->radix_)
        {
            ::Beep(5000,200);
            return;
        }
    }

    CEdit::OnChar(nChar, nRepCnt, nFlags);

    __int64 new_val = value();
    if (pp_->overflow_)
    {
        // If the new character causes overflow of the field beep and ignore it
        ::Beep(5000,200);
        SetWindowText(ss);
        SetSel(start, end, FALSE);
    }
    else
    {
        pp_->current_ = (pp_->current_ & ~pp_->mask_) | (new_val & pp_->mask_);
        add_sep();
    }
    pp_->FixFileButtons();
}

void CCalcEdit::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    ASSERT(pp_ != NULL);
    ASSERT(pp_->visible_);
    CString ss;                         // Text from the window (edit control)
    int start, end;                     // Current selection in the edit control

    pp_->ShowBinop(pp_->op_);           // Show current binop (if any) and clear overflow/error status

    // If we need to clear edit control (eg. binary op just entered) ...
    if (!pp_->in_edit_)
    {
        // Clear control ready for new number unless (arrow, Del etc key pressed)
        if (isprint(nChar))
            SetWindowText("");
        pp_->in_edit_ = TRUE;           // and go into edit mode
    }
    else
    {
        // Fiddle with the selection so that arrows/Del work OK in presence of separator chars
        GetWindowText(ss);
        GetSel(start, end);

        char sep_char = ' ';
        if (pp_->radix_ == 10) sep_char = dec_sep_char_;

        // If no selection and character to delete is separator ...
        if (nChar == VK_DELETE && start == end && ss.GetLength() > start+1 && ss[start+1] == sep_char)
        {
            // Set selection so that the separator and following digit is deleted
            SetSel(start, start+2);
        }
        else if (nChar == VK_LEFT && start == end && start > 0 && ss[start-1] == sep_char)
        {
            // Move cursor back one so we skip over the separator (else add_sep below makes the caret stuck)
            SetSel(start-1, start-1);
        }
    }

    CEdit::OnKeyDown(nChar, nRepCnt, nFlags);

    // Tidy display if hex and char(s) deleted or caret moved
    GetWindowText(ss);
    if ((nChar == VK_DELETE || nChar == VK_RIGHT || nChar == VK_LEFT) &&
        ::GetKeyState(VK_SHIFT) >= 0 && ss.GetLength() > 0)
    {
        add_sep();
    }

    if (nChar == VK_DELETE)
    {
        // If Del key was used the value may have changed so update pp_->current_
        pp_->current_ = (pp_->current_ & ~pp_->mask_) | (value() & pp_->mask_);

        // FixFileButtons is normally called in OnChar but if Del is pressed OnChar is not called
        pp_->FixFileButtons();
    }

    pp_->source_ = pp_->aa_->recording_ ? km_user : km_result;
}
