#include <windows.h>
#include <mmsystem.h>
#include <time.h>
#include <math.h>
#include <stdio.h>
#include "resource.h"
#include "alarmfuncs.h"
#include "sticky.h"

const char*   AMPMStr[]  = {"AM", "PM", NULL};
const char*   WeekStr[]  = {"Sunday",    "Monday",   "Tuesday", 
                            "Wednesday", "Thursday", "Friday", 
                            "Saturday"};
const char*   MonthStr[] = {"January", "February", "March", "April", "May", "June",
                            "July", "August", "September", "October", "November", 
                            "December" };


// ************************************************************************************

long LASTJULDATE = 17520902L;   /* last day to use Julian calendar */
long LASTJULJDN  = 2361221L ;   /* jdn of same */

// Function is used to convert a dates separate components into a 
// single long integer so that it can be incremented/decremented on
// a per day basis.
//
// y, m, d  : The date
// julian   : Should be false if date is past julian conversion date.
long YMDToJdn(int y, int m, int d, BOOL julian)
{
    long jdn;

    if (julian)             // set Julian flag if auto set 
        julian = (((y * 100L) + m) * 100 + d  <=  LASTJULDATE);

    if (y < 0)              // adjust BC year
        y++;

    if (julian)
        jdn = 367L * y - 7 * (y + 5001L + (m - 9) / 7) / 4 + 275 * m / 9 + d + 1729777L;
    else
        jdn = (long)(d - 32076) + 1461L * (y + 4800L + (m - 14) / 12) / 4 + 367 * (m - 2 - (m - 14) / 12 * 12) / 12 - 3 * ((y + 4900L + (m - 14) / 12) / 100) / 4 + 1; 

    return jdn;
}


// Function converts a julian long integer to a real date.
// Date converted is stored int variables within the class in Year, Month, Day.
void jdnToYMD(long jdn, boolean julian, int* pYear, int* pMonth, int* pDay)
{
    long x, z, m, d, y;
    long daysPer400Years        = 146097L;
    long fudgedDaysPer4000Years = 1460970L + 31;

    if (julian)                 /* set Julian flag if auto set */
        julian = (jdn <= LASTJULJDN);

    x = jdn + 68569L;
    if ( julian )
    {
        x += 38;
        daysPer400Years = 146100L;
        fudgedDaysPer4000Years = 1461000L + 1;
    }
    z = 4 * x / daysPer400Years;
    x = x - (daysPer400Years * z + 3) / 4;
    y = 4000 * (x + 1) / fudgedDaysPer4000Years;
    x = x - 1461 * y / 4 + 31;
    m = 80 * x / 2447;
    d = x - 2447 * m / 80;
    x = m / 11;
    m = m + 2 - 12 * x;
    y = 100 * (z - 49) + y + x;

    //-------------------
    if (y <= 0)                   /* adjust BC years */
            y--;

    *pYear  = y;
    *pMonth = m;
    *pDay   = d;
}  

// ************************************************************************************

// returns 0 upon success, -1 uponan erorr.
inline int GetLastStickyWriteTime (SYSTEMTIME* pFT)
{
    FILETIME ft, lft;
    HANDLE stickyFile = CreateFile (STRSTICKYRECORDS, 
                                    GENERIC_READ,
                                    FILE_SHARE_READ,
                                    NULL,
                                    OPEN_EXISTING,
                                    FILE_ATTRIBUTE_NORMAL,
                                    NULL);

    if (stickyFile == NULL)
        return -1;


    if (GetFileTime(stickyFile, NULL, NULL, &ft) == FALSE)
    {
        CloseHandle (stickyFile);
        return -1;
    }

    CloseHandle (stickyFile);

    // convert UTC file time to local file time
    FileTimeToLocalFileTime(&ft, &lft);

    if (FileTimeToSystemTime (&lft, pFT) == FALSE)      
        return -1;

    return 0;
}

// ************************************************************************************

void DoAlarmDownTimeTest (void)
{
    DWORD      threadID;
    SYSTEMTIME fileDateTime;
    SYSTEMTIME curDateTime;

    if (GetLastStickyWriteTime (&fileDateTime) != 0)
        return;

    GetLocalTime (&curDateTime);

    for (int alrmTst = 0; alrmTst < STICKRECNUM; alrmTst++)
    {
        // Sticky was in snooze mode when stickys was shut down!
        if (StickyRecords[alrmTst].stickyAlrm.alarmDlgThread != FALSE)
        {
            StickyRecords[alrmTst].stickyAlrm.alarmDlgThread = FALSE;
            CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                
            continue;
        }

        if (((StickyRecords[alrmTst].stickyWnd != NULL) && 
             (StickyRecords[alrmTst].stickyAlrm.alarmEnabled != FALSE)) &&
             (StickyRecords[alrmTst].stickyAlrm.alarmDlgThread == FALSE))
        {       
            if (((StickyRecords[alrmTst].stickyAlrm.year >= fileDateTime.wYear) && (StickyRecords[alrmTst].stickyAlrm.year <= curDateTime.wYear)) || 
                 (StickyRecords[alrmTst].stickyAlrm.year == ALRMEVERY))
            {
                if (((StickyRecords[alrmTst].stickyAlrm.month >= fileDateTime.wMonth) && (StickyRecords[alrmTst].stickyAlrm.month <= curDateTime.wMonth)) || 
                     (StickyRecords[alrmTst].stickyAlrm.month == ALRMEVERY))
                {                  
                                                            
                    BOOL dayTst   = FALSE;                    
                    BOOL lastDay  = FALSE;
                    BOOL firstDay = FALSE;                   

                    // Test to see if we are only comparing within the first.
                    // If not then compare in the last day, all other days don't
                    // need comparing as they are already gone and so we should fire
                    // the alarm info dialog box.
                    if (((curDateTime.wYear  == fileDateTime.wYear) &&
                         (curDateTime.wMonth == fileDateTime.wMonth)) &&
                         (curDateTime.wDay   == fileDateTime.wDay))
                        firstDay = TRUE;

                    // test week day
                    if (StickyRecords[alrmTst].stickyAlrm.weekDayTest  != FALSE)
                    {
                       int tst = (fileDateTime.wDayOfWeek-1);
                   
                        do 
                        {
                            if (tst == 6)
                                tst = 0;
                            else
                                tst++;

                            if ((StickyRecords[alrmTst].stickyAlrm.day & (1 << tst)) != 0)
                            {
                                dayTst = TRUE;
                                
                                // if the last or the first day then test the hours and minutes...
                                // otherwise they have alrwady god.
                                if ((tst == curDateTime.wDayOfWeek) && (firstDay == FALSE))
                                    lastDay = TRUE;
                                break;
                            }                            
                        } while (tst != curDateTime.wDayOfWeek);
                    }

                    // Test day of month
                    else if (StickyRecords[alrmTst].stickyAlrm.weekDayTest  == FALSE)                            
                    {
                        int tst = (fileDateTime.wDay-1);
                        
                        do 
                        {
                            if (tst == 31)
                                tst = 1;
                            else
                                tst++;

                            if ((StickyRecords[alrmTst].stickyAlrm.day & (1 << (tst-1))) != 0)
                            {
                                dayTst = TRUE;

                                // if the last or the first day then test the hours and minutes...
                                // otherwise they have alrwady god.
                                if ((tst == curDateTime.wDay) && (firstDay == FALSE))
                                    lastDay = TRUE;
                                break;
                            }                            
                        } while (tst != curDateTime.wDay);                        
                    }

                    if (dayTst != FALSE)
                    {
                        int hour;

                        // convert time to 24 hour to make comparisons easier.
                        if (StickyRecords[alrmTst].stickyAlrm.hour != ALRMEVERY)
                        {
                            if (StickyRecords[alrmTst].stickyAlrm.AMPM != 0)
                            {
                                if (StickyRecords[alrmTst].stickyAlrm.hour < 12)
                                    hour = 12 + StickyRecords[alrmTst].stickyAlrm.hour;
                                
                            }
                            else
                                hour = StickyRecords[alrmTst].stickyAlrm.hour;
                        }

                        // if not the first or last day, then dont test the hours and minutes... as this
                        // would have already passed.
                        if ((firstDay == FALSE) && (lastDay == FALSE))
                        {
                            // Create a separate thread to handle number sound sample playbacks.
                            CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                                            
                        }
                        else if ( (firstDay != FALSE)  && 
                                  (((hour >= fileDateTime.wHour) && (hour <= curDateTime.wHour)) || (StickyRecords[alrmTst].stickyAlrm.hour == ALRMEVERY)) )
                        {
                            // if the current hour then test the minutes ... else don't bother.
                            if (hour == curDateTime.wHour)
                            {
                                if ((StickyRecords[alrmTst].stickyAlrm.minute <= curDateTime.wMinute) || 
                                     (StickyRecords[alrmTst].stickyAlrm.minute == ALRMEVERY))
                                {
                                    // Create a separate thread to handle number sound sample playbacks.
                                    CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                                            
                                }
                            }
                            else
                            {
                                // Create a separate thread to handle number sound sample playbacks.
                                CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                                            
                            }
                        }
                        else if ((lastDay != FALSE)  && ((hour <= curDateTime.wHour) || (StickyRecords[alrmTst].stickyAlrm.hour == ALRMEVERY)))
                        {
                            // if the last hour in the current, then test the minutes, otherwise don't bother.
                            if (hour == curDateTime.wHour)
                            {
                                if ((StickyRecords[alrmTst].stickyAlrm.minute <= curDateTime.wMinute) || 
                                    (StickyRecords[alrmTst].stickyAlrm.minute == ALRMEVERY))
                                {
                                    // Create a separate thread to handle number sound sample playbacks.
                                    CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                                            
                                }
                            }
                            else
                            {
                                // Create a separate thread to handle number sound sample playbacks.
                                CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                                            
                            }
                        }
                    }
                }

            }
        }
    }
}

// ************************************************************************************

void DoAlarmTest (void)
{
    DWORD      threadID;
    SYSTEMTIME curDateTime;
    int        hour, ampm;

    GetLocalTime (&curDateTime);
    if (curDateTime.wHour < 12)
    {
        hour = curDateTime.wHour;
        ampm = 0;
    }
    else
    {
        if (curDateTime.wHour > 12)
            hour = (curDateTime.wHour - 12);
        else
            hour = curDateTime.wHour;
        ampm = 1;
    }

    for (int alrmTst = 0; alrmTst < STICKRECNUM; alrmTst++)
    {
        if (((StickyRecords[alrmTst].stickyWnd != NULL) && 
             (StickyRecords[alrmTst].stickyAlrm.alarmEnabled != FALSE)) &&
             (StickyRecords[alrmTst].stickyAlrm.alarmDlgThread == FALSE))
        {            
            if ((StickyRecords[alrmTst].stickyAlrm.year == curDateTime.wYear) || (StickyRecords[alrmTst].stickyAlrm.year == ALRMEVERY))
            {
                if ((StickyRecords[alrmTst].stickyAlrm.month == curDateTime.wMonth) || (StickyRecords[alrmTst].stickyAlrm.month == ALRMEVERY))
                {                    
                    BOOL dayTst = FALSE;

                    // test week day
                    if  ((StickyRecords[alrmTst].stickyAlrm.weekDayTest  != FALSE) &&
                        ((StickyRecords[alrmTst].stickyAlrm.day & (1 << curDateTime.wDayOfWeek)) != 0))
                    {
                        dayTst = TRUE;
                    }
                    // Test day of month
                    else if ( (StickyRecords[alrmTst].stickyAlrm.weekDayTest  == FALSE) &&
                            ((StickyRecords[alrmTst].stickyAlrm.day & (1 << (curDateTime.wDay-1))) != 0))
                    {
                        dayTst = TRUE;
                    }

                    if (dayTst != FALSE)
                    {                        
                        if (((StickyRecords[alrmTst].stickyAlrm.hour == hour) && (StickyRecords[alrmTst].stickyAlrm.AMPM == ampm)) || (StickyRecords[alrmTst].stickyAlrm.hour == ALRMEVERY))
                        {                            
                            if ((StickyRecords[alrmTst].stickyAlrm.minute == curDateTime.wMinute) || (StickyRecords[alrmTst].stickyAlrm.minute == ALRMEVERY))
                            {                                                        
                                // Create a separate thread to handle number sound sample playbacks.
                                CreateThread (NULL, 512, AlarmDispThread, (LPDWORD) alrmTst, 0, &threadID);                                
                            }
                        }
                    }
                }
            }
        }
    }
}


// ************************************************************************************
// ******************************* Dialog Boxs ****************************************
// ************************************************************************************

void constructAlarmInfo (char* buff, int stkIdx)
{        
    SYSTEMTIME curDateTime;   
    
    GetLocalTime (&curDateTime);
    wsprintf (buff, "Date ");

    // select the users day
    if (StickyRecords[stkIdx].stickyAlrm.weekDayTest != FALSE)
        wsprintf (buff+strlen(buff),  "%s/", WeekStr[curDateTime.wDayOfWeek]);
    else
        wsprintf (buff+strlen(buff),  "%02d/", curDateTime.wDay);

    if (StickyRecords[stkIdx].stickyAlrm.month > ALRMEVERY)
        wsprintf (buff+strlen(buff),  "%02d/", StickyRecords[stkIdx].stickyAlrm.month);
    else
        wsprintf (buff+strlen(buff), "~~/");

    if (StickyRecords[stkIdx].stickyAlrm.year > ALRMEVERY)
        wsprintf (buff+strlen(buff),  "%02d", StickyRecords[stkIdx].stickyAlrm.year);
    else
        wsprintf (buff+strlen(buff), "~~");

    wsprintf (buff+strlen(buff), "  Time ");

    if (StickyRecords[stkIdx].stickyAlrm.hour > ALRMEVERY)
        wsprintf (buff+strlen(buff),  "%02d:", StickyRecords[stkIdx].stickyAlrm.hour);
    else
        wsprintf (buff+strlen(buff), "~~:");
    
    if (StickyRecords[stkIdx].stickyAlrm.minute > ALRMEVERY)
        wsprintf (buff+strlen(buff),  "%02d:", StickyRecords[stkIdx].stickyAlrm.minute);
    else
        wsprintf (buff+strlen(buff), "~~:");

    if ((StickyRecords[stkIdx].stickyAlrm.AMPM <= 1) && 
        (StickyRecords[stkIdx].stickyAlrm.AMPM >= 0))
        wsprintf (buff+strlen(buff),  "%s", AMPMStr[StickyRecords[stkIdx].stickyAlrm.AMPM]);
    else
        wsprintf (buff+strlen(buff),  "??");    
}

// *************************** Alarm Threads ***********************************************

DWORD WINAPI AlarmBlinkStickyThread (LPVOID pData) 
{
    OutputDebugString ("1) Starting AlarmBlinkStickyThread\n");

    int      stkIdx    = (int) (DWORD) pData;
    BOOL     flasher   = FALSE;
    COLORREF flashCol  = StickyRecords[stkIdx].stickyAlrm.alarmBlinkCol;

    StickyRecords[stkIdx].stickyAlrm.alarmBlinker = TRUE;

    // Lets start the blinking process...
    while (StickyRecords[stkIdx].stickyAlrm.alarmBlinker != FALSE)
    {
        if (flasher != FALSE)
        {
            StickyRecords[stkIdx].stickyAlrm.alarmBlinkCol = flashCol;
            flasher = FALSE;
        }
        else
        {
            StickyRecords[stkIdx].stickyAlrm.alarmBlinkCol = StickyRecords[stkIdx].backCol;
            flasher = TRUE;
        }

        // change the colour of the sticky note.
        if (StickyRecords[stkIdx].stickyWnd != NULL)
            InvalidateRect (StickyRecords[stkIdx].stickyWnd, NULL, TRUE);
        else
            return 0;   // something killed the sticky note !

        Sleep (500);
    }

    // change the colour of the sticky note.
    if (StickyRecords[stkIdx].stickyWnd != NULL)
        InvalidateRect (StickyRecords[stkIdx].stickyWnd, NULL, TRUE);

    // restore alarm blinking colour
    StickyRecords[stkIdx].stickyAlrm.alarmBlinkCol = flashCol;
    OutputDebugString ("1) Ending AlarmBlinkStickyThread\n");
    return 0;
}

DWORD WINAPI AlarmSndPlaybackThread (LPVOID pData) 
{
    OutputDebugString ("2) Starting AlarmSndPlaybackThread\n");
    int  stkIdx   = (int) (DWORD) pData;
    int  sndCount = StickyRecords[stkIdx].stickyAlrm.alarmSndRpt;

    StickyRecords[stkIdx].stickyAlrm.alarmSndThread = TRUE;
    while ((sndCount > 0 ) && (StickyRecords[stkIdx].stickyAlrm.alarmSndThread != FALSE))
    {
        if (SoundActive != FALSE)
            PlaySound (AlarmSnd, NULL, SND_FILENAME | SND_SYNC);

        sndCount--;
    }
    PlaySound (NULL, NULL, SND_ASYNC | SND_PURGE);
    OutputDebugString ("2) Ending AlarmSndPlaybackThread\n");
    return 0;
}

DWORD WINAPI AlarmDispThread (LPVOID pData) 
{
    OutputDebugString ("3) Starting AlarmDispThread\n");
    DialogBoxParam(HInst, MAKEINTRESOURCE(IDD_ALARMINFO), NULL, (DLGPROC) AlarmInfoDialog, (LPARAM) pData);
    OutputDebugString ("3) Ending AlarmDispThread\n");
    return 0;
}

// ************************************************************************************

BOOL CALLBACK AlarmInfoDialog (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static int stkIdx;   

    switch (message)
    {
        case WM_INITDIALOG:
        {           
            char  buff[100];
            RECT  wndRect;

  
            // Load the dialog box controls with alarm information!
            stkIdx = lParam;                     
            StickyRecords[stkIdx].stickyAlrm.alarmDlgThread = TRUE; // Ignore all other alarms for this sticky while this is still active.
            constructAlarmInfo (buff, stkIdx);
            SetWindowText (GetDlgItem(hwnd, IDC_DTINFOSTATIC), buff);
            SetWindowText (GetDlgItem(hwnd, IDC_ALARMINFO), StickyRecords[stkIdx].winCaption);

            if (StickyRecords[stkIdx].stickyAlrm.LoopPlayback != FALSE)
            {
                if (SoundActive != FALSE)
                    PlaySound (AlarmSnd, NULL, SND_ASYNC | SND_FILENAME | SND_LOOP);
            }
            else
            {
                // Create a separate thread to handle number sound sample playbacks.
                PostMessage (ParentStarter, WM_USER, ALARM_THRD_SOUNDPLAY, stkIdx);                
            }

            // ******* Create thread to blink sticky note being alarmed ******
            if (StickyRecords[stkIdx].stickyAlrm.alarmBlinking != FALSE)
                PostMessage (ParentStarter, WM_USER, ALARM_THRD_BLINKER, stkIdx);                                

            // Load the snooze timer 
            for (int count = 5; count <= 30; count+=5)
            {
                // convert integer to string.
                wsprintf (buff, "%2d mins", count);
                SendDlgItemMessage(hwnd, IDC_SNOOZETIME, CB_ADDSTRING, 0, (LPARAM) ((LPSTR) buff));
                SendDlgItemMessage(hwnd, IDC_SNOOZETIME, CB_SETCURSEL, 0, 0);
            }

            // Make dialog box always on top.
            GetWindowRect (hwnd, &wndRect);
            SetWindowPos (hwnd, 
                          HWND_TOPMOST, 
                          wndRect.left,
                          wndRect.top,
                          wndRect.right  - wndRect.left,
                          wndRect.bottom - wndRect.top,
                          SWP_SHOWWINDOW);
        }
        return TRUE;

        case WM_COMMAND :
            switch (LOWORD (wParam))
            {
                // Process the colour select message
                case IDC_OK:
                    if (SoundActive != FALSE)                    
                        PlaySound (NULL, NULL, SND_ASYNC | SND_PURGE);

                    StickyRecords[stkIdx].stickyAlrm.alarmSndThread = FALSE;

                    // Deassign alarm active if it is not in every state.
                    if ((StickyRecords[stkIdx].stickyAlrm.year         != ALRMEVERY) &&
                        (StickyRecords[stkIdx].stickyAlrm.month        != ALRMEVERY) &&
                        (StickyRecords[stkIdx].stickyAlrm.hour         != ALRMEVERY) &&
                        (StickyRecords[stkIdx].stickyAlrm.minute       != ALRMEVERY) &&
                        (StickyRecords[stkIdx].stickyAlrm.everyDayTest == FALSE))
                    {
                         StickyRecords[stkIdx].stickyAlrm.alarmEnabled = FALSE;
                    }

                    StickyRecords[stkIdx].stickyAlrm.alarmDlgThread = FALSE; // Set alarm testing active again                    
                    EndDialog (hwnd, TRUE);
                    break;      
                    
                // Hind the dialog and wait that amount of time then redisplay
                case IDC_SNOOZE:
                    {
                        int   time = (int)SendDlgItemMessage(hwnd, IDC_SNOOZETIME, CB_GETCURSEL, 0, 0);

                        ShowWindow (hwnd, SW_HIDE);

                        // Halt all threads                         
                        StickyRecords[stkIdx].stickyAlrm.alarmSndThread = FALSE;
                        StickyRecords[stkIdx].stickyAlrm.alarmBlinker   = FALSE;
                        Sleep(500);                        
                        PlaySound (NULL, NULL, SND_ASYNC | SND_PURGE);
                        
                        // Sleep some time                        
                        Sleep ((((1000) * 60) * 5) * (time+1));

                        // determine if sticky is still around 
                        if (StickyRecords[stkIdx].stickyWnd == NULL)
                        {
                            EndDialog (hwnd, FALSE);
                            break;
                        }

                        if (StickyRecords[stkIdx].stickyAlrm.LoopPlayback != FALSE)
                        {
                            if (SoundActive != FALSE)
                                PlaySound (AlarmSnd, NULL, SND_ASYNC | SND_FILENAME | SND_LOOP);
                        }
                        else
                        {
                            // Create a separate thread to handle number sound sample playbacks.                            
                            PostMessage (ParentStarter, WM_USER, ALARM_THRD_SOUNDPLAY, stkIdx);
                        }

                        // ******* Create thread to blink sticky note being alarmed ******
                        if (StickyRecords[stkIdx].stickyAlrm.alarmBlinking != FALSE)
                            PostMessage (ParentStarter, WM_USER, ALARM_THRD_BLINKER, stkIdx);                                                            

                        ShowWindow (hwnd, SW_SHOW);                        
                    }
                    break;
            }
            break;

        case WM_CLOSE:
            SendMessage (hwnd, WM_COMMAND, MAKEWPARAM (IDC_OK, 0), 0);            
            break;
    }

    return FALSE;
}

// ************************************************************************************
// ************************************************************************************

BOOL CALLBACK SetAlarmDialog (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static SYSTEMTIME curDateTime;
    static int        curStkyIdx;

    switch (message)
    {
        case WM_INITDIALOG:
        {
            int  count;
            char buffer[50];

            curStkyIdx = (int) lParam;

            // Load the year into the combo box ...
            GetLocalTime (&curDateTime);
            SendDlgItemMessage(hwnd, IDC_ALMYEAR, CB_ADDSTRING, 0, 
                               (LPARAM) ((LPSTR) "Every Month"));
            for (count = (int)curDateTime.wYear; count < (int)curDateTime.wYear+100; count++)
            {
                // convert integer to string.
                wsprintf (buffer, "%d", count);
                SendDlgItemMessage(hwnd, IDC_ALMYEAR, CB_ADDSTRING, 0,
                (LPARAM) ((LPSTR) buffer));
            }

            // load the month!
            SendDlgItemMessage(hwnd, IDC_ALMMONTH, CB_ADDSTRING, 0,
                               (LPARAM) ((LPSTR) "Every Month"));
            for (count = 1; count <= 12; count++)
            {
                // convert integer to string.
                wsprintf (buffer, "%s", MonthStr[count-1]);
                SendDlgItemMessage(hwnd, IDC_ALMMONTH, CB_ADDSTRING, 0,
                (LPARAM) ((LPSTR) buffer));
            }

            // Load weekday
            for (count = 0; count < 7; count++)
            {
                wsprintf (buffer, "%s", WeekStr[count]);
                SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_ADDSTRING, 0,
                                   (LPARAM) ((LPSTR) buffer));
            }

            //  Load day of month
            for (count = 1; count <= 31; count++)
            {
                // convert integer to string.
                wsprintf (buffer, "Day %2d", count);
                SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_ADDSTRING, 0,
                                   (LPARAM) ((LPSTR) buffer));
            }

            // load the hour!
            SendDlgItemMessage(hwnd, IDC_ALMHOUR, CB_ADDSTRING, 0,
                               (LPARAM) ((LPSTR) "Every Hour"));
            for (count = 0; count <= 12; count++)
            {
                // convert integer to string.
                wsprintf (buffer, "%2d", count);
                SendDlgItemMessage(hwnd, IDC_ALMHOUR, CB_ADDSTRING, 0,
                (LPARAM) ((LPSTR) buffer));
            }

            // load the minute!
            SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_ADDSTRING, 0,
                               (LPARAM) ((LPSTR) "Every Minute"));
            for (count = 0; count <= 59; count++)
            {
                // convert integer to string.
                wsprintf (buffer, "%2d", count);
                SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_ADDSTRING, 0,
                                   (LPARAM) ((LPSTR) buffer));
            }

            // Set the time period.
            SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_ADDSTRING, 0, (LPARAM)((LPSTR) AMPMStr[0]));
            SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_ADDSTRING, 0, (LPARAM)((LPSTR) AMPMStr[1]));
            
            //***** Set controls to the current time if alarm isn't set *****
            if (StickyRecords[curStkyIdx].stickyAlrm.year <= ALRMINIT)
            {
                // Select the current year
                SendDlgItemMessage(hwnd, IDC_ALMYEAR,  CB_SETCURSEL, 1, 0);
                // select the current month
                SendDlgItemMessage(hwnd, IDC_ALMMONTH, CB_SETCURSEL, curDateTime.wMonth, 0);                
                // select the current Day
                SendDlgItemMessage(hwnd, IDC_ALMDAY,   LB_SETSEL, TRUE, curDateTime.wDay+6);

                // Select the current hour
                SendDlgItemMessage(hwnd, IDC_ALMHOUR,   CB_SETCURSEL, (curDateTime.wHour % 12)+1, 0);
                // select the current minute
                SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_SETCURSEL, curDateTime.wMinute+1,  0);

                // select the current time zone section
                if (curDateTime.wHour < 12)                    
                    SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_SETCURSEL, 0, 0);
                else                    
                    SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_SETCURSEL, 1, 0);

                SendDlgItemMessage(hwnd, IDC_ALMCHK, BM_SETCHECK, 1, 0);

                // Set alarm sound playback controls
                SetDlgItemInt     (hwnd, IDC_ALARMSNDRPT, 1, FALSE);                
                SendDlgItemMessage(hwnd, IDC_LOOPSND, BST_UNCHECKED, 1, 0);
                EnableWindow      (GetDlgItem (hwnd, IDC_ALARMSNDRPT), TRUE);
            }
            else     // use values from stricky record to set items in the dialog box.
            {
                if (StickyRecords[curStkyIdx].stickyAlrm.year > ALRMEVERY)
                {
                    // Select the users year
                    if (StickyRecords[curStkyIdx].stickyAlrm.year < (int)curDateTime.wYear)
                        StickyRecords[curStkyIdx].stickyAlrm.year = (int)curDateTime.wYear;
                    
                    SendDlgItemMessage(hwnd, IDC_ALMYEAR,  CB_SETCURSEL, (WPARAM) (StickyRecords[curStkyIdx].stickyAlrm.year-curDateTime.wYear)+1, 0);
                }
                else
                    SendDlgItemMessage(hwnd, IDC_ALMYEAR,  CB_SETCURSEL, 0, 0);

                // select the users month
                if (StickyRecords[curStkyIdx].stickyAlrm.month > ALRMEVERY)                    
                    SendDlgItemMessage(hwnd, IDC_ALMMONTH, CB_SETCURSEL, StickyRecords[curStkyIdx].stickyAlrm.month, 0);
                else
                    SendDlgItemMessage(hwnd, IDC_ALMMONTH, CB_SETCURSEL, 0, 0);

                // set the every day check box.
                if (StickyRecords[curStkyIdx].stickyAlrm.everyDayTest != FALSE)
                    SendDlgItemMessage(hwnd, IDC_EVERYDAY, BM_SETCHECK, 1, 0);
                else
                    SendDlgItemMessage(hwnd, IDC_EVERYDAY, BM_SETCHECK, 0, 0);                

                // select the users day
                if (StickyRecords[curStkyIdx].stickyAlrm.weekDayTest != FALSE)
                {
                    for (int bitTst = 0; bitTst < 8; bitTst++)
                    {
                        if ((StickyRecords[curStkyIdx].stickyAlrm.day & (1 << bitTst)) != 0)
                            SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_SETSEL, TRUE,bitTst);                                                         
                        else
                            SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_SETSEL, FALSE,bitTst); 
                    }
                }
                else
                {
                    for (int bitTst = 0; bitTst < 32; bitTst++)
                    {
                        if ((StickyRecords[curStkyIdx].stickyAlrm.day & (1 << bitTst)) != 0)
                            SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_SETSEL, TRUE,bitTst+7);                                                         
                        else
                            SendDlgItemMessage(hwnd, IDC_ALMDAY, LB_SETSEL, FALSE,bitTst+7);                         
                    }
                }

                // Select the hour
                if (StickyRecords[curStkyIdx].stickyAlrm.hour > ALRMEVERY)                    
                    SendDlgItemMessage(hwnd, IDC_ALMHOUR,   CB_SETCURSEL, (StickyRecords[curStkyIdx].stickyAlrm.hour % 12)+1, 0);
                else
                    SendDlgItemMessage(hwnd, IDC_ALMHOUR,   CB_SETCURSEL, 0, 0);

                // Select the minute
                if (StickyRecords[curStkyIdx].stickyAlrm.minute > ALRMEVERY)
                    SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_SETCURSEL, StickyRecords[curStkyIdx].stickyAlrm.minute+1,  0);
                else
                    SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_SETCURSEL, 0,  0);

                SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_SETCURSEL, StickyRecords[curStkyIdx].stickyAlrm.AMPM, 0);

                if (StickyRecords[curStkyIdx].stickyAlrm.alarmEnabled != FALSE)
                    SendDlgItemMessage(hwnd, IDC_ALMCHK, BM_SETCHECK, 1, 0);
                else
                    SendDlgItemMessage(hwnd, IDC_ALMCHK, BM_SETCHECK, 0, 0);

                // set the dialog alarm sound repeat value.
                SetDlgItemInt(hwnd, IDC_ALARMSNDRPT, StickyRecords[curStkyIdx].stickyAlrm.alarmSndRpt, FALSE);

                // determine what type of sound playback the user wants
                if (StickyRecords[curStkyIdx].stickyAlrm.LoopPlayback != FALSE)
                {
                    SendDlgItemMessage(hwnd, IDC_LOOPSND, BM_SETCHECK, 1, 0);
                    EnableWindow (GetDlgItem (hwnd, IDC_ALARMSNDRPT), FALSE);
                }
                else
                {
                    SendDlgItemMessage(hwnd, IDC_LOOPSND, BST_UNCHECKED, 1, 0);
                    EnableWindow (GetDlgItem (hwnd, IDC_ALARMSNDRPT), TRUE);
                }
            }

        }
        return TRUE;
        
        case WM_COMMAND:
            switch (LOWORD (wParam))
            {
                // process button click upon alarm sound playback.
                case IDC_LOOPSND:
                    if (SendDlgItemMessage(hwnd, IDC_LOOPSND, BM_GETCHECK, 0, 0) != 0)
                    {
                       SendDlgItemMessage(hwnd, IDC_LOOPSND, BM_SETCHECK, 1, 0);
                       EnableWindow (GetDlgItem (hwnd, IDC_ALARMSNDRPT), FALSE);
                    }
                    else
                    {   
                        SendDlgItemMessage(hwnd, IDC_LOOPSND, BST_UNCHECKED, 1, 0);
                        EnableWindow (GetDlgItem (hwnd, IDC_ALARMSNDRPT), TRUE);
                    }   
                    break;

                // Process the colour select message
                case IDC_ALMFORGETIT:
                    EndDialog(hwnd, FALSE);
                    break;

                case IDC_ALMSETIT:
                {
                    // *** test routines ***
                    int hourSel;
                    int ampmSel;
                    int yearSel;
                    
                    
                    // Test alarm enable/disable box.
                    if (SendDlgItemMessage(hwnd, IDC_ALMCHK, BM_GETCHECK, 0, 0) == 0)                        
                    {                                             
                        int responce = MessageBox (hwnd, "You are about to set an alarm, but it is not enable.\n"
                                                   "Do you wish to enable the alarm?", 
                                                   "Alarm Not-Enable Warning", MB_YESNO | MB_ICONQUESTION);
                        if (responce == IDNO)
                            StickyRecords[curStkyIdx].stickyAlrm.alarmEnabled = FALSE;
                        else
                            StickyRecords[curStkyIdx].stickyAlrm.alarmEnabled = TRUE;
                    }
                    else
                        StickyRecords[curStkyIdx].stickyAlrm.alarmEnabled = TRUE;

                    // test the hour to am/pm section in alarm settings.
                    hourSel = (int)SendDlgItemMessage(hwnd, IDC_ALMHOUR, CB_GETCURSEL, 0, 0);
                    StickyRecords[curStkyIdx].stickyAlrm.AMPM = ampmSel = (int)SendDlgItemMessage(hwnd, IDC_ALMAMPM, CB_GETCURSEL, 0, 0);
                    if (((hourSel == 1) && (ampmSel != 0)) || ((hourSel == 13) && (ampmSel != 1)))
                    {
                        MessageBox (hwnd, "AM must have hours between 0 -11, PM must have hours between 12 - 11",
                        "Bad AM/PM setting", MB_ICONINFORMATION | MB_OK);
                        break;
                    }

                    if (hourSel == 0)
                        hourSel = ALRMEVERY;
                    else
                        hourSel -= 1;

                    StickyRecords[curStkyIdx].stickyAlrm.hour = hourSel;

                    // Get loop sound check box
                    if (SendDlgItemMessage(hwnd, IDC_LOOPSND, BM_GETCHECK, 0, 0) != 0)
                        StickyRecords[curStkyIdx].stickyAlrm.LoopPlayback = TRUE;                       
                    else
                        StickyRecords[curStkyIdx].stickyAlrm.LoopPlayback = FALSE;

                    // Get settings from the controls boxes...
                    yearSel = (int)SendDlgItemMessage(hwnd, IDC_ALMYEAR, CB_GETCURSEL, 0, 0);
                    if (yearSel > 0)
                        StickyRecords[curStkyIdx].stickyAlrm.year  = (yearSel-1) + curDateTime.wYear;
                    else
                        StickyRecords[curStkyIdx].stickyAlrm.year = ALRMEVERY;

                    StickyRecords[curStkyIdx].stickyAlrm.month  = (int)SendDlgItemMessage(hwnd, IDC_ALMMONTH,  CB_GETCURSEL, 0, 0);
                    if (StickyRecords[curStkyIdx].stickyAlrm.month == 0)
                        StickyRecords[curStkyIdx].stickyAlrm.month = ALRMEVERY;

                    // --------------------------------------------
                    // ---- Day Selection 
                    // --------------------------------------------

                    if (SendDlgItemMessage(hwnd, IDC_EVERYDAY, BM_GETCHECK, 0, 0) != 0)
                        StickyRecords[curStkyIdx].stickyAlrm.everyDayTest = TRUE;                       
                    else
                        StickyRecords[curStkyIdx].stickyAlrm.everyDayTest = FALSE;

                                        
                    int  lbSel[40];
                    int  numSel = (int)SendDlgItemMessage(hwnd, IDC_ALMDAY,LB_GETSELITEMS, 40, (LPARAM) lbSel);
                    
                    if ((numSel == LB_ERR) || (numSel == 0))
                    {
                        MessageBox (hwnd, "You must select a day for the alarm to occur.",
                                    "Day Select Error", MB_ICONINFORMATION | MB_OK);
                        break;
                    }

                    int dayType = 0;
                    StickyRecords[curStkyIdx].stickyAlrm.day     = 0;                    
                    for (int idx = 0; idx < numSel; idx++)
                    {
                        if (lbSel[idx] < 7) 
                        {
                            switch (dayType)
                            {
                                case 0:
                                case 1:
                                   dayType = 1;
                                   StickyRecords[curStkyIdx].stickyAlrm.weekDayTest = TRUE;
                                   StickyRecords[curStkyIdx].stickyAlrm.day |= (1 << (lbSel[idx]) );
                                   break;

                                default:
                                    MessageBox (hwnd, "You CANNOT mix week and month day selections together.",
                                                "Day Select Error", MB_ICONINFORMATION | MB_OK);
                                   return TRUE;
                            }
                        }
                        else
                        {
                            switch (dayType)
                            {
                                case 0:
                                case 2:
                                   dayType = 2;
                                   StickyRecords[curStkyIdx].stickyAlrm.weekDayTest = FALSE;
                                   StickyRecords[curStkyIdx].stickyAlrm.day |= (1 << (lbSel[idx]-7) );
                                   break;

                                default:
                                    MessageBox (hwnd, "You CANNOT mix week and month day selections together.",
                                                "Day Select Error", MB_ICONINFORMATION | MB_OK);
                                   return TRUE;
                            }
                        }

                    }

                    // -----------------------------------------------------

                    StickyRecords[curStkyIdx].stickyAlrm.minute = (int)SendDlgItemMessage(hwnd, IDC_ALMMINUTE, CB_GETCURSEL, 0, 0);
                    if (StickyRecords[curStkyIdx].stickyAlrm.minute == 0)
                        StickyRecords[curStkyIdx].stickyAlrm.minute = ALRMEVERY;
                    else
                        StickyRecords[curStkyIdx].stickyAlrm.minute -= 1;

                    // Get dialog item repeat count
                    // set the dialog alarm sound repeat value.
                    BOOL goodValue; 
                    int  tmpValue = GetDlgItemInt(hwnd, IDC_ALARMSNDRPT, &goodValue, FALSE);

                    if (goodValue != FALSE)
                         StickyRecords[curStkyIdx].stickyAlrm.alarmSndRpt = tmpValue;

                    EndDialog (hwnd, TRUE);
                }
                break;
            }
            break;

        case WM_CLOSE:
            EndDialog(hwnd, FALSE);
            break;
    }

    return FALSE;
}