/*
 * oracall32.cpp	1.01 03/15/01
 *
 * Copyright (c) 2001 Sergey Yakovlev. All Rights Reserved.
 *
 *    E-mail:  yakovlev@zdnetonebox.com
 * Voice/Fax:  +1 (617) 250-0001 x1385
 *       ICQ:  86060618
 *
 *
 * Author grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Author.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. AUTHOR AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL AUTHOR OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF AUTHOR HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */


// oracall32.cpp : Defines the entry point for the DLL application.
//

#include <windows.h>

extern "C"
{
#include <string.h>
#include <stdio.h>
#include <oci.h>
}

#include "oracall32.h"

/* Error handling function.
   @param errhp - An error handle. 
   @param status - An error code.
   */
void checkerr(OCIError *errhp, sword& status)
{
	text errbuf[512];
	sb4 errcode = 0;
	
	switch (status)
	{
		case OCI_SUCCESS:
			break;
		case OCI_SUCCESS_WITH_INFO:
			MessageBox(NULL, "Error - OCI_SUCCESS_WITH_INFO", NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_NEED_DATA:
			MessageBox(NULL, "Error - OCI_NEED_DATA", NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_NO_DATA:
			MessageBox(NULL, "Error - OCI_NODATA", NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_ERROR:
			(void) OCIErrorGet((dvoid *)errhp, (ub4) 1, (text *) NULL, &errcode,
								errbuf, (ub4) sizeof(errbuf), OCI_HTYPE_ERROR);
			
			if(errcode == 1405)
				status = 0;
			else
				MessageBox(NULL, (char*)errbuf, NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_INVALID_HANDLE:
			MessageBox(NULL, "Error - OCI_INVALID_HANDLE", NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_STILL_EXECUTING:
			MessageBox(NULL, "Error - OCI_STILL_EXECUTE", NULL, MB_OK | MB_ICONWARNING);
			break;
		case OCI_CONTINUE:
			MessageBox(NULL, "Error - OCI_CONTINUE", NULL, MB_OK | MB_ICONWARNING);
			break;
		case CONERR_ALRCON:
			MessageBox(NULL, "Error - CONERR_ALRCON", NULL, MB_OK | MB_ICONWARNING);
			break;
		case CONERR_NOTCON:
			MessageBox(NULL, "Error - CONERR_NOTCON", NULL, MB_OK | MB_ICONWARNING);
			break;
		default:
			break;
	}
}

/* CConnection constructor */
CConnection::CConnection()
{
	state = not_connected;
	stm = NULL;
	init();
}

/* CConnection destructor */
CConnection::~CConnection()
{
	// disconnect if connection exists
	if(state == connected)
	{
		if(disconnect() == 0)
			deallocate();
	}
}

/* creating and initializing an OCI environment. */
int CConnection::init(void)
{
	sword status;
	
	/* initialize the mode to be the threaded and object environment */
	if(status = OCIEnvCreate(&envhp, OCI_THREADED|OCI_OBJECT, (dvoid *)0, 
							0, 0, 0, (size_t) 0, (dvoid **)0))
		return (status);
	
	/* allocate an error handle */
	checkerr(errhp, status = OCIHandleAlloc((dvoid *)envhp, (dvoid **)&errhp,
							OCI_HTYPE_ERROR, 0, (dvoid **) 0));
	
	return (status);
}

/* freeing an allocated OCI environment. */
int CConnection::deallocate(void)
{
	sword status;
	
	/* freeing an error handle */
	if(errhp)
	{
		if(status = OCIHandleFree((dvoid *)errhp, OCI_HTYPE_ERROR))
			return (status);
	}

	/* Detaches the process from the shared memory subsystem and releases the shared memory */
	status = OCITerminate(OCI_DEFAULT);

	return (status);
}

/* This function is used to create a simple logon session.
   @param username - The username. 
   @param password -  The user's password.
   @param dblink - The name of the database to connect to.
   */
int CConnection::connect(const char *username, const char *password, const char *dblink)
{
	sword status;
	
	if(state == connected)
	{
		// this object is already connected
		return (CONERR_ALRCON);
	}
	
	// create a simple logon session
	if(status = OCILogon(envhp, errhp, &svchp, (text*)username, (ub4)strlen(username), 
			(text*)password, (ub4)strlen(password), (text*)dblink, strlen(dblink)))
	{
		checkerr(errhp, status); 
	}
	else
	{
		// successful login
		state = connected;
	}

	return (status);
}

/* This function is used to terminate a connection and session created 
   with connect(). 
   */
int CConnection::disconnect()
{
	sword status;
	
	// delete the statement object
	if(stm != NULL)
	{
		delete stm;
		stm = NULL;
	}

	if(state == not_connected)
	{
		// this object has not been connected
		return (CONERR_NOTCON);
	}
	
	
	if(status = OCILogoff(svchp, errhp))
	{
		checkerr(errhp, status);
	}
	else
	{
		// successful logout
		state = not_connected;
	}
	
	return (status);
}

/* Commits the transaction associated with a specified service context. */
int CConnection::commit()
{
	sword status;
	
	if(state == not_connected)
	{
		// this object has not been connected
		return (CONERR_NOTCON);
	}
	
	checkerr(errhp, status = OCITransCommit(svchp, errhp, OCI_DEFAULT));

	return (status);
}

/* Rolls back the current transaction. */
int CConnection::rollback()
{
	sword status;
	
	if(state == not_connected)
	{
		// this object has not been connected
		return (CONERR_NOTCON);
	}
	
	checkerr(errhp, status = OCITransRollback(svchp, errhp, OCI_DEFAULT));

	return (status);
}

/* SQL statements with or without parameters are normally 
   executed using CStatement objects.
   */
CStatement* CConnection::createStatement()
{
	if(stm != NULL)
	{
		delete stm;
		stm = NULL;
	}

	if(state == connected)
	{
		// this object has been connected
		stm = new CStatement(this);
	}
	
	return stm;
}

/* Gets the table comments. */
CResultSet* CConnection::getTableComments()
{
	if(createStatement()) 
		return stm->executeQuery("SELECT * FROM user_tab_comments");
	else
		return NULL;
}

/* Gets the column comments.
   @param tab_name - The table name. 
   */
CResultSet* CConnection::getColumnComments(const char* tab_name)
{
	if(createStatement())
	{
		if(tab_name == NULL)
			return stm->executeQuery("SELECT * FROM user_col_comments");
		else if(!stm->prepareStatement("SELECT * FROM user_col_comments WHERE UPPER(table_name) = UPPER(:1)") &&
				!stm->setString(1, tab_name))
			return stm->executeQuery();
		else
			return NULL;
	}
	else
		return NULL;
}

/* write error message to the given file */
//void CConnection::display_error(FILE *file) const
//{
/*	if(lda.rc != 0)
	{
		sword n;
		text msg[512];
		
		n = oerhms((cda_def *)&lda, lda.rc, msg, (sword) sizeof(msg));
		err_report(file, msg, lda.fc);
	}
*/
//}

/* CStatement constructor.
   @param connection - The CConnection handle.
   */
CStatement::CStatement(CConnection* connection)
{
	con = connection;
	rs = NULL;
	stmthp = NULL;
	type = 0;
}

/* CStatement destructor. */
CStatement::~CStatement()
{
	// delete the statement object
	if(rs != NULL)
	{
		delete rs;
		rs = NULL;
	}
}

/* Execute a SQL statement that returns a single resultset.
   @param sql - typically this is a static SQL SELECT statement. 
   */
CResultSet* CStatement::executeQuery(const char* sql)
{
	if(rs != NULL)
	{
		delete rs;
		rs = NULL;
	}
	
	if(prepareStatement(sql) == 0 && isStmSelect())
		rs = executeQuery();

	return rs;
}

/* A prepared SQL query is executed and its resultset is returned. */
CResultSet* CStatement::executeQuery()
{
	sword status;
	
	if(rs != NULL)
	{
		delete rs;
		rs = NULL;
	}
	
	/* execute */
	if(status = OCIStmtExecute(con->svchp, stmthp, con->errhp, (ub4) 0, (ub4) 0,
		(CONST OCISnapshot *) NULL, (OCISnapshot *) NULL, OCI_DEFAULT))
	{
		checkerr(con->errhp, status);
	}
	else
	{
		rs = new CResultSet(this);		
	}

	return rs;
}

/* Execute a SQL INSERT, UPDATE or DELETE statement. 
   @param sql - a SQL INSERT, UPDATE or DELETE statement or a SQL statement that returns nothing. 
   */
int CStatement::executeUpdate(const char* sql)
{
	sword status;
	
	if(rs != NULL)
	{
		delete rs;
		rs = NULL;
	}
	
	if(!(status = prepareStatement(sql)) && !isStmSelect())
		status = executeUpdate();

	return (status);
}

/* Execute a SQL INSERT, UPDATE or DELETE statement. */
int CStatement::executeUpdate()
{
	sword status;
	
	if(rs != NULL)
	{
		delete rs;
		rs = NULL;
	}
	
	/* execute and fetch */
	if(status = OCIStmtExecute(con->svchp, stmthp, con->errhp, (ub4) 1, (ub4) 0,
			(CONST OCISnapshot *) NULL, (OCISnapshot *) NULL, OCI_DEFAULT))
	{
		checkerr(con->errhp, status);
	}

	return (status);
}

/* A SQL statement with or without IN parameters can be pre-compiled 
   and stored in a CStatement object.
   @param sql - a SQL statement that may contain one or more '?' IN parameter placeholders. 
   */
int CStatement::prepareStatement(const char* sql)
{
	sword status;
	
	if(status = OCIHandleAlloc((dvoid *) con->envhp, (dvoid **) &stmthp,
		OCI_HTYPE_STMT, (size_t) 0, (dvoid **) 0))
	{
		checkerr(con->errhp, status);
		return status;
	}

	// prepares a SQL or PL/SQL statement for execution
	checkerr(con->errhp, status = OCIStmtPrepare(stmthp, con->errhp, (text*)sql,
								(ub4) strlen(sql) + 1,
								(ub4) OCI_NTV_SYNTAX, (ub4) OCI_DEFAULT));
	
	// determine what type of SQL statement was prepared
	checkerr(con->errhp, status = OCIAttrGet((dvoid*) stmthp, (ub4) OCI_HTYPE_STMT, 
		(dvoid*) &type, (ub4*) 0, (ub4) OCI_ATTR_STMT_TYPE, con->errhp));

	return (status);
}

/* Set a parameter to a Oracle String value.
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
int CStatement::setString(const short index, const char* val)
{
	sword status;
	OCIBind*	bndp = (OCIBind*) 0;	/* the bind handle */	
	
	/* Bind the placeholders in the statement. */
	checkerr(con->errhp, status = OCIBindByPos(stmthp, &bndp, con->errhp, (ub4)index, 
			(text*) val, strlen(val) + 1, SQLT_STR, (dvoid *) 0,
			(ub2*) 0, (ub2*) 0, (ub4) 0, (ub4 *) 0, OCI_DEFAULT));

	return (status);
}

/* Set a parameter to a Oracle Integer value. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
int CStatement::setInt(const short index, const int* val)
{
	sword status;
	OCIBind*	bndp = (OCIBind*) 0;	/* the bind handle */	
	
	/* Bind the placeholders in the statement. */
	checkerr(con->errhp, status = OCIBindByPos(stmthp, &bndp, con->errhp, (ub4)index, 
			(dvoid*) val, (sb4) sizeof(int), SQLT_INT, (dvoid *) 0,
			(ub2*) 0, (ub2*) 0, (ub4) 0, (ub4*) 0, OCI_DEFAULT));

	return (status);
}

/* Set a parameter to a Oracle Float value. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
int CStatement::setFloat(const short index, const float* val)
{
	sword status;
	OCIBind*	bndp = (OCIBind*) 0;	/* the bind handle */	
	
	/* Bind the placeholders in the statement. */
	checkerr(con->errhp, status = OCIBindByPos(stmthp, &bndp, con->errhp, (ub4)index, 
			(dvoid*) val, sizeof(float), SQLT_FLT, (dvoid *) 0,
			(ub2*) 0, (ub2*) 0, (ub4) 0, (ub4 *) 0, OCI_DEFAULT));

	return (status);
}

/* Set a parameter to a Oracle Double value.
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
int CStatement::setDouble(const short index, const double* val)
{
	sword status;
	OCIBind*	bndp = (OCIBind*) 0;	/* the bind handle */	
	
	/* Bind the placeholders in the statement. */
	checkerr(con->errhp, status = OCIBindByPos(stmthp, &bndp, con->errhp, (ub4)index, 
			(dvoid*) val, sizeof(double), SQLT_FLT, (dvoid *) 0,
			(ub2*) 0, (ub2*) 0, (ub4) 0, (ub4 *) 0, OCI_DEFAULT));

	return (status);
}

/* Set a parameter to a Oracle Date value.
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
int CStatement::setDate(const short index, const COCIDate* val)
{
	sword status;
	OCIBind*	bndp = (OCIBind*) 0;	/* the bind handle */	

	/* Bind the placeholders in the statement. */
	checkerr(con->errhp, status = OCIBindByPos(stmthp, &bndp, con->errhp, (ub4)index, 
			(dvoid*) val->datetime, (sb4) sizeof(val->datetime), SQLT_DAT, (dvoid *) 0,
			(ub2*) 0, (ub2*) 0, (ub4) 0, (ub4 *) 0, OCI_DEFAULT));

	return (status);
}

/* Return the type of this statement. */
bool CStatement::isStmSelect()
{
	return (type == OCI_STMT_SELECT ? true : false);
}

/* CResultSet constructor. 
   @param statement - the CStatement handle.
   */
CResultSet::CResultSet(CStatement* statement)
{
	stm = statement;
	columns = 0;
	fieldInfos = NULL;
	fieldBuffers = NULL;
	fieldOffset = NULL;
	describeResult();
}

/* CResultSet destructor */
CResultSet::~CResultSet()
{
	close();
}

/* close the resultset. */
void CResultSet::close()
{
	if(stm == NULL)
		return;
	
	if(fieldOffset != NULL)
	{
		delete [] fieldOffset;
		fieldOffset = NULL;
	}
	
	if(fieldBuffers != NULL)
	{
		for(int i = 0; i < columns; i++)
			if(fieldBuffers[i].tDataBuffer == SQLT_CLOB) // free lob locator
				(void) OCIDescriptorFree((dvoid *) fieldBuffers[i].pDataBuffer, (ub4) OCI_DTYPE_LOB); 
			else
				delete fieldBuffers[i].pDataBuffer;

		delete [] fieldBuffers;
		fieldBuffers = NULL;
	}

	if(fieldInfos != NULL)
	{
		for(int i = 0; i < columns; i++)
			delete fieldInfos[i].name;
		
		delete [] fieldInfos;
		fieldInfos = NULL;
	}

	columns = 0;
}

/* retrieve information about multiple select-list items. */
void CResultSet::describeResult()
{
	if(stm == NULL)
		return;
	
	if(fieldInfos != NULL || fieldBuffers != NULL)
		close();

	OCIParam* pard;		/* the descriptor of the parameter */
	OCIDefine *defnp = (OCIDefine *) 0;
	sword status = 0;
	int counter = 0;
	
	/* get the number of columns in the table */
	while(status == OCI_SUCCESS) 
	{
		counter++;
		status = OCIParamGet(stm->stmthp, OCI_HTYPE_STMT, stm->con->errhp, (dvoid**)&pard,
								(ub4) counter);
	}	
	
	if((--counter) == 0)
		return;
	
	// Allocate buffer and get the OCI meta data
	fieldInfos = new CFieldInfo[counter];
	fieldBuffers = new CFieldBuffer[counter];
	fieldOffset = new unsigned int[counter];	

	/* Loop only if a descriptor was successfully retrieved for
	current position, starting at 1 */
	for(int i = 0; i < counter; i++) 
	{
		fieldInfos[i].name = NULL;
		fieldInfos[i].dtype = 0;
		fieldInfos[i].dsize = 0;
		fieldInfos[i].precision = fieldInfos[i].scale = 0;
		fieldInfos[i].nullability = 0;

		/* increment counter and get next descriptor, if there is one */
		(void) OCIParamGet(stm->stmthp, OCI_HTYPE_STMT, stm->con->errhp, (dvoid**)&pard,
								(ub4) (1 + i));
		
		/* Retrieve the data type attribute */
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid*) &fieldInfos[i].dtype, (ub4 *) 0, (ub4) OCI_ATTR_DATA_TYPE, stm->con->errhp));
		
		/* Retrieve the column name attribute */
		text* tmpbuff;
		ub4 str_len;
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid**) &tmpbuff, (ub4*) &str_len, (ub4) OCI_ATTR_NAME, 
					stm->con->errhp));
		
		fieldInfos[i].name = new char[str_len + 1];
		strncpy(fieldInfos[i].name, (char*)tmpbuff, str_len);
		fieldInfos[i].name[str_len] = '\0';

		/* Retrieve the data size attribute */
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid*) &fieldInfos[i].dsize, (ub4 *) 0, (ub4) OCI_ATTR_DATA_SIZE, stm->con->errhp));
	
		/* Retrieve the data precision attribute */
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid*) &fieldInfos[i].precision, (ub4 *) 0, (ub4) OCI_ATTR_PRECISION, stm->con->errhp));
		
		/* Retrieve the data scale attribute */
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid*) &fieldInfos[i].scale, (ub4 *) 0, (ub4) OCI_ATTR_SCALE, stm->con->errhp));
		
		/* Retrieve the data nullability attribute */
		checkerr(stm->con->errhp, status = OCIAttrGet((dvoid*) pard, (ub4) OCI_DTYPE_PARAM, 
					(dvoid*) &fieldInfos[i].nullability, (ub4 *) 0, (ub4) OCI_ATTR_IS_NULL, stm->con->errhp));
		
		// allocate an output buffer
		switch(fieldInfos[i].dtype)
		{
		case 1:		// VARCHAR2, NVARCHAR2 
		case 96:	// CHAR, NCHAR 
			fieldBuffers[i].pDataBuffer = new byte[1 + fieldInfos[i].dsize];
			fieldBuffers[i].lDataBuffer = 1 + fieldInfos[i].dsize;
			fieldBuffers[i].tDataBuffer = SQLT_STR;
			break;
		case 2: // NUMBER
			fieldBuffers[i].pDataBuffer = new byte[fieldInfos[i].dsize];
			fieldBuffers[i].lDataBuffer = fieldInfos[i].dsize;
			fieldBuffers[i].tDataBuffer = SQLT_VNU;
			break;
		case 12: // DATE
			fieldBuffers[i].pDataBuffer = new byte[fieldInfos[i].dsize];
			fieldBuffers[i].lDataBuffer = fieldInfos[i].dsize;
			fieldBuffers[i].tDataBuffer = SQLT_DAT;
			break;
		case 112: // CLOB
			if(status = OCIDescriptorAlloc((dvoid *) stm->con->envhp, 
				(dvoid **) &fieldBuffers[i].pDataBuffer, 
				(ub4) OCI_DTYPE_LOB, (size_t) 0, (dvoid **) 0))
			{
				checkerr(stm->con->errhp, status);
				fieldBuffers[i].pDataBuffer = NULL;
				fieldBuffers[i].lDataBuffer = 0;
				fieldBuffers[i].tDataBuffer = 0;
			}
			else
			{
				fieldBuffers[i].lDataBuffer = -1;
				fieldBuffers[i].tDataBuffer = SQLT_CLOB;
			}
			break;
		default:
			fieldBuffers[i].pDataBuffer = NULL;
			fieldBuffers[i].lDataBuffer = 0;
			fieldBuffers[i].tDataBuffer = 0;
		}

		// define the output variable
		if(fieldBuffers[i].pDataBuffer != NULL)
		{
			if(fieldBuffers[i].tDataBuffer == SQLT_CLOB)
				checkerr(stm->con->errhp, status = OCIDefineByPos(stm->stmthp, &defnp, stm->con->errhp,
						(ub4)(1 + i), (dvoid*) &fieldBuffers[i].pDataBuffer, (sb4) fieldBuffers[i].lDataBuffer,
						(ub2) fieldBuffers[i].tDataBuffer, (dvoid*) 0, (ub2*) 0, (ub2*)0, OCI_DEFAULT));
			else 
				checkerr(stm->con->errhp, status = OCIDefineByPos(stm->stmthp, &defnp, stm->con->errhp,
						(ub4)(1 + i), (dvoid*) fieldBuffers[i].pDataBuffer, (sb4) fieldBuffers[i].lDataBuffer,
						(ub2) fieldBuffers[i].tDataBuffer, (dvoid*) 0, (ub2*) 0, (ub2*)0, OCI_DEFAULT));
		}
	}

	columns = counter;
}

/* Returns specific kinds of information about the fields in a resultset.
   @param index - The zero-based index of the field.
   @param fieldinfo - A reference to a CFieldInfo structure.
   */
void CResultSet::getFieldInfo(const short index, CFieldInfo& fieldinfo)
{
	if(stm == NULL || fieldInfos == NULL || (1 + index) > columns)
		return;
	
	fieldinfo = fieldInfos[index];
}

/* Returns the number of fields in the recordset. */
short CResultSet::getFieldInfoCount()
{
	return (columns);
}

/* Gets the length of a LOB. 
   @param index - The zero-based index of the field.
   */
unsigned int CResultSet::getLobLenth(const short index)
{
	unsigned int ret = 0;
	sword status;

	if(stm == NULL || fieldBuffers == NULL || (1 + index) > columns || 
		fieldBuffers[index].pDataBuffer == NULL || fieldBuffers[index].tDataBuffer != SQLT_CLOB)
		return ret;
	
	/* Length of the Lob */
	checkerr(stm->con->errhp, status = OCILobGetLength(stm->con->svchp, stm->con->errhp, 
			(OCILobLocator*)fieldBuffers[index].pDataBuffer, &ret));
		
	return ret;
}

/* A CResultSet is initially positioned before its first row; 
   the first call to next makes the first row the current row; 
   the second call makes the second row the current row, etc. 
   */
int CResultSet::next()
{
	if(stm == NULL)
		return -1;
	
	sword status;
	
	// initialize the result buffers
	for(int i = 0; i < columns; i++)
	{
		if(fieldBuffers[i].tDataBuffer == SQLT_STR)
			strcpy((char*)fieldBuffers[i].pDataBuffer, (fieldInfos[i].nullability) ? "NULL" : "");
		else if(fieldBuffers[i].tDataBuffer == SQLT_VNU)
			OCINumberSetZero(stm->con->errhp, (OCINumber*)fieldBuffers[i].pDataBuffer);

		fieldOffset[i] = 1;
	}

	if((status = OCIStmtFetch(stm->stmthp, stm->con->errhp, (ub4)1, 
					OCI_FETCH_NEXT, OCI_DEFAULT)) && (status != OCI_NO_DATA))
		checkerr(stm->con->errhp, status);

	return (status);
}

/* Get the value of a column in the current row as a String.
   @param index - The zero-based index of the field.
   @param dest - Buffer into which the converted string is placed. 
   @param len - Size of the buffer.
   */
char* CResultSet::getString(const short index, char* dest, const short len)
{
	sword status;

	memset((dvoid *)dest, '\0', len);
	
	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return dest;

	if(fieldBuffers[index].tDataBuffer == SQLT_STR)
	{
		strncpy(dest, (char*) fieldBuffers[index].pDataBuffer, len - 1);
	}
	else if(fieldBuffers[index].tDataBuffer == SQLT_VNU)
	{
		text temp[22]; 
		char* fmt = "FM9999999999999999999";
		ub4 temp_size = sizeof(temp);
		
//		if(fieldInfos[index].precision > fieldInfos[index].scale &&
//			fieldInfos[index].scale > 0 && (strlen(fmt) - 2) > fieldInfos[index].scale)
//			strnset(fmt + (strlen(fmt) - (1 + fieldInfos[index].scale)), 'D', 1);

		if(status = OCINumberToText(stm->con->errhp, (OCINumber*)fieldBuffers[index].pDataBuffer, 
				(text*) fmt, (ub4) strlen(fmt), 
				(text*) 0, (ub4) 0,
				(ub4*) &temp_size, (text*) temp))
		{
			checkerr(stm->con->errhp, status);
		}
		else
			strncpy(dest, (char*) temp, len - 1);
	}
	else if(fieldBuffers[index].tDataBuffer == SQLT_DAT)
	{
		byte* pb = (byte*) fieldBuffers[index].pDataBuffer;
		tm time; 
		(time.tm_sec = pb[6])--;	// second
		(time.tm_min = pb[5])--;	// minute
		(time.tm_hour = pb[4])--;	// hour
		time.tm_mday = pb[3];		// day
		(time.tm_mon = pb[2])--;	// month
		time.tm_year = pb[1];		// year
		
		strftime(dest, len, "%c", &time);
	}
	else if(fieldBuffers[index].tDataBuffer == SQLT_CLOB)
	{
		strncpy(dest, "{CLOB}", len - 1);
	}

	return dest;
}

/* Get the value of a column in the current row as an int.
   @param index - The zero-based index of the field.
   */
int CResultSet::getInt(const short index)
{
	sword ret = 0;
	sword status;

	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return ret;

	if(fieldBuffers[index].tDataBuffer == SQLT_VNU)
	{
		if(status = OCINumberToInt(stm->con->errhp, (OCINumber*)fieldBuffers[index].pDataBuffer, 
				sizeof(ret), OCI_NUMBER_SIGNED, (dvoid *)&ret))
		{
			checkerr(stm->con->errhp, status);
		}
	}

	return ret;
}

/* Get the value of a column in the current row as a float.
   @param index - The zero-based index of the field.   
   */
float CResultSet::getFloat(const short index)
{
	float ret = 0;
	sword status;

	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return ret;

	if(fieldBuffers[index].tDataBuffer == SQLT_VNU)
	{
		if(status = OCINumberToReal(stm->con->errhp, (OCINumber*)fieldBuffers[index].pDataBuffer, 
				sizeof(ret), (dvoid*)&ret))
		{
			checkerr(stm->con->errhp, status);
		}
	}

	return ret;
}

/* Get the value of a column in the current row as a double.
   @param index - The zero-based index of the field.
   */
double CResultSet::getDouble(const short index)
{
	double ret = 0;
	sword status;

	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return ret;

	if(fieldBuffers[index].tDataBuffer == SQLT_VNU)
	{
		if(status = OCINumberToReal(stm->con->errhp, (OCINumber*)fieldBuffers[index].pDataBuffer, 
				sizeof(ret), (dvoid*)&ret))
		{
			checkerr(stm->con->errhp, status);
		}
	}

	return ret;
}

/* Get the value of a column in the current row as a long double.
   @param index - The zero-based index of the field.
   */
long double CResultSet::getLongDouble(const short index)
{
	long double ret = 0;
	sword status;

	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return ret;

	if(fieldBuffers[index].tDataBuffer == SQLT_VNU)
	{
		if(status = OCINumberToReal(stm->con->errhp, (OCINumber*)fieldBuffers[index].pDataBuffer, 
				sizeof(ret), (dvoid*)&ret))
		{
			checkerr(stm->con->errhp, status);
		}
	}

	return ret;
}

/* Get the value of a column in the current row as a tm structure.
   @param index - The zero-based index of the field.
   @param time - Pointer to a tm structure.
   */
int CResultSet::getDate(const short index, tm* time)
{
	sword status = -1;
	
	time->tm_sec = 0;	// Seconds after minute (0  59)
	time->tm_min = 0;	// Minutes after hour (0  59)
	time->tm_hour = 0;	// Hours since midnight (0  23)
	time->tm_mday = 1;	// Day of month (1  31)
	time->tm_mon = 0;	// Month (0  11; January = 0)
	time->tm_year = 0;	// Year (current year minus 1900)
	time->tm_wday = 0;	// Day of week (0  6; Sunday = 0)
	time->tm_yday = 0;	// Day of year (0  365; January 1 = 0)
	time->tm_isdst = 0;	// Positive if daylight saving time is in effect; 
						// 0 if daylight saving time is not in effect;

	if(stm == NULL || fieldInfos == NULL || 
			(1 + index) > columns || fieldBuffers[index].pDataBuffer == NULL)
		return status;

	if(fieldBuffers[index].tDataBuffer == SQLT_DAT)
	{
		byte* pb = (byte*) fieldBuffers[index].pDataBuffer;
		
		(time->tm_sec = pb[6])--;	// second
		(time->tm_min = pb[5])--;	// minute
		(time->tm_hour = pb[4])--;	// hour
		time->tm_mday = pb[3];		// day
		(time->tm_mon = pb[2])--;	// month
		time->tm_year = pb[1];		// year

		status = 0;
	}

	return status;
}

/* Set the absolute offset from the beginning of the LOB value.
   @param index - The zero-based index of the field.
   @param offset - The absolute offset from the beginning of the LOB value. 
   */
void CResultSet::setLobOffset(const short index, const unsigned int offset)
{
	
	if(stm == NULL || fieldInfos == NULL || (1 + index) > columns || 
		fieldBuffers[index].tDataBuffer != SQLT_CLOB ||
		fieldBuffers[index].tDataBuffer != SQLT_BLOB)
		return;
	
	fieldOffset[index] = offset;
}

/* Read data from the lob locator.
   @param index - The zero-based index of the field.
   @param dest - The pointer to a buffer into which the piece will be read. 
   @param len - The length of the buffer in octets. 
*/
bool CResultSet::readClob(const short index, char* dest, const short len)
{
	memset((dvoid *)dest, '\0', len);
	ub4 amtp = len - 1;
	sword status = -1;

	if(stm == NULL || fieldInfos == NULL || (1 + index) > columns || 
		fieldBuffers[index].tDataBuffer != SQLT_CLOB)
		return false;
	
	if(fieldBuffers[index].pDataBuffer != NULL)
	{
		status = OCILobRead(stm->con->svchp, stm->con->errhp, (OCILobLocator*)fieldBuffers[index].pDataBuffer, 
				&amtp, fieldOffset[index], (dvoid*) dest, (ub4)len, (dvoid*) 0,
				NULL, (ub2) 0, (ub1) SQLCS_IMPLICIT);
	
		if(status == OCI_NEED_DATA)
		{
			fieldOffset[index] += amtp;
		}
		else
		{
			fieldOffset[index] = 1;
			checkerr(stm->con->errhp, status);
		}
	}

	return (status == OCI_NEED_DATA ? true : false);
}

/* Write data into the selected lob locator.
   @param index - The zero-based index of the field.
   @param source - The pointer to a buffer from which the piece will be written. 
   @param len - The length of the buffer in octets. 
   */
int CResultSet::writeClob(const short index, const char* source, const short len)
{
	sword status = -1;

	if(stm == NULL || fieldInfos == NULL || (1 + index) > columns || 
		fieldBuffers[index].tDataBuffer != SQLT_CLOB)
		return status;
	
	if(fieldBuffers[index].pDataBuffer != NULL)
	{
		/*Write it into the locator */
		ub4 amtp = len;    /* IN/OUT : IN - amount if data to write */
		status = OCILobWrite(stm->con->svchp, stm->con->errhp, (OCILobLocator*)fieldBuffers[index].pDataBuffer, 
					&amtp, fieldOffset[index], (dvoid *) source, (ub4) strlen(source) + 1, OCI_ONE_PIECE,
					(dvoid *)0, NULL, (ub2) 0, (ub1) SQLCS_IMPLICIT);
		
		if(status == OCI_SUCCESS)
		  fieldOffset[index] += amtp;
		else
		  checkerr(stm->con->errhp, status);
			
	}
	
	return status;
}

///////////////////////// Exported function ////////////////////////////

/* Attempt to establish a connection to the given database. 
   @param username - The username. 
   @param password -  The user's password.
   @param dblink - The name of the database to connect to.
   */
ORACALL32_API CConnection* SYAllocConnect(const char *username, const char *password, const char *dblink)
{
	CConnection* connection = new CConnection();
	
	if(connection != NULL) 
		if(connection->connect(username, password, dblink) != 0)
		{
			delete connection;
			connection = NULL;
		}

	return connection;
}

/* This call is used to terminate a session and connection which 
   were created with SYAllocConnect(). This call implicitly deallocates 
   the server, user session, and service context handles.
   @param connection - The CConnection handle. 
   */
ORACALL32_API void SYFreeConnect(CConnection* connection)
{
	if(connection != NULL)
		delete connection;
}

/* Commit makes all changes made since the previous commit/rollback permanent 
   and releases any database locks currently held by the CConnection. This 
   method should only be used when auto commit has been disabled.
   @param connection - The CConnection handle. 
   */
ORACALL32_API int SYCommit(CConnection* connection)
{
	if(connection == NULL)
		return -1;
	
	return connection->commit();
}

/* Rollback drops all changes made since the previous commit/rollback and 
   releases any database locks currently held by the CConnection. This 
   method should only be used when auto commit has been disabled.
   @param connection - The CConnection handle. 
   */
ORACALL32_API int SYRollback(CConnection* connection)
{
	if(connection == NULL)
		return -1;
	
	return connection->rollback();
}

/* Execute a SQL statement that returns a single CResultSet. 
   @param connection - The CConnection handle. 
   @param query - typically this is a static SQL SELECT statement. 
   */
ORACALL32_API CResultSet* SYExecuteSQL(CConnection* connection, const char* query)
{
	if(connection == NULL)
		return NULL;

	CStatement* statement = connection->createStatement();
	
	if(statement == NULL)
		return NULL;
	
	return statement->executeQuery(query);
}

/* Execute a SQL INSERT, UPDATE or DELETE statement. In addition, SQL statements 
   that return nothing such as SQL DDL statements can be executed.
   @param connection - The CConnection handle. 
   @param query - a SQL INSERT, UPDATE or DELETE statement or a SQL statement that returns nothing. 
  */
ORACALL32_API int SYExecuteUpdate(CConnection* connection, const char* query)
{
	if(connection == NULL)
		return -1;
	
	CStatement* statement = connection->createStatement();
	
	if(statement == NULL)
		return -1;
	
	return statement->executeUpdate(query);
}

/* A SQL statement with or without IN parameters can be pre-compiled and 
   stored in a CStatement object. This object can then be used 
   to efficiently execute this statement multiple times. 
   @param connection - The CConnection handle. 
   @param query - a SQL statement that may contain one or more '?' IN parameter placeholders. 
*/
ORACALL32_API CStatement* SYPrepare(CConnection* connection, const char* query)
{
	if(connection == NULL)
		return NULL;

	CStatement* statement = connection->createStatement();
	
	if(statement == NULL || statement->prepareStatement(query))
		return NULL;

	return (statement);
}

/* A prepared SQL query is executed and its CResultSet is returned.
   @param statement - The CStatement handle. 
   */
ORACALL32_API CResultSet* SYExecutePrepared(CStatement* statement)
{
	CResultSet* ret = NULL;
	
	if(statement == NULL)
		return ret;
	
	if(statement->isStmSelect())
		ret = statement->executeQuery();
	else
		statement->executeUpdate();

	return ret;
}

/* Set a parameter to a String value. The driver converts this to a SQL 
   VARCHAR or LONGVARCHAR value (depending on the arguments size relative to 
   the driver's limits on VARCHARs) when it sends it to the database.
   @param statement - The CStatement handle. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
*/
ORACALL32_API void SYSetParamString(CStatement* statement, const short index, const char* val)
{
	if(statement == NULL)
		return;
	
	statement->setString(index, val);
}

/* Set a parameter to a int value. The driver converts this to a SQL INTEGER 
   value when it sends it to the database.
   @param statement - The CStatement handle. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
ORACALL32_API void SYSetParamInteger(CStatement* statement, const short index, const int* val)
{
	if(statement == NULL)
		return;
	
	statement->setInt(index, val);
}

/* Set a parameter to a float value. The driver converts this to a SQL FLOAT 
   value when it sends it to the database.
   @param statement - The CStatement handle. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
ORACALL32_API void SYSetParamFloat(CStatement* statement, const short index, const float* val)
{
	if(statement == NULL)
		return;
	
	statement->setFloat(index, val);
}

/* Set a parameter to a double value. The driver converts this to a SQL DOUBLE 
   value when it sends it to the database.
   @param statement - The CStatement handle. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
ORACALL32_API void SYSetParamDouble(CStatement* statement, const short index, const double* val)
{
	if(statement == NULL)
		return;
	
	statement->setDouble(index, val);
}

/* Set a parameter to a Date value. The driver converts this to a SQL DATE 
   value when it sends it to the database.
   @param statement - The CStatement handle. 
   @param index - the first parameter is 1, the second is 2, ... 
   @param val - the parameter value.
   */
ORACALL32_API void SYSetParamDate(CStatement* statement, const short index, const COCIDate* val)
{
	if(statement == NULL)
		return;
	
	statement->setDate(index, val);
}

/* Returns specific kinds of information about the fields in a resultset.
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param fieldinfo - A reference to a CFieldInfo structure.
   */
ORACALL32_API void SYGetFieldInfo(CResultSet* rs, const short index, CFieldInfo& fieldinfo)
{
	if(rs != NULL)
		rs->getFieldInfo(index, fieldinfo);
}

/* Returns the number of fields in the resultset. 
   @param rs - The CResultSet handle. 
   */
ORACALL32_API short SYGetFieldInfoCount(CResultSet* rs)
{
	if(rs == NULL)
		return 0;

	return rs->getFieldInfoCount();
}

/* Returns the table comments. 
   @param connection - The CConnection handle. 
   */
ORACALL32_API CResultSet*	SYGetTableComments(CConnection* connection)
{
	if(connection == NULL)
		return NULL;
	
	return connection->getTableComments();
}

/* Returns the column comments. 
   @param connection - The CConnection handle. 
   @param tab_name - The table name. 
   */
ORACALL32_API CResultSet*	SYGetColumnComments(CConnection* connection, const char* tab_name)
{
	if(connection == NULL)
		return NULL;
	
	return connection->getColumnComments(tab_name);
}

/* A CResultSet is initially positioned before its first row; the first call 
   to next makes the first row the current row; the second call makes the 
   second row the current row, etc. 
   @param rs - The CResultSet handle. 
   */
ORACALL32_API int SYNext(CResultSet* rs)
{
	if(rs == NULL)
		return -1;
	
	return rs->next();
}

/* Get the value of a column in the current row as a String. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param dest - Buffer into which the converted string is placed. 
   @param len - Size of the buffer.
   */
ORACALL32_API char*	SYGetString(CResultSet* rs, const short index, char* dest, const short len)
{
	if(rs == NULL)
		return dest;
	
	return rs->getString(index, dest, len);
}

/* Get the value of a column in the current row as int. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   */
ORACALL32_API int SYGetInteger(CResultSet* rs, const short index)
{
	if(rs == NULL)
		return 0;

	return rs->getInt(index);
}

/* Get the value of a column in the current row as float. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   */
ORACALL32_API float SYGetFloat(CResultSet* rs, const short index)
{
	if(rs == NULL)
		return 0;

	return rs->getFloat(index);
}

/* Get the value of a column in the current row as double. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   */
ORACALL32_API double SYGetDouble(CResultSet* rs, const short index)
{
	if(rs == NULL)
		return 0;

	return rs->getDouble(index);
}

/* Get the value of a column in the current row as long double. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   */
ORACALL32_API long double SYGetLongDouble(CResultSet* rs, const short index)
{
	if(rs == NULL)
		return 0;

	return rs->getLongDouble(index);
}

/* Get the value of a column in the current row as a Date. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param time - Pointer to a tm structure.
   */
ORACALL32_API int SYGetDate(CResultSet* rs, const short index, tm* time)
{
	if(rs == NULL)
		return 0;

	return rs->getDate(index, time);
}

/* Set the absolute offset from the beginning of the LOB value. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param offset - The absolute offset from the beginning of the LOB value. 
   */
ORACALL32_API void SYSetLobOffset(CResultSet* rs, const short index, const unsigned int offset)
{
	if(rs != NULL)
		rs->setLobOffset(index, offset);
}

/* Read data from the lob locator. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param dest - The pointer to a buffer into which the piece will be read. 
   @param len - The length of the buffer in octets. 
   */
ORACALL32_API bool SYReadClob(CResultSet* rs, const short index, char* dest, const short len)
{
	if(rs == NULL)
		return false;

	return rs->readClob(index, dest, len);
}

/* Write data into the selected lob locator. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
   @param source - The pointer to a buffer from which the null-terminated source string will be written. 
   */
ORACALL32_API int SYWriteClob(CResultSet* rs, const short index, const char* source)
{
	if(rs == NULL)
		return -1;

	return rs->writeClob(index, source, strlen(source));
}

/* Returns the size of a field. 
   @param rs - The CResultSet handle. 
   @param index - The zero-based index of the field.
*/
ORACALL32_API unsigned int SYGetLobLenth(CResultSet* rs, const short index)
{
	if(rs == NULL)
		return 0;

	return rs->getLobLenth(index);
}
