// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- //
// C++ Header File Name: gxbtree.h 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 08/22/2000 
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ---------- Include File Description and Details  ---------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The gxBtree class is a disk-base B-tree used by various database 
applications for searching and sorting structured data and indexing 
associated objects or records stored in a data file. The gxBtree 
uses the 32/64-bit gxDatabase engine to perform all disk operations, 
supporting large files and proprietary file system interfaces. 
*/
// ----------------------------------------------------------- //   
#ifndef __GX_DATABASE_BTREE_HPP__
#define __GX_DATABASE_BTREE_HPP__

#include "btnode.h"
#include "btstack.h"

// Btree header
struct gxBtreeHeader
{ 
  BtreeNodeOrder_t node_order; // Btree node order
  BtreeSize_t key_size;        // Database key size
  BtreeSize_t n_keys;          // Total number of entry keys
  BtreeSize_t n_nodes;         // Total number of nodes
  BtreeSize_t btree_height;    // Height of this Btree
  FAU root;                    // Btree root address  
  gxClassID class_id;          // Optional class ID for Btree object keys
};

// Btree class
class gxBtree
{  
public:
  gxBtree(DatabaseKeyB &key_type, BtreeNodeOrder_t order,
	 gxClassID cid = (gxClassID_t)-1);
  virtual ~gxBtree();

public: // Btree database file functions
  gxDatabaseError Create(const char *fname, int num_trees = 1);
  gxDatabaseError Open(const char *fname, 
		       gxDatabaseAccessMode mode = gxDBASE_READWRITE);
  gxDatabaseError Open(const char *fname, FAU header_address,
		       gxDatabaseAccessMode mode = gxDBASE_READWRITE); 
  gxDatabaseError InitBtree(gxDatabase *fptr, int create, 
			    FAU header_address);
  gxDatabaseError InitBtree(int create, FAU header_address);
  gxDatabaseError Close();
  void Release();
  gxDatabaseError Flush();
  gxDatabaseError ReadBtreeHeader(); 
  gxDatabaseError WriteBtreeHeader();
  gxDatabaseError ReadNode(BtreeNode &node, FAU
			   node_address = gxCurrAddress);
  gxDatabaseError WriteNode(const BtreeNode &node,
			    FAU node_address = gxCurrAddress);

public: // Btree insert and delete functions
  int Insert(DatabaseKeyB &key, DatabaseKeyB &compare_key, int flush = 1);
  int Delete(DatabaseKeyB &key, DatabaseKeyB &compare_key, int flush = 1);
  int LazyDelete(DatabaseKeyB &key, DatabaseKeyB &compare_key, int flush = 1);

public: // Btree search functions
  int Find(DatabaseKeyB &key, DatabaseKeyB &compare_key, int test_tree = 1);
  int FindFirst(DatabaseKeyB &key, int test_tree = 1);
  int FindNext(DatabaseKeyB &key, DatabaseKeyB &compare_key,
	       int test_tree = 1);
  int FindPrev(DatabaseKeyB &key, DatabaseKeyB &compare_key,
	       int test_tree = 1);
  int FindLast(DatabaseKeyB &key, int test_tree = 1);
  
public:
  int TestTree(int reinit = 1);
  gxDatabaseError ReInit(int flush = 0);
  gxDatabaseError ReInit(FAU header_address, int flush = 0);
  int IsEmpty() { return btree_header.n_keys == (BtreeSize_t)0; }
  int IsEmpty() const { return btree_header.n_keys == (BtreeSize_t)0; }
  size_t TotalNodeSize();
  int HeaderInSync() { return header_in_sync == 1; }
  int HeaderInSync() const { return header_in_sync == 1; }
  gxDatabase *gxDatabasePtr() { return f; }
  gxBtreeHeader *BtreeHeader() { return &btree_header; }
  FAU Root() { return btree_header.root; }
  FAU Root() const { return btree_header.root; }
  FAU HeaderAddress() { return btree_header_address; }
  FAU HeaderAddress() const { return btree_header_address; }
  BtreeSize_t NumKeys() { return btree_header.n_keys; }
  BtreeSize_t NumKeys() const { return btree_header.n_keys; }
  BtreeSize_t NumNodes() { return btree_header.n_nodes; }
  BtreeSize_t NumNodes() const { return btree_header.n_nodes; }
  BtreeSize_t KeySize() { return btree_header.key_size; }
  BtreeSize_t KeySize() const { return btree_header.key_size; }
  BtreeNodeOrder_t NodeOrder() { return btree_header.node_order; }
  BtreeNodeOrder_t NodeOrder() const { return btree_header.node_order; }
  BtreeSize_t BtreeHeight() { return btree_header.btree_height; }
  BtreeSize_t BtreeHeight() const { return btree_header.btree_height; }
  gxClassID ClassID() { return btree_header.class_id; }
  gxClassID ClassID() const { return btree_header.class_id; }

public:  // Database exception handling functions
  gxDatabaseError GetDatabaseError() { return f->GetDatabaseError(); }
  gxDatabaseError GetDatabaseError() const {  return f->GetDatabaseError(); }
  gxDatabaseError SetDatabaseError(gxDatabaseError err) {
    return f->SetDatabaseError(err);
  }
  gxDatabaseError ResetDatabaseError() {
    return f->SetDatabaseError(gxDBASE_NO_ERROR);
  }
  const char *DatabaseExceptionMessage();

protected: // Internal Insert, delete, and balancing functions
  gxDatabaseError GrowNewRoot(DatabaseKeyB &key);
  gxDatabaseError CollapseRoot();
  gxDatabaseError CollapseRoot(BtreeNode &root_node);
  gxDatabaseError InsertDatabaseKey(DatabaseKeyB &key,
				    DatabaseKeyB &compare_key);
  gxDatabaseError LazyDeleteDatabaseKey(DatabaseKeyB &key,
					DatabaseKeyB &compare_key);
  gxDatabaseError DeleteDatabaseKey(DatabaseKeyB &key,
				    DatabaseKeyB &compare_key);
  gxDatabaseError InsertBalance(BtreeNode &parent_node,
				BtreeKeyLocation_t key_location,
				DatabaseKeyB &compare_key);
  gxDatabaseError InsertRightRotation(BtreeNode &parent_node,
				      BtreeKeyLocation_t key_location,
				      DatabaseKeyB &compare_key);
  gxDatabaseError InsertLeftRotation(BtreeNode &parent_node,
				     BtreeKeyLocation_t key_location,
				     DatabaseKeyB &compare_key);
  gxDatabaseError Merge(BtreeNode &parent_node,
			BtreeKeyLocation_t key_location,
			DatabaseKeyB &compare_key);

protected:
  gxDatabase *f;              // Database the Btree is connected to
  FAU btree_header_address;   // File address of the Btree header
  gxBtreeHeader btree_header; // In memory copy of the Btree header
  
  // True if the in-memory copy of the header is in sync with the disk copy
  int header_in_sync; 
};

#endif // __GX_DATABASE_BTREE_HPP__
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

