// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- //
// C++ Source Code File Name: pod.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 09/18/1997
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The POD (Persistent Object Database) manager is used to manage
file pointers to database data files and index files.
*/
// ----------------------------------------------------------- //   
#include "pod.h"

PODIndexFile::PODIndexFile(unsigned num_indexes)
{
  // Ensure that the maximum number of index file is not exceeded
  if(num_indexes > POD_MAX_INDEX) num_indexes = POD_MAX_INDEX;

  // Clear the index file array
  unsigned i;
  for(i = 0; i < POD_MAX_INDEX; i++) index_file[i] = 0;

  n_indexes = num_indexes; 
  curr_index = 0;

  // Allocate memory for the specified number of index files
  for(i = 0; i < n_indexes; i++) {
    index_file[i] = new PODIndexFile_t;
    if(!index_file[i]) { // Check for any memory allocation errors
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#endif
    }
  }
}

PODIndexFile::~PODIndexFile()
{
  unsigned i;
  for(i = 0; i < n_indexes; i++) delete index_file[i];
}

gxDatabaseError PODIndexFile::Open(const char *ifname,
				   unsigned index_number, 
				   DatabaseKeyB &key_type,
				   BtreeNodeOrder_t order,
				   gxDatabaseAccessMode mode,
				   gxClassID cid)
// Function used to open an existing index file or create a new
// one if the specified file name does not exist. Returns a non-zero
// value to indicate an error condition or zero if successful.
{
  // Construct a new Btree for the specified index
  gxBtree *btree = new gxBtree(key_type, order, cid);
  if(!btree) { 
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  index_file[index_number]->btree = btree;  
  curr_index = index_number;

  if(gxDatabase::Exists(ifname)) { // Open the the file if it exists
    return btree->Open(ifname, mode);
  }
  else { // Create a the file
    return btree->Create(ifname);
  }
}

gxDatabaseError PODIndexFile::Close(unsigned index_number)
// Function used to close an open index file. Returns a non-zero
// value to indicate an error condition or zero if successful.
{

  if(index_file[index_number]->btree) {
    if(index_file[index_number]->btree->Close() != gxDBASE_NO_ERROR) {
      return index_file[index_number]->btree->GetDatabaseError();
    }
    delete index_file[index_number]->btree;
    index_file[index_number]->btree = 0;
  }
  return gxDBASE_NO_ERROR;
}

POD::POD()
{
  opendatafile = 0;
  openindexfile = 0;
  exists = 0;
  using_index = 0;
}

POD::~POD() 
{ 
  Close();
}

gxDatabaseError POD::Open(const char *dfname, const char *ifname,
			  DatabaseKeyB &key_type, BtreeNodeOrder_t order,
			  gxDatabaseAccessMode mode, int use_index,
			  FAU static_size, int num_trees)
// Function used to open an existing database or create a new one using
// a single index file. Returns a non-zero value to indicate an error
// condition or zero if successful.
{
  // Close any currently open data and/or index files
  gxDatabaseError err = Close();
  if(err != gxDBASE_NO_ERROR) return err;
  
  unsigned num_indexes = 1;

  if(use_index) { // Initialize the indexing subsystem
    openindexfile = new PODIndexFile(num_indexes);
    if(!openindexfile) { // Could not allocate memory for the index file
      using_index = 0;
    }
    else {
      using_index = use_index;
      openindexfile->n_indexes = num_indexes;
      openindexfile->curr_index = 0;
      openindexfile->index_file[0]->dbkey_size = key_type.SizeOfDatabaseKey();
      openindexfile->index_file[0]->order = order;
      openindexfile->index_file[0]->use_index = 1;
    }
  }

  if(!gxDatabase::Exists(dfname)) { // Create a new data file
    if(CreateDataFile(dfname, static_size) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 0;
    if(using_index) {
      // Overwrite the index file if it exists since the index keys will
      // be useless without a valid data file.
      if(CreateIndex(ifname, 0, key_type, order, num_trees) !=
	 gxDBASE_NO_ERROR) {
	return GetIndexFileError();
      }
    }
  }
  else { // Open the existing data file
    if(OpenDataFile(dfname, mode) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 1;
    if(using_index) { // Open the index file if it exist
      if(!gxDatabase::Exists(ifname)) {
	// This index file will have to be rebuilt by the application
	openindexfile->index_file[0]->rebuild_index = 1;
	if(CreateIndex(ifname, 0, key_type, order, num_trees) !=
	   gxDBASE_NO_ERROR) {
	  return GetIndexFileError();
	}
      }
      else {
	if(OpenIndex(ifname, 0, key_type, order, mode) != gxDBASE_NO_ERROR) {
	  return GetIndexFileError();
	}
      }
    }
  }
  return gxDBASE_NO_ERROR;
}

gxDatabaseError POD::Open(const char *dfname, PODIndexFile *ix_ptr,
			  gxDatabaseAccessMode mode, FAU static_size)
// Function used to open an existing database or create a new one using
// multiple index files with different key types. Returns a non-zero value
// to indicate an error condition or zero if successful.
{
  // Close any currently open data and/or index files
  gxDatabaseError err = Close();
  if(err != gxDBASE_NO_ERROR) return err;

  if(!gxDatabase::Exists(dfname)) { // Create a new datafile
    if(CreateDataFile(dfname, static_size) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 0;
    using_index = 1;
    openindexfile = ix_ptr;
  }
  else { // Open the existing data file
    if(OpenDataFile(dfname, mode) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 1;
    using_index = 1;
    openindexfile = ix_ptr;
  }

  return gxDBASE_NO_ERROR;
}

gxDatabaseError POD::Open(const char *dfname, char *ifname[POD_MAX_INDEX],
			  unsigned num_indexes, DatabaseKeyB &key_type,
			  BtreeNodeOrder_t order, gxDatabaseAccessMode mode,
			  FAU static_size)
// Function used to open an existing database or create a new one using
// multiple index files that use the same key type. Returns a non-zero
// value to indicate an error condition or zero if successful.
{
  // Close any currently open data and/or index files
  gxDatabaseError err = Close();
  if(err != gxDBASE_NO_ERROR) return err;

  unsigned i;
  openindexfile = new PODIndexFile(num_indexes);
  if(!openindexfile) {  // Memory allocation error
    using_index = 0;
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  else {
    using_index = 1;
    openindexfile->n_indexes = num_indexes;
    openindexfile->curr_index = 0;
    for(i = 0; i < num_indexes; i++) {
      openindexfile->index_file[i]->dbkey_size = key_type.SizeOfDatabaseKey();
      openindexfile->index_file[i]->order = order;
      openindexfile->index_file[i]->use_index = 1;
    }
  }
  
  if(!gxDatabase::Exists(dfname)) { // Create a new data file
    if(CreateDataFile(dfname, static_size) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 0;
    for(i = 0; i < num_indexes; i++) {
      // Overwrite all the index files since the index keys will
      // be useless without a valid data file.
      if(CreateIndex((const char *)ifname[i], i, key_type, order) !=
	 gxDBASE_NO_ERROR) {
	return GetIndexFileError();
      }
    }
  }
  else { // Open the existing data file
    if(OpenDataFile(dfname, mode) != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    exists = 1;
    for(i = 0; i < num_indexes; i++) {
      if(!gxDatabase::Exists((const char *)ifname[i])) {
	// These index files will have to be rebuilt by the application
	openindexfile->index_file[i]->rebuild_index = 1;
	if(CreateIndex((const char *)ifname[i], i, key_type, order) !=
	   gxDBASE_NO_ERROR) {
	  return GetIndexFileError();
	}
	
      }
      else {
	if(OpenIndex((const char *)ifname[i], i, key_type, order, mode) !=
	   gxDBASE_NO_ERROR) {
	  return GetIndexFileError();
	}
      }
    }
  }
  return gxDBASE_NO_ERROR;
}

void POD::Release()
// Reset the data file and index file pointers to zero without
// closing the files or performing any other action. NOTE: This
// is function used to reset the file pointers when more the one
// object is referencing this POD database and the database has
// been closed or the file pointers have been deleted.
{
  opendatafile = 0;
  openindexfile = 0;
}

gxDatabaseError POD::Close()
// Function used to close the database, disconnecting both the data and
// index files. Returns a non-zero value to indicate an error condition
// or zero if successful.
{
  if(CloseDataFile() != gxDBASE_NO_ERROR) {
    return GetDataFileError();
  }
  if(openindexfile) {
    for(unsigned i = 0; i < openindexfile->n_indexes; i++) {
      if(CloseIndex(i) != gxDBASE_NO_ERROR) {
	openindexfile->curr_index = i;
	return GetIndexFileError(i);
      }
    }
    delete openindexfile;
    openindexfile = 0;
  }
  return gxDBASE_NO_ERROR;
}

gxDatabaseError POD::CreateDataFile(const char *fname, FAU static_size)
// Function used to create a new data file. The "static_size"
// variable is used to reserve a specified number of bytes that
// will not be affected by the dynamic allocation routines.
// Returns a non-zero value to indicate an error condition or
// zero if successful.
{
  // Close the current data file before creating a new one
  if(CloseDataFile() != gxDBASE_NO_ERROR) {
    return GetDataFileError();
  }
  opendatafile = new gxDatabase;
  if(!opendatafile) { // Check for any memory allocation errors
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  return opendatafile->Create(fname, static_size);
}

gxDatabaseError POD::CreateIndex(const char *fname, unsigned index_number, 
				 DatabaseKeyB &key_type,
				 BtreeNodeOrder_t order, 
				 int num_trees)
{
  if(num_trees <= 0) num_trees = 1;
  if(CloseIndex(index_number) != gxDBASE_NO_ERROR) {
    return GetIndexFileError(index_number);
  }
  gxDatabase *db_file = new gxDatabase;
  if(!db_file) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  
  gxBtree *btree = new gxBtree(key_type, order);
  if(!btree) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  openindexfile->index_file[index_number]->btree = btree;
  
  if(db_file->Create(fname, (sizeof(gxBtreeHeader) * num_trees)) !=
     gxDBASE_NO_ERROR) {
    return db_file->GetDatabaseError();
  }

  return btree->InitBtree(db_file, 1, (FAU)db_file->FileHeaderSize());
}

void POD::ReleaseDataFile()
// Reset the data file pointer to zero without closing the
// file or performing any other action. NOTE: This function is
// used to reset the file pointer when more the one object is
// referencing this file pointer and the file has been closed
// or the pointer has been deleted.
{
  opendatafile = 0;
}

gxDatabaseError POD::CloseDataFile()
// Disconnects the database from the file.
{
  if(opendatafile) {
    if(opendatafile->Close() != gxDBASE_NO_ERROR) {
      return GetDataFileError();
    }
    delete opendatafile;
    opendatafile = 0;
  }
  return gxDBASE_NO_ERROR;
}

void POD::ReleaseIndexFile()
// Reset the index file pointer to zero without closing the
// file or performing any other action. NOTE: This function is
// used to reset the file pointer when more the one object is
// referencing this file pointer and the file has been closed
// or the pointer has been deleted.
{
  openindexfile = 0;
}

gxDatabaseError POD::CloseIndex(unsigned index_number)
// Disconnects the database from the index file.
{
  gxBtree *btree = openindexfile->index_file[index_number]->btree;
  if(btree) {
    if(btree->Close() != gxDBASE_NO_ERROR) {
      return GetIndexFileError(index_number);
    }
    delete btree;
    openindexfile->index_file[index_number]->btree = 0;
    openindexfile->index_file[index_number]->use_index = 0;
  }
  return gxDBASE_NO_ERROR;
}

gxDatabaseError POD::OpenDataFile(const char *fname, gxDatabaseAccessMode mode)
{
  if(CloseDataFile() != gxDBASE_NO_ERROR) {
    return GetDataFileError();
  }
  opendatafile = new gxDatabase;
  if(!opendatafile) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  return opendatafile->Open(fname, mode);
}

gxDatabaseError POD::OpenIndex(const char *fname, unsigned index_number, 
			       DatabaseKeyB &key_type, BtreeNodeOrder_t order, 
			       gxDatabaseAccessMode mode)
{
  if(CloseIndex(index_number) != gxDBASE_NO_ERROR) {
    return GetIndexFileError(index_number);
  }
  gxBtree *btree = new gxBtree(key_type, order);
  if(!btree) {
#ifdef __CPP_EXCEPTIONS__
    throw gxCDatabaseException();
#else
    return gxDBASE_MEM_ALLOC_ERROR;
#endif
  }
  openindexfile->index_file[index_number]->btree = btree;
  return btree->Open(fname, mode);
}

gxBtree *POD::Index(unsigned index_number)
// Returns a pointer to the Btree index file for the specified
// index number.
{
  if(index_number > openindexfile->n_indexes) return 0;
  return openindexfile->index_file[index_number]->btree;
}

gxBtree *POD::Index(unsigned index_number) const
// Returns a pointer to the Btree index file for the specified
// index number.
{
  if(index_number > openindexfile->n_indexes) return 0;
  return openindexfile->index_file[index_number]->btree;
}

gxDatabaseError POD::GetDataFileError()
{
  if(!opendatafile) return gxDBASE_NO_DATABASE_OPEN;
  return opendatafile->GetDatabaseError();
}

gxDatabaseError POD::GetDataFileError() const
{
  if(!opendatafile) return gxDBASE_NO_DATABASE_OPEN;
  return opendatafile->GetDatabaseError();
}

gxDatabaseError POD::SetDataFileError(gxDatabaseError err)
{
  if(!opendatafile) return gxDBASE_NO_DATABASE_OPEN;
  return opendatafile->SetDatabaseError(err);
}

gxDatabaseError POD::GetIndexFileError(unsigned index_number)
{
  if(!Index(index_number)) return gxDBASE_NO_DATABASE_OPEN;
  return Index(index_number)->GetDatabaseError();
}

gxDatabaseError POD::GetIndexFileError(unsigned index_number) const
{
  if(!Index(index_number)) return gxDBASE_NO_DATABASE_OPEN;
  return Index(index_number)->GetDatabaseError();
}

gxDatabaseError POD::SetIndexFileError(gxDatabaseError err,
				       unsigned index_number)
{
  if(!Index(index_number)) return gxDBASE_NO_DATABASE_OPEN;
  return Index(index_number)->SetDatabaseError(err);
}

const char *POD::DataFileExceptionMessage()
{
  if(!opendatafile)
    return gxDatabaseExceptionMessage(gxDBASE_NO_DATABASE_OPEN);
  return opendatafile->DatabaseExceptionMessage();
}

const char *POD::IndexFileExceptionMessage(unsigned index_number)
{
  if(!Index(index_number))
    return gxDatabaseExceptionMessage(gxDBASE_NO_DATABASE_OPEN);
  return Index(index_number)->DatabaseExceptionMessage();
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
