{ OPTLIB.PAS : Argument option handling routine library

  Title   : OPTLIB
  Version : 2.0
  Date    : Nov 11,1996
  Author  : J R Ferguson
  Language: Borland Turbo Pascal 4.0 through 7.0
            Targets: DOS Real and Protected mode
  Usage   : Unit

  This unit contains code to process command-line options of the form
    /<option>[[/]<option>...]
  where <option> has the form
    <idn>[<value>]
  <idn>   is a unique one-character option identifier. Upper and lower
          case letters are treated as equal.
  <value> describes the value that will be assigned to the specified
          option. Its syntax depends on the option type as described
          below.

  Five option types are recognized:
  1. Switch options can have the values true or false.
     For these options <value> can be + (for true) or - (for false).
     The default for <value> is +.
  2. Number options describe lists of one or more signed or unsigned
     integer values.
     For these options <value> has the format
        [<integer>][,[<integer>]...]
     where <integer> is empty or consists of an optional + or - sign
     followed by a sequence of 1 to 4 digits. For each item in the
     list a switch specifying whether the <integer> value may be omitted
     is provided when the option item is defined. Also a default as well
     as a minimum and a maximum value for each integer item is specified
     at definition time.
  5. Range option have integer range set values as defined in unit RngLib,
     and can be used to specify ranges of line numbers or text columns.
     For these options <value> specifies a left and right boundary pair,
     using the format
       [<integer>][,[<integer>]]
     where <integer> is empty or consists of an optional + or - sign
     followed by a sequence of 1 to 4 digits. For both integers a switch
     specifying whether the value may be omitted is provided when the
     option is defined. Also a default as well as a minimum and
     a maximum value for each integer is specified at definition time.
     When the same range option is parsed more than once the ranges will
     be combined (merged).
  4. Character options have character values. Their <value> consist of
     the character immediately following the option identifier and can
     only be omitted with the last option in a command argument. The
     default value for a character option is specified when the option is
     defined.
  5. string options have string values. Their <value> consist of all
     remaining characters in the command-argument. If <value> is omitted,
     an empty string is assigned to the option.

  To use this unit, a program should declare a variable of type OptLstPtr
  that is referenced to as the option list variable in all procedures and
  functions in the unit.
  The program must contain initializing code executed once, containing:
  - A call to procedure OptInit.
  - A call to procedures OptDefSw, OptDefNm, OptDefRg, OptDefCh or
    OptDefSt to define each option, specifying its identifying character.
    For number options procedure OptDefNm is called once for each item in
    the number value list.
  - A call to function OptParse for each command-line argument that might
    contain an option.

  During program execution, the functions OptScanned, OptValSw, OptValNm,
  OptValRg, OptValCh, OptValSt and OptRgInside can be used as often as
  desired to find out if an option has been scanned by OptParse, to get
  the current value of an option, or to check if an integer is inside a
  range option value.

  At the end of the program or after all option processing is completed an
  optional call to procedure OptDone can be executed to free all heap
  space allocated for the option list.

  Remarks:
  - It is possible to maintain separate groups of options in one program,
    for instance to support command-argument syntax of the form
      COMMAND <option-group-1> <file-argument> [...] <option-group-2>
    In this case each option-group has its own option list variable.
  - As option identifiers have to be unique within an option list, any
    redefinition of an option identifier that has been defined before
    is ignored, except for number options where redefinition results in
    adding an entry to the list of numbers that go with that option.
}

UNIT OPTLIB;
{$V-}


INTERFACE
Uses ArgLib, ChrLib, StpLib, RngLib;

const
  OPTPREFIX   = '/';  { prefix character for option arguments }
  OPTNUMSEP   = ',';  { separator character for number option value lists }
  OPTMAXDIG   = 4;    { maximum number of digits in a number option item }

type
{ The following type definitions can be referenced externally: }
  OptSwTyp    = boolean;
  OptNmTyp    = integer;
  OptRgTyp    = RngTyp;
  OptChTyp    = char;
  OptStTyp    = CmdStp;
  OptStInd    = CmdInd;
  OptLstPtr   = ^OptLstRec;

{ The remaining type definitions are for internal use only }
  OptTyp      = (OptSwitch, OptNumber, OptRange, OptChar, Optstring);
  OptNmRec    = record
                  NmReq: boolean;
                  NmDfl, NmMin, NmMax, NmVal: OptNmTyp
                end;
  OptNmLstPtr = ^OptNmLstRec;
  OptNmLstRec = record inh: OptNmRec; nxt: OptNmLstPtr end;
  OptRgBndRec = record
                  RgReq: boolean;
                  RgDfl, RgMin, RgMax : OptNmTyp;
                end;
  OptRgPtr    = ^OptRgRec;
  OptRgRec    = record
                  RgLft, RgRgt : OptRgBndRec;
                  RgVal        : OptRgTyp;
                end;
  OptStPtr    = ^OptStTyp;
  OptRec      = record
                  Letter     : char;
                  Scanned    : boolean;
                  case Soort : OptTyp of
                    OptSwitch: (SwVal: OptSwTyp);
                    OptNumber: (NmLst: OptNmLstPtr);
                    OptRange : (RgPtr: OptRgPtr);
                    OptChar  : (ChDfl, ChVal: OptChTyp);
                    Optstring: (StPtr: OptStPtr);
                end;
  OptLstRec   = record inh: OptRec; nxt: OptLstPtr end;

procedure OptInit(var OptLst: OptLstPtr);
{ Initialize option list OptLst. Execute this routine once at the start
  of the program. }

procedure OptDone(var OptLst: OptLstPtr);
{ Clear heap space allocated for option list OptLst. This routine can be
  executed once at the end of the program or as soon as all option
  processing is done (including references to any options in the list).
  Execution of this procedure is not neccessary. }

procedure OptDefSw(var OptLst: OptLstPtr; idn: char);
{ Add definition of switch type option idn into option list OptLst, giving
  it the initial value false. }

procedure OptDefNm(var OptLst: OptLstPtr; idn: char;
                   required: boolean; min, max, dfl: OptNmTyp);
{ Add definition of number type option idn into option list OptLst,
  specifying if it is required and also specifying the minimum, maximum
  and default values min, max and dfl. The value is initially set to the
  default value.
  When a number option with the same identifier has been defined before
  using this procedure, a number item is added to the list of items for
  this number option. }

procedure OptDefRg(var OptLst: OptLstPtr; idn: char;
                   Lrequired: boolean; Lmin, Lmax, Ldfl: OptNmTyp;
                   Rrequired: boolean; Rmin, Rmax, Rdfl: OptNmTyp);
{ Add definition of range type option idn into option list OptLst,
  specifying for both left and right boundary if it is required,
  and what values are its minimum, maximum and default. The initial
  value is the empty range. }

procedure OptDefCh(var OptLst: OptLstPtr; idn: char; dfl: OptChTyp);
{ Add definition of character type option idn into option list OptLst,
  specifying its default value, which also is its initial value. }

procedure OptDefSt(var OptLst: OptLstPtr; idn: char);
{ Add definition of string type option idn into option list OptLst, giving
  it the empty string as an initial value. }

function  OptParse(var OptLst:OptLstPtr; arg:CmdStp; var ok:boolean)
          :boolean;
{ Parse command-line argument arg for valid options, updating option list
  OptLst.
  The routine OptParse is typically executed once for each command-line
  argument in the program initialization part.
  The function result reflects if the command-line argument contains one
  or more options.
  - If the function result is true, variable ok reflects successfull
    scanning of the argument, so a value ok=false signals a command-syntax
    error.
  - If the function result is false, the argument does not represent one
    or more options. It could be a file-name or some other argument and
    should be interpreted separately. In this case the value of variable
    ok is not defined. If no other argument type is expected this
    condition signals a command-syntax error as well.

  Each option recognized by this routine will be marked as being scanned,
  which can be tested by function OptScanned.
  When the same range option is parsed more than once, the range values
  are combined (merged) into one possibly broken range of integer values.
  When other options are parsed more then once, the last value
  interpreted overwrites that of any previous instances. This property can
  be used to force certain dependencies between options by calling
  OptParse with a program-supplied "arg" variable after scanning the real
  command arguments. }

function OptScanned(OptLst: OptLstPtr; idn: char): boolean;
{ Test if option idn was scanned by procedure OptParse. This function can
  be used to see if the option was specified at all.
  If the option was not defined, the value false will be returned. }

function  OptValSw(OptLst: OptLstPtr; idn: char): OptSwTyp;
{ Returns the current value of switch type option idn in option list
  OptLst. If no option idn was defined, or if it is not defined as a
  switch option, the value false will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefSw and
  the command-line arguments have been scanned for options using function
  OptParse. }

function  OptValNm(OptLst: OptLstPtr; idn: char; i: integer): OptNmTyp;
{ Returns the current value of item i in the list of values for the
  number type option idn in option list OptLst. If no option idn was
  defined, or if it is not defined as a number option, or i is less than
  one or greater than the number of items defined for this number option,
  the value 0 will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefNm and
  the command-line arguments have been scanned for options using function
  OptParse. }

function  OptValRg(OptLst: OptLstPtr; idn: char): OptRgTyp;
{ Returns the current value of range option idn in option list OptLst.
  If no option idn was defined, or if it is not defined as a range
  option, the nil pointer value will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefRg and
  the command-line arguments have been scanned for options using function
  OptParse. }

function  OptRgInside(OptLst: OptLstPtr; idn: char; n: OptNmTyp): boolean;
{ Specifies if the value n is within the current range for range option
  idn in option list OptLst.
  If no option idn was defined, or if it is not defined as a range
  option, the value false will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefRg and
  the command-line arguments have been scanned for options using function
  OptParse. }

function  OptValCh(OptLst: OptLstPtr; idn: char): OptChTyp;
{ Returns the current value of character option idn in option list OptLst.
  If no option idn was defined, or if it is not defined as a character
  option, the character with ord-value 0 will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefCh and
  the command-line arguments have been scanned for options using function
  OptParse. }

function  OptValSt(OptLst: OptLstPtr; idn: char): OptStTyp;
{ Returns the current value of string option idn in option list OptLst. If
  no option idn was defined, or if it is not defined as a string option,
  an empty string will be returned.
  This function can be called as often as required during the execution of
  a program after the option has been defined using procedure OptDefSt and
  the command-line arguments have been scanned for options using function
  OptParse. }


IMPLEMENTATION


{
-- local definitions
}


{ === Number List routines === }
{ Single linked List }

procedure NmLstCreate(var NmLst: OptNmLstPtr);
begin NmLst:= nil end;

procedure NmLstDone(var NmLst: OptNmLstPtr);
begin
  if NmLst<>nil then begin
    NmLstDone(NmLst^.nxt);
    dispose(NmLst)
  end
end;

procedure NmLstInsert(var NmLst: OptNmLstPtr; var elm: OptNmRec);
{ Insert at tail }
var p: OptNmLstPtr;
begin
  if NmLst<>nil then NmLstInsert(NmLst^.nxt,elm)
  else begin
    new(p); with p^ do begin inh:= elm; nxt:= nil end;
    NmLst:= p;
  end;
end;

function NmLstFind(var NmLst: OptNmLstPtr; i: integer;
                                       var p: OptNmLstPtr): boolean;
var ok: boolean;
begin
  if (NmLst=nil) or (i<1) then ok:= false
  else begin
    p:= NmLst;
    while (i>1) and (p^.nxt<>nil) do begin p:= p^.nxt; dec(i) end;
    ok:= i=1;
  end;
  NmLstFind:= ok;
end;

function NmLstUpdate(p: OptNmLstPtr; n: OptNmTyp): boolean;
begin with p^.inh do begin
  if (n < NmMin) or (n > NmMax) then NmLstUpdate:= false
  else begin
    NmVal:= n;
    NmLstUpdate:= true;
  end;
end end;


{ === Option List routines === }
{ Single linked ordered circular list with dummy header item }

function OptLstOrder(e1, e2: OptRec): integer;
begin OptLstOrder:= UppOrder(e1.Letter,e2.Letter) end;

procedure OptLstCreate(var OptLst: OptLstPtr);
begin new(OptLst); OptLst^.nxt:= OptLst end;

procedure OptLstClear(var OptLst: OptLstPtr);
begin dispose(OptLst) end;

function OptLstEmpty(var OptLst: OptLstPtr): boolean;
begin OptLstEmpty:= OptLst^.nxt=OptLst end;

procedure OptLstInsert(var elm: OptRec; var OptLst: OptLstPtr);
var pt0,pt1: OptLstPtr;
begin
  OptLst^.inh:= elm; {place sentinel}
  pt0:= OptLst;
  while OptLstOrder(pt0^.nxt^.inh,elm) < 0 do pt0:= pt0^.nxt;
  new(pt1); with pt1^ do begin inh:= elm; nxt:= pt0^.nxt end;
  pt0^.nxt:= pt1
end;

procedure OptLstRetrieve(var elm: OptRec; var OptLst: OptLstPtr);
{ Retrieve smallest elm and discard from OptLst. If OptLst is empty the
  value of elm is not defined. }
var ptr: OptLstPtr;
begin
  ptr:= OptLst^.nxt;
  elm:= ptr^.inh;
  OptLst^.nxt:= ptr^.nxt;
  if ptr^.nxt <> ptr then dispose(ptr)
end;

function OptLstFind(idn: char; OptLst: OptLstPtr;
                                var p: OptLstPtr): boolean;
{ Find pointer p to option list record with option idn. The returnvalue
  indicates success of find action. The value of p is not defined if idn
  could not be found in OptLst. }
begin
  OptLst^.inh.Letter:= idn; {plaats sentinel}
  p:= OptLst; while OptLstOrder(p^.nxt^.inh,OptLst^.inh) < 0 do p:= p^.nxt;
  p:= p^.nxt;
  OptLstFind:= (p <> OptLst) and (OptLstOrder(p^.inh,OptLst^.inh) = 0);
end;


procedure ParseSwitch(var arg: CmdStp; var SwVal: OptSwTyp);
begin case StpcRet(arg,1) of
  '+' : begin StpDel(arg,1,1); SwVal:= true  end;
  '-' : begin StpDel(arg,1,1); SwVal:= false end;
  else  SwVal:= true;
end end;


function ParseInt(var arg : CmdStp;    { current command argument }
                  var n   : OptNmTyp;  { result value if ok }
                      req : boolean;   { value required or not }
                      dfl : OptNmTyp;  { default value }
                      min : OptNmTyp;  { minimum value accepted }
                      max : OptNmTyp   { maximum value accepted }
                 ): boolean;
{ Parse integer.
  Report success in function result.
  In case of error the value of n is left unchanged.
  If no value is given and req is false n will get the value dfl.
  If a value is found and all is correct it is placed into n.
}
var c: char; plus: boolean; digits: integer; value: OptNmTyp; ok: boolean;
begin
  ok:= false;
  c:= StpcRet(arg,1);
  if not ( IsDigit(c) or (c='-') or (c='+') ) then begin
    if not req then begin n:= dfl; ok:= true; end;
  end
  else begin
    case c of
      '+' : begin StpDel(arg,1,1); plus:= true  end;
      '-' : begin StpDel(arg,1,1); plus:= false end;
      else  plus:= true;
    end;
    value:= 0; c:= StpcRet(arg,1);
    if IsDigit(c) then begin
      digits:= 0;
      repeat
        value:= 10*value - (ord(c)-48);
        inc(digits);
        StpDel(arg,1,1);
        c:= StpcRet(arg,1);
      until not IsDigit(c) or (digits=OPTMAXDIG);
      if plus then value:= -value;
      ok:= not IsDigit(c) and (value >= min) and (value <= max)
    end;
    if ok then n:= value;
  end;
  ParseInt:= ok;
end;


function ParseNumber(var arg: CmdStp; var NmLst: OptNmLstPtr;
                                              i: integer): boolean;
var p: OptNmLstPtr;
begin
  if not NmLstFind(NmLst,i,p) then ParseNumber:= false
  else with p^.inh do begin
    ParseNumber:= ParseInt(arg,NmVal,NmReq,NmDfl,NmMin,NmMax);
  end;
end;


function ParseNmLst(var arg: CmdStp; var NmLst: OptNmLstPtr): boolean;
var ok: boolean; i: integer; var p: OptNmLstPtr;
begin
  i:= 1;
  ok:= ParseNumber(arg,NmLst,i);
  while ok and (StpcRet(arg,1) = OPTNUMSEP) do begin
    StpDel(arg,1,1);
    inc(i);
    ok:= ParseNumber(arg,NmLst,i);
  end;
  inc(i);
  while ok and NmLstFind(NmLst,i,p) do with p^.inh do begin
    if NmReq then ok:= false else NmVal:= NmDfl;
    inc(i);
  end;
  ParseNmLst:= ok;
end;


function ParseRgNum(var arg : CmdStp;
                    var bnd : OptRgBndRec;
                    var n   : OptNmTyp
                   ): boolean;
begin with bnd do begin
  ParseRgNum:= ParseInt(arg,n,RgReq,RgDfl,RgMin,RgMax)
end end;


function ParseRange(var arg: CmdStp; var RgPtr: OptRgPtr): boolean;
var ok: boolean; low,high: OptNmTyp; dummy: CmdStp;
begin with RgPtr^ do begin
  ok:=ParseRgNum(arg,RgLft,low);
  if ok then begin
    if (StpcRet(arg,1) = OPTNUMSEP) then begin
      StpDel(arg,1,1);
      ok:= ParseRgNum(arg,RgRgt,high);
    end
    else begin dummy:= ''; ok:= ParseRgNum(dummy,RgRgt,high) end;
  end;
  if ok then RngInsert(RgVal,low,high);
  ParseRange:= ok;
end end;


function ParseChar(var arg: CmdStp; var ChVal: OptChTyp): boolean;
begin
  if StpEmpty(arg) then ParseChar:= false
  else begin ChVal:= StpcGet(arg); ParseChar:= true end;
end;


procedure Parsestring(var arg: CmdStp; p: OptStPtr);
begin p^:= arg; arg:= '' end;


{
-- global routines ---
}

procedure OptInit(var OptLst: OptLstPtr);
begin
  OptLstCreate(OptLst);
end;


procedure OptDone(var OptLst: OptLstPtr);
var r: OptRec;
begin
  while not OptLstEmpty(OptLst) do begin
    OptLstRetrieve(r,OptLst);
    with r do case Soort of
      OptSwitch : { nothing };
      OptNumber : NmLstDone(NmLst);
      OptRange  : begin RngDispose(RgPtr^.RgVal); dispose(RgPtr) end;
      OptChar   : { nothing };
      Optstring : dispose(StPtr);
    end;
  end;
  OptLstClear(OptLst);
end;


procedure OptDefSw(var OptLst: OptLstPtr; idn: char);
var r: OptRec; p: OptLstPtr;
begin if not OptLstFind(idn,OptLst,p) then with r do begin
  Letter     := UpCase(idn);
  Scanned    := false;
  Soort      := OptSwitch;
  SwVal      := false;
  OptLstInsert(r,OptLst);
end end;


procedure OptDefNm(var OptLst: OptLstPtr; idn: char;
                   required: boolean; min, max, dfl: OptNmTyp);
var r: OptRec; p: OptLstPtr; NmRec: OptNmRec;
begin
  with NmRec do begin
    NmReq:= required; NmDfl:= dfl; NmMin:= min; NmMax:= max; NmVal:= dfl;
  end;
  if OptLstFind(idn,OptLst,p) then NmLstInsert(p^.inh.NmLst,NmRec)
  else with r do begin
    Letter := UpCase(idn);
    Scanned:= false;
    Soort  := OptNumber;
    NmLstCreate(NmLst);
    NmLstInsert(NmLst,NmRec);
    OptLstInsert(r,OptLst);
  end;
end;


procedure OptDefRg(var OptLst: OptLstPtr; idn: char;
                   Lrequired: boolean; Lmin, Lmax, Ldfl: OptNmTyp;
                   Rrequired: boolean; Rmin, Rmax, Rdfl: OptNmTyp);
var r: OptRec; p: OptLstPtr;
begin if not OptLstFind(idn,OptLst,p) then with r do begin
  Letter     := UpCase(idn);
  Scanned    := false;
  Soort      := OptRange;
  new(RgPtr);
  with RgPtr^ do begin
    with RgLft do begin
      RgReq:= Lrequired; RgDfl:= Ldfl; RgMin:= Lmin; RgMax:= Lmax;
    end;
    with RgRgt do begin
      RgReq:= Rrequired; RgDfl:= Rdfl; RgMin:= Rmin; RgMax:= Rmax;
    end;
    RngCreate(RgVal);
  end;
  OptLstInsert(r,OptLst);
end end;


procedure OptDefCh(var OptLst: OptLstPtr; idn: char; dfl: OptChTyp);
var r: OptRec; p: OptLstPtr;
begin if not OptLstFind(idn,OptLst,p) then with r do begin
  Letter := UpCase(idn);
  Scanned:= false;
  Soort  := OptChar;
  ChDfl  := dfl;
  ChVal  := dfl;
  OptLstInsert(r,OptLst);
end end;


procedure OptDefSt(var OptLst: OptLstPtr; idn: char);
var r: OptRec; p: OptLstPtr;
begin if not OptLstFind(idn,OptLst,p) then with r do begin
  Letter := UpCase(idn);
  Scanned:= false;
  Soort  := Optstring;
  new(StPtr); StPtr^:= '';
  OptLstInsert(r,OptLst);
end end;


function OptParse(var OptLst:OptLstPtr; arg:CmdStp; var ok:boolean)
         :boolean;
var p: OptLstPtr;
begin
  ok:= true;
  if StpcRet(arg,1) <> OPTPREFIX then OptParse:= false
  else begin
    StpDel(arg,1,1);
    repeat
      if not OptLstFind(StpcGet(arg),OptLst,p) then ok:= false
      else with p^.inh do begin
        Scanned:= true;
        case Soort of
          OptSwitch : ParseSwitch(arg,SwVal);
          OptNumber : ok:= ParseNmLst(arg,NmLst);
          OptRange  : ok:= ParseRange(arg,RgPtr);
          OptChar   : if not ParseChar(arg,ChVal) then
                        ChVal:= ChDfl;
          Optstring : Parsestring(arg,StPtr);
        end;
      end;
      if ok then if StpcRet(arg,1) = OPTPREFIX then begin
        StpDel(arg,1,1);
        ok:= not StpEmpty(arg);
      end;
    until StpEmpty(arg) or not ok;
    if ok then ok:= StpEmpty(arg);
    OptParse:= true;
  end;
end;


function OptScanned(OptLst: OptLstPtr; idn: char): boolean;
var p: OptLstPtr;
begin
  if OptLstFind(idn,OptLst,p) then OptScanned:= p^.inh.Scanned
  else OptScanned:= false;
end;


function  OptValSw(OptLst: OptLstPtr; idn: char): OptSwTyp;
var p: OptLstPtr; ok: boolean;
begin
  ok:= false;
  if OptLstFind(idn,OptLst,p) then
    if p^.inh.Soort = OptSwitch then
      ok:= true;
  if ok then OptValSw:= p^.inh.SwVal else OptValSw:= false;
end;


function  OptValNm(OptLst: OptLstPtr; idn: char; i: integer): OptNmTyp;
var pl: OptLstPtr; pn: OptNmLstPtr; ok: boolean;
begin
  ok:= false;
  if OptLstFind(idn,OptLst,pl) then
    if pl^.inh.Soort = OptNumber then
      if NmLstFind(pl^.inh.NmLst,i,pn) then
        ok:= true;
  if ok then OptValNm:= pn^.inh.NmVal else OptValNm:= 0;
end;


function  OptValRg(OptLst: OptLstPtr; idn: char): OptRgTyp;
var p: OptLstPtr; ok: boolean;
begin
  ok:= false;
  if OptLstFind(idn,OptLst,p) then
    if p^.inh.Soort = OptRange then
        ok:= true;
  if ok then OptValRg:= p^.inh.RgPtr^.RgVal else OptValRg:= nil;
end;


function  OptRgInside(OptLst: OptLstPtr; idn: char; n: OptNmTyp): boolean;
begin OptRgInside:= RngInside(OptValRg(OptLst,idn),n) end;


function  OptValCh(OptLst: OptLstPtr; idn: char): OptChTyp;
var p: OptLstPtr; ok: boolean;
begin
  ok:= false;
  if OptLstFind(idn,OptLst,p) then
    if p^.inh.Soort = OptChar then
      ok:= true;
  if ok then OptValCh:= p^.inh.ChVal else OptValCh:= chr(0);
end;


function  OptValSt(OptLst: OptLstPtr; idn: char): OptStTyp;
var p: OptLstPtr; ok: boolean;
begin
  ok:= false;
  if OptLstFind(idn,OptLst,p) then
    if p^.inh.Soort = Optstring then
      ok:= true;
  if ok then OptValSt:= p^.inh.StPtr^ else OptValSt:= '';
end;


END.
