/*
  movies.cpp - Display Movie Select's MOVCARD.DAT file.

  Jason Hood, 30 October to 21 November, 2000.
*/


#define PVERSION "1.00"
#define PDATE	 "21 November, 2000"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pc.h>

#include "jptui.h"

TApplication MovieSelect( ENGLISH );


#define MOVCARD  "db/movcard.dat"       // The format of these files
#define TITLEIDX "db/movietit.nx1"      //  is described at the end
#define TITLEREF "db/movietit.in1"      //  of this file
#define SUBCAT	 "db/subcat.lst"


#define RATING	0x25			// Offsets for MOVCARD
#define CATEG	0x28
#define TITLE	0x32

#define MTIDX	0x18			// Offsets for TITLEIDX
#define MTOFS	0x50


char   dbpath[260];	// Path for Movie Select (NOT the database)
FILE*  movcard;
long*  m_index;		// Ordered index of each movie in the db
int    indices; 	// Number of above
long*  movie;		// Index of movie numbers
int    movies;		// Number of above

char** subcat;		// Array of subcategories (from SUBCAT.LST)
int    subcats; 	// Number of above
#define SUBCATS 7	// Number of subcategories for each movie

#define PREFS 14
boolean pref[PREFS];	// Category preferences

#define TITLES 18	// Number of titles displayed in the list
int  number[TITLES];	// Index of each title
int  list_film = -1;	// Index of title chosen from list

int* film;		// Array of titles from the cmdline or find dialog
int  films;		// Number of above
int  filmp;		// Position within the above array

int* match;		// Array of matching titles from the find dialog
int  matches;		// Number of above

struct
{
  int	id;
  char* name;
}
categ[] =
{
/*{ 920, "Laser VideoDiscs"        },   // These are >= tests
  { 910, "Close-Captioned Films"   },*/
  { 900, "Adult Audience"          },   // There is one 908
  { 700, "Education/Gen. Interest" },
  { 500, "Sports/Recreation"       },
  { 300, "Foreign Film"            },
  { 170, "Western"                 },
  { 160, "Sci-Fi/Fantasy"          },
  { 150, "Religious"               },
  { 140, "Musical & Perf. Arts"    },
  { 135, "Horror Suspense"         },
  { 130, "Drama"                   },
  { 120, "Comedy"                  },
  { 110, "Childrens"               },
  { 100, "Adventures"              },
  {   5, "New Release"             },   // These match exactly
  {   0, "Very New Release"        },   // Only three
/*{  10, "Titles"                  },
  {  20, "Stars"                   },
  {  30, "Directors"               },*/
};

#define CATEGS	13			// Number of >= categories
#define CATEGS2  2			// Number of exact-matching categories
#define VNEWRLS 14			// Index of Very New Release


// ---------------------------------------------------------------------------
// Movie Window - Displays the details of a movie.

TWindow FilmWindow( DIALOG1, 1,1, 80,25, "Movie Info",
		    NO_INFO_BAR, NOT_MODAL, NOT_MOVABLE );

#define     DESCLEN 60
#define     MOVIELEN (DESCLEN+3)
TTextZone   MovieText(	&FilmWindow,  3, 3, MOVIELEN,7, "", 255 );
TPushButton PBPrev(	&FilmWindow, 70, 3,  6,    "~Prev" );
TPushButton PBNext(	&FilmWindow, 70, 5,  6,    "~Next", PB_DEFAULT );
TPushButton PBFind(	&FilmWindow, 70, 7,  6,    "~Find" );
TPushButton PBList(	&FilmWindow, 70, 9,  6,    "~List" );
TListBox    LBStars(	&FilmWindow,  3,11, 35, 7, "~Stars" );
TListBox    LBDir(	&FilmWindow,  3,19, 35, 4, "~Directed By" );
TFrame	    FInfo(	&FilmWindow, 44,11, 32,13 );

TLabel	    Year(			&FInfo, 1, 1, 30,1 );
TLabel	    Length(			&FInfo, 1, 2, 30,1 );
TLabel	    Categ(			&FInfo, 1, 3, 30,1 );
TLabel	    Type(			&FInfo, 1, 4, 30,1 );
TLabel	    Subcat[SUBCATS] = { TLabel( &FInfo, 1, 5, 30,1 ),
				TLabel( &FInfo, 1, 6, 30,1 ),
				TLabel( &FInfo, 1, 7, 30,1 ),
				TLabel( &FInfo, 1, 8, 30,1 ),
				TLabel( &FInfo, 1, 9, 30,1 ),
				TLabel( &FInfo, 1,10, 30,1 ),
				TLabel( &FInfo, 1,11, 30,1 ) };


// ---------------------------------------------------------------------------
// Find Window - Asks for keywords and displays matching titles.

TWindow FindWindow( DIALOG1, 1,1, 80,25, "Find Titles",
		    NO_INFO_BAR, NOT_MODAL, NOT_MOVABLE );

TEditZone   SearchText( &FindWindow,  4,3,  0,-1, "~Enter keywords", 68, 255 );
TCheckBox   CBMatch(	&FindWindow,  4,5, 14,	  "~Match order", CHECKED );
TPushButton PBSearch(	&FindWindow, 24,5,  9,	  "~Find", PB_DEFAULT );
TPushButton PBPrefs(	&FindWindow, 35,5, 13,	  "~Preferences" );
TPushButton PBCancel(	&FindWindow, 50,5,  8,	  "~Cancel", PB_CANCEL );
TListBox    LBFound(	&FindWindow,  4,7, 70,16, " " );


// ---------------------------------------------------------------------------
// List Window - Displays a list of movies, according to preferences.

TWindow ListWindow( DIALOG1, 1,1, 80,25, "Movie Listing",
		    NO_INFO_BAR, NOT_MODAL, NOT_MOVABLE );

#define     TITLELEN 65
TFrame	    FList(   &ListWindow,  2, 2, TITLELEN+2,20 );
PLabel	    Title[TITLES];
PPushButton Xref[28];		// Alphabet plus first and last
TPushButton PBLPref( &ListWindow,  2,22, 13, "Preferences" );
TPushButton PBLFind( &ListWindow, 17,22,  6, "Find" );
TPushButton PBPgUp2( &ListWindow, 52,22,  4, "<<", PB_NORMAL, SHC_NONE );
TPushButton PBPgUp(  &ListWindow, 57,22,  3, "~<", PB_NORMAL, SHC_NONE );
TPushButton PBPgDn(  &ListWindow, 61,22,  3, "~>", PB_NORMAL, SHC_NONE );
TPushButton PBPgDn2( &ListWindow, 65,22,  4, ">>", PB_NORMAL, SHC_NONE );


// ---------------------------------------------------------------------------
// Preferences Dialog - Select categories to display/find.

TWindow PrefDialog( DIALOG2, 9,2, 63,22, "Preferences",
		    NO_INFO_BAR, MODAL, MOVABLE );

PCheckBox   CBPref[PREFS];
TPushButton PBPOK(     &PrefDialog, 10,17,  8, "OK",     PB_DEFAULT );
TPushButton PBPCancel( &PrefDialog, 10,19,  8, "Cancel", PB_CANCEL );
TPushButton PBPNone(   &PrefDialog, 36,17, 11, "C~lear All" );
TPushButton PBPAll(    &PrefDialog, 36,19, 11, "Chec~k All" );


// ---------------------------------------------------------------------------
// Callbacks

void QuitCall( PObject, char* );	// Close windows and Preferences Cancel

void PrNxCall( PObject, char* );
void FindCall( PObject, char* );	// Both FilmWindow and ListWindow
void ListCall( PObject, char* );

void SearchCall( PObject, char* );
void CancelCall( PObject, char* );
void FoundCall( PObject, int, char* );
void FoundFocus( PObject, char* );

void TitleCall( PObject, char* );
void XrefCall( PObject, char* );
void PageCall( PObject, char* );

void PrefsCall( PObject, char* );	// Both FindWindow and ListWindow
void PBPOkCall( PObject, char* );
void AllCall( PObject, char* );


// ---------------------------------------------------------------------------

// Read a two-byte big-endian value from file.
int read2( FILE* file )
{
  return (fgetc( file ) << 8) |	fgetc( file );
}

// Read a four-byte big-endian value from file.
long read4( FILE* file )
{
  return (fgetc( file ) << 24) | (fgetc( file ) << 16) |
	 (fgetc( file ) <<  8) |  fgetc( file );
}

// Read a two-byte (word) little-endian value from file.
int readw( FILE* file )
{
  return fgetc( file ) | (fgetc( file ) << 8);
}

// Read a four-byte (long) little-endian value from file.
long readl( FILE* file )
{
  return  fgetc( file )        | (fgetc( file ) << 8) |
	 (fgetc( file ) << 16) | (fgetc( file ) << 24);
}


// Capitalise str - lowercase everything but the first letter.
// Complicated by abbreviations, acronyms and apostrophes.
// This routine is actually better than Movie Select's own.
// Why couldn't the db be in "proper" case?
char* lower( char* str )
{
  static char* nolo[] = { "ABC", "BBC",  "CC",  "FA", "FBI",  "GPS",
			   "II", "III",  "IV", "JFK", "KGB",  "NBA",
			  "NFL",  "PC", "PGA", "TNT",  "TV",  "UFO",
			  "USA", "VHF", "WCW",  "WW", "WWI", "WWII",
			   "ZZ", "~" };
  char* tmp = str;
  int	cnt, j;

  for (;;)
  {
    while (*tmp && !isupper( *tmp )) ++tmp;	// Get the first uppercase char
    if (!*tmp) break;

    if (tmp > str && isdigit( tmp[-1] ))	// Was the prev. char a digit?
    {
      if (tmp[1] && !isalnum( tmp[2] )) 	// Are there only two chars?
      {
	if ((tmp[0] == 'T' && tmp[1] == 'H') || // Lowercase positional
	    (tmp[0] == 'S' && tmp[1] == 'T') || //  suffixes (1st not 1St)
	    (tmp[0] == 'N' && tmp[1] == 'D') ||
	    (tmp[0] == 'R' && tmp[1] == 'D'))
	  --tmp;
      }
      else if (tmp[0] == 'S' && !isalnum( tmp[1] ))     // Lowercase plural
	--tmp;						//  (90s)
    }
    else if (*tmp == 'S' && tmp-1 > str && tmp[-1] == '\'' && isdigit(tmp[-2]))
      --tmp;						// Ownership (90's)
    else
    {
      for (cnt = 1; isupper( tmp[cnt] ) && cnt < 5; ++cnt) ;
      if (cnt == 2 && *tmp == 'V' && tmp[1] == 'S') --tmp; // versus abbr.
      else if (cnt >= 2 || cnt <= 4)
      {
	for (j = 0; *nolo[j] < *tmp; ++j) ;
	while (*nolo[j] == *tmp)
	{
	  if ((int)strlen( nolo[j] ) == cnt &&
	      !strncmp( nolo[j]+1, tmp+1, cnt-1 ))
	  {
	    tmp += cnt - 1;		// Don't lowercase abbreviations
	    break;
	  }
	  ++j;
	}
      }
    }
    ++tmp;				// Leave the first letter upper
    while (*tmp && (isupper( *tmp ) || *tmp == '\''))
    {
      // Ownership and contractions, complicated by names (O'Toole, I've)
      if (*tmp == '\'' && isupper( tmp[-1] ) && tmp[-1] != 'I' &&
	  isalnum( tmp[1] ) && isalnum( tmp[2] )) break;
      *tmp = tolower( *tmp );
      ++tmp;
    }
  }

  return str;
}


// Read a string from a file, where size indicates the number of bytes
// containing the length of the string (assumed big-endian). The string
// is dynamically allocated - deleting is the caller's responsibility.
// The string is case-converted.
char* reads( FILE* file, int size )
{
  int	len;
  char* buf;

  len = (size == 1) ? fgetc( file ) : read2( file );
  if (len)
  {
    buf = new char[len+1];
    fread( buf, len, 1, file );
    buf[len] = 0;
    lower( buf );
  }
  else buf = NULL;

  return buf;
}


// Make an array out of CR-separated strings contained in str. The array is
// dynamically allocated - it is the caller's responsibility to delete it.
char** makea( char* str, int& cnt )
{
  int	 len;
  char** buf;

  cnt = 0;
  for (len = 0; str[len]; ++len)
  {
    if (str[len] == '\r') ++cnt;
  }

  if (cnt == 0) buf = NULL;
  else
  {
    buf = new char*[cnt];
    for (len = 0; len < cnt; ++len)
    {
      buf[len] = str;
      while (*str != '\r') ++str;
      *str++ = 0;
    }
  }

  return buf;
}


// Word-wrap str at width characters. Simply replaces space with newline at
// appropriate places.
char* wrap( char* str, int width )
{
  int	len = strlen( str );
  int	j;
  char* tmp = str;

  while (len > width)
  {
    // Find the first space before wrap would occur
    for (j = width; j > 0 && tmp[j] != ' '; --j) ;
    if (j == 0) 		// A very long word
    {
      // Find the first space
      for (j = width; j < len && tmp[j] != ' '; ++j) ;
      if (j == len) break;	// No spaces at all, and can't insert newline
    }
    len -= j + 1;
    tmp += j;			// Point to the space
    *tmp++ = '\n';              // Replace space with newline
  }

  return str;
}


// Split (word-wrap) a string in half, if it is longer than max_len. Returns
// a pointer to the second half, or to str if halving was not required.
// Assumes the string can be split (no words longer than max_len).
char* halve( char* str, int max_len )
{
  int len = strlen( str );
  int j = len / 2;
  int k = j;

  if (len <= max_len) j = -1;		// Prime for return value
  else
  {
    if (str[j] != ' ')
    {
      if (j > max_len)
      {
	j = max_len;
	while (str[j] != ' ') --j;
      }
      else
      {
	while (str[--j] != ' ') ;       // Space before the word being split
	while (str[++k] != ' ') ;       // Space after the word being split
	if (k <= max_len)
	{
	  if (k < len - j)		// Use the one closest to half
	    j = k;			// (k - len/2 < len/2 - j)
	}
      }
    }
    str[j] = 0;				// Replace space with NUL
  }

  return str + j + 1;
}


// Given a category number, return the index into the categ[] array.
int categ_index( int cat )
{
  int j;

  for (j = 0; j < CATEGS; ++j)	  if (cat >= categ[j].id) return j;
  for (; j < CATEGS+CATEGS2; ++j) if (cat == categ[j].id) return j;

  return VNEWRLS;	// Very New Release - should never be reached
}


// Determine if the movie matches the category preferences. Note that
// idx is the file position of the movie, not a movie index or number.
// If the movie matches, the file pointer is left at the title of the movie.
boolean test_categ( long idx )
{
  // Translate the categ[] index into a pref[] index.
  static int cat_xlat[] = { 12, 11, 10, 6, 5, 4, 9, 7, 3, 2, 1, 8, 0, 13, 13 };
  boolean rc;

  fseek( movcard, idx + CATEG, SEEK_SET );
  rc = pref[cat_xlat[categ_index( read2( movcard ) )]];
  if (rc) fseek( movcard, TITLE - (CATEG + 2), SEEK_CUR );

  return rc;
}


// Search for the next (previous) film after (before) num. dir should be
// plus or minus one. The search is wrapped.
int next_film( int num, int dir )
{
  do
  {
    num += dir;
    if (num >= indices) num = 0;
    else if (num < 0) num = indices - 1;
  }
  while (!test_categ( m_index[num] ));

  return num;
}


// Display all titles starting from num. dir should be plus or minus one.
void display_titles( int num, int dir )
{
  char* title;
  int	pos;
  int	cnt;

  pos = (dir > 0) ? 0 : TITLES - 1;
  num -= dir;	// next_film starts after the supplied number
  for (cnt = 0; cnt < TITLES; ++cnt)
  {
    number[pos] = num = next_film( num, dir );
    title = reads( movcard, 1 );
    Title[pos]->m_set_text( title );
    delete title;
    pos += dir;
  }
}


// Display all the details of a movie. If idx is negative, it is taken to be
// a movie number, rather than an index.
void display_film( int idx )
{
  char*  title;
  char*  desc;
  char*  cast_s;
  char*  dir_s;
  char*  misc[4];
  char** cast;
  char** dir;
  char	 buf[256];
  int	 cast_c, dir_c, subc_c;
  int	 rate, length, cat, subc[SUBCATS];
  int	 len;
  int	 j;

  #define RATINGS 7
  static char* rating[] = { "", "G", "PG", "PG-13", "R", "NC-17", "X", "??" };

  idx = (idx < 0) ? movie[-idx] : m_index[idx];
  fseek( movcard, idx + RATING, SEEK_SET );

  rate	 = fgetc( movcard );
  length = read2( movcard );
  cat	 = read2( movcard );
  subc_c = fgetc( movcard );
  for (j = 0; j < SUBCATS; ++j)
    subc[j] = fgetc( movcard ) - 1;	// Subcat number is 1-based.
  title  = reads( movcard, 1 );
  dir_s  = reads( movcard, 1 );
  cast_s = reads( movcard, 2 );
  desc	 = reads( movcard, 2 );
  len = read2( movcard );		// Skip Director xref
  fseek( movcard, len, SEEK_CUR );
  len = read2( movcard );		// Skip Actor xref
  fseek( movcard, len, SEEK_CUR );
  misc[0] = reads( movcard, 1 );	// Year~TV~B&W/Color~Type

  char* split = halve( title, MOVIELEN );
  if (split == title )
  {
    FilmWindow.m_set_normal_attr( DIALOG1 );
    FilmWindow.m_gotoxy( 3, 1 );
    FilmWindow.m_putnch( MOVIELEN, ' ' );
    FilmWindow.m_textattr( (BLUE<<4) | WHITE );
  }
  else
  {
    FilmWindow.m_textattr( (BLUE<<4) | WHITE );
    FilmWindow.m_gotoxy( 3, 1 );
    FilmWindow.m_put_caption( title, FALSE, MOVIELEN, CENTERED );
  }
  FilmWindow.m_gotoxy( 3, 2 );
  FilmWindow.m_put_caption( split, FALSE, MOVIELEN, CENTERED );

  if (desc) wrap( desc, DESCLEN );
  MovieText.m_enable_modification();
  MovieText.m_clear_text();
  MovieText.m_insert_text( desc );
  MovieText.m_disable_modification();

  for (j = 1; j < 4; ++j)
  {
    misc[j] = strchr( misc[j-1], '~' );
    *misc[j]++ = 0;
  }
  sprintf( buf, "%s%s%s%s%s", misc[0], *misc[0] ? " " : "",     // Year
			      misc[1], *misc[1] ? " " : "",     // B&W/Color
			      misc[2] ); 			// TV
  Year.m_set_text( buf );

  if (length) sprintf( buf, "%d min. ", length );
  else *buf = 0;
  if (rate > RATINGS) rate = RATINGS;
  strcat( buf, rating[rate] );
  Length.m_set_text( buf );

  Categ.m_set_text( categ[categ_index( cat )].name );
  Type.m_set_text( misc[3] );
  for (j = 0; j < subc_c; ++j)
    Subcat[j].m_set_text( subcat[subc[j]] );
  for (; j < SUBCATS; ++j)
    Subcat[j].m_set_text( "" );

  delete title;
  delete desc;
  delete misc[0];

  LBStars.m_clear_list();
  if (cast_s)
  {
    cast = makea( cast_s, cast_c );
    for (j = 0; j < cast_c; ++j)
      LBStars.m_add_item( cast[j] );
    delete cast;
    delete cast_s;
  }

  LBDir.m_clear_list();
  if (dir_s)
  {
    dir = makea( dir_s, dir_c );
    if (dir)				// Some are stuffed
    {
      for (j = 0; j < dir_c; ++j)
	LBDir.m_add_item( dir[j] );
      delete dir;
    }
    delete dir_s;
  }
}


// Display the movie selected from the list.
void TitleCall( PObject /*sender*/, char* idx )
{
  ListWindow.m_close();
  FilmWindow.m_open();
  list_film = number[*idx - 1];
  display_film( list_film );
}


// Display all titles from a given letter, from the start of the db,
// or from its end.
void XrefCall( PObject /*sender*/, char* let )
{
  int num;
  int dir = 1;
  // The number of the first movie for each letter of the alphabet.
  static int xref[] = {	  164,	1470,  3425,  5219,  6531, 26343, 38929,
			 9286, 10508, 11081, 11646, 27174, 13240, 45175,
			15715, 40702, 29425, 17424, 38283, 28723, 22782,
			22991, 23520, 24797, 24806, 25020 };

  if (*let == '#') num = 0;
  else if (*let == '~') num = indices - 1, dir = -1;
  else
  {
    num = xref[*let - 'A'];
    fseek( movcard, movie[num] + 4, SEEK_SET );
    num = read4( movcard );		// Use the index, not number
  }

  display_titles( num, dir );
}


// Display the next (previous) page of titles, possibly after skipping a page.
void PageCall( PObject /*sender*/, char* dir )
{
  int num;
  int ofs = 1;

  if (*dir == '-')                      // Page Up
  {
    num = number[0] - 1;
    if (num < 0) num = indices - 1;
    ofs = -1;
  }
  else if (*dir == '+')                 // Page Down
  {
    num = number[TITLES-1] + 1;
    if (num >= indices) num = 0;
  }
  else					// Two pages up or down
  {
    num = (*dir == '/') ? number[0] : number[TITLES-1];
    if (*dir == '/') ofs = -1;
    for (int j = 0; j <= TITLES; ++j)
      num = next_film( num, ofs );
  }

  display_titles( num, ofs );
}


// Search for titles containing wrds words in wrd[]. Words that do not match
// are ignored. If CBMatch is checked, words must match in order; if only one
// word, it must be first.
void search( char* wrd[], int wrds )
{
  char	path[260];
  FILE* idx;
  FILE* ref;
  int	pos[28];			// First, alphabet, last
  int*	title;				// Matches for this word
  int	titles = 0;
  char* buf;
  char* num;
  int	i, j, k = 0, cnt;
  long	p;
  int	c, t = 0;

  strcat( strcpy( path, dbpath ), TITLEIDX );
  idx = fopen( path, "rb" );
  strcat( strcpy( path, dbpath ), TITLEREF );
  ref = fopen( path, "rb" );

  fseek( idx, MTIDX, SEEK_SET );
  for (j = 0; j < 28; ++j)
    pos[j] = readw( idx );

  matches = 0;
  if (match) delete match, match = NULL;
  LBFound.m_clear_list();

  SetMousePointer( MP_HOURGLASS );

  for (j = 0; j < wrds; ++j)
  {
    strupr( wrd[j] );
    i = (isupper( *wrd[j] )) ? *wrd[j] - 'A' + 1 : 0;
    fseek( idx, pos[i] * 4 + MTOFS, SEEK_SET );
    p = readl( idx );
    fseek( idx, p, SEEK_SET );
    cnt = pos[i+1] - pos[i] + (i == 26);	// The last index, not one
    for (; cnt; --cnt)				//  more than last
    {
      titles = readw( idx );		// Number of matching titles
      k = readw( idx ); 		// Length of string containing titles
      p = readl( idx ); 		// Position of string
      c = 0;
      while (wrd[j][c] && (t = fgetc( idx )) == wrd[j][c]) ++c;
      if (wrd[j][c] == 0 && fgetc( idx ) == 0)
	break;					// A successful match
      titles = 0;
      if (t > wrd[j][c]) break; 		// Gone too far
      if (t) while (fgetc( idx )) ;		// Skip to the NUL
      if (ftell( idx ) & 1) fgetc( idx );	// Skip second NUL
    }
    if (titles)
    {
      fseek( ref, p, SEEK_SET );
      buf = new char[k+1];
      fread( buf, k, 1, ref );
      buf[k] = 0;
      if (matches == 0) 		// Read an initial list of movies
      {
	match = new int[titles];
	num = strtok( buf, "," );
	while (titles)
	{
	  match[matches++] = atoi( num );
	  num = strtok( NULL, "," );
	  --titles;
	}
      }
      else				// Narrow the list down
      {
	title = new int[matches];
	t = 0;
	num = strtok( buf, "," );
	while (titles)
	{
	  k = atoi( num );
	  for (i = 0; i < matches; ++i)
	  {
	    if (match[i] == k)
	    {
	      title[t++] = k;
	      break;
	    }
	  }
	  num = strtok( NULL, "," );
	  --titles;
	}
	for (i = 0; i < t; ++i) match[i] = title[i];
	matches = t;
	delete title;
	if (t == 0) j = wrds;		// No matches to words so far, so stop
      }
      delete buf;
    }
  }

  // Test the category and word order of the titles found.
  titles = 0;
  for (j = 0; j < matches; ++j)
  {
    if (!test_categ( movie[match[j]] )) continue;

    if (CBMatch.m_is_checked())
    {
      buf = strupr( reads( movcard, 1 ) );
      if (wrds == 1)
      {
	// Skip the non-alphanumeric characters.
	for (num = buf; !isalnum( *num ); ++num) ;
	// Test all characters in the word.
	k = !(strncmp( num, wrd[0], strlen( wrd[0] ) ));
	// Check if the word in the title is finished.
	if (k) k = (isupper( num[strlen( wrd[0] )] ) == 0);
      }
      else
      {
	num = strstr( buf, wrd[0] ) + strlen( wrd[0] );
	for (k = 1; k < wrds; ++k)
	{
	  num = strstr( num, wrd[k] );
	  if (num == NULL) break;
	  num += strlen( wrd[k] );
	}
      }
      delete buf;
      if (k != wrds) continue;
    }
    match[titles++] = match[j];
  }
  matches = titles;

  if (matches)				// Place the titles in the list
  {
    for (j = 0; j < matches; ++j)
    {
      fseek( movcard, movie[match[j]] + TITLE, SEEK_SET );
      buf = reads( movcard, 1 );
      LBFound.m_add_item( buf );
      delete buf;
    }
    char tmp[50];
    sprintf( tmp, "%d title%s found", matches, "s" + (matches == 1) );
    LBFound.m_set_caption( tmp );
    LBFound.m_set_focus();
  }
  else LBFound.m_set_caption( "No titles found" );

  SetMousePointer( MP_ARROW );

  fclose( ref );
  fclose( idx );
}


// Translate the EditZone into a word array and call search().
// If it was selected from the listbox, display the matching title, instead.
void SearchCall( PObject sender, char* /*arg*/ )
{
  char* word[50];
  int	words;
  char	buf[256];
  char* tmp;

  if (sender->m_get_window()->m_get_previous_focused_object() == &LBFound)
  {
    FoundCall( &LBFound, LBFound.m_get_selected_item_index(), "" );
    return;
  }

  strcpy( buf, SearchText.m_get_string() );
  words = 0;
  tmp = strtok( buf, " " );
  while (tmp)
  {
    word[words++] = tmp;
    tmp = strtok( NULL, " " );
  }
  if (words == 0) return;

  search( word, words );
}


// Display the next (previous) film in either the list, or found titles.
void PrNxCall( PObject /*sender*/, char* dir )
{
  if (list_film >= 0)
  {
    list_film = next_film( list_film, (*dir == '-') ? -1 : 1 );
    display_film( list_film );
  }
  else
  {
    if (*dir == '-')
    {
      if (--filmp < 0) filmp = films - 1;
    }
    else // (*dir == '+')
    {
      if (++filmp == films) filmp = 0;
    }
    display_film( film[filmp] );
  }
}


void ListCall( PObject /*sender*/, char* /*arg*/ )
{
  FilmWindow.m_close();
  ListWindow.m_open();
}


char prevwin = 'l';             // The window to display after cancelling

void FindCall( PObject /*sender*/, char* win )
{
  prevwin = *win;
  if (*win == 'm') FilmWindow.m_close();
  else		   ListWindow.m_close();
  FindWindow.m_open();
  SearchText.m_set_focus();
}


void CancelCall( PObject /*sender*/, char* /*arg*/ )
{
  FindWindow.m_close();
  if (prevwin == 'm') FilmWindow.m_open();
  else		      ListWindow.m_open();
}


// Create the array of matching movies and display the selected movie.
void FoundCall( PObject /*sender*/, int idx, char* /*arg*/ )
{
  FindWindow.m_close();
  FilmWindow.m_open();
  list_film = -1;		// Indicate the use of the film[] array
  if (film) delete film;
  films = matches;
  film	= new int[films];
  for (int j = 0; j < films; ++j) film[j] = -match[j];
  filmp = idx - 1;
  display_film( film[filmp] );
}


// Change the "Find" button to "Display" when the listbox has focus,
// or the button takes focus from the listbox.
void FoundFocus( PObject sender, char* foc )
{
  if (*foc == '-' || *foc == '/')
    PBSearch.m_set_caption( "~Find" );
  else if (*foc == '+' || /* *foc == '*' && */
	   sender->m_get_window()->m_get_previous_focused_object() == &LBFound)
    PBSearch.m_set_caption( "Display" );
}


// Check or clear all categories.
void AllCall( PObject /*sender*/, char* which )
{
  if (*which == '+')
    for (int j = 0; j < PREFS; ++j) CBPref[j]->m_check();
  else
    for (int j = 0; j < PREFS; ++j) CBPref[j]->m_uncheck();
}


// Update the list and found titles, based on the new preferences.
void PBPOkCall( PObject /*sender*/, char* /*arg*/ )
{
  int j, k = 0;

  for (j = 0; j < PREFS; ++j) k += CBPref[j]->m_is_checked();
  if (k == 0)
    MessageBox( "Oops", "You must select at least one category", ALERT );
  else
  {
    for (j = 0; j < PREFS; ++j) pref[j] = CBPref[j]->m_is_checked();
    SearchCall( &PBPOK, "" );
    display_titles( number[0], 1 );
    JPStop();
  }
}


// Create and process the Preferences dialog.
void PrefsCall( PObject /*sender*/, char* /*arg*/ )
{
  int j;
  static char* pref_str[PREFS] =
  {
    "Ad~venture",
    "C~omedies",
    "~Dramas",
    "~Horror/Suspense",
    "~SciFi/Fantasy",
    "~Westerns",
    "~Foreign Films",
    "~Musical & Performing Arts",
    "~Childrens",
    "~Religious",
    "S~ports/Recreation",
    "~Education/General Interest",
    "~Adult Audience",
    "~New Release"
  };

  for (j = 0; j < PREFS; ++j)
  {
    CBPref[j] = new TCheckBox( &PrefDialog, 4 + (j / (PREFS/2)) * 26,
					    2 + (j % (PREFS/2)) * 2,
					    DisplayLength( pref_str[j] ) + 3,
					    pref_str[j], pref[j] );
  }

  PBPOK.m_set_pressed_callback( PBPOkCall, "" );
  PBPCancel.m_set_pressed_callback( QuitCall, "" );
  PBPNone.m_set_pressed_callback( AllCall, "-" );
  PBPAll.m_set_pressed_callback( AllCall, "+" );

  PrefDialog.m_open();
  JPRunDialog();
  PrefDialog.m_close();

  for (j = 0; j < PREFS; ++j) delete CBPref[j];
}


void QuitCall( PObject /*sender*/, char* /*arg*/ )
{
  JPStop();
}


// Read the list of subcategories from SUBCAT.LST - a CR-separated list
// of strings.
void read_subcats( void )
{
  char	path[260];
  FILE* file;
  char*	buf;
  int	len;

  strcat( strcpy( path, dbpath ), SUBCAT );
  file = fopen( path, "rb" );
  fseek( file, 0, SEEK_END );
  len = ftell( file );
  rewind( file );
  buf = new char[len+1];
  fread( buf, len, 1, file );
  buf[len] = 0;
  fclose( file );
  subcat = makea( buf, subcats );
}


int main( int argc, char* argv[] )
{
  long	idx, mov;
  int	len;
  char	path[260];
  char* db;

  if (argc > 1 && (argv[1][1] == '?' || !strcmp( argv[1], "--help" )))
  {
    printf( "Movies - a viewer for Movie Select's database.\n"
	    "Jason Hood. Version "PVERSION" ("PDATE"). Freeware.\n"
	    "jadoxa@hotmail.com. http://adoxa.homepage.com/movies/\n"
	    "\n"
	    "Usage: %s [+index...] [-number...]\n"
	    "       %s [~] keyword...\n"
	    "\n"
	    "where:\n"
	    "  index is the position of a movie;\n"
	    "  number is the movie number;\n"
	    "  keyword is part of a title.\n"
	    "\n"
	    "Title searches will find the keywords in order, or first if\n"
	    "only one keyword; using '~' will find the keywords anywhere.\n"
	    "\n"
	    "Use the MOVSEL environment variable to indicate the path of\n"
	    "Movie Select (eg. set MOVSEL=d:/ if your CD is D:).\n"
	    , argv[0], argv[0] );
    return 0;
  }

  db = getenv( "MOVSEL" );
  if (db == NULL)
  {
    db = stpcpy( dbpath, argv[0] ) - 1;
    while (db > dbpath && *db != '/' /* && *db != '\\' */) --db;
    db[1] = 0;
  }
  else
  {
    db = stpcpy( dbpath, db ) - 1;
    if (*db != '\\' && *db != '/') db[1] = '/', db[2] = 0;
  }
  strcat( strcpy( path, dbpath ), MOVCARD );
  movcard = fopen( path, "rb" );
  if (movcard == NULL)
  {
    printf( "Can't find the Movie Select database:\n%s\n", path );
    return 1;
  }

  fseek( movcard, 8, SEEK_SET );
  indices = read4( movcard );  		// Number of movies
  idx	  = read4( movcard );		// Position of index
  mov	  = read4( movcard );		// Position of movie number index
  movies  = read4( movcard );		// Number of movie numbers
  m_index = new long[indices + movies];
  movie   = m_index + indices;

  fseek( movcard, idx, SEEK_SET );
  for (len = 0; len < indices; ++len)
    m_index[len] = read4( movcard );

  fseek( movcard, mov, SEEK_SET );
  for (len = 0; len < movies; ++len)
    movie[len] = read4( movcard );

  read_subcats();

  int curs_x, curs_y, lines;
  unsigned char* text_screen;
  ScreenGetCursor( &curs_x, &curs_y );
  lines = ScreenRows();
  text_screen = (unsigned char*)malloc( lines * ScreenCols() * 2 );
  ScreenRetrieve( text_screen );

  SetFullScreenMode();
  JPInit();

  FilmWindow.m_set_close_button_pressed_callback( QuitCall, "" );
  PBPrev.m_set_pressed_callback( PrNxCall, "-" );
  PBNext.m_set_pressed_callback( PrNxCall, "+" );
  PBFind.m_set_pressed_callback( FindCall, "m" );
  PBList.m_set_pressed_callback( ListCall, "" );

  FindWindow.m_set_close_button_pressed_callback( QuitCall, "" );
  PBPrefs.m_set_pressed_callback( PrefsCall, "" );
  PBSearch.m_set_pressed_callback( SearchCall, "" );
  PBCancel.m_set_pressed_callback( CancelCall, "" );
  LBFound.m_set_item_dbl_click_callback( FoundCall, "" );
  LBFound.m_set_focus_taken_callback( FoundFocus, "+" );
  LBFound.m_set_focus_lost_callback( FoundFocus, "-" );
  PBSearch.m_set_focus_taken_callback( FoundFocus, "*" );
  PBSearch.m_set_focus_lost_callback( FoundFocus, "/" );

  ListWindow.m_set_close_button_pressed_callback( QuitCall, "" );
  // Create the labels displaying the titles.
  for (len = 0; len < TITLES; ++len)
  {
    char buf[] = "?";
    Title[len] = new TLabel( &FList, 1,len+1, TITLELEN,1, "", TRUE );
    *buf = len+1;
    Title[len]->m_set_clicked_callback( TitleCall, buf );
  }
  // Create the cross-reference pushbuttons.
  for (len = 0; len < 28; ++len)
  {
    char buf[] = "~?";
    buf[1] = (len == 0) ? '#' : (len == 27) ? '~' : (len - 1 + 'A');
    Xref[len] = new TPushButton( &ListWindow, 71 + len / 14 * 4, 5 + (len % 14),
				 3, buf, PB_NORMAL, SHC_NONE );
    Xref[len]->m_set_pressed_callback( XrefCall, buf+1 );
  }
  PBPgUp2.m_set_pressed_callback( PageCall, "/" );
  PBPgUp.m_set_pressed_callback(  PageCall, "-" );
  PBPgDn.m_set_pressed_callback(  PageCall, "+" );
  PBPgDn2.m_set_pressed_callback( PageCall, "*" );
  PBLFind.m_set_pressed_callback( FindCall, "l" );
  PBLPref.m_set_pressed_callback( PrefsCall, "" );

  // Initialise the default categories.
  for (len = 0; len < 9; ++len) pref[len] = CHECKED;
  pref[13] = CHECKED;
  pref[7] = NOT_CHECKED;
  for (len = 9; len < 13; ++len) pref[len] = NOT_CHECKED;

  display_titles( 0, 1 );
  if (argc == 1)
  {
    ListWindow.m_open();
  }
  else
  {
    // A list of index or movie numbers.
    if (*argv[1] == '+' || *argv[1] == '-')
    {
      FilmWindow.m_open();
      film  = new int[argc - 1];
      films = 0;
      for (len = 1; len < argc; ++len)
      {
	mov = atoi( argv[len] );
	if (mov < 0)
	{
	  idx = -mov;
	  if (idx >= movies || movie[idx] == 0) continue;
	  idx = movie[idx];
	}
	else
	{
	  if (mov >= indices) continue;
	  idx = m_index[mov];
	}
	film[films++] = mov;
	fseek( movcard, idx + TITLE, SEEK_SET );
	char* buf = reads( movcard, 1 );
	LBFound.m_add_item( buf );
	delete buf;
      }
      filmp = 0;
      display_film( film[0] );		// Always assume one match
    }
    else				// A list of keywords
    {
      FindWindow.m_open();
      if (*argv[1] == '~')              // "Approximate" match
      {
	if (argv[1][1]) ++argv[1];
	else ++argv, --argc;
	CBMatch.m_uncheck();
      }
      char buf[256];
      strcpy( buf, argv[1] );		// Combine all words into one string
      for (len = 2; len < argc; ++len)
	strcat( strcat( buf, " " ), argv[len] );
      SearchText.m_set_string( buf );
      search( argv+1, argc-1 );
    }
  }

  JPRun();

  _set_screen_lines( lines );
  ScreenUpdate( text_screen );
  ScreenSetCursor( curs_x, curs_y );
  free( text_screen );

  if (film)  delete film;
  if (match) delete match;
  for (len = 0; len < TITLES; ++len) delete Title[len];
  delete *subcat;
  delete subcat;
  delete m_index;

  fclose( movcard );
  return 0;
}


/*
  Actor Xref:  18936,	800, 24173,  3895, 24795,  5224, 21212,
		6776,  7911,  7994, 27084,  9162, 27147, 11984,
	       12366, 12636, 19117, 13589, 14582, 16266, 16916,
	       25582, 17304, 18186, 18191, 18322,

 Director Xref:    3,  136,  496,  809, 1027, 5169, 1253,
		5656, 1751, 1773, 1849, 4545, 4687, 2605,
		2679, 2737, 4421, 2935, 3168, 5181, 3672,
		3681, 4302, 4451, 4451, 3957,


 Format of the MOVCARD.DAT file (values are big-endian):

   0000: Signature ('mtDB')
   0004: Length of header (?) excluding the above and this
   0008: Number of movies
   000C: Position of index (ordered)
   0010: Position of index (movie numbers)
   0014: Number of movie numbers
   0018: 16 zeros (?)
   0028: First movie

 Movie format:

   0000: Movie number
   0004: Index number
   0008: Position of previous movie
   000C: Length of this movie (including this header)
   000E: offset of title (after this header, always zero)
   0010: offset of directors
   0012: offset of description
   0014: offset of stars
   0016: offset of title (again?, another zero)
   0018: offset of director cross-reference
   001A: offset of actor cross-reference
   001C: offset of year/tv/color/type
   001E: more zeros
   0025: Rating
   0026: Length (minutes)
   0028: Category
   002A: Number of subcategories
   002B: Space for seven subcategories

   0032: Length of movie title
   0033: Title
   ????: Length of directors string
     +1: CR-separated list of directors
   ????: Length of stars string
     +2: CR-separated list of stars
   ????: Length of description string
     +2: Description
   ????: Length of directors cross-reference
     +2: CR-separated list of director numbers
   ????: Length of actors cross-reference
     +2: CR-separated list of actor numbers
   ????: Length of year/tv/color/type string
     +1: Year/tv/color/type (tilde-separated)


 Format of MOVIETIT.NX1 (values are little-endian):

   0000: Signature ('1NDI')
   0004: Length of following (0010)
   0008: Offset of alphabetical index (0018)
   000C: Length of above (0038)
   0010: Position of index (0050)
   0014: Length of indices
   0018: Index of first keyword
   001A: Index of 'A'
   ....
   004C: Index of 'Z'
   004E: Index of last keyword

 Format of keyword:

   0000: Number of titles containing keyword
   0002: Length of string containing movie numbers
   0004: Position of above string in MOVIETIT.IN1
   0008: Keyword (NUL terminated, extra added to make even)


 Format of MOVIETIT.IN1:

   Comma-separated list of movie numbers.
*/
