//******************************************************************************
// PLC LANGUAGE MODULE: C                                                     **
//                                                                            **
// Copyright 1997 Adept Software, All Rights Reserved                         **
//                                                                            **
//   Check for errors before executing a statement.                           **
//******************************************************************************

#define	PLC_C_C
	#include "PLC_C.H"
#undef	PLC_C_C

//**************************************
// CONSTANTS

// TOKEN SUBTYPES (EXECUTION)
enum
{
	// OPERATORS
	TT_OPR_NEGATE=0,TT_OPR_COMPLEMENT,TT_OPR_UNARY_PLUS,TT_OPR_UNARY_MINUS,
		TT_OPR_INCREMENT_PRE,TT_OPR_DECREMENT_PRE,TT_OPR_INCREMENT_POST,TT_OPR_DECREMENT_POST,
	TT_OPR_MULTIPLY,TT_OPR_DIVIDE,TT_OPR_MODULUS,
	TT_OPR_ADD,TT_OPR_SUBTRACT,
	TT_OPR_SHL,TT_OPR_SHR,
	TT_OPR_LESS,TT_OPR_LESS_EQUAL,TT_OPR_GREATER,TT_OPR_GREATER_EQUAL,
	TT_OPR_EQUAL,TT_OPR_NOT_EQUAL,
	TT_OPR_AND,
	TT_OPR_XOR,
	TT_OPR_OR,
	TT_OPR_LOGICAL_AND,
	TT_OPR_LOGICAL_OR,
	TT_OPR_CONDITIONAL_Q,TT_OPR_CONDITIONAL_C,
	TT_OPR_ASSIGN,TT_OPR_ASSIGN_MULT,TT_OPR_ASSIGN_DIV,TT_OPR_ASSIGN_MOD,TT_OPR_ASSIGN_ADD,TT_OPR_ASSIGN_SUB,
		TT_OPR_ASSIGN_AND,TT_OPR_ASSIGN_XOR,TT_OPR_ASSIGN_OR,TT_OPR_ASSIGN_SHL,TT_OPR_ASSIGN_SHR,
	TT_OPR_COMMA,
	// KEYWORDS
	TT_KEY_IF=0,TT_KEY_ELSE,
	TT_KEY_DO,TT_KEY_WHILE,
	TT_KEY_FOR,
	TT_KEY_SWITCH,TT_KEY_CASE,TT_KEY_DEFAULT,
	TT_KEY_BREAK,TT_KEY_CONTINUE,
	TT_KEY_RETURN,
	TT_KEY_GOTO,
};

//**************************************
// LOCAL VARIABLES

// COMMON STRINGS (COMPILATION)
PLC_COMMON	Common=
{
	// single char
	",",
	"(",")",
	"(",")",
	"\"","\"",
	// char list
	" \t\n", "\r",
	";", ";",
	"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ",
	"0123456789\".",
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_@$",
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_",
	// string
	"//",
	"/*","*/",
	"{","}",
	"",":",
	"(",")",
	":",
};

// KEYWORDS (COMPILATION)
PLC_KEYWORD	KeywordList[]=
{
	{	"IF","","",
		PLC_KEY_OPEN,PLC_KEY_EXPRESSION,PLC_KEY_CLOSE,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"ELSE","","",
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{   "DO","WHILE","",
		PLC_KEY_STATEMENT,
		PLC_KEY_STRING1,
		PLC_KEY_OPEN,PLC_KEY_EXPRESSION,PLC_KEY_CLOSE,
		PLC_KEY_DONE,
	},
	{	"WHILE","","",
		PLC_KEY_OPEN,PLC_KEY_EXPRESSION,PLC_KEY_CLOSE,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"FOR","","",
		PLC_KEY_OPEN,PLC_KEY_EXPRESSION,PLC_KEY_DIVIDER,
		PLC_KEY_EXPRESSION,PLC_KEY_DIVIDER,
		PLC_KEY_EXPRESSION,PLC_KEY_CLOSE,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"SWITCH","","",
		PLC_KEY_OPEN,PLC_KEY_EXPRESSION,PLC_KEY_CLOSE,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"CASE",":","",
		PLC_KEY_DATATYPE,PLC_KEY_STRING2,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"DEFAULT",":","",
		PLC_KEY_STRING2,
		PLC_KEY_STATEMENT,
		PLC_KEY_DONE,
	},
	{	"BREAK","","",
		PLC_KEY_DONE,
	},
	{	"CONTINUE","","",
		PLC_KEY_DONE,
	},
	{	"RETURN","","",
		PLC_KEY_EXPRESSION,
		PLC_KEY_DONE,
	},
/*	{	"GOTO","","",
		PLC_KEY_SYMBOL,
		PLC_KEY_DONE,
	},
*/};

// OPERATORS (COMPILATION)
PLC_OPERATOR	OperatorList[]=
{
	{"!",	0,	TRUE,	PLC_OP_NEXT,},
	{"~",	0,	TRUE,	PLC_OP_NEXT,},
	{"+",	0,	TRUE,	PLC_OP_NEXTIFNOPREV,},
	{"-",	0,	TRUE,	PLC_OP_NEXTIFNOPREV,},
	{"++",	0,	TRUE,	PLC_OP_NEXTIFNOPREV,},
	{"--",	0,	TRUE,	PLC_OP_NEXTIFNOPREV,},
	{"++",	0,	TRUE,	PLC_OP_PREV,},
	{"--",	0,	TRUE,	PLC_OP_PREV,},
	{"*",	1,	FALSE,	PLC_OP_BOTH,},
	{"/",	1,	FALSE,	PLC_OP_BOTH,},
	{"%",	1,	FALSE,	PLC_OP_BOTH,},
	{"+",	2,	FALSE,	PLC_OP_BOTH,},
	{"-",	2,	FALSE,	PLC_OP_BOTH,},
	{"<<",	3,	FALSE,	PLC_OP_BOTH,},
	{">>",	3,	FALSE,	PLC_OP_BOTH,},
	{"<",	4,	FALSE,	PLC_OP_BOTH,},
	{"<=",	4,	FALSE,	PLC_OP_BOTH,},
	{">",	4,	FALSE,	PLC_OP_BOTH,},
	{">=",	4,	FALSE,	PLC_OP_BOTH,},
	{"==",	5,	FALSE,	PLC_OP_BOTH,},
	{"!=",	5,	FALSE,	PLC_OP_BOTH,},
	{"&",	6,	FALSE,	PLC_OP_BOTH,},
	{"^",	7,	FALSE,	PLC_OP_BOTH,},
	{"|",	8,	FALSE,	PLC_OP_BOTH,},
	{"&&",	9,	FALSE,	PLC_OP_BOTH,},
	{"||",	10,	FALSE,	PLC_OP_BOTH,},
	{"?",	11,	TRUE,	PLC_OP_PREV,},
	{":",	11,	TRUE,	PLC_OP_BOTH,},
	{"=",	12,	TRUE,	PLC_OP_BOTH,},
	{"*=",	12,	TRUE,	PLC_OP_BOTH,},
	{"/=",	12,	TRUE,	PLC_OP_BOTH,},
	{"%=",	12,	TRUE,	PLC_OP_BOTH,},
	{"+=",	12,	TRUE,	PLC_OP_BOTH,},
	{"-=",	12,	TRUE,	PLC_OP_BOTH,},
	{"&=",	12,	TRUE,	PLC_OP_BOTH,},
	{"^=",	12,	TRUE,	PLC_OP_BOTH,},
	{"|=",	12,	TRUE,	PLC_OP_BOTH,},
	{"<<=",	12,	TRUE,	PLC_OP_BOTH,},
	{">>=",	12,	TRUE,	PLC_OP_BOTH,},
	{",",	13,	FALSE,	PLC_OP_BOTH,},
};

//******************************************************************************
//******************************************************************************
// LOCAL ROUTINES

// execute a keyword statement
void	C_ExecKeyword	(PLC_TOKEN *Token,flag *IfResultPtr)
{
	PLC_DATATYPE	Value;
	PLC_TOKEN		*Statement,*Expression1,*Expression2,*Expression3,
					*TokenTemp;

	ERR_START
	{
		switch(Token->SubType)
		{
		// IF
		case TT_KEY_IF:
			Expression1=LNK_Next(PLC_TOK_Anchor(Token));
			Statement=LNK_Next(&Expression1->Link);
			PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
			*IfResultPtr=PLC_DT_GetBool(&PLC_Result);
			ERR_CHECKBREAK();
			if(*IfResultPtr) PLC_ExecStatement(LNK_Next(&Expression1->Link),NULL);
			break;

		// ELSE
		case TT_KEY_ELSE:
			Statement=LNK_Next(PLC_TOK_Anchor(Token));
			if(!*IfResultPtr) PLC_ExecStatement(Statement,NULL);
			break;

		// DO
		case TT_KEY_DO:
			Statement=LNK_Next(PLC_TOK_Anchor(Token));
			Expression1=LNK_Next(&Statement->Link);
			do
			{
				ERR_CHECKBREAK();
				PLC_ExecStatement(Statement,NULL);
				if(PLC_FlowStatus==PLC_FLOW_BREAK) {PLC_FlowStatus=PLC_FLOW_NORMAL; break;}
				else if(PLC_FlowStatus==PLC_FLOW_RETURN || PLC_FlowStatus==PLC_FLOW_GOTO) break;
				else if(PLC_FlowStatus==PLC_FLOW_CONTINUE) PLC_FlowStatus=PLC_FLOW_NORMAL;
				PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
			}while(*IfResultPtr=PLC_DT_GetBool(&PLC_Result));
			break;

		// WHILE
		case TT_KEY_WHILE:
			Expression1=LNK_Next(PLC_TOK_Anchor(Token));
			Statement=LNK_Next(&Expression1->Link);
			while(1)
			{
				PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
				if(*IfResultPtr=PLC_DT_GetBool(&PLC_Result)) break;
				else
				{
					ERR_CHECKBREAK();
					PLC_ExecStatement(Statement,NULL);
					if(PLC_FlowStatus==PLC_FLOW_BREAK) {PLC_FlowStatus=PLC_FLOW_NORMAL; break;}
					else if(PLC_FlowStatus==PLC_FLOW_RETURN || PLC_FlowStatus==PLC_FLOW_GOTO) break;
					else if(PLC_FlowStatus==PLC_FLOW_CONTINUE) PLC_FlowStatus=PLC_FLOW_NORMAL;
				}
			}
			break;

		// FOR
		case TT_KEY_FOR:
			Expression1=LNK_Next(PLC_TOK_Anchor(Token));
			Expression2=LNK_Next(&Expression1->Link);
			Expression3=LNK_Next(&Expression2->Link);
			Statement=LNK_Next(&Expression3->Link);
			PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
			while(1)
			{
				if(Expression2!=NULL)
				{
					PLC_ExecExpression(PLC_TOK_Anchor(Expression2));
					if(!PLC_DT_GetBool(&PLC_Result)) break;
				}
				ERR_CHECKBREAK();
				PLC_ExecStatement(Statement,NULL);
				if(PLC_FlowStatus==PLC_FLOW_BREAK) {PLC_FlowStatus=PLC_FLOW_NORMAL; break;}
				else if(PLC_FlowStatus==PLC_FLOW_RETURN || PLC_FlowStatus==PLC_FLOW_GOTO) break;
				else if(PLC_FlowStatus==PLC_FLOW_CONTINUE) PLC_FlowStatus=PLC_FLOW_NORMAL;
				PLC_ExecExpression(PLC_TOK_Anchor(Expression3));
			}
			break;

		// SWITCH
		case TT_KEY_SWITCH:
			Expression1=LNK_Next(PLC_TOK_Anchor(Token));
			Statement=LNK_Next(&Expression1->Link);
			PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
			ERR_CHECKBREAK();
			// SEARCH FOR CASE that matches value
			PLC_DT_Assign(&Value,&PLC_Result);
			TokenTemp=NULL;		// default token
			if(Statement->Type==PLC_TT_COMPOUND) Statement=LNK_Next(PLC_TOK_Anchor(Statement));
			while(Statement!=NULL)
			{
				if(Statement->Type==PLC_TT_KEYWORD && (Statement->SubType==TT_KEY_CASE || Statement->SubType==TT_KEY_DEFAULT))
				{
					if(Statement->SubType==TT_KEY_DEFAULT) TokenTemp=Statement;
					else
					{
						PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
						ERR_CHECKBREAK();
						if(PLC_DT_Compare(&PLC_Result,&Value)==0) break;
					}
				}
				Statement=LNK_Next(&Statement->Link);
			}
			ERR_CHECKBREAK();
			// EXECUTE CURRENT CASE
			if(Statement==NULL) Statement=TokenTemp;
			if(Statement!=NULL)
			{
				PLC_ExecBlock(LNK_Next(&Statement->Link));
				ERR_CHECKBREAK();
				if(PLC_FlowStatus==PLC_FLOW_BREAK) PLC_FlowStatus=PLC_FLOW_NORMAL;
			}
			break;

		// IGNORE
		case TT_KEY_CASE:
		case TT_KEY_DEFAULT:
			break;

		// PROGRAM FLOW KEYWORDS
		case TT_KEY_CONTINUE:
			PLC_FlowStatus=PLC_FLOW_CONTINUE;
			break;

		case TT_KEY_BREAK:
			PLC_FlowStatus=PLC_FLOW_BREAK;
			break;

		case TT_KEY_RETURN:
			Expression1=LNK_Next(PLC_TOK_Anchor(Token));
			PLC_ExecExpression(PLC_TOK_Anchor(Expression1));
			PLC_FlowStatus=PLC_FLOW_RETURN;
			break;

		case TT_KEY_GOTO:
			PLC_GotoToken=PLC_LabelSearch(((PLC_SYMBOL*)PLC_TOK_Pointer(Token))->Name);
			if(PLC_GotoToken==NULL) {PLC_Err(PLC_ERR_UNKNOWN_SYMBOL_,((PLC_SYMBOL*)PLC_TOK_Pointer(Token))->Name); errbreak;}
			break;
		}
	}ERR_END;
}

//******************************************************************************

void	C_ApplyOperator(byte OperNum,PLC_DATATYPE *Val1,PLC_DATATYPE *Val2)
{
	ERR_START
	{
		PLC_Result.Type=PLC_DT_NONE;
		if(OperNum==TT_OPR_ASSIGN);
		else if(OperNum==TT_OPR_EQUAL) PLC_DT_SetBool(&PLC_Result,!PLC_DT_Compare(Val1,Val2));
		else if(OperNum==TT_OPR_NOT_EQUAL) PLC_DT_SetBool(&PLC_Result,PLC_DT_Compare(Val1,Val2));
		else if(Val1->Type==PLC_DT_FLOAT || (Val2!=NULL && Val2->Type==PLC_DT_FLOAT))
		{
			// result: floating point
			switch(OperNum)
			{
			case TT_OPR_INCREMENT_POST:	PLC_DT_Assign(&PLC_Result,Val1);
			case TT_OPR_INCREMENT_PRE:	PLC_DT_SetFloat(Val1,PLC_DT_GetFloat(Val1)+1);	break;
			case TT_OPR_DECREMENT_POST:	PLC_DT_Assign(&PLC_Result,Val1);
			case TT_OPR_DECREMENT_PRE:	PLC_DT_SetFloat(Val1,PLC_DT_GetFloat(Val1)-1);	break;
			// one operand
			case TT_OPR_NEGATE:			PLC_DT_SetFloat(&PLC_Result,!PLC_DT_GetFloat(Val1));	break;
			case TT_OPR_UNARY_PLUS:		PLC_DT_SetFloat(&PLC_Result,+PLC_DT_GetFloat(Val1));	break;
			case TT_OPR_UNARY_MINUS:	PLC_DT_SetFloat(&PLC_Result,-PLC_DT_GetFloat(Val1));	break;
			// two operands
			case TT_OPR_ASSIGN_MULT:
			case TT_OPR_MULTIPLY:		PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)*PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_ASSIGN_DIV:
			case TT_OPR_DIVIDE:
				if(PLC_DT_GetFloat(Val2)==0) PLC_Err(PLC_ERR_DIV_ZERO);
				else PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)/PLC_DT_GetFloat(Val2));
				break;
			case TT_OPR_ASSIGN_ADD:
			case TT_OPR_ADD:			PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)+PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_ASSIGN_SUB:
			case TT_OPR_SUBTRACT:		PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)-PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_LESS:			PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)<PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_LESS_EQUAL:		PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)<=PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_GREATER:		PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)>PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_GREATER_EQUAL:	PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)>=PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_LOGICAL_AND:	PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)&&PLC_DT_GetFloat(Val2));	break;
			case TT_OPR_LOGICAL_OR:		PLC_DT_SetFloat(&PLC_Result,PLC_DT_GetFloat(Val1)||PLC_DT_GetFloat(Val2));	break;
			default:
				PLC_Err(PLC_ERR_ILLEGAL_OPERATION);
				break;
			}
		}
		else if(Val1->Type==PLC_DT_INT)
		{
			// result: integer
			switch(OperNum)
			{
			case TT_OPR_INCREMENT_POST:	PLC_DT_Assign(&PLC_Result,Val1);
			case TT_OPR_INCREMENT_PRE:	PLC_DT_SetInt(Val1,PLC_DT_GetInt(Val1)+1);	break;
			case TT_OPR_DECREMENT_POST:	PLC_DT_Assign(&PLC_Result,Val1);
			case TT_OPR_DECREMENT_PRE:	PLC_DT_SetInt(Val1,PLC_DT_GetInt(Val1)-1);	break;
			// one operand
			case TT_OPR_NEGATE:			PLC_DT_SetInt(&PLC_Result,!PLC_DT_GetInt(Val1));	break;
			case TT_OPR_COMPLEMENT:		PLC_DT_SetInt(&PLC_Result,~PLC_DT_GetInt(Val1));	break;
			case TT_OPR_UNARY_PLUS:		PLC_DT_SetInt(&PLC_Result,+PLC_DT_GetInt(Val1));	break;
			case TT_OPR_UNARY_MINUS:	PLC_DT_SetInt(&PLC_Result,-PLC_DT_GetInt(Val1));	break;
			// two operands
			case TT_OPR_ASSIGN_MULT:
			case TT_OPR_MULTIPLY:		PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)*PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_DIV:
			case TT_OPR_DIVIDE:
				if(PLC_DT_GetInt(Val2)==0) PLC_Err(PLC_ERR_DIV_ZERO);
				else PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)/PLC_DT_GetInt(Val2));
				break;
			case TT_OPR_ASSIGN_MOD:
			case TT_OPR_MODULUS:
				if(PLC_DT_GetInt(Val2)==0) PLC_Err(PLC_ERR_DIV_ZERO);
				else PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)%PLC_DT_GetInt(Val2));
				break;
			case TT_OPR_ASSIGN_ADD:
			case TT_OPR_ADD:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)+PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_SUB:
			case TT_OPR_SUBTRACT:		PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)-PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_SHL:
			case TT_OPR_SHL:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)<<PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_SHR:
			case TT_OPR_SHR:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)>>PLC_DT_GetInt(Val2));	break;
			case TT_OPR_LESS:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)<PLC_DT_GetInt(Val2));	break;
			case TT_OPR_LESS_EQUAL:		PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)<=PLC_DT_GetInt(Val2));	break;
			case TT_OPR_GREATER:		PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)>PLC_DT_GetInt(Val2));	break;
			case TT_OPR_GREATER_EQUAL:	PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)>=PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_AND:
			case TT_OPR_AND:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)&PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_XOR:
			case TT_OPR_XOR:			PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)^PLC_DT_GetInt(Val2));	break;
			case TT_OPR_ASSIGN_OR:
			case TT_OPR_OR:				PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)|PLC_DT_GetInt(Val2));	break;
			case TT_OPR_LOGICAL_AND:	PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)&&PLC_DT_GetInt(Val2));	break;
			case TT_OPR_LOGICAL_OR:		PLC_DT_SetInt(&PLC_Result,PLC_DT_GetInt(Val1)||PLC_DT_GetInt(Val2));	break;
			default:
				PLC_Err(PLC_ERR_ILLEGAL_OPERATION);
				break;
			}
		}
		else if(Val1->Type==PLC_DT_STRING)
		{
			switch(OperNum)
			{
			case TT_OPR_ASSIGN_ADD:
			case TT_OPR_ADD:
				PLC_DT_SetString(&PLC_Result,PLC_DT_GetString(Val1));
				strncat(PLC_DT_GetString(&PLC_Result),PLC_DT_GetString(Val2),PLC_DT_STRMAX-strlen(PLC_DT_GetString(&PLC_Result)));
				break;
			case TT_OPR_LESS:			PLC_DT_SetBool(&PLC_Result,(strcmp(PLC_DT_GetString(Val1),PLC_DT_GetString(Val2))<0));	break;
			case TT_OPR_LESS_EQUAL:		PLC_DT_SetBool(&PLC_Result,(strcmp(PLC_DT_GetString(Val1),PLC_DT_GetString(Val2))<=0));	break;
			case TT_OPR_GREATER:		PLC_DT_SetBool(&PLC_Result,(strcmp(PLC_DT_GetString(Val1),PLC_DT_GetString(Val2))>0));	break;
			case TT_OPR_GREATER_EQUAL:	PLC_DT_SetBool(&PLC_Result,(strcmp(PLC_DT_GetString(Val1),PLC_DT_GetString(Val2))>=0));	break;
			default:
				PLC_Err(PLC_ERR_ILLEGAL_OPERATION);
				break;
			}
		}
		// ASSIGNMENT
		switch(OperNum)
		{
		case TT_OPR_ASSIGN:			PLC_DT_Assign(Val1,Val2);	break;
		case TT_OPR_ASSIGN_MULT:
		case TT_OPR_ASSIGN_DIV:
		case TT_OPR_ASSIGN_MOD:
		case TT_OPR_ASSIGN_ADD:
		case TT_OPR_ASSIGN_SUB:
		case TT_OPR_ASSIGN_SHL:
		case TT_OPR_ASSIGN_SHR:
		case TT_OPR_ASSIGN_AND:
		case TT_OPR_ASSIGN_XOR:
		case TT_OPR_ASSIGN_OR:
			PLC_DT_Assign(Val1,&PLC_Result);
			break;
		}
		if(PLC_Result.Type==PLC_DT_NONE) PLC_DT_Assign(&PLC_Result,Val1);
	}ERR_END;
}

//******************************************************************************
//******************************************************************************
// PUBLIC ROUTINES

void	PLC_C_Init			(void)
{
	// compilation
	PLC_KeywordMax=sizeof(KeywordList)/sizeof(PLC_KEYWORD);
	PLC_OperatorMax=sizeof(OperatorList)/sizeof(PLC_OPERATOR);
	PLC_Common=Common;
	PLC_Keywords=&KeywordList;
	PLC_Operators=&OperatorList;
	// execution
	PLC_ExecKeyword=C_ExecKeyword;
	PLC_ApplyOperator=C_ApplyOperator;
}

//******************************************************************************
//******************************************************************************
