//***************************************************************************
//
//  DemoView.cpp
//
//***************************************************************************

#include "stdafx.h"
#include "LVDemo.h"
#include "DemoDoc.h"
#include "DemoView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNCREATE (CMyListView, CListView)

BEGIN_MESSAGE_MAP (CMyListView, CListView)
    ON_WM_DRAWITEM_REFLECT ()
    ON_WM_MEASUREITEM_REFLECT ()
    ON_NOTIFY_REFLECT (LVN_COLUMNCLICK, OnColumnClick)
    ON_WM_DESTROY ()
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CMyListView initialization

BOOL CMyListView::PreCreateWindow (CREATESTRUCT& cs)
{
    if (!CListView::PreCreateWindow (cs))
        return FALSE;

    //
    // Make the list view an owner-drawn control.
    //
    cs.style |= LVS_OWNERDRAWFIXED;
    return TRUE;
}

void CMyListView::OnInitialUpdate()
{
    CListView::OnInitialUpdate();

    //
    // Switch to report view.
    //
    ModifyStyle (LVS_TYPEMASK, LVS_REPORT);

    //
    // Add columns to the list view.
    //
    GetListCtrl ().InsertColumn (0, "File Name", LVCFMT_LEFT, 192);
    GetListCtrl ().InsertColumn (1, "Size", LVCFMT_RIGHT, 128);
    GetListCtrl ().InsertColumn (2, "Created", LVCFMT_CENTER, 128);
    GetListCtrl ().InsertColumn (3, "Last Modified", LVCFMT_CENTER, 128);

    //
    // Add items to the list view.
    //
    AddFileInfo ();
}

/////////////////////////////////////////////////////////////////////////////
// CMyListView message handlers

void CMyListView::OnColumnClick (NMHDR* pNMHDR, LRESULT* pResult) 
{
    //
    // When a header button is clicked, call the list view's SortItems
    // function to sort the data in the column. The column number passed
    // in SortItem's second parameter becomes the lParamSort parameter
    // passed to the comparison function.
    //
    NM_LISTVIEW* pNMListView = (NM_LISTVIEW*) pNMHDR;
    GetListCtrl ().SortItems (CompareFunc, pNMListView->iSubItem);
    *pResult = 0;
}

void CMyListView::OnDestroy ()
{
    //
    // Delete the ITEMINFO structures allocated by AddItemToListView
    // before the list view is destroyed.
    //
    FreeItems ();
}

/////////////////////////////////////////////////////////////////////////////
// CMyListView drawing

void CMyListView::MeasureItem (LPMEASUREITEMSTRUCT lpmis)
{
    //
    // Compute the height of each line in the list view from the font height.
    //
    LOGFONT lf;
    CFont* pFont = GetFont ();
    pFont->GetLogFont (&lf);
    lpmis->itemHeight = abs ((int) lf.lfHeight) + 6;
}

void CMyListView::DrawItem (LPDRAWITEMSTRUCT lpdis)
{
    //
    // Create a CDC object from the DC handle provided to us.
    //
    CDC dc;
    dc.Attach (lpdis->hDC);

    //
    // Initialize a CRect object with a rectangle that surrounds the item
    // and its subitems.
    //
    CRect rect = lpdis->rcItem;

    //
    // Retrieve a pointer to the ITEMINFO structure that holds the item's
    // data. lpdis->itemID is the 0-based index of the item that we're being
    // being asked to paint.
    //
    ITEMINFO* pItem = (ITEMINFO*) GetListCtrl ().GetItemData (lpdis->itemID);

    //
    // If the item is selected, paint the rectangle with the system color
    // COLOR_HIGHLIGHT. Otherwise, use COLOR_WINDOW.
    //
    CBrush* pBrush = new CBrush (::GetSysColor ((lpdis->itemState &
        ODS_SELECTED) ? COLOR_HIGHLIGHT : COLOR_WINDOW));
    dc.FillRect (rect, pBrush);
    delete pBrush;

    //
    // If the item has the input focus, draw a dotted focus rectangle.
    //
    if (lpdis->itemState & ODS_FOCUS)
        dc.DrawFocusRect (rect);

    //
    // Draw the text of the item and its subitems. If the item is selected,
    // the text color is set to COLOR_HIGHLIGHTTEXT first. Values for the
    // item and subitems are retrieved through the ITEMINFO pointer
    // initialized earlier, converted to text, and displayed by our own
    // DrawItemText function.
    //
    if (lpdis->itemID != -1) {
        COLORREF crOld;
        BOOL bReset = FALSE;

        if (lpdis->itemState & ODS_SELECTED) {
            crOld = dc.SetTextColor (::GetSysColor (COLOR_HIGHLIGHTTEXT));
            bReset = TRUE;
        }

        CString string;
        for (int i=0; i<4; i++) {
            LV_COLUMN lvc;
            lvc.mask = LVCF_WIDTH | LVCF_FMT;
            GetListCtrl ().GetColumn (i, &lvc);
            int nFormat = lvc.fmt & LVCFMT_JUSTIFYMASK;
            int nWidth = lvc.cx;

            switch (i) {

            case 0: // File name
                DrawItemText (&dc, pItem->strFileName, rect, nWidth,
                    nFormat);
                break;

            case 1: // File size
                string.Format ("%u", pItem->dwFileSize);
                DrawItemText (&dc, string, rect, nWidth, nFormat);
                break;

            case 2: // Date and time created
                DrawItemText (&dc,
                    (LPCTSTR) TimeToString (pItem->ftCreationTime),
                    rect, nWidth, nFormat);
                break;

            case 3: // Date and time modified
                DrawItemText (&dc,
                    (LPCTSTR) TimeToString (pItem->ftLastWriteTime),
                    rect, nWidth, nFormat);
                break;
            }
            rect.left += nWidth;
        }

        if (bReset)
            dc.SetTextColor (crOld);
    }

    //
    // Detach the control's DC from our own CDC object so the former won't
    // be deleted when this function ends.
    //
    dc.Detach ();
}

void CMyListView::DrawItemText (CDC* pDC, CString text, CRect rect,
    int nWidth, int nFormat)
{
    //
    // Make sure the text will fit in the prescribed rectangle, and truncate
    // it if it won't.
    //
    BOOL bNeedDots = FALSE;
    int nMaxWidth = nWidth - 4;

    while (GetListCtrl ().GetStringWidth ((LPCTSTR) text) > (nMaxWidth - 4)) {
        text = text.Left (text.GetLength () - 1);
        bNeedDots = TRUE;
    }

    if (bNeedDots) {
        if (text.GetLength () >= 1)
            text = text.Left (text.GetLength () - 1);
        text += "...";
    }

    //
    // Draw the text into the rectangle using MFC's handy CDC::DrawText
    // function.
    //
    rect.left += 2; 
    rect.right = rect.left + nMaxWidth;

    UINT nStyle = DT_VCENTER | DT_SINGLELINE;
    if (nFormat == LVCFMT_LEFT)
        nStyle |= DT_LEFT;
    else if (nFormat == LVCFMT_CENTER)
        nStyle |= DT_CENTER;
    else // nFormat == LVCFMT_RIGHT
        nStyle |= DT_RIGHT;

    pDC->DrawText (text, rect, nStyle);
}

void CMyListView::FreeItems ()
{
    //
    // Delete the ITEMINFO structures allocated when the control was created.
    //
    int nCount = GetListCtrl ().GetItemCount ();
    if (nCount) {
        for (int i=0; i<nCount; i++)
            delete (ITEMINFO*) GetListCtrl ().GetItemData (i);
    }
}

/////////////////////////////////////////////////////////////////////////////
// CMyListView helpers

void CMyListView::AddFileInfo ()
{
    //
    // Get the path to the Window directory.
    //
    char szPath[MAX_PATH];
    ::GetWindowsDirectory (szPath, sizeof (szPath));
    ::lstrcat (szPath, "\\*.*");

    //
    // Enumerate the files in the directory and add items to the list view.
    //
    WIN32_FIND_DATA fd;
    HANDLE hFind = ::FindFirstFile (szPath, &fd);
    if (hFind == INVALID_HANDLE_VALUE)
        return;

    do {
        if (!(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
            if (!AddItemToListView (fd))
                break;
        }
    } while (::FindNextFile (hFind, &fd));

    ::FindClose (hFind);
}

BOOL CMyListView::AddItemToListView (WIN32_FIND_DATA& fd)
{
    //
    // Allocate an ITEMINFO structure to hold the item's data.
    //
    ITEMINFO* pItem;
    try {
        pItem = new ITEMINFO;
    }
    catch (CMemoryException* e) {
        e->Delete ();
        return FALSE;
    }

    //
    // Initialize the ITEMINFO structure's fields.
    //
    pItem->strFileName = fd.cFileName;
    pItem->dwFileSize = fd.nFileSizeLow;
    pItem->ftCreationTime = fd.ftCreationTime;
    pItem->ftLastWriteTime = fd.ftLastWriteTime;

    //
    // Add an item to the list view control, and set the item's lParam
    // equal to the address of the corresponding ITEMINFO structure.
    //
    LV_ITEM lvi;
    lvi.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM;
    lvi.iItem = GetListCtrl ().GetItemCount ();
    lvi.iSubItem = 0;
    lvi.iImage = 0;
    lvi.pszText = NULL;
    lvi.lParam = (LPARAM) pItem;

    if (GetListCtrl ().InsertItem (&lvi) == -1)
        return FALSE;

    return TRUE;
}

CString CMyListView::TimeToString (FILETIME ft)
{
    //
    // Convert the date and time in a FILETIME structure into a CString.
    //
    CTime time (ft);

    BOOL pm = FALSE;
    int nHour = time.GetHour ();
    if (nHour == 0)
        nHour = 12;
    else if (nHour == 12)
        pm = TRUE;
    else if (nHour > 12) {
        nHour -= 12;
        pm = TRUE;
    }

    CString string;
    string.Format ("%d/%0.2d/%0.2d (%d:%0.2d%c)",
        time.GetMonth (), time.GetDay (), time.GetYear () % 100,
        nHour, time.GetMinute (), pm ? 'p' : 'a');

    return string;
}

/////////////////////////////////////////////////////////////////////////////
// Callback function for sorting

int CALLBACK CMyListView::CompareFunc (LPARAM lParam1, LPARAM lParam2,
    LPARAM lParamSort)
{
    ITEMINFO* pItem1 = (ITEMINFO*) lParam1;
    ITEMINFO* pItem2 = (ITEMINFO*) lParam2;
    int nResult;

    switch (lParamSort) { // lParamSort == Column number (0-3)

    case 0: // File name
        nResult = pItem1->strFileName.CompareNoCase (pItem2->strFileName);
        break;

    case 1: // File size
        nResult = pItem1->dwFileSize - pItem2->dwFileSize;
        break;

    case 2: // Date and time created
        nResult = ::CompareFileTime (&pItem1->ftCreationTime,
            &pItem2->ftCreationTime);
        break;

    case 3: // Date and time modified
        nResult = ::CompareFileTime (&pItem1->ftLastWriteTime,
            &pItem2->ftLastWriteTime);
        break;
    }
    return nResult;
}
