unit list;

// VBList Object implementation code: Delphi 2.0 Implementation
// (C)1995 By John Lam
interface

uses
  Ole2, OleAuto, Classes, WinProcs, WinTypes, SysUtils, Windows;

type
  TVBList = class;

  // My declaration of the IEnumVariant interface
  IMyEnumVariant = class(IUnknown)
    constructor Create(pStringList: TVBList);

    // Standard IUnknown interface functions
    function QueryInterface(const iid: TIID; var obj): HResult; override;
    function AddRef: Longint; override;
    function Release: Longint; override;

    // IEnumVariant interface functions
    function Next(celt: Integer; elt: PVariantArgList; pceltFetched: Integer): HResult; virtual; stdcall;
    function Skip(celt: Longint): HResult; virtual; stdcall;
    function Reset: HResult; virtual; stdcall;
    function Clone(var enum: IEnumVariant): HResult; virtual; stdcall;
  private
    m_pVBList: TVBList;
    m_CurrentPosition : Integer;
    FRefCount: Integer;
  end;

  TVBList = class(TAutoObject)
    constructor Create; override;
  private
    m_StringList: TStringList;
    m_CurrentPosition: Integer;
    FEnumVariant: IMyEnumVariant;
    function GetNewEnum: Variant;
  automated
    // Property for controller to obtain the IEnumVariant interface
    property _NewEnum: Variant read GetNewEnum dispid -4;

    // Required property for Collection
    function Count: Integer;

    procedure AddHead(const S: String);
    procedure AddTail(const S: String);
    function GetHead: String;
    function GetTail: String;
    procedure RemoveHead;
    procedure RemoveTail;
    procedure RemoveAll;
    function GetNext: String;
    function GetPrev: String;
    function IsEmpty: Integer;
    procedure GotoHead;
    procedure GotoTail;
    function GetCurrent: String;
    procedure SetCurrent(const S: String);
    procedure RemoveCurrent;
    function GetIndex: Integer;
    procedure SetIndex(I: Integer);
    function FindFromCurrent(const S: String): Integer;
    function FindFromStart(const S: String): Integer;
    procedure InsertBefore(const S: String);
    procedure InsertAfter(const S: String);
  end;

implementation

// Initialize the VBList Control
constructor TVBList.Create;
begin
  inherited Create;
  m_StringList := TStringList.Create;
  m_CurrentPosition := -1;
end;

// Wrapper functions for TStringList

// Add an item to the start of the list
procedure TVBList.AddHead(const S: String);
begin
  m_StringList.Insert(0, S);
  m_CurrentPosition := 0;
end;

// Add an item to the end of the list
procedure TVBList.AddTail(const S: String);
begin
  m_CurrentPosition := m_StringList.Add(S);
end;

// Get an item from the start of the list
function TVBList.GetHead: String;
begin
  Result := '';
  with m_StringList do begin
    if Count > 0 then begin
      Result := m_StringList.Strings[0];
      m_CurrentPosition := 0;
    end
    else
      m_CurrentPosition := -1;
  end;
end;

// Get an item from the end of the list
function TVBList.GetTail: string;
begin
  Result := '';
  with m_StringList do begin
    if Count = 0 then
      m_CurrentPosition := -1
    else begin
      Result := Strings[Count - 1];
      m_CurrentPosition := Count - 1;
    end;
  end;
end;

// Remove an item from the start of the list
procedure TVBList.RemoveHead;
begin
  if Count > 0 then m_StringList.Delete(0);

  // Update current position index
  if Count > 0 then
    m_CurrentPosition := 0
  else
    m_CurrentPosition := -1;
end;

// Remove an item from the end of the list
procedure TVBList.RemoveTail;
begin
  with m_StringList do begin
    if Count > 0 then Delete(Count - 1);
    if Count > 0 then
      m_CurrentPosition := Count - 1
    else
      m_CurrentPosition := -1;
  end;
end;

// Clear the list
procedure TVBList.RemoveAll;
begin
  m_StringList.Clear;
  m_CurrentPosition := -1;
end;

// Get the next element in the list
function TVBList.GetNext: String;
begin
  Result := '';
  with m_StringList do begin
    if m_CurrentPosition < Count then begin
      Inc(m_CurrentPosition);
      Result := Strings[m_CurrentPosition];
    end;
  end;
end;

// Get the previous element in the list
function TVBList.GetPrev: String;
begin
  Result := '';
  with m_StringList do begin
    if m_CurrentPosition > 0 then begin
      Dec(m_CurrentPosition);
      Result := Strings[m_CurrentPosition];
    end;
  end;
end;

// Return a count of the number of elements in the list
function TVBList.Count: Integer;
begin
  Result := m_StringList.Count;
end;

// Boolean function: IsListEmpty?
function TVBList.IsEmpty: Integer;
begin
  Result := 0;
  if m_StringList.Count = 0 then Result := -1;
end;

// Goto the head of the list
procedure TVBList.GotoHead;
begin
  m_CurrentPosition := 0;
end;

// Goto the tail of the list
procedure TVBList.GotoTail;
begin
  if m_StringList.Count = 0 then
    m_CurrentPosition := -1
  else
    m_CurrentPosition := m_StringList.Count - 1;
end;

// Get current entry from the list
function TVBList.GetCurrent: String;
begin
  Result := '';
  with m_StringList do
    if Count > 0 then Result := Strings[m_CurrentPosition];
end;

// Change the current entry in the list
procedure TVBList.SetCurrent(Const S: String);
begin
  if m_CurrentPosition <> -1 then
    m_StringList.Strings[m_CurrentPosition] := S;
end;

// Remove the current entry in the list
procedure TVBList.RemoveCurrent;
begin
  if m_CurrentPosition <> -1 then
    m_StringList.Delete(m_CurrentPosition);
end;

// Return the current index into the list
function TVBList.GetIndex: Integer;
begin
  Result := m_CurrentPosition;
end;

// Set the current index into the list: if it is a legal index
procedure TVBList.SetIndex(I: Integer);
begin
  if (I > -1) and (I < m_StringList.Count) then m_CurrentPosition := I;
end;

// Search for a string in the list starting from the current position
// Return the index to the string if found, -1 if not
function TVBList.FindFromCurrent(const S: String): Integer;
begin
  Result := -1;
  if m_CurrentPosition > -1 then
    // Find returns, if true, the index of the found element in m_CurrentPosition
    if m_StringList.Find(S, m_CurrentPosition) then
      Result := m_CurrentPosition;
end;

// Search for a string in the list starting from the start of the list
// Return the index to the string if found, -1 if not
function TVBList.FindFromStart(const S: String): Integer;
var
  I: Integer;
begin
  Result := -1;
  I := 0;
  with m_StringList do
    if Count > 0 then
      if Find(S, I) then Result := I;
end;

// Insert a new element before the current element
procedure TVBList.InsertBefore(const S: String);
begin
  if m_CurrentPosition > -1 then
    m_StringList.Insert(m_CurrentPosition, S);
end;

// Insert a new element after the current element
procedure TVBList.InsertAfter(const S: String);
begin
  if m_CurrentPosition > -1 then begin
    Inc(m_CurrentPosition);
    m_StringList.Insert(m_CurrentPosition, S);
  end;
end;

// Return a pointer to the contained IMyEnumVariant object
function TVBList.GetNewEnum: Variant;
begin
  // Create the object on the fly
  FEnumVariant := IMyEnumVariant.Create(Self);
  VarClear(Result);
  TVarData(Result).VType := varUnknown;
  TVarData(Result).VUnknown := FEnumVariant;
  FEnumVariant.Reset;
end;

// IEnumVariant object

// Create a new instance of the Enumerator by keeping a reference to the TStringList
// object that is contained in the TVBList object
constructor IMyEnumVariant.Create(pStringList: TVBList);
begin
  m_pVBList := pStringList;
  m_CurrentPosition := -1;
  FRefCount := 1;
end;

// Return the next celt elements from the Collection
function IMyEnumVariant.Next(celt: Integer; elt: PVariantArgList; pceltFetched: Integer): HResult;
var
  hr: HResult;
  I: Integer;
  OleStr: TBStr;
begin
  // Validate the arguments from the controller
  pceltFetched := 0;
  if pceltFetched <> 0 then
    pceltFetched := 0
  else if celt > 1 then begin
    Result := E_INVALIDARG;
    Exit;
  end;

  // Initialize the Variant Array
  for I := 0 to celt - 1 do
    VariantInit(Variant(elt^[I]));

  hr := S_OK;
  
  // Now attempt to return the number of elements that was requested
  for I := 0 to celt - 1 do begin
    if m_CurrentPosition < m_pVBList.Count then begin

      m_pVBList.SetIndex(m_CurrentPosition);
      OleStr := StringToOleStr(m_pVBList.GetCurrent);
      PVarData(@elt^[I])^.VType := varOleStr;
      PVarData(@elt^[I])^.VOleStr := OleStr;

      Inc(m_CurrentPosition);
      Inc(pceltFetched);
    end
    else
      hr := S_FALSE;
  end;

  Result := hr;
end;

function IMyEnumVariant.Skip(celt: Longint): HResult;
begin
end;

// Reset the current position in the list to the start of the list
function IMyEnumVariant.Reset: HResult;
begin
  m_CurrentPosition := 0;
  Result := S_OK;
end;

function IMyEnumVariant.Clone(var enum: IEnumVariant): HResult;
begin
end;

// Standard IUnknown interface implementation
function IMyEnumVariant.QueryInterface(const iid: TIID; var obj): HResult;
begin
  if IsEqualIID(iid, IID_IUnknown) or IsEqualIID(iid, IID_IEnumVariant) then begin
    Pointer(obj) := Self;
    AddRef;
    Result := S_OK;
  end
  else begin
    Pointer(obj) := nil;
    Result := E_NOINTERFACE;
  end;
end;

function IMyEnumVariant.AddRef: Longint;
begin
  Inc(FRefCount);
  Result := FRefCount;
end;

function IMyEnumVariant.Release: Longint;
begin
  Dec(FRefCount);
  Result := FRefCount;
  if FRefCount = 0 then Free;
end;

// Register the VBList object
procedure RegisterVBList;
const
  AutoClassInfo: TAutoClassInfo = (
    AutoClass: TVBList;
    ProgID: 'vblist.VBList';
    ClassID: '{61797060-37C4-11CF-AE93-80ECFDC00000}';
    Description: 'Visual Basic List Control';
    Instancing: acMultiInstance);
begin
  Automation.RegisterClass(AutoClassInfo);
end;

initialization
  RegisterVBList;
end.
