/* ------------------------------------------------------------------------ */
/* SND2MIDI.C (C) CopyLeft Bill Buckels 1993-1999                           */
/* All Rights Reversed.                                                     */
/*                                                                          */
/* Licence Agreement                                                        */
/* -----------------                                                        */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* If you don't agree, remove this source code from your computer now.      */
/*                                                                          */
/* Written by   : Bill Buckels                                              */
/*                589 Oxford Street                                         */
/*                Winnipeg, Manitoba, Canada R3M 3J2                        */
/*                                                                          */
/* Email: bbuckels@escape.ca                                                */
/* WebSite: http://www.escape.ca/~bbuckels                                  */
/*                                                                          */
/* Purpose      : Converts .SND files to .MID (MIDI) files.                 */
/*                Usage  is "SND2MIDI MySong.SND"                           */
/*                Output is MySong.MID                                      */
/*                                                                          */
/*                This utility will allow you to convert the sound files    */
/*                (.SND files) used by the author's programs (like          */
/*                WorkBook and Mickiano) to play a simple melody            */
/*                through the PC speaker, into an equivalently simple       */
/*                MIDI file that you can use in programs that playback      */
/*                MIDI through a sound card (i.e. Windows Media Player).    */
/*                                                                          */
/* Revision     : 2.0 Second Release                                        */
/* ------------------------------------------------------------------------ */
/* Written in Large Model MSC Version 6.00a                                 */
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include <io.h>
#include <string.h>

#define FALSE 0
#define TRUE  1

#define META_EVENT       0xff

/* meta event 1st data byte constants */
#define SEQUENCE_NUMBER  0
#define TEXT_EVENT       1
#define COPYRIGHT_NOTICE 2
#define TRACK_NAME       3
#define INSTRUMENT_NAME  4
#define LYRIC            5
#define MARKER           6
#define CUE_POINT        7
#define CHANNEL_PREFIX   0x20
#define END_OF_TRACK     0x2f
#define SET_TEMPO        0x51
#define SMPTE_OFFSET     0x54
#define TIME_SIGNATURE   0x58
#define KEY_SIGNATURE    0x59
#define SPECIFIC         0x7f

/* system common status byte constants */

#define SYSTEM_EXCLUSIVE       0xf0
#define MTC_QUARTER_FRAME      0xf1
#define SONG_POSITION_POINTER  0xf2
#define SONG_SELECT            0xf3
#define TUNE_REQUEST           0xf6
#define END_OF_EXCLUSIVE       0xf7

/* system real time status byte constants */

#define MIDI_CLOCK            0xf8
#define START_MIDI            0xfa
#define CONTINUE_MIDI         0xfb
#define STOP_MIDI             0xfc
#define ACTIVE_SENSING        0xfe
#define SYSTEM_RESET          0xff

/* midi channel status bytes */
/* midi channel equivalent 0-15 is used as a subscript here */

/* drum channel on 10th or 16th (subscript 9 or 15) */
unsigned char NOTE_OFF[16]={
          0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
          0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f};
unsigned char NOTE_ON[16]={
          0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
          0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f};
unsigned char POLY_KEY_PRESSURE[16]={
          0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
          0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf};
unsigned char CONTROL_CHANGE[16]={
          0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
          0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf};
unsigned char PROGRAM_CHANGE[16]={
          0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
          0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf};
unsigned char CHANNEL_PRESSURE[16]={
          0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
          0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf};
unsigned char PITCH_WHEEL_CHANGE[16]={
          0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
          0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef};


// MIDI Files are made up of chunks.  Each chunk has a 4-character type
// and a 32-bit length, which is the number of bytes in the chunk. MIDI
// Files contain two types of chunks: header chunks and track chunks.

// However, it just so happens that the "chunk metaphor" fits many if
// not most of the event "packets" within the midifile. As a case in
// point I have used some basic defined constants to capitalize on this
// "chunk" relationship (below)...

#define LONG_CHUNK_SIZE  4
#define LONG_SIZE        4L

#define SHORT_CHUNK_SIZE 2
#define SHORT_SIZE       2L

char *midi_header = "MThd";
char *midi_track = "MTrk";

/* array of voices for each channel mapped on the command line */
/* set to piano by default */
unsigned iVoices[16]={
          1, 1, 1, 1, 1, 1, 1, 1,
          1, 1, 1, 1, 1, 1, 1, 1};

/* function prototypes */

void MotorolaLong (unsigned long lval, unsigned char *psnMotorolaBuf);
void MotorolaShort (unsigned val, unsigned char *psnMotorolaBuf);
int FreqToMid (int i);
long WriteNoteToMidiFile (FILE *fp,
	 unsigned freq,
	 unsigned duration,
	 unsigned uiDurationBeforeNote,
	 int channel);
void DoTempo (unsigned long lval, unsigned char *psnBuf);
long WriteTempo (FILE *fp, unsigned uiSeconds);
long StringEvent (FILE *fp, unsigned uiEventType, char *buf);
long CopyLeft (FILE *fp, char *infile, char *outfile);
long TrackName (FILE *fp, int iTrackNum, int iVoice);
long SetVoice (FILE *fp, int voice, int channel);
long WriteKeySignature (FILE *fp);
long WriteTimeSignature (FILE *fp);
long WriteMidiHeader (FILE *fp, int NumTracks);
long WriteBeginningOfTrack (FILE *fp);
long MarkTheTrackLength (FILE *fp, long lSavePos, long lBodyChunkLength);
long WriteEndOfTrack (FILE *fp, unsigned delay);

/* translate an intel long to a motorola long */
void MotorolaLong(unsigned long lval, unsigned char *psnMotorolaBuf)
{
  int i;
  unsigned long accumulator;
  unsigned char temp;

  for(i = 3;i > - 1;i--)
  {
    accumulator = lval&255;
    temp = (unsigned char )accumulator;
    psnMotorolaBuf[i] = temp;
    lval >>= 8;
  }
}

/* translate an intel short to a motorola short */
void MotorolaShort(unsigned val, unsigned char *psnMotorolaBuf)
{
  int i;
  unsigned accumulator;
  unsigned char temp;
  
  for(i = 1;i > - 1;i--)
  {
    accumulator = val&255;
    temp = (unsigned char )accumulator;
    psnMotorolaBuf[i] = temp;
    val >>= 8;
  }
  
}

/* ------------------------------------------------------------------- */
/* FreqToMid                                                           */
/*                                                                     */
/* Translates a frequency within a range to a corresponding midi note. */
/* Does not attempt to do half-tones nor anything resembling a         */
/* microtonal note map. Sound effects proably wouldn't translate too   */
/* well using this function, but it is intended for music anyway.      */
/* ------------------------------------------------------------------- */
int FreqToMid(int i)
{
  /* input is note frequency */
  /* returns midi note       */
  
  /* note is not in range */
  if(i < 64) return - 1;
  if(i > 4096)return - 1;
  
  /* octave 0 */
  if(i < 127)
  {
    if(i < 67) return  36;
    if(i < 71) return  37;
    if(i < 76) return  38;
    if(i < 80) return  39;
    if(i < 85) return  40;
    if(i < 90) return  41;
    if(i < 96) return  42;
    if(i < 101) return  43;
    if(i < 107) return  44;
    if(i < 114) return  45;
    if(i < 120) return  46;
    return  47;
  }
  
  /* first octave */
  if(i < 255)
  {
    if(i < 136) return 48;
    if(i < 143) return 49;
    if(i < 152) return 50;
    if(i < 160) return 51;
    if(i < 170) return 52;
    if(i < 180) return 53;
    if(i < 190) return 54;
    if(i < 202) return 55;
    if(i < 215) return 56;
    if(i < 227) return 57;
    if(i < 240) return 58;
    return 59;
  }
  
  /* second octave */
  if(i < 512)
  {
    if(i < 270)return 60;
    if(i < 286)return 61;
    if(i < 304)return 62;
    if(i < 322)return 63;
    if(i < 340)return 64;
    if(i < 360)return 65;
    if(i < 382)return 66;
    if(i < 406)return 67;
    if(i < 428)return 68;
    if(i < 454)return 69;
    if(i < 480)return 70;
    return 71;
  }
  
  /* third octave */
  if(i < 1024)
  {
    if(i < 540)return 72;
    if(i < 572)return 73;
    if(i < 605)return 74;
    if(i < 643)return 75;
    if(i < 680)return 76;
    if(i < 720)return 77;
    if(i < 765)return 78;
    if(i < 810)return 79;
    if(i < 855)return 80;
    if(i < 910)return 81;
    if(i < 965)return 82;
    return 83;
  }
  
  /* fourth octave */
  if(i < 2048)
  {
    if(i < 1080) return 84;
    if(i < 1144) return 85;
    if(i < 1210) return 86;
    if(i < 1286) return 87;
    if(i < 1340) return 88;
    if(i < 1440) return 89;
    if(i < 1530) return 90;
    if(i < 1620) return 91;
    if(i < 1710) return 92;
    if(i < 1820) return 93;
    if(i < 1930) return 94;
    return 95;
  }
  
  /* fifth octave */
  /* sixth octave ignored */
  if(i < 2160)return 96;
  if(i < 2288)return 97;
  if(i < 2420)return 98;
  if(i < 2572)return 99;
  if(i < 2680)return 100;
  if(i < 2880)return 101;
  if(i < 3060)return 102;
  if(i < 3240)return 103;
  if(i < 3420)return 104;
  if(i < 3640)return 105;
  if(i < 3860)return 106;
  
  return 107;
  
}

/* ------------------------------------------------------------------- */
/* WriteNoteToMidiFile                                                 */
/*                                                                     */
/* Translates the frequency, duration pairs used in our .SND files to  */
/* midi noteon, noteoff sets, and writes these to the midifile.        */
/*                                                                     */
/* I could have made a more conscise version of this function and      */
/* put the various values in arrays etc. but it would have been        */
/* harder to read and to "muck about" with...                          */
/* ------------------------------------------------------------------- */

#define NORMAL_VELOCITY  127
#define LOW_VELOCITY     96

long WriteNoteToMidiFile(FILE *fp,
            unsigned freq,                   /* frequency of note */
            unsigned duration,               /* length of note */
            unsigned uiDurationBeforeNote,   /* length of pause before note */
            int channel)                     /* voice to map to */
{
  int note1, note2, note3, note4, note5;
  unsigned char cduration;

  long WriteNoteCount = 0L;

  /* we are 18.2 ticks per second  */
  /* use groups of 5 notes maximum */
  
  if((note1 = FreqToMid(freq)) == - 1)return 0L;  // get the note

  // avoid major/minor and melodic conflicts
  // use only full octaves for pseudo-chord
  // reduce velocity for chord

  note2 = note1 + 12;
  note3 = note1 - 12;

  note4 = note1 + 24;
  note5 = note1 - 24;
  
  // 1. use the pause between notes (if any)
  //    as a prolog to the first noteon message
  //    this has been saved in the value... "uiDurationBeforeNote"

  // 2. to control the note length
  //    precede the first noteoff with the "real duration"

  if(uiDurationBeforeNote > 127)uiDurationBeforeNote = 127;

  cduration = (unsigned char)uiDurationBeforeNote;    // pause this long
  
  // turn the note on
  fputc(cduration, fp);            // pause before note
  fputc(NOTE_ON[channel], fp);     // play root note
  fputc(note1, fp);
  fputc(NORMAL_VELOCITY, fp);      // volume
  WriteNoteCount += LONG_CHUNK_SIZE;

  if (note2 < 128) {
    fputc(0, fp);
    fputc(NOTE_ON[channel], fp);  // play one octave higher
    fputc(note2, fp);
    fputc(LOW_VELOCITY, fp);      // volume
    WriteNoteCount += LONG_CHUNK_SIZE;
  }

  if (note3 < 128) {
    fputc(0, fp);
    fputc(NOTE_ON[channel], fp);  // play one octave lower
    fputc(note3, fp);
    fputc(LOW_VELOCITY, fp);      // volume
    WriteNoteCount += LONG_CHUNK_SIZE;
  }
  
  // if we have a duration length greater than 1
  // add a couple more notes (2 octaves higher and lower)
  // i.e. give the longer notes a little fuller "chord"

  if(duration > 1)
  {
    if (note4 < 128) {
      fputc(0, fp);
      fputc(NOTE_ON[channel], fp);
      fputc(note4, fp);
      fputc(LOW_VELOCITY, fp);
      WriteNoteCount += LONG_CHUNK_SIZE;
    }
  
    if (note5 < 128) {
      fputc(0, fp);
      fputc(NOTE_ON[channel], fp);
      fputc(note5, fp);
      fputc(LOW_VELOCITY, fp);
      WriteNoteCount += LONG_CHUNK_SIZE;
    }
  }
  
  // turn the note off...

  if(duration > 127)duration = 127;
  cduration = (unsigned char)duration;

  fputc(cduration, fp);            // pause before noteoff
  fputc(NOTE_OFF[channel], fp);    // root note
  fputc(note1, fp);
  fputc(NORMAL_VELOCITY, fp);
  WriteNoteCount += LONG_CHUNK_SIZE;

  if (note2 < 128) {
    fputc(0, fp);
    fputc(NOTE_OFF[channel], fp);  // one octave higher
    fputc(note2, fp);
    fputc(LOW_VELOCITY, fp);
    WriteNoteCount += LONG_CHUNK_SIZE;
  }

  if (note3 < 128) {
    fputc(0, fp);
    fputc(NOTE_OFF[channel], fp);  // one octave lower
    fputc(note3, fp);
    fputc(LOW_VELOCITY, fp);
    WriteNoteCount += LONG_CHUNK_SIZE;
  }
  
  // if we have a duration length greater than 1
  // we added a couple more notes (2 octaves higher and lower)
  // turm these off too...
  if(duration > 1)
  {
    if (note4 < 128) {
      fputc(0, fp);
      fputc(NOTE_OFF[channel], fp);
      fputc(note4, fp);
      fputc(LOW_VELOCITY, fp);
      WriteNoteCount += LONG_CHUNK_SIZE;
    }
  
    if (note5 < 128) {
      fputc(0, fp);
      fputc(NOTE_OFF[channel], fp);
      fputc(note5, fp);
      fputc(LOW_VELOCITY, fp);
      WriteNoteCount += LONG_CHUNK_SIZE;
    }
  }

  return (WriteNoteCount);
  
}

/* ------------------------------------------------------------------- */
/*  DoTempo - used in the "control track"                              */
/*                                                                     */
/*  FF 51 03 tttttt    Set Tempo, microseconds per MIDI quarter-note   */
/*                                                                     */
/*  This event indicates a tempo change.  tttttt is a 24-bit value,    */
/*  stored most-significant-byte first.  Another way of putting        */
/*  "microseconds per quarter-note" is "24ths of a microsecond per     */
/*  MIDI clock".                                                       */
/*                                                                     */
/* ------------------------------------------------------------------- */
void DoTempo(unsigned long lval, unsigned char *psnBuf)
{
  // tempo is microseconds per midi quarter note
  int i;
  unsigned long accumulator;
  unsigned char temp;
  
  psnBuf[0] = META_EVENT;         // fill in the first half of the buffer
  psnBuf[1] = SET_TEMPO;
  psnBuf[2] = 3;
  for(i = 5;i > 2;i--)            // fill in the last half of the buffer
  {
    accumulator = lval&255;
    temp = (unsigned char )accumulator;
    psnBuf[i] = temp;
    lval >>= 8;
  }
  
}

/* ------------------------------------------------------------------- */
/* WriteTempo                                                          */
/*                                                                     */
/* Write the tempo value to file...                                    */
/* ------------------------------------------------------------------- */
long WriteTempo(FILE *fp, unsigned uiSeconds)
{

#define TEMPO_SIZE    6
#define TEMPO_RETURN  ((long )(TEMPO_SIZE + 1))

  char szTempoChunk[TEMPO_SIZE];
  long dwMicroSeconds = (long)(1000000L * uiSeconds);

  fputc(0, fp);
  DoTempo(dwMicroSeconds, szTempoChunk);        // microseconds...
  fwrite(szTempoChunk, TEMPO_SIZE, 1, fp);

  return TEMPO_RETURN;
}

/* ------------------------------------------------------------------- */
/* StringEvent                                                         */
/*                                                                     */
/* Create Various Text Events and Write Them Into The MidiFile.        */
/* ------------------------------------------------------------------- */
long StringEvent(FILE *fp, unsigned uiEventType, char *buf)
{

  unsigned len;
  long dwRetVal = 0L;

  // all text events are supported
  switch(uiEventType)
  {
    case TEXT_EVENT:
    case COPYRIGHT_NOTICE:
    case TRACK_NAME:
    case INSTRUMENT_NAME:
    case LYRIC:
    case MARKER:
    case CUE_POINT:

      len = strlen(buf);
      if (len > 0)
        break;
    default:
      return 0L;    // if nothing written, return 0 bytes

  }

  fputc(0, fp);                 // reset delta time
  fputc(META_EVENT,fp);         // super is meta event
  fputc(uiEventType, fp);       // text event type
  fputc(len,fp);                // string length

  // length of text event prefix...
  dwRetVal += LONG_SIZE;

  // write text event
  fwrite(buf, len, 1, fp);

  // return bytecount
  dwRetVal += len;
  return (dwRetVal);
  
}

/* ------------------------------------------------------------------- */
/* CopyLeft                                                            */
/*                                                                     */
/* Write the Producer's Credit into a text field.                      */
/* We can hardly claim copyright to the song...                        */
/* But our resourcefulness in creating midi files from nothing         */
/* deserves some credit:)                                              */
/* ------------------------------------------------------------------- */
long CopyLeft(FILE *fp, char *infile, char *outfile)
{

  char szNoticeBuf[256];
  
  // create text event
  sprintf(szNoticeBuf, "%s was converted to %s using SND2MIDI(C) "
                       "Copyright 1993-1999 Bill Buckels.",
                       infile, outfile);

  return StringEvent(fp, TEXT_EVENT, szNoticeBuf);
  
}

/* ------------------------------------------------------------------- */
/* TrackName                                                           */
/* ------------------------------------------------------------------- */
long TrackName(FILE *fp, int iTrackNum, int iVoice)
{
  char buf[40];
  
  if (iTrackNum == 0)
    strcpy(buf, "Control Track");
  else
    sprintf(buf, "Track%0003d - MidiVoice %d", iTrackNum, iVoice);

  return StringEvent(fp, TRACK_NAME, buf);
}

/* ------------------------------------------------------------------- */
/* SetVoice                                                            */
/*                                                                     */
/* Select alternate voices based on the General Midi instrument        */
/* patch assignment. Used at the beginning of each track.              */
/*                                                                     */
/* To cause the MIDI device to change to a particular Program (which   */
/* some devices refer to as Patch, or Instrument, or Preset, or        */
/* whatever). Most sound modules have a variety of instrumental        */
/* sounds, such as Piano, and Guitar, and Trumpet, and Flute, etc.     */
/* Each one of these instruments is contained in a Program. So,        */
/* changing the Program changes the instrumental sound that the MIDI   */
/* device uses when it plays Note On messages. Of course, other MIDI   */
/* messages also may modify the current Program's (ie, instrument's)   */
/* sound. But, the Program Change message actually selects which       */
/* instrument currently plays. There are 128 possible program          */
/* numbers, from 0 to 127. If the device is a MultiTimbral unit,       */
/* then it usually can play 16 "Parts" at once, each receiving data    */
/* upon its own MIDI channel. This message will then change the        */
/* instrument sound for only that Part which is set to the message's   */
/* MIDI channel.                                                       */
/*                                                                     */
/* ------------------------------------------------------------------- */
long SetVoice(FILE *fp, int voice, int channel)
{

  long lVoiceOffset = 0L;

  // default voice if out of range
  if (voice < 1 || voice > 127)
    voice = 1;                     // Bright Acoustic Piano

  // Bank Select

  // Some MIDI devices have more than 128 Programs (ie, Patches,
  // Instruments, Preset, etc). MIDI Program Change messages only support
  // switching between 128 programs.

  // So, Bank Select Controller (sometimes called Bank Switch) is sometimes
  // used to allow switching between groups of 128 programs. For example,
  // let's say that a device has 512 Programs. It may divide these into 4
  // banks of 128 programs apiece. So, if you want program #129, that would
  // actually be the first program within the second bank. You would send a
  // Bank Select Controller to switch to the second bank (ie, the first
  // bank is #0), and then follow with a Program Change to select the first
  // Program in this bank.


  /* Control Change - data bytes = 2 */
  fputc(0, fp);                           // delay = 0
  fputc(CONTROL_CHANGE[channel], fp);
  fputc(0, fp);                           // bank select
  fputc(0, fp);                           // select bank 0
  lVoiceOffset += LONG_CHUNK_SIZE;

  // NOTE: When a Bank Select is received, the MIDI module doesn't actually
  // change to a patch in the new bank. Rather, the Bank Select value is
  // simply stored by the MIDI module without changing the current patch.
  // Whenever a subsequent Program Change is received, the stored Bank
  // Select is then utilized to switch to the specified patch in the new
  // bank. For this reason, Bank Select must be sent before a Program
  // Change, when you desire changing to a patch in a different bank. (Of
  // course, if you simply wish to change to another patch in the same
  // bank, there is no need to send a Bank Select first).


  /* Program Change - data bytes = 1 */
  fputc(0, fp);                          // 0 delay
  fputc(PROGRAM_CHANGE[channel], fp);    // select instrument channel
  fputc(voice, fp);                      // assign GM voice to channel
  lVoiceOffset += 3;

  return (lVoiceOffset);                 // return bytes written

}

/* ------------------------------------------------------------------- */
/* WriteKeySignature                                                   */
/*                                                                     */
/* Write a "generic" key signature. Basically a placeholder.           */
/* ------------------------------------------------------------------- */
long WriteKeySignature(FILE *fp)
{

  fputc(0, fp);
  fputc(META_EVENT, fp);
  fputc(KEY_SIGNATURE, fp);
  fputc(2, fp);              // count 2 bytes of data
  fputc(0, fp);              // Key of C
  fputc(0, fp);              // Major Key

  return (6L);               // bytes written
  
}

/* ------------------------------------------------------------------- */
/* WriteTimeSignature                                                  */
/*                                                                     */
/* Write a "generic" timing signature. Basically a placeholder.        */
/* ------------------------------------------------------------------- */
long WriteTimeSignature(FILE *fp)
{

  fputc(0, fp);
  fputc(META_EVENT, fp);
  fputc(TIME_SIGNATURE, fp);
  fputc(4, fp);   // 4 bytes follow

  fputc(4, fp);   // nn numerator     4/4 time
  fputc(2, fp);   // dd denominator
  fputc(96, fp);  // cc clockclicks - midiclocks per metronome clicks
  fputc(8, fp);   // bb 32nd notes in a midi quarter note

  return (8L);               // bytes written
  
}




/* ------------------------------------------------------------------- */
/* WriteMidiHeader                                                     */
/*                                                                     */
/* Write The Midi File Header                                          */
/*                                                                     */
/* The header chunk at the beginning of the file specifies some        */
/* basic information about the data in the file.  Here's the syntax    */
/* of the complete chunk:                                              */
/*                                                                     */
/* <Header Chunk> = <chunk type><length><format><ntrks><division>      */
/*                                                                     */
/* As described above, <chunk type> is the four ASCII characters       */
/* 'MThd'; <length> is a 32-bit representation of the number 6 (high   */
/* byte first).                                                        */
/*                                                                     */
/* The data section contains three 16-bit words, stored                */
/* most-significant byte first.                                        */
/*                                                                     */
/* ------------------------------------------------------------------- */
long WriteMidiHeader(FILE *fp, int NumTracks)
{
  char buf[LONG_CHUNK_SIZE];

  long lPos = 0L;

  /* write header chunk */
  fwrite(midi_header, LONG_CHUNK_SIZE, 1, fp);
  lPos += LONG_SIZE;                                  // 4L

  /* write the length of the data portion of the header chunk */
  MotorolaLong(6L, buf);
  fwrite(buf, LONG_CHUNK_SIZE, 1, fp);
  lPos += LONG_SIZE;                                  // 8L
  
  /* data portion of header chunk starts... */

  // The first word, format, specifies the overall organization of the
  // file. Only three values of format are specified:
  //    0    the file contains a single multi-channel track
  //    1    the file contains one or more simultaneaous tracks
  //    2    the file contains one or more sequentially independent
  //         single-track patterns

  /* write file format 1 */
  MotorolaShort(1, buf);
  fwrite(buf, SHORT_CHUNK_SIZE, 1, fp);
  lPos += SHORT_SIZE;                                  // 10L
  
  /* write ??? tracks */
  /* The next word, ntrks, is the number of track chunks in the file. */

  MotorolaShort(NumTracks, buf);
  fwrite(buf, SHORT_CHUNK_SIZE, 1, fp);
  lPos += SHORT_SIZE;                                  // 12L

  // The third word, division, specifies the meaning of the delta-times.
  // It has two formats, one for metrical time, and one for time-code-based
  // time.  If bit 15 of division is zero, the bits 14 thru 0 represent the
  // number of delta-time "ticks" which make up a quarter-note.

  /* Write delta time resolution of 91 ticks per quarter note.      */
  /* 5 X the motorola timer tick of 18.2 ticks per second           */
  /* that we use in our .SND files.                                 */

  /* we also set the tempo at 5 seconds per quarter note.           */
  /* in track 1 (our control track)                                 */

  MotorolaShort(91, buf);
  fwrite(buf, SHORT_CHUNK_SIZE, 1, fp);
  lPos += SHORT_SIZE;                                  // 14L

  return (lPos);

}

/* ------------------------------------------------------------------- */
/* WriteBeginningOfTrack                                               */
/*                                                                     */
/* Write the standard 8 byte packet that marks the beginning of a      */
/* midi track.                                                         */
/* ------------------------------------------------------------------- */
long WriteBeginningOfTrack(FILE *fp)
{
  unsigned char snBuf[LONG_CHUNK_SIZE];
  long LengthOffset = 0L;

  /* write track header */
  fwrite(midi_track, LONG_CHUNK_SIZE, 1, fp);
  LengthOffset += LONG_SIZE;

  /* write the length of the track body chunk */
  /* 0 bytes for now... we will go back and change this */
  MotorolaLong(0L, snBuf);
  fwrite(snBuf, LONG_CHUNK_SIZE, 1, fp);

  return LengthOffset;

}

/* ------------------------------------------------------------------- */
/* MarkTheTrackLength                                                  */
/*                                                                     */
/* Rewind the file to the track length field for the current track     */
/* and mark the track length. This is done after the track is complete.*/
/*                                                                     */
/* Assumptions - the position of the track length field was previously */
/*               saved and the length of the track was calculated      */
/*               outside of this function...                           */
/* ------------------------------------------------------------------- */
long MarkTheTrackLength(FILE *fp, long lSavePos, long lBodyChunkLength)
{
    unsigned char snBuf[LONG_CHUNK_SIZE];

    /* rewind and mark the track length */
    rewind(fp);
    fseek(fp, lSavePos, SEEK_SET);
    MotorolaLong(lBodyChunkLength, snBuf);
    fwrite(snBuf, LONG_CHUNK_SIZE, 1, fp);

    return 4L;
}

/* ------------------------------------------------------------------- */
/* WriteEndOfTrack                                                     */
/*                                                                     */
/* Write the standard 4 byte packet that marks the ending of a         */
/* midi track.                                                         */
/* ------------------------------------------------------------------- */
long WriteEndOfTrack(FILE *fp, unsigned delay)
{

  if(delay > 127)delay = 127;

  /* mark the end of the track */
  fputc(delay, fp);              // delta time = ??? (last delay if any)
  fputc(META_EVENT, fp);         // status = META_EVENT
  fputc(END_OF_TRACK, fp);
  fputc(0, fp);

  return (4L);
}

int main(int argc, char **argv)
{
  FILE *fp, *MIDI_FILE;
  unsigned long dwChunkCtr=0L, dwSaveCtr=0L;
  unsigned char c;
  int frequency,
      duration,
      iDurationBeforeNote,
      iChannel,
      iNumChannels,
      iNumTracks,
      idx,
      bStrict;
  char infile[128], outfile[128], scratch[128], buf[128], *wordptr;
  
  puts("SND2MIDI(C) CopyLeft Bill Buckels 1993-1999\nAll Rights Reversed.");
  puts("PC speaker sound file to MIDI file converter.");
  
  if(argc < 2)
  {
    puts("Usage is: "
         "\"SND2MIDI [sndfile.snd] [patch1 patch2 patch3...] STRICT\"");
    puts("          Patches are entered on the command line as numeric args.");
    puts("          Patch Range 1-127 (General Midi Instrument Voice Numbers)");
    puts("Each Patch will occupy a channel on a separate consecutive track.");
    puts("The STRICT keyword will force timing signature and key signature.");
    puts("Required by some sequencers...");
    printf("Enter .SND SoundFile Name (Blank to Exit): ");
    gets(scratch);
    if (scratch[0] == 0)
      return (1);
    puts("Enter Instrument Patches (Blank When Done)");
  }
  else {
    strcpy(scratch, argv[1]);
  }

  bStrict = FALSE; // no strict creation unless requested...

  /* get the voice patches for the instruments from the command line */
  /* other command line args as well... */

  iNumChannels = 0;

  if (argc < 3) {
    if (argc == 1) {
      for (idx = 0; idx < 16;) {
        printf("Enter Channel %d : ", idx+1);
        gets(buf);
        if (strcmpi(buf, "STRICT") == 0) {
          bStrict = TRUE;  // strict file creation
          continue;
        }
        if (buf[0] < '1' || buf[0] > '9')
          break;
        iVoices[iNumChannels] = atoi(buf);
        iNumChannels++;
        idx++;
      }
    }
  }
  else {
    for (idx = 2; idx < argc; idx++) {
      if (argv[idx][0] < '1' || argv[idx][0] > '9') {
        if (strcmpi(argv[idx], "STRICT") == 0)
          bStrict = TRUE;  // strict file creation
        continue;
      }
      iVoices[iNumChannels] = atoi(argv[idx]);
      iNumChannels++;
      if (iNumChannels > 15)     // maximum channels = 16
        break;
    }
  }

  // by default we have one channel
  if (iNumChannels == 0)
    iNumChannels = 1;


  // number of tracks is control track + voice tracks
  iNumTracks = (iNumChannels + 1);

  /* open the files */
  strcat(scratch, ".");

  wordptr = strtok(scratch, ".");

  sprintf(infile, "%s.SND", scratch);
  sprintf(outfile, "%s.MID", scratch);
  
  if((fp = fopen(infile, "rb")) == NULL)
  {
    perror(infile);
    return (1);
  }

  /* don't overwrite... they may have a midi file that has been  */
  /* painstakenly editted of the same name as the rough one that */
  /* snd2midi creates... */

  if((MIDI_FILE = fopen(outfile, "rb")) != NULL) {
    fclose(MIDI_FILE);
    printf("%s already exists. Overwrite? (Y/N) ", outfile);
    if(0 == (c = toupper(getch())))getch();
    if (c == 'Y') {
      puts("Y");
    }
    else {
      puts("N");
      fclose(fp);
      return (1);
    }
  }

  /* we want to remove the old midi file completely... i have a */
  /* doubt in my mind whether old files would be appended, so safety play. */
  remove(outfile);
  if((MIDI_FILE = fopen(outfile, "wb+r")) == NULL)
  {
    perror(outfile);
    fclose(fp);
    return (1);
  }
  
  printf("infile  >> %s\n", infile);
  printf("outfile >> %s\n", outfile);

  // save position
  dwSaveCtr = WriteMidiHeader(MIDI_FILE, (iNumChannels+1));
  dwSaveCtr += WriteBeginningOfTrack(MIDI_FILE);          // 18L
  
  /* write body chunk */
  dwChunkCtr = 0L;

  /* track name = control track*/
  dwChunkCtr += TrackName(MIDI_FILE, 0, 0);

  /* sn2midi midifile creator signature */
  dwChunkCtr += CopyLeft(MIDI_FILE, infile, outfile);
  
  /* set the tempo at 5 seconds per quarter note */
  /* 5 X the motorola timer tick of 18.2 ticks per second */
  /* that we use in our .SND files. */
  /* Using delta time resolution of 91 ticks per quarter note. */

  dwChunkCtr += WriteTempo(MIDI_FILE, 5);

  if (TRUE == bStrict)  {
    dwChunkCtr += WriteTimeSignature(MIDI_FILE);
    dwChunkCtr += WriteKeySignature(MIDI_FILE);
  }

  dwChunkCtr += WriteEndOfTrack(MIDI_FILE, 0);

  /* rewind and mark the track length */
  MarkTheTrackLength(MIDI_FILE, dwSaveCtr, dwChunkCtr);

  // Write the music into the midifile
  // This is the part that makes the music tracks...

  for (iChannel = 0; iChannel < iNumChannels; iChannel++) {

    printf("        << Midi Track %d Data Checksum.", iChannel + 1);

    /* seek forward again to start of next track */
    dwSaveCtr += LONG_CHUNK_SIZE;  // add length of track length packet
    dwSaveCtr += dwChunkCtr;       // add length of data chunk
    dwChunkCtr = 0L;               // reset length of data chunk

    printf("\r%ld", dwChunkCtr);

    fseek(MIDI_FILE, dwSaveCtr, SEEK_SET);

    /* save the start of the track */
    /* we need to seek back to here to write the track body length */
    /* after we finish writing the track */

    dwSaveCtr +=  WriteBeginningOfTrack(MIDI_FILE);

    dwChunkCtr += TrackName(MIDI_FILE, (iChannel+1), iVoices[iChannel]);
    printf("\r%ld", dwChunkCtr);

    dwChunkCtr += SetVoice(MIDI_FILE, iVoices[iChannel], iChannel);
    printf("\r%ld", dwChunkCtr);

    /* convert from snd to midi */
    rewind(fp); // reset to start of .snd file at start of each loop

    iDurationBeforeNote = 0; // pause 0 ticks before note...

    /* I used the number -1 to encode the logical EOF in my sound files */
    while((frequency = getw(fp)) != - 1)
    {
      duration = fgetc(fp);

      /* I used the number 32767 to encode silence in my sound files */
      /* this is actually a "throwback" to the GWBASIC sound statement. */
      if(frequency != 32767)
      {
        dwChunkCtr += WriteNoteToMidiFile(MIDI_FILE,
                                        frequency, duration,
                                        iDurationBeforeNote, iChannel);

        iDurationBeforeNote = 0;       // reset saved delay after each note
        printf("\r%ld", dwChunkCtr);
      }
      else
      {
        /* for periods of silence, add to the space between notes       */
        /* this is added to the beginning of the noteon message         */
        /* for the root note...                                         */

        /* we save this away until the next valid note.                 */
        iDurationBeforeNote += duration;
      }
    }

    /* mark the end of the track */

    dwChunkCtr += WriteEndOfTrack(MIDI_FILE, iDurationBeforeNote);
    printf("\r%ld\n", dwChunkCtr);

    /* rewind and mark the track length */
    MarkTheTrackLength(MIDI_FILE, dwSaveCtr, dwChunkCtr);

  } // end of track loop

  fclose(fp);
  fclose(MIDI_FILE);
  puts("Done!");
  return 0;
}

