// Music Driver head file (MUSDRV.H)
//
// Written 1996, 1997 by Roland Acton - public domain.
//
// This file is part of the Game Music System 1.1 distribution.

// Music driver class contains the actual song playing code.

#ifndef INCLUDED_MUSDRV
#define INCLUDED_MUSDRV

#ifdef TARGET_WIN95
  #include <pc.h>
#else
  #ifdef TARGET_MSDOS
    #ifdef USING_GCC
      #include <dpmi.h>
      #include <pc.h>
      #ifdef COMPILING_MUSDRV
        #include <crt0.h>
        int _crt0_startup_flags = _CRT0_FLAG_NONMOVE_SBRK
          | _CRT0_FLAG_LOCK_MEMORY;
      #endif
    #endif
    #ifdef USING_WATCOM
*** Need to find out what the Watcom equivalent of _CRT0_FLAG_LOCK_MEMORY is.
*** _CRT0_FLAG_NONMOVE_SBRK prevents the base address of the code, data, and
*** stack segments from changing when memory is allocated. This memory
*** allocation style is necessary for interrupt routines.
    #endif
  #endif
#endif

#define TIMER_INTERRUPT_VECTOR 0x1C
#define TIMER_CONTROL_PORT 0x43
#define TIMER_COUNTER_PORT 0x40
#define AM_VIB_OFFSET 0x20
#define SCALING_VOLUME_OFFSET 0x40
#define A_D_OFFSET 0x60
#define S_R_OFFSET 0x80
#define OP1MOD_FEEDBACK_OFFSET 0xC0
#define WAVE_OFFSET 0xE0
#define FREQ_LO_OFFSET 0xA0
#define OCT_FREQ_OFFSET 0xB0
#define STEREO_LEFT_MASK 0x10
#define STEREO_RIGHT_MASK 0x20

#define VIBRATO_DIVISOR 6
   /* This setting changes the sweep of the vibrato. A lower value increases
      the vibrato depth. A value of 6 approximates the vibrato in a
      Protracker song. */

union oct_freq_union {
  unsigned char byte;
  struct {
    unsigned freq_hi: 2;
    unsigned octave: 3;
    unsigned key_on: 1;
  } bitmap;
};

struct freq_lo_hi {
  unsigned char freq_lo;
  oct_freq_union oct_freq;
};

class music_driver {
public:
  static unsigned char shadow_regs_left[256];
  static unsigned char shadow_regs_right[256];
     // Copy of the sb card's current register contents.
  static const unsigned char channel_offset[9+9];
  static const unsigned char percussion_channel_map[15];
  static const unsigned int note_freqs[13];
  static freq_lo_hi note_to_freq_table[127];
  static unsigned char last_note_number[MAX_TRACKS];
  static unsigned char channel_keyed_on[MAX_TRACKS];
  static unsigned char arpeggio_index[MAX_TRACKS];
  static unsigned char arpeggio_offset[MAX_TRACKS];
  static long int frequency_offset[MAX_TRACKS];
  static unsigned int channel_instrument[MAX_TRACKS];
     // The instrument number being played on the channel.
  static unsigned int accumulated_ticks;
     // Ticks spent updating the current line.
  static unsigned long int portamento_dest[MAX_TRACKS];
  static unsigned int last_portamento[MAX_TRACKS];
  static const unsigned int vibrato_table[32];
  static int vibrato_index[MAX_TRACKS];
  static int vibrato_offset[MAX_TRACKS];
  static unsigned int vibrato_speed[MAX_TRACKS];
  static unsigned int vibrato_depth[MAX_TRACKS];
  static int primary_volume[MAX_TRACKS];
     /* For FM tracking, the carrier volume. For sample tracking, the sample
        volume. Valid values are 0-63, with 63 being loudest. */
  static int secondary_volume[MAX_TRACKS];
     /* Used only in FM tracking, and is the modulator volume. If the
        channel is using frequency modulation, this variable is
        unnecessary. */
  static unsigned int volume_table[64][64];
     // A lookup table for volume controls.
#ifdef TARGET_WIN95
#else
  #ifdef USING_BORLAND
    static void interrupt (*old_timer_interrupt)(...);
    static void interrupt (*old_dsp_interrupt)(...);
  #endif
  #ifdef USING_GCC
    static _go32_dpmi_seginfo GMS_timer_interrupt, old_PM_timer_interrupt;
    static _go32_dpmi_seginfo GMS_dsp_interrupt, old_PM_dsp_interrupt;
  #endif
  #ifdef USING_WATCOM
    static void (__interrupt __far *old_timer_interrupt)();
    static void (__interrupt __far *old_dsp_interrupt)();
  #endif
#endif
  static volatile unsigned int should_song_play;
  static int jump_indicator;
     /* If not -1, jump to the indicated pattern when the current line is
        over. */
  static volatile unsigned int forbid_count;
     // If greater than zero, the interrupt routine should just exit.
#ifdef EDITOR_HOOKS
  static unsigned int play_block_mode;
  static unsigned int track_active[MAX_TRACKS];
  static unsigned int show_interrupt;
  static volatile unsigned int interrupt_ticks;
#endif
  static unsigned int sb_base_address;
  static unsigned int dma_channel;
  static const unsigned int dma_page_reg_num[4];
  static unsigned int sb_interrupt_number;
  static const unsigned int vector_table[16];
  static unsigned char pic1memory;
  static unsigned char pic2memory;
  static unsigned int is_sample_playing;
  static unsigned int is_high_speed_mode_on;
  static unsigned int sample_piece_instrument;
  static unsigned long int sample_piece_address;
  static unsigned long int sample_piece_length;
// Information about the sample the DSP interrupt should play next.
  static unsigned int subroutine_just_ended;
  static unsigned int handling_effects;
  static unsigned int effect_finished;
  static unsigned int global_volume;

  static void set_up_driver(song::sound_cards sound_sys,
    unsigned int sb_base, unsigned int sb_int_num, unsigned int dma_chan);
  static void write_adlib(unsigned char reg, unsigned char value);
  static void write_sb_left(unsigned char reg, unsigned char value);
  static void write_sb_right(unsigned char reg, unsigned char value);
  static void force_write_adlib(unsigned char reg, unsigned char value)
    {shadow_regs_left[reg] = value + 1;write_adlib(reg, value);}
  static void force_write_sb_left(unsigned char reg, unsigned char value)
    {shadow_regs_left[reg] = value + 1;write_sb_left(reg, value);}
  static void force_write_sb_right(unsigned char reg, unsigned char value)
    {shadow_regs_right[reg] = value + 1;write_sb_right(reg, value);}
/* The force_write_<card> routines will force a write even if the register
   already contains the value. */
  static void load_instrument(unsigned int inst_number, unsigned int channel);
  static void reset_card();
  static void set_up_card();
  static void play_one_note(unsigned char *note, unsigned int channel,
    unsigned int just_effects);
  static unsigned int check_for_portamento(unsigned char *note);
  static void write_frequency(unsigned int channel);
  static void set_volume(unsigned int channel, unsigned int new_volume);
  static void shift_volume(unsigned int channel, int volume_shift);
  static void do_volume(unsigned int channel, unsigned int mod2_volume,
    unsigned int mod1_volume = 9999);
  static void play_one_line(unsigned int just_effects);
  static void update();
  static void reset_accumulated_ticks() {accumulated_ticks = 1;
    jump_indicator = -1;}
  static unsigned int compute_timer_setting(unsigned int tempo);
  static unsigned int compute_tempo_milliseconds(unsigned int tempo);
#ifdef USING_BORLAND
  static void interrupt timer_int(...);
  static void interrupt dsp_int(...);
#endif
#ifdef USING_GCC
  static void timer_int();
  static void dsp_int();
#endif
#ifdef USING_WATCOM
  static void __interrupt __far music_driver::timer_int();
  static void __interrupt __far music_driver::dsp_int();
#endif
  static gms_function_return_codes wedge_player();
  static gms_function_return_codes remove_player();
  static gms_function_return_codes wedge_dsp_handler();
  static gms_function_return_codes remove_dsp_handler();
  static void set_timer(unsigned int new_setting);
  static void start_playing() {should_song_play = YES;}
  static void stop_playing() {should_song_play = NO;}
  static unsigned int is_song_playing() {return should_song_play;}
  static void forbid_playing() {forbid_count++;}
  static void permit_playing() {forbid_count--;}
  static void update_am_vib();
  static void write_dsp(unsigned char value);
  static unsigned char read_dsp();
  static gms_function_return_codes reset_dsp();
  static void cancel_sample_playback();
  static unsigned char find_time_constant(unsigned long int frequency);
  static void prepare_dma(unsigned long int address,
    unsigned long int length);
  static unsigned long int max_sample_piece_size
    (unsigned long int linear_address, unsigned long int sample_size);
  static void trigger_sample(unsigned int inst_num)
    {load_instrument(inst_num, 0);}
  static void set_global_volume(unsigned int new_volume);
};



// Checks to see if the given note has a portamento effect.
inline unsigned int music_driver::check_for_portamento(unsigned char *note) {

  do {
    if ((*note & 127) == 0x3)
      return YES;
    else
      note += 2;
  }
  while (*(note - 2) & 128);
  return NO;
}



// Computes the hardware timer setting that corresponds to the given tempo.
inline unsigned int music_driver::compute_timer_setting(unsigned int tempo) {

  return 0x2D8426 / (tempo + 45);
}



// Computes the number of milliseconds between updates at the given tempo.
inline unsigned int music_driver::compute_tempo_milliseconds
    (unsigned int tempo) {

  return 1000 / (1193180 / compute_timer_setting(tempo));
}



// Sets the hardware timer.
inline void music_driver::set_timer(unsigned int new_setting) {

  outportb(TIMER_CONTROL_PORT, 0x3C);
  outportb(TIMER_COUNTER_PORT, new_setting);
  outportb(TIMER_COUNTER_PORT, new_setting >> 8);
}



// Writes the overall_am_vib byte to the sound card.
inline void music_driver::update_am_vib() {

  if (song::soundsystem == song::adlib)
    write_adlib(0xBD, song::overall_am_vib.byte);
  else
    write_sb_left(0xBD, song::overall_am_vib.byte);
}



// Writes the given value to the DSP.
inline void music_driver::write_dsp(unsigned char value) {

  while (inportb(sb_base_address + 0xC) & 128);
  outportb(sb_base_address + 0xC, value);
}



// Reads a value from the DSP.
inline unsigned char music_driver::read_dsp() {

  while (!(inportb(sb_base_address + 0xE) & 128));
  return inportb(sb_base_address + 0xA);
}



// Stops the currently playing sample.
inline void music_driver::cancel_sample_playback() {

  if (is_sample_playing) {
    if (is_high_speed_mode_on)
// When in high speed mode, the DSP can only be stopped by a reset.
      reset_dsp();
    else
      write_dsp(0xD0);
    is_sample_playing = NO;
  }
}



// Determines the time constant which corresponds to the given frequency.
inline unsigned char music_driver::find_time_constant
    (unsigned long int frequency) {

  return (65536 - (256000000 / frequency)) >> 8;
}



// Sets the DMA hardware up for a transfer.
inline void music_driver::prepare_dma(unsigned long int address,
    unsigned long int length) {

// The address must be a linear ("flat") address.
// Length needs to be a long int because 65536 can be passed.
/* DMA transfers cannot cross page boundries (page size is 64K for 8-bit
   DMA). This routine assumes that legal values have been passed! */
  outportb(0x0A, 4 + dma_channel);
  outportb(0x0C, 0);
  outportb(0x0B, 0x48 + dma_channel);
  outportb(dma_channel << 1, address);
  outportb(dma_channel << 1, address >> 8);
  outportb((dma_channel << 1) + 1, length - 1);
  outportb((dma_channel << 1) + 1, (length - 1) >> 8);
  outportb(dma_page_reg_num[dma_channel], address >> 16);
  outportb(0x0A, dma_channel);
}



/* The DMA hardware can't transfer across page boundries. This routine
   figures out the maximum amount that can be transferred without crossing a
   page. */
inline unsigned long int music_driver::max_sample_piece_size
    (unsigned long int linear_address, unsigned long int sample_size) {
  unsigned long int next_page;

  next_page = (linear_address & 0xFFFF0000) + 0x10000;
  if (linear_address + sample_size > next_page)
    return next_page - linear_address;
  else
    return sample_size;
}



#ifdef COMPILING_MUSDRV
const unsigned char music_driver::channel_offset[9+9] =
    {0x00, 0x01, 0x02, 0x08, 0x09, 0x0a, 0x10, 0x11, 0x12,
     0x03, 0x04, 0x05, 0x0b, 0x0c, 0x0d, 0x13, 0x14, 0x15};
// Lookup table for the sb card's weird register configuration

const unsigned char music_driver::percussion_channel_map[15] =
    {0, 1, 2, 3, 4, 5, 9, 10, 11, 12, 13, 14, 15, 16, 17};
// The percussion-mode correspondence between tracks and melodic channels.

const unsigned int music_driver::note_freqs[13] =
    {0x157, 0x16B, 0x181, 0x198, 0x1B0, 0x1CA, 0x1E5, 0x202,
     0x220, 0x241, 0x263, 0x287, 0x2AE};
// Frequencies of notes, from C (octave x) to C (octave x+1)

const unsigned int music_driver::vibrato_table[32] =
    {0, 24, 49, 74, 97, 120, 141, 161,
     180, 197, 212, 224, 235, 244, 250, 253,
     255, 253, 250, 244, 235, 224, 212, 197,
     180, 161, 141, 120, 97, 74, 49, 24};
// This is the vibrato table Protracker uses.

const unsigned int music_driver::dma_page_reg_num[4] =
    {0x87, 0x83, 0x81, 0x82};
// I/O port numbers for the page registers of the first four DMA channels

const unsigned int music_driver::vector_table[16] =
    {0, 0, 0xA, 0, 0, 0xD, 0, 0xF, 0, 0, 0x72, 0, 0, 0, 0, 0};
/* The actual interrupt vectors which correspond to the interrupt numbers.
   Only 2, 5, 7, and 10 are valid for Soundblaster cards. */

  unsigned char music_driver::shadow_regs_left[256];
  unsigned char music_driver::shadow_regs_right[256];
  freq_lo_hi music_driver::note_to_freq_table[127];
  unsigned char music_driver::last_note_number[MAX_TRACKS];
  unsigned char music_driver::channel_keyed_on[MAX_TRACKS];
  unsigned char music_driver::arpeggio_index[MAX_TRACKS];
  unsigned char music_driver::arpeggio_offset[MAX_TRACKS];
  long int music_driver::frequency_offset[MAX_TRACKS];
  unsigned int music_driver::channel_instrument[MAX_TRACKS];
  unsigned int music_driver::accumulated_ticks;
  unsigned long int music_driver::portamento_dest[MAX_TRACKS];
  unsigned int music_driver::last_portamento[MAX_TRACKS];
  int music_driver::vibrato_index[MAX_TRACKS];
  int music_driver::vibrato_offset[MAX_TRACKS];
  unsigned int music_driver::vibrato_speed[MAX_TRACKS];
  unsigned int music_driver::vibrato_depth[MAX_TRACKS];
  int music_driver::primary_volume[MAX_TRACKS];
  int music_driver::secondary_volume[MAX_TRACKS];
  unsigned int music_driver::volume_table[64][64];
#ifdef TARGET_WIN95
#else
  #ifdef USING_BORLAND
    void interrupt (*music_driver::old_timer_interrupt)(...);
    void interrupt (*music_driver::old_dsp_interrupt)(...);
  #endif
  #ifdef USING_GCC
    _go32_dpmi_seginfo music_driver::GMS_timer_interrupt,
      music_driver::old_PM_timer_interrupt;
    _go32_dpmi_seginfo music_driver::GMS_dsp_interrupt,
      music_driver::old_PM_dsp_interrupt;
  #endif
  #ifdef USING_WATCOM
    void (__interrupt __far *music_driver::old_timer_interrupt)();
    void (__interrupt __far *music_driver::old_dsp_interrupt)();
  #endif
#endif
  volatile unsigned int music_driver::should_song_play;
  int music_driver::jump_indicator;
  volatile unsigned int music_driver::forbid_count;
#ifdef EDITOR_HOOKS
  unsigned int music_driver::play_block_mode;
  unsigned int music_driver::track_active[MAX_TRACKS];
  unsigned int music_driver::show_interrupt;
  volatile unsigned int music_driver::interrupt_ticks;
#endif
  unsigned int music_driver::sb_base_address;
  unsigned int music_driver::dma_channel;
  unsigned int music_driver::sb_interrupt_number;
  unsigned char music_driver::pic1memory;
  unsigned char music_driver::pic2memory;
  unsigned int music_driver::is_sample_playing;
  unsigned int music_driver::is_high_speed_mode_on;
  unsigned int music_driver::sample_piece_instrument;
  unsigned long int music_driver::sample_piece_address;
  unsigned long int music_driver::sample_piece_length;
  unsigned int music_driver::subroutine_just_ended;
  unsigned int music_driver::handling_effects;
  unsigned int music_driver::effect_finished;
  unsigned int music_driver::global_volume;
#endif   // #ifdef COMPILING_MUSDRV

#ifdef EDITOR_HOOKS
  #ifdef COMPILING_MUSDRV
    music_driver music;
  #else
    extern music_driver music;
  #endif
#endif

#endif   // #ifndef INCLUDED_MUSDRV
