IMPLEMENTATION MODULE VgaLib2;

(*  This is FREE software, as described in the GNU General Public Licences.
    Therefore it comes WITH THE FULL SOURCES. Please feel free to improve this
    code when necessary. You OWN this code. 

    I would appreciate that people let in this message when extending this
    library, as a small tribute to me (for laying the foundation).

    In case people need extra information, contact me via:

        snail mail: Jan Verhoeven, 5012 GH 272, The Netherlands
   electronic mail: jverhoeven@bigfoot.com

    I remain full copyrights to these sources. If you want to send me a small
    "thanks", please send me a postcard of your hometown to the above shown
    snailmail address. Yes it is in code; the internal code of our national
    mail deliverer.

    Use this software at your own risk. Please find yourself a GNU GPL if you
    are in any doubt. I use these functions for all my own software, but there
    is NO GUARANTEE OF ANY KIND covering it.            *)

FROM    barith      IMPORT  shr;
FROM    SYSTEM      IMPORT  ASSEMBLER, ADDRESS;
FROM    FileSystem  IMPORT  File, Lookup, Close, ReadNBytes, Response;
FROM    InOut       IMPORT  WriteString;
FROM    Strings     IMPORT  Length;
FROM    Storage     IMPORT  Available, ALLOCATE, DEALLOCATE;
FROM    System      IMPORT  Terminate;

VAR     fp      : File;
        Bread   : CARDINAL;
        BitMap  : ADDRESS;


PROCEDURE   SetVGA;         (*  Set screen to VGA mode 12h: 640 x 480.  *)

BEGIN
    ASM
        PUSH BP
        MOV  AX, 0012H
        INT  010H
        POP  BP
    END;
END SetVGA;


PROCEDURE   SetText;        (*  Set screen to 80 x 25 textmode      *)
BEGIN
    ASM
        PUSH BP
        MOV  AX, 0003H
        INT  010H
        POP  BP
    END;
END SetText;


PROCEDURE   SetColour (Foreground, Background   : COLOUR);
                            (*  Define colour to work with.         *)

VAR     Col     : CARDINAL;

BEGIN
    Col := ORD (Foreground) + 16 * ORD (Background);
    ASM
        MOV  DX, 03C4H          (* VGA controller port *)
        MOV  AH, Col
        MOV  AL, 2
        OUT  DX, AX
    END;
END SetColour;


PROCEDURE   SetMask (Mask : CHAR);          (*  Set up mask for plotting in VGA memory.  *)

BEGIN
    ASM
        MOV  DX, 03CEH          (* VGA controller port *)
        MOV  AH, Mask
        MOV  AL, 8
        OUT  DX, AX
    END;
END SetMask;


PROCEDURE   Plot (VAR InWin : WinData);     (*  Plot point on CurX, CurY.   *)

VAR x, y    : CARDINAL;

BEGIN
    x := InWin.CurX + InWin.TopX;
    y := InWin.CurY + InWin.TopY;

    ASM
        MOV  AX, 0A000H
        MOV  ES, AX         (* Set up segment register *)
        MOV  CX, x
        AND  CX, 7          (* Which bit to plot? *)
        MOV  AH, 80H
        SHR  AH, CL         (* Compose plotting mask *)
        MOV  AL, 8
        MOV  DX, 03CEH
        OUT  DX, AX         (* Set plottingmask *)
        MOV  AX, y          (* Calculate offset in Video RAM *)
        MOV  BX, AX
        ADD  AX, AX
        ADD  AX, AX
        ADD  AX, BX         (* AX := 5 * Y *)
        MOV  CL, 4
        SHL  AX, CL         (* AX := 16 * 5 * Y *)
        MOV  BX, x
        SHR  BX, 1
        SHR  BX, 1
        SHR  BX, 1
        ADD  BX, AX         (* plus X / 8 *)
        MOV  AL, ES:[BX]
        MOV  AL, 0FFH
        MOV  ES:[BX], AL    (* and plot it *)
    END;
END Plot;


PROCEDURE UnPlot (VAR InWin : WinData);         (*  Erase pixel on CurX, CurY.  *)

VAR x, y    : CARDINAL;

BEGIN
    x := InWin.CurX + InWin.TopX;
    y := InWin.CurY + InWin.TopY;

    ASM
        MOV  AX, 0A000H
        MOV  ES, AX         (* Set up segment register *)
        MOV  CX, x
        AND  CX, 7          (* Which bit to plot? *)
        MOV  AH, 80H
        SHR  AH, CL         (* Compose plotting mask *)
        MOV  AL, 8
        MOV  DX, 03CEH
        OUT  DX, AX         (* Set plottingmask *)
        MOV  AX, y          (* Calculate offset in Video RAM *)
        MOV  BX, AX
        ADD  AX, AX
        ADD  AX, AX
        ADD  AX, BX         (* AX := 5 * Y *)
        MOV  CL, 4
        SHL  AX, CL         (* AX := 16 * 5 * Y *)
        MOV  BX, x
        SHR  BX, 1
        SHR  BX, 1
        SHR  BX, 1
        ADD  BX, AX         (* plus X / 8 *)
        MOV  AL, ES:[BX]
        MOV  AL, 0
        MOV  ES:[BX], AL    (* and erase it *)
    END;
END UnPlot;


PROCEDURE DrawH (VAR InWin : WinData; Flag : BOOLEAN);

(*  Draw a horizontal line from CurX, CurY for DeltaX pixels.  *)

VAR Index, Stop         : CARDINAL;
    x, dx, y, Kval      : CARDINAL;
    Emask, Lmask, Val   : CHAR;

BEGIN
    IF Flag THEN        (* Flag = TRUE => Plot, else UnPlot *)
        Val := 0FFX;
    ELSE
        Val := 0X;
    END;
    IF InWin.DeltaX < 18 THEN
        FOR Index := 0 TO InWin.DeltaX DO       (* For short lines *)
            Plot (InWin);
            INC (InWin.CurX);
        END;
    ELSE
         x := InWin.TopX + InWin.CurX;          (* For long lines *)
         y := InWin.TopY + InWin.CurY;
        dx := InWin.DeltaX;
        ASM
            MOV  AX, 0A000H
            MOV  ES, AX         (* Set up segment register *)
            MOV  CX, x
            AND  CX, 7
            MOV  BX, 8
            SUB  BX, CX
            MOV  AL, 0FFH
            SHR  AL, CL
            MOV  Emask, AL      (* compose plotting mask *)
            MOV  CX, dx
            SUB  CX, BX
            MOV  AX, CX
            AND  AX, 7
            PUSH AX             (* Save L-val *)
            SUB  CX, AX
            SHR  CX, 1
            SHR  CX, 1
            SHR  CX, 1
            MOV  Kval, CX
            MOV  AL, 0
            POP  CX             (* retrieve L-val *)
            JCXZ L0
            MOV  AL, 080H
        L0: DEC  CX
            SAR  AL, CL
            MOV  Lmask, AL

            MOV  AX, y              (* Calculate offset in Video RAM *)
            MOV  BX, AX
            ADD  AX, AX
            ADD  AX, AX
            ADD  AX, BX             (* AX := 5 * Y *)
            MOV  CL, 4
            SHL  AX, CL             (* AX := 16 * 5 * Y *)
            MOV  BX, x
            SHR  BX, 1
            SHR  BX, 1
            SHR  BX, 1
            ADD  BX, AX             (* plus X / 8 *)

            MOV  AH, Emask
            MOV  DX, 03CEH
            MOV  AL, 8
            OUT  DX, AX             (* Set plotting mask *)

            MOV  AL, Val
            MOV  AH, ES:[BX]
            MOV  ES:[BX], AL        (* Do the plotting ... *)

            INC  BX
            MOV  CX, Kval
            JCXZ L2
            MOV  AX, 0FF08H
            OUT  DX, AX
            MOV  AH, Val
        L1: MOV  AL, ES:[BX]
            MOV  ES:[BX], AH
            INC  BX
            LOOP L1
        L2: MOV  AH, Lmask
            MOV  AL, 8
            OUT  DX, AX
            MOV  AL, ES:[BX]
            MOV  AL, Val
            MOV  ES:[BX], AL
        END;
        INC (InWin.CurX, dx);
    END;
END DrawH;


PROCEDURE DrawV (VAR InWin : WinData);
  
(*  Draw a vertical line from CurX, CurY for DeltaY pixels. 100% optimized for speed.  *)

VAR x, y, dy    : CARDINAL;

BEGIN
     x := InWin.CurX + InWin.TopX;
     y := InWin.CurY + InWin.TopY;
    dy := InWin.DeltaY;
    ASM
        MOV  AX, 0A000H
        MOV  ES, AX         (* Set up segment register *)
        MOV  CX, x
        AND  CX, 7          (* Which bit to plot? *)
        MOV  AH, 80H
        SHR  AH, CL         (* Compose plotting mask *)
        MOV  AL, 8
        MOV  DX, 03CEH
        OUT  DX, AX         (* Set plottingmask *)
        MOV  AX, y          (* Calculate offset in Video RAM *)
        MOV  BX, AX
        ADD  AX, AX
        ADD  AX, AX
        ADD  AX, BX         (* AX := 5 * Y *)
        MOV  CL, 4
        SHL  AX, CL         (* AX := 16 * 5 * Y *)
        MOV  BX, x
        SHR  BX, 1
        SHR  BX, 1
        SHR  BX, 1
        ADD  BX, AX         (* plus X / 8 *)
        MOV  CX, dy
    L0: MOV  AL, ES:[BX]
        MOV  AL, 0FFH
        MOV  ES:[BX], AL    (* and plot it *)
        ADD  BX, 80
        LOOP L0
    END;
    INC (InWin.CurY, dy);
END DrawV;


PROCEDURE PlotChar (VAR InWin : WinData; Letter : CHAR);

(*  Plot character on InWin.(CurX,CurY).    *)

VAR xpos,   ypos,   MapOfs,
    VGApos, VGAseg, Pmask   : CARDINAL;
    Cval                    : CHAR;

BEGIN
    IF Letter = 0AX THEN
        INC (InWin.CurY, 16);           (* Process LF *)
        RETURN;
    END;
    IF Letter = 0DX THEN
        InWin.CurX := InWin.Indent;     (* Process CR *)
        RETURN;
    END;
    IF InWin.CurX >= InWin.Width - ChrWid THEN
        InWin.CurX := InWin.Indent;
        INC (InWin.CurY, 16);
    END;
    xpos := InWin.CurX + InWin.TopX;
    ypos := InWin.CurY + InWin.TopY;
    VGApos := 80 * ypos + shr (xpos, 3);
    VGAseg := 0A000H;
    MapOfs := ORD (Letter) * 16;
    ASM
        PUSH ES             (* save ES *)
        MOV  CX, xpos
        AND  CX, 7
        MOV  Cval, CL       (* nr of bits "off center" *)
        MOV  BX, 0FF00H
        SHR  BX, CL
        MOV  Pmask, BX      (* mask to use for left and right halves *)
        MOV  AX, BX
        MOV  AL, 8
        MOV  DX, 03CEH
        OUT  DX, AX         (* set plotting mask for left part *)
        MOV  CX, 16
        MOV  BX, VGApos
        LES  SI, BitMap     (* here are the pixels that make the tokens *)
        ADD  SI, MapOfs
    L0: PUSH CX
        LES  AX, BitMap     (* load ES, AX is just scrap *)
        MOV  AH, ES:[SI]    (* load pattern *)
        MOV  CL, Cval
        SHR  AX, CL         (* compose left half *)
        MOV  ES, VGAseg
        MOV  AL, ES:[BX]
        MOV  ES:[BX], AH    (* and "print" it *)
        ADD  BX, 80         (* point to next row *)
        INC  SI             (* and next pixel pattern *)
        POP  CX
        LOOP L0             (* repeat until done *)
        MOV  AX, Pmask
        CMP  AL, 0          (* if Cval = 0 => perfect allignment *)
        JE   ex             (*   skip second half *)
        XCHG AH, AL         (* else repeat the story once more *)
        MOV  AL, 8
        OUT  DX, AX         (* set up mask for right half *)
        MOV  CX, 16
        SUB  BX, 1279       (* 16 x 80 - 1 *)
        SUB  SI, CX
    L1: PUSH CX
        LES  AX, BitMap
        MOV  AH, ES:[SI]
        MOV  AL, 0
        MOV  CL, Cval
        SHR  AX, CL
        MOV  ES, VGAseg
        MOV  AH, ES:[BX]
        MOV  ES:[BX], AL
        ADD  BX, 80
        INC  SI
        POP  CX
        LOOP L1
    ex: POP  ES
    END;
    INC (InWin.CurX, ChrWid);   (* point to next printing position *)
END PlotChar;


PROCEDURE PlotText (VAR InWin : WinData; String : ARRAY OF CHAR);

(*  Print a string of text to the screen.  *)

VAR n   : CARDINAL;

BEGIN
    n := 0;
    SetColour (InWin.TexCol, InWin.BckCol);
    WHILE (n <= HIGH (String)) & (String [n] <> 0X) DO
        PlotChar (InWin, String [n]);
        INC (n);
    END;
END PlotText;


PROCEDURE Center (InWin : WinData; String : ARRAY OF CHAR);

(*  Center a line of text in InWin box.     *)

VAR Pixels, Chars, Spaces,
    Index, Between          : CARDINAL;
    Letter                  : CHAR;

BEGIN
    Pixels := InWin.Width - 2 * InWin.Indent - InWin.CurX;
    Spaces := 0;
    Chars  := 0;
    Index  := 0;
    WHILE (Index <= HIGH (String)) & (String [Index] <> 0X) DO
        Letter := String [Index];
        IF Letter = ' ' THEN
            INC (Spaces);           (* space counter *)
        ELSE
            INC (Chars, ChrWid);    (* pixels needed for letters *)
        END;
        INC (Index);
    END;

    IF Spaces > 0 THEN
        Between := (Pixels - Chars) DIV Spaces;     (* calculate spacing *)
        SetColour (InWin.MnuCol, InWin.BckCol)
    END;

    Index := 0;
    WHILE (Index <= HIGH (String)) & (String [Index] <> 0X) DO
        Letter := String [Index];
        CASE Letter OF
            ' ' :   INC (InWin.CurX, Between); |
            '_' :   PlotChar (InWin, ' ')       (* Underscore is printed as space *)
        ELSE
            PlotChar (InWin, Letter)            (* letters are printed as such *)
        END;
        INC (Index)
    END
END Center;


PROCEDURE ClickBar (InWin : WinData; String : ARRAY OF CHAR; VAR ClickPoints : ARRAY OF CARDINAL);

(*  Center a line of text in InWin box and mark the clickpoints for the mouse.  *)

VAR Pixels, Chars, Spaces, Count,
    Index, Between                  : CARDINAL;
    Letter                          : CHAR;

BEGIN
    Pixels := InWin.Width - 2 * InWin.Indent - InWin.CurX;
    Spaces := 0;
    Chars  := 0;
    Index  := 0;
    WHILE (Index <= HIGH (String)) & (String [Index] <> 0X) DO
        Letter := String [Index];
        IF Letter = ' ' THEN
            INC (Spaces);           (* space counter *)
        ELSE
            INC (Chars, ChrWid);    (* pixels needed for letters *)
        END;
        INC (Index);
    END;

    Between := (Pixels - Chars) DIV Spaces;     (* calculate spacing *)
    SetColour (InWin.MnuCol, InWin.BckCol);

    Index := 0;
    Count := 0;
    WHILE (Index <= HIGH (String)) & (String [Index] <> 0X) DO
        Letter := String [Index];
        CASE Letter OF
            '_' :   PlotChar (InWin, ' ');  |   (* Underscore is printed as space *)
            ' ' :   IF Count > 0 THEN
                        ClickPoints [Count] := InWin.CurX;
                        INC (Count);
                        ClickPoints [Count] := InWin.CurY;
                        INC (Count);
                    END;
                    INC (InWin.CurX, Between);
                    ClickPoints [Count] := InWin.CurX;
                    INC (Count);
                    ClickPoints [Count] := InWin.CurY;
                    INC (Count);
        ELSE
            PlotChar (InWin, Letter);           (* letters are printed as such *)
        END;
        INC (Index);
    END;
END ClickBar;


PROCEDURE MakeBox (InWin : WinData);

(*  Make a box on screen starting at (TopX, TopY).  *)

BEGIN
    InWin.CurX := 0;
    InWin.CurY := 0;                        (* Make sure pointers are correct *)
    InWin.DeltaX := InWin.Width - 1;
    InWin.DeltaY := InWin.Height - 1;       (* setup parameters for drawing lines *)
    SetColour (InWin.BoxCol, InWin.BckCol);
    DrawH (InWin, TRUE);        (* draw horizontal line *)
    DrawV (InWin);              (* draw vertical line   *)
    InWin.CurX := 0;
    InWin.CurY := 1;            (* adjust coordinates   *)
    DrawV (InWin);              (* draw last vertical line  *)
    DEC (InWin.CurY);
    INC (InWin.CurX);           (* adjust coordinates once more *)
    DrawH (InWin, TRUE);        (* draw final line  *)
END MakeBox;


PROCEDURE EraseBox (InWin : WinData);

(*  Fill inside of the InWin box with BLACK.    *)

VAR k       : CARDINAL;

BEGIN
    SetColour (WHITE, black);
    InWin.DeltaX := InWin.Width - 2;    (* Do not erase window borders  *)
    InWin.CurY := 0;                    (* prime Y coordinate *)
    FOR k := 3 TO InWin.Height DO
        InWin.CurX := 1;                (* prepare to ... *)
        INC (InWin.CurY);
        DrawH (InWin, FALSE);           (* ... UNdraw a line *)
    END;
END EraseBox;


PROCEDURE FillBox (InWin : WinData);

(*  Fill inside of the InWin box with BckCol.    *)

VAR k       : CARDINAL;

BEGIN
    SetColour (InWin.BckCol, InWin.BckCol);     (* do not overwrite window borders  *)
    InWin.DeltaX := InWin.Width - 2 ;
    InWin.CurY := 0;
    FOR k := 3 TO InWin.Height DO
        InWin.CurX := 1;                (* fill up by ...   *)
        INC (InWin.CurY);
        DrawH (InWin, TRUE);            (* ... drawing lines in succession. *)
    END;
END FillBox;


PROCEDURE WriteNumber (VAR InWin : WinData; Num, Len : CARDINAL);

(*  Print Num right justified in Len positions. *)

VAR Digit   : CHAR;                     (* Most recent digit    *)
    n       : CARDINAL;                 (* Loop counter         *)
    Store   : ARRAY [0..8] OF CHAR;     (* Store result         *)

BEGIN
    FOR n := 0 TO Len DO
        Store [n] := ' ';       (* Clear character buffer   *)
    END;
    n := Len;                   (* Start from the back      *)
    REPEAT
        Digit := CHR ((Num MOD 10) + ORD ("0"));                (* compose numeral  *)
        DEC (n);                                                (* prepare index    *)
        Store [n] := Digit;                         (* temporaly store numeral      *)
        Num := Num DIV 10;                          (* prepaer number for next try  *)
    UNTIL (Num = 0) OR (n = 0);                     (* until ready or string full   *)
    FOR n := 0 TO Len - 1 DO
        PlotChar (InWin, Store [n]);        (* and now plot the result to screen    *)
    END;
END WriteNumber;


BEGIN
    IF NOT Available (4096) THEN                (* enough memory for BitMap data?   *)
        WriteString ("Insufficient memory to run this program.");
        Terminate (1);
    END;
    ALLOCATE (BitMap, 4096);                (* if so, grab it *)
    Lookup (fp, "bitmap.vga", FALSE);       (* is the pixeldata file at hand?   *)
    IF fp.res = notdone THEN
        WriteString ("Characterset datafile not present.");
        Terminate   (2);                    (* if not, get out with errormessage    *)
    END;
    ReadNBytes (fp, BitMap, 4096, Bread);   (* else read data into BitMap buffer    *)
    IF Bread <> 4096 THEN
        WriteString ("Wrong size of BITMAP.VGA file.");
        Terminate   (3);                    (* unless file is too short!    *)
    END;
    Close (fp);             (* That's all folks!    *)
END VgaLib2.
