/* $ pcibios.cpp 28/06/99 21:24 $
 * PCI BIOS functions. Revision 1.1
 *
 * Copyright (C) 1999  Dmitry Uvarov <mit@pc2o205a.pnpi.spb.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $ 13/07/99 22:19 $
 * Fixed bug with BIOS32 detection. (?)Can't detect BOIS32 under QEMM (maybe
 * qemm remaps bios code area to high memory).
 *
 */

#include "..\include\pcibios.h"
#include "..\include\portio.h"
#include "pcibios.def"


/* pragma to read code segment value */
word  GetCS();
#pragma aux GetCS = \
       " mov ax,cs "   \
       value [ax];

#ifdef __PCIBIOS__DEBUG__
  #include <stdio.h>
  #define debug_msg(msg)  printf(msg)
#else
  #define debug_msg(msg)
#endif

#define  PCI_ID_STRING  0x20494350  /* ' ICP' */
#define  PCI_INT        0x1A
#define  BIOS32_SIG     '_23_'      /* "_32_" */
#define  PCI_SERVICE    'ICP$'      /* "$PCI" */
#define  PCI_SIGNATURE  ' ICP'      /* "PCI " */

word PCIBIOS_ver;  /* PCI version */

static ibool pcibios_initiated = FALSE;

/* structures for far calls */
static struct {
        dword address;
        word  segment;
} bios32_entry, pci_entry;


union bios32dir {
        struct {
                dword signature;        /* _32_ */
                dword entry;            /* 32 bit physical address */
                byte  revision;         /* Revision level, 0 */
                byte  length;           /* Length in paragraphs should be 01 */
                byte  checksum;         /* All bytes must add up to zero */
                byte  reserved[5];      /* Must be zero */
        } fields;
        byte bytes[16];
  };

typedef struct {
    ibool (*find_device)(word, word, word, byte*, byte*);
    ibool (*find_class)(dword, word, byte*, byte*);
    byte  (*read_config_byte)(byte, byte, word);
    word  (*read_config_word)(byte, byte, word);
    dword (*read_config_dword)(byte, byte, word);
    ibool (*write_config_byte)(byte, byte, word, byte);
    ibool (*write_config_word)(byte, byte, word, word);
    ibool (*write_config_dword)(byte, byte, word, dword);
  } pci_access;

dword __find_device(word,word,word);
#pragma aux __find_device =      \
      " mov ax,0xB102      "     \
      " callf [pci_entry]  "     \
      " shl ebx,16         "     \
      " mov bx,ax          "     \
      parm [dx] [cx] [si] modify [eax] value [ebx];
static ibool  pci_bios32_find_device(word ven_id, word dev_id, word dev_indx, byte *bus_num, byte *dev_num)
  {
     dword pack = __find_device(ven_id,dev_id,dev_indx);
     *bus_num = pack >> 24;
     *dev_num = (pack >> 16) & 0xFF;
     return (pack & 0xFF00)==0;
  }


dword __find_class(dword,word);
#pragma aux __find_class  =      \
      " mov ax,0xB103      "     \
      " callf [pci_entry]  "     \
      " shl ebx,16         "     \
      " mov bx,ax          "     \
      parm [ecx] [si] modify [eax edx] value [ebx];
static ibool  pci_bios32_find_class(dword class_code, word dev_indx, byte *bus_num, byte *dev_num)
  {
     dword pack = __find_class(class_code,dev_indx);
     *bus_num = pack >> 24;
     *dev_num = (pack >> 16) & 0xFF;
     return (pack & 0xFF00)==0;
  }

byte __readbyte(byte,byte,word);
#pragma aux __readbyte =      \
    " mov ax,0xB108    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] modify [eax ebx ecx edx] value [cl];
static byte   pci_bios32_read_config_byte(byte dev_num, byte bus_num, word reg_num)
  {
     return __readbyte(dev_num,bus_num,reg_num);
  }

word __readword(byte,byte,word);
#pragma aux __readword =      \
    " mov ax,0xB109    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] modify [eax ebx ecx edx] value [cx];
static word   pci_bios32_read_config_word(byte dev_num, byte bus_num, word reg_num)
  {
     return __readword(dev_num,bus_num,reg_num);
  }

dword __readdword(byte,byte,word);
#pragma aux __readdword =     \
    " mov ax,0xB10A    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] modify [eax ebx edx] value [ecx];
static dword  pci_bios32_read_config_dword(byte dev_num, byte bus_num, word reg_num)
  {
     return __readdword(dev_num,bus_num,reg_num);
  }

byte __writebyte(byte,byte,word,byte);
#pragma aux __writebyte =     \
    " mov ax,0xB10B    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] [cl] modify [eax ebx ecx edx] value [ah];
static ibool  pci_bios32_write_config_byte(byte dev_num, byte bus_num, word reg_num, byte data)
  {
     return __writebyte(dev_num,bus_num,reg_num,data) == 0;
  }

byte __writeword(byte,byte,word,word);
#pragma aux __writeword =     \
    " mov ax,0xB10C    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] [cx] modify [eax ebx ecx edx] value [ah];
static ibool  pci_bios32_write_config_word(byte dev_num, byte bus_num, word reg_num, word data)
  {
     return __writeword(dev_num,bus_num,reg_num,data) == 0;
  }

byte __writedword(byte,byte,word,dword);
#pragma aux __writedword =    \
    " mov ax,0xB10D    "    \
    " callf [pci_entry]"    \
    parm [bl] [bh] [di] [ecx] modify [eax ebx ecx edx] value [ah];
static ibool  pci_bios32_write_config_dword(byte dev_num, byte bus_num, word reg_num, dword data)
  {
     return __writedword(dev_num,bus_num,reg_num,data) == 0;
  }


/*
 * Functions for direct access with access type 1
 */
#define CONFIG_CMD(dev_num, bus_num, reg_num) (0x80000000 | (bus_num << 16) | (dev_num << 8) | (reg_num & ~3))

static byte   pci_conf1_read_config_byte(byte dev_num, byte bus_num, word reg_num)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     return inpb(0xCFC + (reg_num & 3));
  }

static word   pci_conf1_read_config_word(byte dev_num, byte bus_num, word reg_num)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     return inpw(0xCFC + (reg_num & 2));
  }

static dword  pci_conf1_read_config_dword(byte dev_num, byte bus_num, word reg_num)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     return inpd(0xCFC);
  }

static ibool  pci_conf1_write_config_byte(byte dev_num, byte bus_num, word reg_num, byte data)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     outpb(0xCFC + (reg_num & 3), data);
     return PCIBIOS_SUCCESSFUL;
  }

static ibool  pci_conf1_write_config_word(byte dev_num, byte bus_num, word reg_num, word data)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     outpw(0xCFC + (reg_num & 2), data);
     return PCIBIOS_SUCCESSFUL;
  }

static ibool  pci_conf1_write_config_dword(byte dev_num, byte bus_num, word reg_num, dword data)
  {
     outpd(0xCF8, CONFIG_CMD(dev_num,bus_num,reg_num));
     outpd(0xCFC, data);
     return PCIBIOS_SUCCESSFUL;
  }
#undef CONFIG_CMD


/*
 * Functions for direct access with access type 2
 */
#define IOADDR(dev_num, reg_num)   ((0xC000 | ((dev_num & 0x78) << 5)) + reg_num)
#define FUNC(dev_num)            (((dev_num & 7) << 1) | 0xf0)

static byte   pci_conf2_read_config_byte(byte dev_num, byte bus_num, word reg_num)
  {
     outpb(0xCF8, FUNC(dev_num));
     outpb(0xCFA, bus_num);
     byte val = inpb(IOADDR(dev_num, reg_num));
     outpb(0xCF8, 0);
     return val;
  }

static word   pci_conf2_read_config_word(byte dev_num, byte bus_num, word reg_num)
  {
     outpb(0xCF8, FUNC(dev_num));
     outpb(0xCFA, bus_num);
     word val = inpw(IOADDR(dev_num, reg_num));
     outpb(0xCF8, 0);
     return val;
  }

static dword  pci_conf2_read_config_dword(byte dev_num, byte bus_num, word reg_num)
  {
     outpb(0xCF8, FUNC(dev_num));
     outpb(0xCFA, bus_num);
     dword val = inpd(IOADDR(dev_num, reg_num));
     outpb(0xCF8, 0);
     return val;
  }

static ibool  pci_conf2_write_config_byte(byte dev_num, byte bus_num, word reg_num, byte data)
  {
    outpb(0xCF8, FUNC(dev_num));
    outpb(0xCFA, bus_num);
    outpb(IOADDR(dev_num,reg_num), data);
    outpb(0xCF8, 0);
    return PCIBIOS_SUCCESSFUL;
  }

static ibool  pci_conf2_write_config_word(byte dev_num, byte bus_num, word reg_num, word data)
  {
    outpb(0xCF8, FUNC(dev_num));
    outpb(0xCFA, bus_num);
    outpw(IOADDR(dev_num,reg_num), data);
    outpb(0xCF8, 0);
    return PCIBIOS_SUCCESSFUL;
  }

static ibool  pci_conf2_write_config_dword(byte dev_num, byte bus_num, word reg_num, dword data)
  {
    outpb(0xCF8, FUNC(dev_num));
    outpb(0xCFA, bus_num);
    outpd(IOADDR(dev_num,reg_num), data);
    outpb(0xCF8, 0);
    return PCIBIOS_SUCCESSFUL;
  }
#undef IOADDR
#undef FUNC


static ibool pci_direct_find_device(word ven_id, word dev_id, word dev_indx, byte *bus_num, byte *dev_num)
  {
    int indx = 0;
    for (int bus=0; bus<16; bus++) {
        for (int dev=0; dev<256; dev++) {
             dword data = PCIBIOS_ReadConfigDWord(dev,bus,0);
             if (((data & 0xFFFF) == ven_id) && ((data >> 16) == dev_id)) {
                 if (indx == dev_indx) {
                     *bus_num = bus;
                     *dev_num = dev;
                     return PCIBIOS_SUCCESSFUL;
                 }
                 indx++;
             }
        }
    }
    return PCIBIOS_DEVICE_NOT_FOUND;
  }

static ibool pci_direct_find_class(dword class_code, word dev_indx, byte *bus_num, byte *dev_num)
  {
    /*
     * class code is set of 3 bytes: base class, sub class and programming
     *                               interface type
     */
    int indx = 0;
    for (int bus=0; bus<16; bus++) {
        for (int dev=0; dev<256; dev++) {
             /* check if device is present */
             dword venid = PCIBIOS_ReadConfigDWord(dev,bus,0);
             if ((venid == 0xFFFFFFFF) || (venid == 0x00000000)) continue;
             /* read class revision and remove revision code from class specifier */
             dword data = PCIBIOS_ReadConfigWord(dev,bus,PCI_CLASS_REVISION)>>8;
             if (data == class_code) {
                 if (indx == dev_indx) {
                     *bus_num = bus;
                     *dev_num = dev;
                     return PCIBIOS_SUCCESSFUL;
                 }
                 indx++;
             }
        }
    }
    return PCIBIOS_DEVICE_NOT_FOUND;
  }

/* pointer to current access functions */
static pci_access *access_pci = NULL;

/* bios32 functions table */
static pci_access pci_bios32_access = {
       pci_bios32_find_device,
       pci_bios32_find_class,
       pci_bios32_read_config_byte,
       pci_bios32_read_config_word,
       pci_bios32_read_config_dword,
       pci_bios32_write_config_byte,
       pci_bios32_write_config_word,
       pci_bios32_write_config_dword
};

/* PCI direct configuration 1 functions table */
static pci_access pci_direct_conf1_access = {
       pci_direct_find_device,
       pci_direct_find_class,
       pci_conf1_read_config_byte,
       pci_conf1_read_config_word,
       pci_conf1_read_config_dword,
       pci_conf1_write_config_byte,
       pci_conf1_write_config_word,
       pci_conf1_write_config_dword
};

/* PCI direct configuration 2 functions table */
static pci_access pci_direct_conf2_access = {
       pci_direct_find_device,
       pci_direct_find_class,
       pci_conf2_read_config_byte,
       pci_conf2_read_config_word,
       pci_conf2_read_config_dword,
       pci_conf2_write_config_byte,
       pci_conf2_write_config_word,
       pci_conf2_write_config_dword
};

/*
 * return entry point to bios32 service.
 */
dword bios32_service(dword service);
#pragma aux bios32_service  =   \
   " xor ebx,ebx          "     \
   " callf [bios32_entry] "     \
   " cmp al, 0            "     \
   " je noerror           "     \
   "   xor ebx,ebx        "     \
   "   xor edx,edx        "     \
   " noerror: add ebx,edx "     \
   parm [eax] modify [ecx edx] value [ebx];

word check_bios32_presence(dword *sig);
#pragma aux check_bios32_presence = \
   " push esi             "         \
   " mov ax,0xB101        "         \
   " callf [pci_entry]    "         \
   " pop esi              "         \
   " xor al,al            "         \
   " cmp ah,0             "         \
   " je noerror           "         \
   "   xor edx,edx        "         \
   " noerror: mov [esi],edx"        \
   parm [esi] modify [eax ebx ecx edx] value [ax];


static ibool check_bios()
  {
     if ((pci_entry.address = bios32_service(PCI_SERVICE))!=NULL) {
        pci_entry.segment = GetCS();
        dword signature;
        int res = check_bios32_presence(&signature);
        if ((res==0) && (signature==PCI_SIGNATURE)) return TRUE;
#ifdef __PCIBIOS__DEBUG__
        else
            printf("pcibios_init() warning: BIOS32 Service Directory says PCI BIOS is present,\n"
                   "                        but PCI_BIOS_PRESENT subfunc fails.\n");
#endif
     };
     debug_msg("pcibios_init(): BIOS32 does not provide PCI Services\n");
     return FALSE;
  };

static ibool check_bios32()
  {
     debug_msg("pcibios_init(): scanning for BIOS32 presence\n");
    /*
     * follow standart procedure for detection BIOS32 Service Directory by
     * scanning memory addresses 0xE0000-0xFFFFF for BIOS32 signature
     */
     for (bios32dir *bios32 = (bios32dir*)0xE0000;
                     bios32<= (bios32dir*)0xFFFF0;
                     bios32++)  {
         /* check length and signature fields */
         if ((bios32->fields.signature != BIOS32_SIG) ||
             (bios32->fields.length != 1)) continue;
         /* check for checksum */
         byte sum=0;
         for (int i = 0; i<16; i++) sum+=bios32->bytes[i];
         if (sum != 0) continue;
         bios32_entry.address = bios32->fields.entry;
         if (!bios32_entry.address) continue;
#ifdef __PCIBIOS__DEBUG__
         printf("pcibios_init(): found BIOS32 Service Directory at 0x%x\n",bios32_entry.address);
#endif
         /* fill calling sructure */
         bios32_entry.segment=GetCS();
         if (check_bios()) {
            access_pci = &pci_bios32_access;
            return TRUE;
         }
         return FALSE;
     }
     debug_msg("pcibios_init(): BIOS32 Service Directory not founded\n");
     return FALSE;
  }

#ifdef __PCIBIOS__DEBUG__
  static char *pci_use_config_msg = "pcibios_init(): direct access avaible. using configuration type %d\n";
#endif
static ibool check_direct()
  {
     debug_msg("pcibios_init(): attempting to use PCI direct access\n");
     /*
      * attempt to use direct PCI access throught I/O ports
      * check for access type
      */
    int pci_type;
    cli();
    outpb(0xCFB,0x01);
    int tmp = inpd(0xCF8);
    outpd(0xCF8,0x80000000);
    if (inpd(0xCF8) == 0x80000000) {
        outpd(0xCF8,tmp);
        sti();
        pci_type = 1;
#ifdef __PCIBIOS__DEBUG__
        printf(pci_use_config_msg,1);
#endif
        access_pci = &pci_direct_conf1_access;
        return TRUE;
    }
    outpd(0xCF8,tmp);

    outpb(0xCFB,0);
    outpb(0xCF8,0);
    outpb(0xCFA,0);
    if ((inpb(0xCF8) == 0) && (inpb(0xCFB) == 0)) {
        sti();
#ifdef __PCIBIOS__DEBUG__
        printf(pci_use_config_msg,2);
#endif
        access_pci = &pci_direct_conf2_access;
        return TRUE;
    }
    sti();
    debug_msg("pcibios_init(): not supported chipset for direct PCI access!\n");
    return FALSE;
  }

ibool PCIBIOS_init()
  {
    if (pcibios_initiated) return TRUE;
    if (!check_bios32())
       if (!check_direct())
          return FALSE;
    pcibios_initiated = TRUE;
    return TRUE;
  }

ibool PCIBIOS_present()
  {
    return access_pci ? TRUE:FALSE;
  }



/****************************************************************************
*                                                                           *
* Function: PCIBIOS_FindDevice();                                           *
* Description: find PCI device by given vendor id, device id and device     *
*              index                                                        *
* Returns: bus number                                                       *
*          device/function number (bits 7-3 device, bits 2-0 function)      *
*          true if successful                                               *
*                                                                           *
****************************************************************************/
ibool PCIBIOS_FindDevice(word ven_id, word dev_id, word dev_indx, byte *bus_num, byte *dev_num)
  {
     if (access_pci && access_pci->find_device)
         return access_pci->find_device(ven_id,dev_id,dev_indx,bus_num,dev_num);
     return PCIBIOS_FUNC_NOT_SUPPORTED;
  }

/****************************************************************************
*                                                                           *
* Function: PCIBIOS_FindClassCode();                                        *
* Description: find PCI device by given class code and device index         *
* Returns: bus number                                                       *
*          device/function number (bits 7-3 device, bits 2-0 function)      *
*          true if successful                                               *
*                                                                           *
****************************************************************************/
ibool PCIBIOS_FindClassCode(dword class_code, word dev_indx, byte *bus_num, byte *dev_num)
  {
     if (access_pci && access_pci->find_class)
         return access_pci->find_class(class_code,dev_indx,bus_num,dev_num);
     return PCIBIOS_FUNC_NOT_SUPPORTED;
  }

/****************************************************************************
*                                                                           *
* Function: PCIBIOS_ReadConfigXXXX();                                       *
* Description: read configuration byte/word/dword from given device         *
*              configuration space                                          *
* Returns: readed byte/word/dword                                           *
*                                                                           *
****************************************************************************/
byte  PCIBIOS_ReadConfigByte(byte dev_num, byte bus_num, word reg_num)
  {
     if (access_pci && access_pci->read_config_byte)
         return access_pci->read_config_byte(dev_num,bus_num,reg_num);
     return PCIBIOS_FUNC_NOT_SUPPORTED;
  }

word  PCIBIOS_ReadConfigWord(byte dev_num, byte bus_num, word reg_num)
  {
     if (access_pci && access_pci->read_config_word)
         return access_pci->read_config_word(dev_num,bus_num,reg_num);
     return PCIBIOS_FUNC_NOT_SUPPORTED;
  }

dword PCIBIOS_ReadConfigDWord(byte dev_num, byte bus_num, word reg_num)
  {
     if (access_pci && access_pci->read_config_dword)
         return access_pci->read_config_dword(dev_num,bus_num,reg_num);
     return PCIBIOS_FUNC_NOT_SUPPORTED;
  }

/****************************************************************************
*                                                                           *
* Function: PCIBIOS_WriteConfigXXXX();                                      *
* Description: write configuration byte/word/dword into given device        *
*              configuration space                                          *
* Returns: nothing                                                          *
*                                                                           *
****************************************************************************/
ibool PCIBIOS_WriteConfigByte(byte dev_num, byte bus_num, word reg_num, byte data)
  {
     if (access_pci && access_pci->write_config_byte)
         return access_pci->write_config_byte(dev_num,bus_num,reg_num,data);
     return FALSE;
  }

ibool PCIBIOS_WriteConfigWord(byte dev_num, byte bus_num, word reg_num, word data)
  {
     if (access_pci && access_pci->write_config_word)
         return access_pci->write_config_word(dev_num,bus_num,reg_num,data);
     return FALSE;
  }

ibool PCIBIOS_WriteConfigDWord(byte dev_num, byte bus_num, word reg_num, dword data)
  {
     if (access_pci && access_pci->write_config_dword)
         return access_pci->write_config_dword(dev_num,bus_num,reg_num,data);
     return FALSE;
  }

ibool PCIBIOS_ReadConfigSpace(byte dev_num, byte bus_num, pci_config_space *pciconf)
  {
     dword *conf = (dword*)pciconf;
     for (int i=0; i<16; i++)
        conf[i] = PCIBIOS_ReadConfigDWord(dev_num,bus_num,i<<2);
     return TRUE;
  }

#ifdef __PCIBIOS__DEBUG__
char *PCIBIOS_strerror(int code)
  {
        switch (code) {
            case PCIBIOS_SUCCESSFUL:          return "SUCCESSFUL";
            case PCIBIOS_FUNC_NOT_SUPPORTED:  return "FUNC_NOT_SUPPORTED";
            case PCIBIOS_BAD_VENDOR_ID:       return "SUCCESSFUL";
            case PCIBIOS_DEVICE_NOT_FOUND:    return "DEVICE_NOT_FOUND";
            case PCIBIOS_BAD_REGISTER_NUMBER: return "BAD_REGISTER_NUMBER";
            case PCIBIOS_SET_FAILED:          return "SET_FAILED";
            case PCIBIOS_BUFFER_TOO_SMALL:    return "BUFFER_TOO_SMALL";
            default:                          return "UNKNOWN ERROR";
        }
  }
#endif