/*
  dfill.c: Determine the "best" way to put files onto floppy disks.

  Jason Hood, 4 March, 2000.
	      15 July to 8 September, 2003.

  Given a disk size and a list of files, try and minimise the number of disks,
  whilst maximising the free space.
*/


#define PVERS "1.00"
#define PDATE "8 September, 2003"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#ifdef __DJGPP__
# include <pc.h>
# define  _kbhit kbhit
# define  _getch getkey
#else
# include <conio.h>
#endif


typedef struct
{
  char* name;		// Filename
  long	size;		// Its size, rounded to a multiple of 512
} File;


#define SEEN_MAX 50	// Maximum number of previously seen files to store
#define SEEN_MIN 12	// Minimum number of files required to be seen
typedef struct s_fl
{
  struct s_fl* next;
  int  len;
  File list[1]; 	// Dynamically allocated
} FileList;
FileList** seen;	// An array of lists for each disk


struct
{
  int	order;		// Use command line order, not sorted
  int	limit;		// Limit the number of possibilities
  int	display;	// Number of disks for the progress display
  int	graph;		// Use a "graphical" progress display
  int	num;		// Use numbers instead of files
  int	test;		// Don't write the batch file
  char* cmd;		// Command to copy the files
  int	multi;		// Command allows multiple filespecs
  char* dest;		// Destination path
}
option =
{
  0,			// Use decreasing size
  -1,			// Default limit is half the number of disks
  -1,			// Default display is third the number of disks
  0,			// No graph
  0,			// Command line contains filenames
  0,			// Create the batch file
  "copy",               // Default command
  0,			// Default command allows only one filename
  "a:"                  // Default destination
};


long  desired;		// The size of the disks
long  last_total;	// The total of the last disk when other disks are full
File* cur_set;		// The current set of disks
File* best_set; 	// The best set of disks
int   files;		// The total number of files to process
typedef struct
{
  int  copied;		// Files processed
  int  disks;		// Disks used
  long least;		// The lowest sum
  long full;		// The overall "fullness" of the disks
} State;
State cur, best;
int   stop;		// Stop processing (best found or key pressed)

FILE* screen;		// Progress display goes directly to the console


// Sort by decreasing size.
int size_cmp( const void* e1, const void* e2 )
{
  return ((const File*)e2)->size - ((const File*)e1)->size;
}


/*
  Add a set of files to the set of disks.

  file: list of files being processed
  len:	number of files to add
  idx:	indices into file of the files to add
  sum:	total size of the files
*/
void add_files( File* file, int len, int* idx, long sum )
{
  int j;

  for (j = 0; j < len; ++j)
    cur_set[cur.copied++] = file[idx[j]];
  cur_set[cur.copied-1].size = -cur_set[cur.copied-1].size;
  ++cur.disks;

  if (sum < cur.least)
    cur.least = sum;
  sum >>= 9;
  cur.full += sum * sum;
}


// Determine if the current set of disks is better than the best set.
int better( State* cur, File* src, State* best, File* dest )
{
  int copy;

  copy = (cur->disks < best->disks);
  if (cur->disks == best->disks)
  {
    copy = (cur->least < best->least);
    if (cur->least == best->least)
      copy = (cur->full < best->full);
  }
  if (copy)
  {
    memcpy( dest, src, files * sizeof(File) );
    *best = *cur;
  }

  return copy;
}


/*
  Determine the best combination of files that sums close to desired,
  continuing recursively until no files remain.

  count: the number of files
  file:  the files
  total: the sum of the file sizes
*/
void process( int count, File* file, long total )
{
  long	sum,		// Current sum
	best_sum,	// Highest sum so far
	hi_sum; 	// Highest total to which a file can be added
  int*	val;		// Indices of the summed files
  int	len;		// Number of above
  struct		// Stack of partial combinations
  {
    int  idx;		// Position to continue
    long sum;		// Sum at this point
    int  len;		// Number of files at this point
  } *stack;
  int	ptr;		// Position into the stack
  File* list;		// List of remaining files
  State old;		// Restore the state after recursion
  int	comb;		// Number of combinations processed
  int	disp;		// Progress has been displayed
  int	i, j, k;

  val	 = malloc( count * sizeof( int ) );
  val[0] = 0;

  if (total <= desired)
  {
    j = count;
    while (--j) val[j] = j;
    add_files( file, count, val, total );
  }
  else if (count == 2)
  {
    add_files( file,   1, val, file[0].size );
    add_files( file+1, 1, val, file[1].size );
  }
  else
  {
    if (option.order && count >= SEEN_MIN)
    {
      FileList** p = &seen[cur.disks];
      j = 0;
      while (*p)
      {
	if ((*p)->len == count &&
	    !memcmp( (*p)->list, file, count * sizeof(File) ))
	{
	  free( val );
	  return;
	}
	++j;
	p = &(*p)->next;
      }
      if (j < SEEN_MAX)
      {
	*p = malloc( sizeof(FileList) + (count - 1) * sizeof(File) );
	(*p)->len = count;
	memcpy( (*p)->list, file, count * sizeof(File) );
	(*p)->next = NULL;
      }
    }

    stack = malloc( (count - 2) * sizeof(*stack) +
		    (count - 1) * sizeof(File) );
    list  = (File*)(stack + count - 2);

    comb = 0;
    disp = 0;

    best_sum = 0;
    if (option.order)
    {
      // Find the highest size that can be added to the first size,
      // and the lowest size (excluding the first).
      hi_sum = desired - file[0].size;
      sum    = file[1].size;
      j = count;
      while (--j)
      {
	if (file[j].size <= hi_sum)
	{
	  if (file[j].size > best_sum)
	    best_sum = file[j].size;
	  if (file[j].size < sum)
	    sum = file[j].size;
	}
      }
      best_sum += file[0].size;
      hi_sum	= desired - sum;
    }
    else
    {
      hi_sum = desired - file[count-1].size;
    }

    stack[0].sum = file[0].size;
    stack[0].len = 1;
    stack[0].idx = 1;
    ptr = 0;
    do
    {
      sum = stack[ptr].sum;
      len = stack[ptr].len;
      for (k = stack[ptr].idx; k < count; ++k)
      {
	if (sum + file[k].size <= desired)
	{
	  if (k < count - 1)
	  {
	    i = k + 1;
	    while (file[i-1].size == file[i].size && i < count)
	      ++i;
	    if (i < count)
	    {
	      stack[ptr].idx = i;
	      stack[ptr].sum = sum;
	      stack[ptr].len = len;
	      ++ptr;
	    }
	  }
	  val[len++] = k;
	  sum += file[k].size;
	  if (sum > hi_sum)
	  {
	    if (!option.order && k == count - 2 && ptr > 0)
	      --ptr;
	    break;
	  }
	}
      }
      if (sum >= best_sum)
      {
	best_sum = sum;

	old = cur;

	add_files( file, len, val, sum );

	++comb;
	if (old.disks < option.display)
	{
	  disp = (option.graph) ? fprintf( screen, "\e[%dC*\n", comb + 3 )
				: fprintf( screen, "%2d ", comb );
	  fflush( screen );
	}

	for (k = 0, i = j = 1; i < len; ++j, ++i)
	  for (; j < val[i]; ++j)
	    list[k++] = file[j];
	for (; j < count; ++j)
	  list[k++] = file[j];

	process( k, list, total - sum );

	if (disp)
	{
	  if (option.graph)
	    fputs( "\e[A", screen );
	  else
	  {
	    j = disp;
	    do fputc( '\b', screen ); while (--j);
	  }
	  fflush( screen );
	}

	cur = old;

	if (comb == option.limit || (sum == desired && total - sum <= desired)
	    || stop)
	  break;
      }
      if (!option.order && val[len-1] == count - 1)
      {
	while (--len > 0 && val[len] == val[len-1] + 1)
	  --ptr;
	if (ptr <= 0)
	  break;
      }
    } while (ptr--);

    free( stack );

    if (disp)
    {
      if (option.graph)
      {
	fputs( "\e[4C", screen );
	j = comb;
        do fputc( '.', screen ); while (--j);
	fputc( '\r', screen );
      }
      else
      {
	j = k = disp - 1;
	do fputc( ' ',  screen ); while (--j);
	do fputc( '\b', screen ); while (--k);
      }
      fflush( screen );
    }
  }

  free( val );

  if (cur.copied == files)
  {
    if (better( &cur, cur_set, &best, best_set ))
      if (best.least == last_total)
	stop = 1;

    if (_kbhit())
    {
      _getch();
      stop = 2;
    }
  }
}


/*
  Initialisation for process().

  count: the number of files
  file:  the files
  total: the sum of the files
*/
void dfill( int count, File* file, long total )
{
  int j;
  FileList *p, *q;

  files      = count;
  cur.copied = 0;
  cur.disks  = 0;
  best.disks = count;
  best.least = cur.least = desired;
  best.full  = cur.full  = 0;

  if (option.order)
    seen = calloc( count, sizeof(FileList*) );

  process( count, file, total );

  if (option.order)
  {
    j = count;
    do
    {
      p = seen[--j];
      while (p)
      {
	q = p->next;
	free( p );
	p = q;
      }
    } while (j);
    free( seen );
  }
}


int main( int argc, char* argv[] )
{
  File* file;
  File* sort;
  File* list;
  File* disk;
  int	disk_count;
  int	count, skip;
  long	sum, total;
  int	opt_disks, disks;
  State set;
  File	skipped;
  FILE* bat;
  int	files;
  int	j, k;
  struct stat st;
  char* bp;
  int	clen = 0, dlen = 0, len = 0;

  if (argc < 3 || !strcmp( argv[1], "/?" ) || !strcmp( argv[1], "-?" ) ||
		  !strcmp( argv[1], "--help" ))
  {
    puts(
"Disk Fill by Jason Hood <jadoxa@yahoo.com.au>\n"
"Version "PVERS" ("PDATE"). Freeware.\n"
"http://diskfill.adoxa.cjb.net/\n"
"\n"
"dfill [-o] [-l<limit>] [-p<disks>] [-g] [-t] [-n]\n"
"      [-c<cmd>] [-m] [-d<dest>] [size] files...\n"
"\n"
"o          use the command line order (don't sort by decreasing size)\n"
"l          limit the amount of testing (default is half the optimal number\n"
"            of disks; 0 is no limit)\n"
"p          display <disks> for the progress (default is third of optimal)\n"
"g          use a \"graphical\" progress display (requires ANSI;\n"
"            <limit> and <disks> must be within screen width and height)\n"
"t          test - don't write the batch file\n"
"n          use numbers instead of files (note that size is\n"
"            required with this option)\n"
"c<cmd>     use <cmd> to perform the copy (default is \"copy\")\n"
"m          indicate <cmd> allows multiple filenames\n"
"d<dest>    copy files to <dest> (default is \"a:\")\n"
"size       the size of the disks (default is 1457664)\n"
"files...   the files (or sizes with -n) to copy\n"
"\n"
"Notes: empty files, files bigger than size and directories are ignored;\n"
"       all files are assumed to fit on the disk;\n"
"       all disks are assumed to be the same size, with a 512-byte cluster;\n"
"       when using -n the numbers are rounded to multiples of 512."
	);
    return 0;
  }

  for (j = 1; argv[j] && argv[j][0] == '-'; ++j)
  {
    switch (argv[j][1])
    {
      case 'o': option.order   = 1;                   break;
      case 'l': option.limit   = atoi( argv[j] + 2 ); break;
      case 'p': option.display = atoi( argv[j] + 2 ); break;
      case 'g': option.graph   = 1;                   break;
      case 't': option.test    = 1;                   break;
      case 'n': option.num     = 1;                   break;
      case 'c': option.cmd     = argv[j] + 2;         break;
      case 'm': option.multi   = 1;                   break;
      case 'd': option.dest    = argv[j] + 2;         break;
      default:	goto break_for;
    }
  }
  if (!argv[j])
  {
    fputs( "Nothing to optimise!\n", stderr );
    return 1;
  }
  break_for:

  // If the first non-option argument is a number, use it as the size;
  // otherwise treat it as a file and use the default size.
  desired = strtol( argv[j], &bp, 0 );
  if (*bp != '\0')
  {
    desired = 1457664L;
  }
  else
  {
    if (desired < 0)
      desired = -desired;
    desired = ((desired + 511) >> 9) << 9;
    ++j;
  }

#ifdef __DJGPP__
  _djstat_flags = (unsigned short)-1;	// Don't want anything special
#endif

  file	= malloc( (argc - j) * sizeof(File) );
  count = 0;
  sum	= 0;	// Bytesize
  total = 0;	// Realsize
  for (; j < argc; ++j)
  {
    if (option.num)
    {
      st.st_size = strtol( argv[j], NULL, 0 );
    }
    else
    {
      if (stat( argv[j], &st ))
      {
	fprintf( stderr, "%s : unable to stat\n", argv[j] );
	continue;
      }
      if (S_ISDIR( st.st_mode ))
      {
	fprintf( stderr, "%s : directory\n", argv[j] );
	continue;
      }
    }
    if (st.st_size == 0)
    {
      fprintf( stderr, "%s : empty\n", argv[j] );
    }
    else if (st.st_size > desired)
    {
      fprintf( stderr, "%s : too big\n", argv[j] );
    }
    else
    {
      file[count].name = argv[j];
      file[count].size = ((st.st_size + 511) >> 9) << 9;
      sum   += st.st_size;
      total += file[count].size;
      ++count;
    }
  }
  if (count < 2)
  {
    fputs( "Nothing to optimise!\n", stderr );
    return 1;
  }

  opt_disks = (int)(total / desired);
  if (total % desired)
    ++opt_disks;

  printf( "Processing %d files totalling %ld (%ld) bytes.\n"
	  "Optimal solution requires %d disks of %ld bytes.\n\n",
	  count, sum, total, opt_disks, desired );

  // The files on the disks. A negative size indicates the last file.
  disk = malloc( count * sizeof(File) );
  disk_count = 0;

  // Keep a working copy.
  list = sort = malloc( count * sizeof(File) );
  memcpy( sort, file, count * sizeof(File) );
  qsort( sort, count, sizeof(File), size_cmp );

  // Separate the files too big to be combined.
  j = 0;
  while (sort[j].size + sort[count-1].size > desired)
  {
    disk[disk_count].name   =  sort[j].name;
    disk[disk_count++].size = -sort[j].size;
    total -= sort[j].size;
    if (++j == count-1)
    {
      disk[disk_count].name   =  sort[j].name;
      disk[disk_count++].size = -sort[j].size;
      goto display;
    }
  }
  count -= disk_count;
  list	+= disk_count;
  disk	+= disk_count;

  // Separate the files that won't combine.
  skip = 0;
  while (list[skip].size + list[skip+1].size > desired)
    ++skip;

  // If the original command line order was requested search for the
  // separated files and move them to the beginning.
  if (option.order)
  {
    for (j = 0; j < disk_count; ++j)
    {
      k = j;
      while (memcmp( file + k, sort + j, sizeof(File) ))
	++k;
      memmove( file + j + 1, file + j, (k - j) * sizeof(File) );
      file[j] = sort[j];
    }
    file += disk_count;
    for (j = 0; j < skip; ++j)
    {
      k = j;
      while (memcmp( file + k, list + j, sizeof(File) ))
	++k;
      memmove( file + j + 1, file + j, (k - j) * sizeof(File) );
      file[j] = list[j];
    }
    list  = file;
    file -= disk_count;
    skip  = count;
  }

  // Have a (not-very-good) guess for limit.
  if (option.limit < 0)
    option.limit = (opt_disks + 1) / 2;
  if (option.display < 0)
    option.display = (opt_disks + 2) / 3;
  if (option.graph)
    if (option.display == 0 || option.limit == 0 || option.limit > 126)
      option.graph = 0;

  // The total of the last disk if all previous disks are completely full.
  last_total = total - (opt_disks - disk_count - 1) * desired;

  screen = fopen( "con", "w" );
  if (option.graph)
  {
    for (j = 1; j <= option.display; ++j)
    {
      fprintf( screen, "%2d [", j );
      for (k = 0; k < option.limit; ++k)
	fputc( '.', screen );
      fputs( "]\n", screen );
    }
    fprintf( screen, "\e[%dA", option.display );
    fflush( screen );
  }

  cur_set  = malloc( 2 * count * sizeof(File) );
  best_set = disk;
  dfill( count, list, total );

  // See if leaving the big files out will produce a better set.
  best_set = cur_set + count;
  set	   = best;
  for (j = 0; j < skip && !stop; list[0] = skipped, ++j)
  {
    skipped = list[j];			// Skip this value,
    list[j] = list[0];			// swapping it with the previous

    if (skipped.size >= set.least)
      continue;

    dfill( count - 1, list + 1, total - skipped.size );

    if (best.disks == set.disks - 1)
    {
      k = 0;
      cur = best;
      memcpy( cur_set, best_set, (count - 1) * sizeof(File) );
      add_files( &skipped, 1, &k, skipped.size );
      set = cur;
      memcpy( disk, cur_set, count * sizeof(File) );
    }
    else
    {
      best.least += skipped.size;
      if (better( &best, best_set, &set, disk ))
      {
	disk[count-1] = disk[count-2];	// Assume last disk
	disk[count-2] = skipped;
      }
    }
  }

  if (option.graph)
  {
    j = option.display;
    do fputs( "\e[K\n", screen ); while (--j);
    fprintf( screen, "\e[%dA", option.display );
  }
  fclose( screen );

  disk	-= disk_count;
  count += disk_count;
  disk_count += set.disks;

  printf( "Best result found uses %d disks", disk_count );
  if (stop != 2)
  {
    printf(" (-l%d", option.limit );
    if (option.order)
      fputs( " -o", stdout );
    putchar( ')' );
  }
  puts( ":\n" );

display:
  puts( "Bytes Free\tFiles\n"
	"----------\t-----" );

  if (!option.test)
  {
    bat = fopen( "fill.bat", "w" );
    if (bat == NULL)
    {
      fputs( "Unable to create \"fill.bat\".\n", stderr );
      option.test = 1;
    }
    else
    {
      fprintf( bat, "@echo off\n"
		    "echo Filling %d disks with %d files.\n"
		    "echo.\n",
		    disk_count, count );
      clen = strlen( option.cmd );
      dlen = strlen( option.dest ) + 1;
    }
  }
  else
    bat = NULL; // Keep GCC happy

  disks = 0;
  for (j = 0; j < count;)
  {
    sum   = 0;
    files = 1;
    k = j;
    while (disk[k].size > 0)
    {
      sum += disk[k].size;
      ++files;
      ++k;
    }
    sum += -disk[k].size;
    printf( " %7ld  \t %2d  \n", desired - sum, files );
    if (option.test)
      j = k + 1;
    else
    {
      ++disks;
      fprintf( bat, "echo Please insert the %s disk...\npause >nul\n",
	       (disks == 1)	     ? "first" :
	       (disks == disk_count) ? "last"  :
				       "next" );
      if (option.multi)
      {
	fputs( option.cmd, bat );
	len = clen;
      }
      for (; j <= k; ++j)
      {
	if (!option.multi)
	  fputs( option.cmd, bat );
	// Don't exceed the command line length limit.
	else if (len + strlen( disk[j].name ) + 3 + dlen > 127)
	{
	  fprintf( bat, " %s\n", option.dest );
	  fputs( option.cmd, bat );
	  len = clen;
	}
	len += fprintf( bat, (strpbrk( disk[j].name, " +;,=" ))
			      ? " \"%s\"" : " %s", disk[j].name );
	if (!option.multi)
	  fprintf( bat, " %s\n", option.dest );
      }
      if (option.multi)
	fprintf( bat, " %s\n", option.dest );
    }
  }
  if (!option.test)
    fclose( bat );

  free( cur_set );
  free( sort );
  free( disk );
  free( file );

  return 0;
}
