/*
 *  Cheshiresoft Calendar-Almanac
 *  Copyright (c) 2003 by Andrew Ziem.  All rights reserved.
 *  http://chesire.freeservers.com
 *  http://cday.sourceforge.net
 *
 *     MAYA.C - JDN to Mayan calendar
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/*
 * NOTES
 *
 *   I use 3114BC-Sep-6 (Julian) or 3114BC-Aug-11 (Gregorian) as the
 *   beginning date of the calendar.  There are several possibilities,
 *   but authorities disagree.
 * 
 *
 * REFERENCES
 *
 *   The Calendar FAQ v2.0 (http://www.pip.dknet.dk/~c-t/calendar.html).
 *
 */

#include "maya.h"
char *MayaTzolkinDays[] = { "Ahau",	/* 0 */
  "Imix",			/* 1 */
  "Ik", "Akbal", "Kan", "Chicchan", "Cimi", "Manik", "Lamat",
  "Muluc", "Oc", "Chuen", "Eb", "Ben", "Ix", "Men", "Cib", "Caban",
  "Etznab", "Caunac"		/* 19 */
};
char *MayaHaabMonths[] = { "Uayeb",	/* extra period at end of year */
  "Pop",			/* 1, 1st month */
  "Uo", "Zip", "Zotz", "Tzec", "Xul", "Yaxkin", "Mol", "Chen",
  "Yax", "Zac", "Ceh", "Mac", "Kankin", "Muan", "Pax", "Kayab",
  "Cumku"			/* 18 */
};

/*
 * this function needs expansion to accept JDN dates outside of the
 * current baktun
 *
 * returns: number of days in era (0 through MAYA_FULL_ERA)
 */
UINT32
SdnToMayaDays (UINT32 JDN)
{
  UINT32 ret;
  if (MAYA_FIRST_JDN > JDN)
    return (UINT32) 0;
  ret = JDN - MAYA_FIRST_JDN;

/*    printf("ret1 = %i\n", ret); */

  /* get correct era */
  while (ret >= MAYA_FULL_ERA)

    {
      ret = ret - MAYA_FULL_ERA;

/*          printf("ret2 = %i\n", ret); */
    }
  return (UINT32) ret;
}


/*
 * given: JDN
 *
 * returns: Maya Loungcount components 
 *
 */
void
SdnToMayaLongcount (UINT32 JDN, int *pBaktun, int *pKatun, int *pTun,
		    int *pUnial, int *pKin)
{
  UINT32 i;
  i = SdnToMayaDays (JDN);
  *pBaktun = i / 144000;
  i = i % 144000;

  /* baktun is 1..13, not 0..12 */
  if (*pBaktun == 0)
    *pBaktun = 13;
  *pKatun = i / 7200;
  i = i % 7200;
  *pTun = i / 360;
  i = i % 360;
  *pUnial = i / 20;
  *pKin = i % 20;
}


/*
 * given: JDN
 *
 * returns: string containing Maya long count
 *
 */
char *
SdnToMayaLongcountStr (UINT32 JDN, char *retstr)
{
  UINT32 i;
  MAYALONGCOUNT mlc;
  i = SdnToMayaDays (JDN);
  i = i % 260;
  SdnToMayaLongcount (JDN, &mlc.baktun, &mlc.katun, &mlc.tun, &mlc.unial,
		      &mlc.kin);
  sprintf (retstr, "%i.%i.%i.%i.%i", mlc.baktun, mlc.katun, mlc.tun,
	   mlc.unial, mlc.kin);
  return (char *) retstr;
}


/*
 * returns: 1 to 13 of Maya Tzolkin date given a JDN 
 */
int
SdnToTzolkin13 (UINT32 JDN)
{
  UINT32 i;
  i = (SdnToMayaDays (JDN) % 13) + 4;

/*    printf("debug: t13i = %i\n", i); */
  if (i < 0)
    i = i + 13;
  if (i > 13)
    i = i - 13;
  if (i == 0)
    i = 13;
  return (int) i;
}


/*
 * returns: 0 (Ahau) to 19 (Caunac) of Maya Tzolkin date given a JDN
 */
int
SdnToTzolkin20 (UINT32 JDN)
{
  MAYALONGCOUNT mlc;
  SdnToMayaLongcount (JDN, &mlc.baktun, &mlc.katun, &mlc.tun, &mlc.unial,
		      &mlc.kin);
  return mlc.kin;
}


/*
 * returns: month (1..18, 0=Uayeb) and day (0..19) of Maya Haab calendar
 */
void
SdnToHaab (UINT32 JDN, int *pMonth, int *pDay)
{
  int doy;

  /* number of days in current year */
  doy = (SdnToMayaDays (JDN) % 365) - 12 - 5;
  if (0 > doy)
    doy = doy + 365;

/*
   if (365 < doy)
   doy = doy - 365;
 */
  if (doy > 359)

    {

      /* the month of Uayeb */
      *pDay = doy - 360;
      *pMonth = 0;
    }

  else

    {

      /* days of month and month of civil year */
      *pDay = doy % 20;
      *pMonth = (doy / 20) + 1;
    }
}


#ifdef TESTMAYA
int
main ()
{
  UINT32 JDN;
  char str[100];
  int hm, hd;
  printf ("APRIL 15 1999\n");
  JDN = 2451284;		/* apr-14-1999 */
  SdnToMayaLongcountStr (JDN, str);
  printf ("longcount (%lu): %s\n", JDN, str);
  printf ("Tzolkin: %i  %s(%i)\n", SdnToTzolkin13 (JDN),
	  MayaTzolkinDays[SdnToTzolkin20 (JDN)], SdnToTzolkin20 (JDN));
  SdnToHaab (JDN, &hm, &hd);
  printf ("Haab: %i %s(%i)\n", hd, MayaHaabMonths[hm], hm);
  printf ("\nBEGINNING OF CALENDAR\n");
  JDN = MAYA_FIRST_JDN;
  SdnToMayaLongcountStr (MAYA_FIRST_JDN, str);
  printf ("longcount (%lu): %s\n", JDN, str);
  printf (" --> should be 13.0.0.0.0\n");
  printf ("Tzolkin: %i  %s(%i)\n", SdnToTzolkin13 (JDN),
	  MayaTzolkinDays[SdnToTzolkin20 (JDN)], SdnToTzolkin20 (JDN));
  printf (" --> should be 4 %s(0)\n", MayaTzolkinDays[0]);
  SdnToHaab (JDN, &hm, &hd);
  printf ("Haab: %i %s(%i)\n", hd, MayaHaabMonths[hm], hm);
  printf (" --> should be 8 Cumku\n");

#ifdef not
  /* on 21 or 23 December AD 2012 */
  printf ("\n``Y2K'' OF 2012AD-DEC-21\n");
  JDN = 2456283;
  SdnToMayaLongcountStr (MAYA_FIRST_JDN, str);
  printf ("SdnToMayaDays = %lu\n", SdnToMayaDays (JDN));
  printf ("longcount (%lu): %s\n", JDN, str);
  printf ("Tzolkin: %i  %s(%i)\n", SdnToTzolkin13 (JDN),
	  MayaTzolkinDays[SdnToTzolkin20 (JDN)], SdnToTzolkin20 (JDN));
  SdnToHaab (JDN, &hm, &hd);
  printf ("Haab: %i %s(%i)\n", hd, MayaHaabMonths[hm], hm);

#endif /*  */
  return 0;
}


#endif /*  */
