/****************************************************************************/
/* JPDATA                                                                   */
/*--------------------------------------------------------------------------*/
/* JPTUI Data Access Functions                                              */
/*--------------------------------------------------------------------------*/
/* Code adapted from EXEDAT 1.0 (Ren Olsthoorn - 1996) by JP Delprat       */
/* LZSS_DECODE(PD) by Haruhiko Okumura                                      */
/* LZARI_DECODE(PD) by Haruhiko Okumura					    */
/* EXEDAT by  in 1996                                                       */
/*--------------------------------------------------------------------------*/
/* Author      : DELPRAT Jean-Pierre / Ren Olsthoorn / Haruhiko Okumura    */
/* Created on  : 05-Jul-96                                                  */
/****************************************************************************/

#include <io.h>
#include <fcntl.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\stat.h>

#include "Compat.h"

#include "Files.h"
#include "JPData.h"
#include "Vocab.h"

#include "Types.h"

#ifdef __TCPLUSPLUS__
#else
extern char *__dos_argv0;
#endif

#define FILE_NAME_LENGTH   12
static char s_file_name[FILE_NAME_LENGTH+1]="????????.???";

/****************************************************************************
*************************  BEGIN EXEDAT HEADER  *****************************
****************************************************************************/


#define LZ_COMPR_NONE 		0
#define LZ_COMPR_LZSS 		1
#define LZ_COMPR_LZARI 		2
#define LZ_N			4096
#define LZ_F			18
#define LZ_THRESHOLD  		2
#define LZ_F2         		60
#define LZ_M          		15
#define LZ_Q1  			(1UL << LZ_M)
#define LZ_Q2  			(2 * LZ_Q1)
#define LZ_Q3  			(3 * LZ_Q1)
#define LZ_Q4  			(4 * LZ_Q1)
#define LZ_MAX_CUM 		(LZ_Q1 - 1)
#define LZ_N_CHAR  		(256 - LZ_THRESHOLD + LZ_F2)

static FILE          *s_file;
static unsigned char *s_to_buffer;
static unsigned long s_from_pointer,
		     s_from_max,
		     s_to_pointer;
static unsigned long s_low,
		     s_high,
		     s_value;

static int           s_char_to_sym[LZ_N_CHAR],
		     s_sym_to_char[LZ_N_CHAR + 1];

static unsigned int  s_sym_freq[LZ_N_CHAR + 1],
		     s_sym_cum[LZ_N_CHAR + 1],
		     s_position_cum[LZ_N + 1];

static unsigned char s_text_buf [LZ_N + LZ_F2 - 1];
static unsigned long s_textsize;

static unsigned int  LZ_GB_buffer,
		     LZ_GB_mask;



#pragma pack(1)
#pragma option -a-
struct exedat_header
{
  char header[8];       // identification = "EXEDAT"
  short version;        // 1
  long number_files;    // number of files in archive
} header;

struct exedat_indexblock
{
  char name[13];        // name of file in archive
  long lokation;        // offset of filedata from end of .exe file
  long length;          // filesize (could be compressed size)
  long original_length; // filesize without compression
  short compression;    // compression type (see LZ_COMPR_LZSS at top)
} indexblock;
#pragma pack()


static int PutcBuffer(int outc)
{
  #ifdef __TCPLUSPLUS__
  s_to_buffer[(unsigned)s_to_pointer++] = (char) outc;
  #else
  s_to_buffer[s_to_pointer++] = (char) outc;
  #endif
  return(1);
}

static int GetcLZSS(void)
{
  if (s_from_pointer++ == s_from_max)
    return(EOF);

  return(getc(s_file));
}

static void LZSSDecode(void)
{
  int i, j, k, r, c;

  unsigned int flags;

  memset(s_text_buf,' ',LZ_N-LZ_F);

  r = LZ_N - LZ_F;
  flags = 0;

  for ( ; ; )
    {
      if (((flags >>= 1) & 256) == 0)
	{
	  if ((c = GetcLZSS()) == EOF)
	     break;
	  flags = c | 0xff00;
	}

      if (flags & 1)
	{
	  if ((c = GetcLZSS()) == EOF)
	    break;
	  PutcBuffer(c);
	  s_text_buf[r++] = c;
	  r &= (LZ_N - 1);
	}
      else
	{
	  if ((i = GetcLZSS()) == EOF)
	    break;
	  if ((j = GetcLZSS()) == EOF)
	    break;
	  i |= ((j & 0xf0) << 4);
	  j = (j & 0x0f) + LZ_THRESHOLD;
	  for (k = 0; k <= j; k++)
	    {
	      c = s_text_buf[(i + k) & (LZ_N - 1)];
	      PutcBuffer(c);
	      s_text_buf[r++] = c;
	      r &= (LZ_N - 1);
	    }
	}
    }
}

static void LZSSDecompress(unsigned char* tbuffer, long comp_size)
{
  s_from_pointer = 0;
  s_from_max = comp_size;
  s_to_pointer = 0;
  s_to_buffer = tbuffer;
  LZSSDecode();
}

static int GetBit(void)
{
  if ((LZ_GB_mask >>= 1) == 0)
    {
      LZ_GB_buffer = GetcLZSS();
      LZ_GB_mask = 128;
    }
  return ((LZ_GB_buffer & LZ_GB_mask) != 0);
}

static void StartModel(void)
{
  int ch, sym, i;

  s_sym_cum[LZ_N_CHAR] = 0;
  for (sym = LZ_N_CHAR; sym >= 1; sym--)
    {
      ch = sym - 1;
      s_char_to_sym[ch] = sym;  s_sym_to_char[sym] = ch;
      s_sym_freq[sym] = 1;
      s_sym_cum[sym - 1] = s_sym_cum[sym] + s_sym_freq[sym];
    }
  s_sym_freq[0] = 0;
  s_position_cum[LZ_N] = 0;
  for (i = LZ_N; i >= 1; i--)
    s_position_cum[i - 1] = s_position_cum[i] + 10000 / (i + 200);
}


static void UpdateModel(int sym)
{
  int i, c, ch_i, ch_sym;

  if (s_sym_cum[0] >= LZ_MAX_CUM)
    {
      c = 0;
      for (i = LZ_N_CHAR; i > 0; i--)
	{
	  s_sym_cum[i] = c;
	  c += (s_sym_freq[i] = (s_sym_freq[i] + 1) >> 1);
	}
      s_sym_cum[0] = c;
    }

  for (i = sym; s_sym_freq[i] == s_sym_freq[i - 1]; i--) ;

  if (i < sym)
    {
      ch_i = s_sym_to_char[i];
      ch_sym = s_sym_to_char[sym];
      s_sym_to_char[i] = ch_sym;
      s_sym_to_char[sym] = ch_i;
      s_char_to_sym[ch_i] = sym;
      s_char_to_sym[ch_sym] = i;
    }
  s_sym_freq[i]++;

  while (--i >= 0)
    s_sym_cum[i]++;
}


static int BinarySearchSym(unsigned int x)
{
  int i, j, k;

  i = 1;
  j = LZ_N_CHAR;
  while (i < j)
    {
      k = (i + j) / 2;
      if (s_sym_cum[k] > x) i = k + 1;  else j = k;
    }
  return i;
}

static int BinarySearchPos(unsigned int x)
{
  int i, j, k;

  i = 1;
  j = LZ_N;
  while (i < j)
    {
      k = (i + j) / 2;
      if (s_position_cum[k] > x) i = k + 1;  else j = k;
    }
  return i - 1;
}

static void StartDecode(void)
{
  int i;
  for (i = 0; i < LZ_M + 2; i++)
    s_value = 2 * s_value + GetBit();
}

static int DecodeChar(void)
{
  int 		     sym, ch;
  unsigned long int  range;

  range = s_high - s_low;
  sym = BinarySearchSym((unsigned int)
	  (((s_value - s_low + 1) * s_sym_cum[0] - 1) / range));
  s_high = s_low + (range * s_sym_cum[sym - 1]) / s_sym_cum[0];
  s_low +=       (range * s_sym_cum[sym    ]) / s_sym_cum[0];
  for ( ; ; )
    {
      if (s_low >= LZ_Q2)
	{
	  s_value -= LZ_Q2;
	  s_low -= LZ_Q2;
	  s_high -= LZ_Q2;
	}
      else if (s_low >= LZ_Q1 && s_high <= LZ_Q3)
	{
	  s_value -= LZ_Q1;
	  s_low -= LZ_Q1;
	  s_high -= LZ_Q1;
	}
      else if (s_high > LZ_Q2)
	break;

      s_low += s_low;
      s_high += s_high;
      s_value = 2 * s_value + GetBit();
    }
  ch = s_sym_to_char[sym];
  UpdateModel(sym);
  return ch;
}

static int DecodePosition(void)
{
  int position;
  unsigned long int  range;

  range = s_high - s_low;
  position = BinarySearchPos((unsigned int)
	  (((s_value - s_low + 1) * s_position_cum[0] - 1) / range));
  s_high = s_low + (range * s_position_cum[position    ]) / s_position_cum[0];
  s_low +=       (range * s_position_cum[position + 1]) / s_position_cum[0];
  for ( ; ; )
    {
      if (s_low >= LZ_Q2)
	{
	  s_value -= LZ_Q2;
	  s_low -= LZ_Q2;
	  s_high -= LZ_Q2;
	}
      else if (s_low >= LZ_Q1 && s_high <= LZ_Q3)
	{
	  s_value -= LZ_Q1;  s_low -= LZ_Q1;  s_high -= LZ_Q1;
	}
      else if (s_high > LZ_Q2)
	break;

      s_low += s_low;
      s_high += s_high;
      s_value = 2 * s_value + GetBit();
    }
  return position;
}

static void LZARIDecode(void)
{
  int  i, j, k, r, c;
  unsigned long int  count, temp;

  s_textsize = 0;
  temp = GetcLZSS() << 24;
  s_textsize |= temp;
  temp = GetcLZSS() << 16;
  s_textsize |= temp;
  temp = GetcLZSS() << 8;
  s_textsize |= temp;
  temp = GetcLZSS();
  s_textsize |= temp;

  if (s_textsize == 0)
    return;

  StartDecode();
  StartModel();

  for (i = 0; i < LZ_N - LZ_F2; i++)
    s_text_buf[i] = ' ';

  r = LZ_N - LZ_F2;
  for (count = 0; count < s_textsize; )
    {
      c = DecodeChar();
      if (c < 256)
	{
	  PutcBuffer(c);
	  s_text_buf[r++] = c;
	  r &= (LZ_N - 1);
	  count++;
	}
      else
	{
	  i = (r - DecodePosition() - 1) & (LZ_N - 1);
	  j = c - 255 + LZ_THRESHOLD;
	  for (k = 0; k < j; k++)
	    {
	      c = s_text_buf[(i + k) & (LZ_N - 1)];
	      PutcBuffer(c);
	      s_text_buf[r++] = c;
	      r &= (LZ_N - 1);  count++;
	   }
	}
    }
}


static void LZARIDecompress(unsigned char* tbuffer, long comp_size)
{
  s_textsize = 0;
  s_from_pointer = 0;
  s_to_pointer = 0;
  LZ_GB_buffer = 0;
  LZ_GB_mask = 0;
  s_low = 0;
  s_high = LZ_Q4;
  s_value = 0;
  s_from_max = comp_size;
  s_to_buffer = tbuffer;
  LZARIDecode();
}


/****************************************************************************
*************************  END EXEDAT HEADER  *******************************
****************************************************************************/

static boolean ExedatOpen(char *archive_name)
{
  long seeklong;
  int handle;

  handle = sopen(archive_name, O_RDONLY | O_BINARY, SH_DENYWR, S_IREAD);
  if (handle<0)
    return(FALSE);

  s_file=fdopen(handle,"rb");
  if (s_file==NULL)
    {
      close(handle);
      return(FALSE);
    }

  memset(header.header,0,sizeof(header.header));

  seeklong = sizeof(header);
  fseek(s_file, -seeklong, SEEK_END);
  fread(&header, (unsigned)seeklong,1,s_file);

  if (strcmp(header.header,"EXEDAT"))
    {
      fclose(s_file);
      return(FALSE);
    }

  return(TRUE);
}


// Allocates a buffer and load a data file into this buffer.
// This buffer must be freed by delete [].
// Returns a pointer to the buffer or NULL if an error occurs.

void LoadDataFile(char *file_name,char *(&buffer),long &length)
{
  int handle;

  char *exe_name;
  char archive_name[MAX_PATH];
  long i, seeklong;
  boolean found;

  char *buf;

  strncpy(s_file_name,file_name,FILE_NAME_LENGTH);
  s_file_name[FILE_NAME_LENGTH]=0;

  buffer=NULL;
  length=0;

  #ifdef __TCPLUSPLUS__
  exe_name=_argv[0];
  #else
  exe_name=__dos_argv0;
  #endif

  // On cherche d'abord le fichier  charger dans le rpertoire courant,
  // puis on le cherche  la fin de l'excutable

  FullPathOfFileInExeDir(archive_name,file_name);

  handle=sopen(file_name, O_RDONLY | O_BINARY, SH_DENYWR, S_IREAD);
  if (handle>=0)
    {
      length=filelength(handle);

      #ifdef __TCPLUSPLUS__
      buffer=new char [(unsigned)length];
      read(handle,buffer,(unsigned)length);
      #else
      buffer=new char [length];
      read(handle,buffer,length);
      #endif

      close(handle);
      return;
    }

  FullPathOfFileInExeDir(archive_name,"JPTUI.DAT");
  if (!ExedatOpen(archive_name))
    {
      if (!ExedatOpen(exe_name))
	return;
    }

  found=FALSE;
  seeklong = sizeof(header);
  for (i = 0; i < header.number_files; i++)
    {
      seeklong += sizeof(indexblock);
      fseek(s_file, -seeklong, SEEK_END);
      fread(&indexblock, sizeof(indexblock),1,s_file);
      if (stricmp(indexblock.name, file_name) == 0)
	{
	  found=TRUE;
	  break;
	}
    }

  if (!found)
    return;

  #ifdef __TCPLUSPLUS__
  buf=new char [(unsigned)indexblock.original_length];
  #else
  buf=new char [indexblock.original_length];
  #endif

  fseek(s_file, -indexblock.lokation, SEEK_END);
  switch (indexblock.compression)
    {
      case LZ_COMPR_LZARI: LZARIDecompress((unsigned char *)buf, indexblock.length);
			break;
      case LZ_COMPR_LZSS : LZSSDecompress((unsigned char *)buf, indexblock.length);
			break;
      case LZ_COMPR_NONE : fread(buf,(unsigned)indexblock.length,1,s_file);
			break;
      default         : fclose(s_file);
			delete []buf;
			return;
    }
  fclose(s_file);

  buffer=buf;
  length=indexblock.original_length;
}

char *GetInvalidDataFileMessage()
{
  char *text;
  int temp;
  int length=strlen(s_file_name);

  switch (GetLanguage())
    {
      case FRENCH : text="xxxxxxxx.xxx (JPTUI.DAT)\nFichier de donnes absent ou invalide !!!";
		    break;
      case GERMAN : text="xxxxxxxx.xxx (JPTUI.DAT)\nDatendatei ungltig oder nicht gefunden !!!";
		    break;
      case ITALIAN: text="xxxxxxxx.xxx (JPTUI.DAT)\nIl file di dati  non valido o mancante !!!";
		    break;
      case SPANISH: text="xxxxxxxx.xxx (JPTUI.DAT)\nEl fichero de datos es invlido o no se encuentra!";
		    break;
      case DUTCH  : text="xxxxxxxx.xxx (JPTUI.DAT)\nGeen of ongeldig gegevensbestand !!!";
                    break;
      case GREEK  : text="xxxxxxxx.xxx (JPTUI.DAT)\n         !!!";
                    break;

      default     : text="xxxxxxxx.xxx (JPTUI.DAT)\nData file is invalid or missing !!!";
		    break;
    }

  memcpy(text,s_file_name,length);
  temp=FILE_NAME_LENGTH-length;
  if (temp>0)
    memset(text+length,' ',temp);
  return(text);
}
