	include	mmclock.mac
	title	Test for MM58167A clock/calendar -- Copyright 1990 Wales

; ======================================================================
;
; Test for the existence of an MM58167A clock/calendar.
; (C) Copyright 1990 Richard B. Wales.  All Rights Reserved.
;
; Global variables:
;
;	None.
;
; Global routines:
;
;	LookForClock
;		Attempts to locate a clock/calendar.

; ======================================================================
;
; Global symbols.

	public	LookForClock

; ======================================================================
;
; Default list of I/O ports at which a clock might be found.

PortList dw	340h, 240h, 2C0h, 0C0h, 0

; ======================================================================
;
; LookForClock
;
;	Looks for an MM58167A clock/calendar chip.  If a clock is found,
;	its I/O port address is copied into "ClkPort".  This routine
;	relies on various idiosyncrasies of the MM58167A chip; and, in
;	order not to disturb other devices, does as much "read-only"
;	checking at each address as possible before writing anything.
;
;	Arguments:
;
;		BX	Should be set to point to the start of an array
;			of initial I/O port addresses (2-byte words) at
;			which to look for a clock.  The list should end
;			with a zero value.  If BX = 0, a default port
;			list is used.
;
;	Returned values:
;
;		DX	Set to the initial I/O port address where the
;			clock was actually found.  If no clock was found,
;			DX will be set to zero.

LookForClock proc near

	; Set up the registers.
	INIT_DS				; need DS to get port # from [BX]
	mov	cx, 4			; CL used in SHL/SHR and added to DX
	and	bx, bx
	jnz	short try_next_port	; if BX = 0 initially,
	mov	bx, offset PortList	;     use default port list

try_next_port:
	; Select a candidate port number.
	mov	dx, [bx]		; get port number
	and	dx, dx			; is port number zero?
	jz	short give_up		;     if so, give up

	; Read from the starting port.  If there is a clock present,
	; the four low-order bits of the value read will be zero,
	; and the four high-order bits will be in the range 0-9.
	in	al, dx
	SPLIT	al
	jnz	short no_clock		; are 4 low-order bits zero?
	cmp	ah, 9
	ja	short no_clock		; are 4 high-order bits <= 9?

	; Try reading the first byte of clock RAM.  Since the low-order
	; four bits are not supported in this byte, the value read should
	; have zeros in those bit positions.
	add	dx, cx
	add	dx, cx			; DX = addr of first RAM byte
	in	al, dx
	shl	al, cl
	jnz	short no_clock

	; Similarly, try reading the sixth byte of clock RAM.  The high-
	; order four bits are not supported in this byte, so the value
	; read should have zeros in those bit positions.
	add	dx, cx
	inc	dx			; DX = addr of sixth RAM byte
	in	al, dx
	shr	al, cl
	jnz	short no_clock

	; Try writing to the first byte of clock RAM, then read the
	; value back.  The value read back should have zeros in the four
	; low-order bit positions, regardless of the value written out.
	dec	dx
	sub	dx, cx			; DX = addr of first RAM byte
	mov	al, 05Ah
	out	dx, al			; write 05Ah . . .
	DELAY	3
	in	al, dx
	cmp	al, 050h		; . . . better read back 050h
	jne	short no_clock
	mov	al, 0A5h
	out	dx, al			; write 0A5h . . .
	DELAY	3
	in	al, dx
	cmp	al, 0A0h		; . . . better read back 0A0h
	jne	short no_clock

	; Similarly, try writing to the sixth byte of clock RAM, then
	; reading the value back.  The value read back should have zeros
	; in the high-order four bit positions, regardless of the value
	; written out.
	add	dx, cx
	inc	dx			; DX = addr of sixth RAM byte
	mov	al, 05Ah
	out	dx, al			; write 05Ah . . .
	DELAY	3
	in	al, dx
	cmp	al, 0Ah			; . . . better read back 0Ah
	jne	short no_clock
	mov	al, 0A5h
	out	dx, al			; write 0A5h . . .
	DELAY	3
	in	al, dx
	cmp	al, 05h			; . . . better read back 05h
	jne	short no_clock

;	We found the clock.
	mov	dx, [bx]		; I/O port address of clock
	ret

no_clock:
	; If any of the tests failed, there isn't an MM58167A chip
	; at the port address in question.  Try next port (if any).
	inc	bx
	inc	bx
	jmp	short try_next_port

give_up:
	; No clock found at any of the addresses.
	xor	dx, dx			; no I/O port address
	ret

LookForClock endp

code	ends

	end