;*************************************************************************
;* FROM:   George Hug
;*
;* This method uses a "linear maximal-length shift register" to generate
;* a series of pseudo-random bits. The register has 31 bits, and must be
;* seeded initially with some non-zero value. A random bit is generated
;* by shifting the entire register one bit to the left, with the
;* exclusive-OR of the 28th and 31st bits being shifted into the first
;* bit from the right. The random bit stream can be read from any bit
;* position in the register, so long as the same position is read each
;* time.  The stream produced will repeat every (2^31 - 1) shifts.
;*
;* If you want to generate random integers falling within a "convenient"
;* range, such as 0 through 1023, inclusive, then you only have to shift
;* the register N=10 times (2^10 = 1024), thereby generating 10 random
;* bits, and read the resulting value out of the bottom 10 bits of the
;* shift register. However, an "inconvenient" range, such as 0 through
;* 999, presents some problems. You could generate the next largest
;* convenient range, and just ignore any value which is greater than
;* 999, but that might affect the randomness of the numbers (I don't
;* know for sure).
;*
;* A safer approach is to convert each generated integer in the
;* convenient range to its proportional equivalent in the desired range.
;* However, in that case the convenient range must be a good bit larger
;* than the desired range so as to avoid rounding anomalies.
;*
;* The code below generates 16-bit integers (0-65565), then converts
;* them to the range 0-999. Each of the 1000 possible integers will
;* occur 65.536 times on average out of every 65536 tries, but with
;* rounding (actually truncation), some values will occur 65 times on
;* average, while others will occur 66 times. That means that even if
;* the 16-bit integers are truly random, some values within the desired
;* range are about 1.5% more likely to occur than the rest.
;*
;* Here are some publications that will give you more information about
;* linear maximal-length shift registers:
;*
;* Horowitz and Hill, The Art of Electronics, 2nd Ed., pp. 656-7.
;* Gregory D. Knox, "Random Number Generation in Machine Language",
;*     Transactor Magazine, May 1987.
;* Don Lancaster, "Hardware Hacker", Radio-Electronics, Feb. 1989, p.79.
;*************************************************************************

ShiftRegLo    dw    5555h            ;seed with any non-zero value
ShiftRegHi    dw    5555h

;***************************************
;* Generate random number [0..AX-1]
;***************************************
PROC        RandInt
        push    bx cx dx
        push    ax            ;Save limiter
        mov    ebx,ShiftRegLo
        mov    dx,ShiftRegHi
        mov    cx,16            ;generate 16 random bits

RandIntLoop:    mov    ah,dh            ;line up the taps
                shl     ah,1
                shl     ah,1
        shl    ah,1            ;28th & 31st bits now aligned
        xor    ah,dh            ;XOR the taps
        shl    ah,1            ;shift result into Carry
                shl     ah,1
        rcl    bx,1            ;now shift the register
                rcl     dx,1
        loop    RandIntLoop        ;do it CX times

        mov    ShiftRegLo,bx        ;save back for next time
        mov    ShiftRegHi,dx

        pop    ax            ;prepare for ax*bx/65536
        mul    bx            ;product in DX:AX
        mov    ax,dx            ;truncate & move answer to AX
        pop    dx cx bx
        ret                ;return with rand # in AX
ENDP
