/*
   LCDCall -- C subroutines used to call LCDBios

   version 1.31
   written by Donald Sawdai, 11/95
   copyright 1995, 1996 Donald Sawdai

   This code may be freely used, distributed, and compiled royalty-free into
   applications as long as it is unmodified in any way.  Donald Sawdai
   gives no warantee of any sort for this code.  However, I am always
   interested in comments (send to dsawdai@engin.umich.edu).

   requirements:  AT class machine, 386 or higher
		  register-compatible VGA card
		  compatible 3D glasses

   note:	  This has been compiled and tested using 386 instruction set
		  and (1) real-mode small model on Borland C/C++ 3.1
		  and (2) protected-mode on Watcom C/C++ 10.5 with the
		  DOS/4GW extender.

   See documentation for details.

   Current limitations/bugs:
   (1) Any program interference with the AT Real-Time Clock (RTC) or with VGA
       registers (CRTC Offset register, CRTC Max Scan Line register,
       CRTC Vertical Retrace End register, or CRTC Start Address High/Low
       registers) while 3D mode is on may conflict with this program.  The
       most likely conflict is screen flipping for buffered screen draws.
       The screen flipping (multiple buffering) must be done through
       LCDBios functions.
*/

#ifdef __BORLANDC__
/* conditional definitions for Borland C/C++ version 3.1 */
#include <alloc.h>
#define WORD x			/* word registers in REGS */
#define DOS_PTR far	       	/* pointer to DOS area is "far" */
#define INT386 int86		/* use "int86" and "int86x" calls */
#define LCD_FLAGS LCDFlagsR 	/* use real-mode pointer */

#elif __WATCOMC__
/* conditional definitions for Watcom C/C++ version 10.5
   (with DOS/4GW extender) */
#include <i86.h>
#include <malloc.h>
#define WORD w			/* word registers in REGS */
#define DOS_PTR			/* pointer to DOS area is near in flat model */
#define INT386 int386		/* use "int86" and "int86x" calls */
#define LCD_FLAGS LCDFlagsP	/* use protected-mode pointer */

#endif

#include <mem.h>
#include <dos.h>
#include "lcdcall.h"

#define VIDEO 0x10		/* video/LCDBios interrupt number */

#pragma pack ( 2 );       	/* make sure data structure is word aligned */
typedef struct {
/* Buffer used in calling Set 3D mode function */
  unsigned short int leftX;
  unsigned short int leftY;
  unsigned short int rightX;
  unsigned short int rightY;
  short int pageAdvanced;
  long int interruptCount;
  long int delayCount;
} LCDBuffer;
#pragma pack;


volatile LCDFlags far *LCDFlagsR;
/* LCDFlagsR is a real-mode far pointer */

volatile LCDFlags *LCDFlagsP;
/* LCDFlagsP is a protected-mode flat pointer.  It is the absolute
address of the structure in the DOS 1MB memory. */



LCD3DMode	mode = Off; 	/* current LCDBIOS mode */
int 		numPages,	/* number of pages (after init LCDBios) */
		curPage;	/* current page being shown */
PageAddresses 	*pages;		/* pointer to array of pages */
int		width;		/* for Alternate* modes, value added to column
				   address of right eye (offset from left) */
LCDBuffer DOS_PTR *buffer; 	/* pointer to buffer in LCDBios */

int isLCDBiosInstalled( int *majorVersion, int *minorVersion)
/* Return:     1 if LCDBIOS is installed.
	       0 if it is not.
   Output:     majorVersion = major version number
	       minorVersion = minor version number (0-99)
*/
{
  union REGS regs;

  memset(&regs, 0, sizeof(regs));
  regs.WORD.ax = 0x4fd3;  	/* LCDBios call */
  regs.WORD.bx = 0x3d00;  	/* get version */
  INT386( VIDEO, &regs, &regs);

  if ((regs.WORD.ax == 0x004f) && (regs.WORD.dx == 0x3344) &&
      (regs.WORD.cx == 0x4c42))
  {
    /* LCDBios is loaded */
    *majorVersion = regs.h.bh;
    *minorVersion = regs.h.bl;
    return(1);
  } else {
    /* LCDBios is not loaded */
    return(0);
  }

}


int LCDBiosInit3DMode( LCD3DMode setMode, int setNumberOfPages,
		       PageAddresses *setPages )
/* Return:     1 on success.  0 on failure.
   Input:      mode = [ TwoBuffer, AlternateFull, AlternateHalf]
	       numberOfPages = number of stereo pages to use.
		    Each page consists of a left image and a
		    right buffer.  For example, in TwoBuffer
		    mode, if there are 2 left buffers and 2 right
		    buffers, it is still considered to be 2
		    pages.
	       pages = pointer to array of PageAddresses (one
		    per page, as specified by numberOfPages).
		    For TwoBuffer mode, both left- and right-
		    image addresses must be specified for each
		    page.  For Alternate... modes, only the left-
		    image address should be used.  The pages are
		    numbered sequentially starting at 0.
Description:
LCDBIOS puts the screen in 3D mode (as specified by "setMode") and
activates the shutter glasses.  The current page is page 0.
LCDFlags... pointers are set to point to the LCDFlag structure in LCDBios.
The video card must be in a VGA or VESA graphics mode prior to calling
this function.  If 3D mode is active in the same mode as "setMode", the
page array is reloaded with the current page set to 0 (but no other action).
*/
{
  union  REGS regs;
  int	i;

  if (mode != Off)
  {
    /* 3D mode is already on */
    if (mode != setMode)
    {
      /* we are in a different mode! */
      return(0);
    } else {
      /* we are in same mode, so reset pages and exit */
      pages = (PageAddresses *) realloc(pages,
			 setNumberOfPages * sizeof(PageAddresses));
      memcpy( pages, setPages, setNumberOfPages * sizeof(PageAddresses));

      numPages = setNumberOfPages;
      curPage = 0;

      if ((mode == AlternateFull) || (mode == AlternateHalf)) {
	/* calculate right-eye pages for left-eye data */
	for (i=0; i<numPages; i++) {
	  pages[i].VESA.rightX = pages[i].VESA.leftX + width;
	  pages[i].VESA.rightY = pages[i].VESA.leftY;
	}
      }

      LCDBiosSetPage( 0 );

      return(1);
    }
  }

  mode = setMode;
  pages = (PageAddresses *)
	      malloc(setNumberOfPages * sizeof(PageAddresses));
  memcpy( pages, setPages, setNumberOfPages * sizeof(PageAddresses));

  numPages = setNumberOfPages;
  curPage = 0;

  switch(setMode) {
  case AlternateFull:
  case AlternateHalf:
    /* switch to TwoBuffer screen mode */
    memset(&regs, 0, sizeof(regs));
    regs.WORD.ax = 0x4fd3;  	/* LCDBios call */
    regs.WORD.bx = 0x3d04;  	/* Alternate* to TwoBuffer */
    regs.WORD.cx = 2 * (setMode == AlternateHalf); /* AlternateHalf or Full */
    INT386( VIDEO, &regs, &regs);
    if (regs.WORD.ax == 0x014f) {
      free( pages );
      mode = Off;
      return(0);
    }

    width = regs.WORD.bx;
    /* calculate right-eye pages for left-eye data */
    for (i=0; i<numPages; i++) {
      pages[i].VESA.rightX = pages[i].VESA.leftX + width;
      pages[i].VESA.rightY = pages[i].VESA.leftY;
    }

  case TwoBuffer:
    break;
  default:
    free( pages );
    mode = Off;
    return(0);
  }

  /* enter 3D mode, displaying page 0 */
  memset(&regs, 0, sizeof(regs));
  regs.WORD.ax = 0x4fd3;  	/* LCDBios call */
  regs.WORD.bx = 0x3d01;  	/* enter 3D mode */
  regs.WORD.cx = pages[0].VESA.rightX;
  regs.WORD.dx = pages[0].VESA.rightY;
  regs.WORD.si = pages[0].VESA.leftX;
  regs.WORD.di = pages[0].VESA.leftY;
  INT386( VIDEO, &regs, &regs );
  if (regs.WORD.ax == 0x014f) {
    free( pages );
    mode = Off;
    return(0);
  }

  LCDFlagsR = (LCDFlags far *) MK_FP( regs.WORD.dx, regs.WORD.di + 8 );
  LCDFlagsP = (LCDFlags *) ( (((long) regs.WORD.dx) << 4) +
				  ((long) (regs.WORD.di & 0xffff)) + 8 );
  buffer = (LCDBuffer DOS_PTR *) ( (char DOS_PTR *)LCD_FLAGS - 8);

  LCDBiosSetPage( 0 );
  LCD_FLAGS->pageAdvanced = 2;		/* we are in calibration mode */

  /* success! */
  return(1);
}


int LCDBiosDeinit3DMode( void )
/* Return:     1 on success.  0 on failure.
Description:
LCDBIOS removes the screen from 3D mode.  LCDFlags... pointers are
set to NULL.
*/
{
  union REGS regs;

  if (mode == Off)
  {
    /* 3D mode is already off */
    return(0);
  }

  free(pages);

  numPages = 0;
  curPage = 0;

  buffer = NULL;
  LCDFlagsR = NULL;
  LCDFlagsP = NULL;

  /* exit 3D mode, displaying left page of page 0 */
  LCDBiosSetPage( 0 );
  memset(&regs, 0, sizeof(regs));
  regs.WORD.ax = 0x4fd3;  	/* LCDBios call */
  regs.WORD.bx = 0x3d02;  	/* exit 3D mode */
  INT386( VIDEO, &regs, &regs);
  if (regs.WORD.ax == 0x014f) {
    mode = Off;
    return(0);
  }

  switch(mode) {
  case AlternateFull:
  case AlternateHalf:
    /* switch back to Alternate screen mode */
    memset(&regs, 0, sizeof(regs));
    regs.WORD.ax = 0x4fd3;  	/* LCDBios call */
    regs.WORD.bx = 0x3d05;  	/* TwoBuffer to Alternate* */
    INT386( VIDEO, &regs, &regs);
    if (regs.WORD.ax == 0x014f) {
      mode = Off;
      return(0);
    }
  case TwoBuffer:
    break;
  default:
    mode = Off;
    return(0);
  }

  mode = Off;

  /* success! */
  return(1);
}


int LCDBiosAdvancePage( void )
/* Return:     1 on success.  0 on failure.
Description:
Advances to next page in "pages" array specified by LCDBiosInit3DMode.
If the current page is the last page in the array, it is advanced
circularly to page 0.  This function also resets the pageAdvanced flag
to 0.  When the page is actually advanced, the pageAdvanced flag is
set to 1 by the timer interrupt.
*/
{
  if (++curPage >= numPages) curPage = 0;

  return( LCDBiosSetPage(curPage) );
}


int LCDBiosSetPage( int page )
/* Return:     1 on success.  0 on failure.
Description:
Sets the current page to "page" in pages array specified by
LCDBiosInit3DMode.  Resets pageAdvanced flag to 0.  When the page is
actually advanced, pageAdvanced flag is set to 1 by the timer
interrupt.
*/
{
  if (mode == Off)
  {
    /* 3D mode is off */
    return(0);
  }

  /* make 0 <= curPage < numPages */
  curPage = ( (page<0) ? 0 : page);
  curPage = ( (curPage>=numPages) ? numPages-1 : curPage);

  /* advance page */
  _disable();
  LCD_FLAGS->pageAdvanced = 0;
  buffer->leftX = pages[curPage].VESA.leftX;
  buffer->leftY = pages[curPage].VESA.leftY;
  buffer->rightX = pages[curPage].VESA.rightX;
  buffer->rightY = pages[curPage].VESA.rightY;
  _enable();

  /* advance page, alternate method
  { union REGS regs;
  memset(&regs, 0, sizeof(regs));
  regs.WORD.ax = 0x4fd3;      // LCDBios call
  regs.WORD.bx = 0x3d03;      // Change start address
  regs.WORD.cx = pages[curPage].VESA.rightX;
  regs.WORD.dx = pages[curPage].VESA.rightY;
  regs.WORD.si = pages[curPage].VESA.leftX;
  regs.WORD.di = pages[curPage].VESA.leftY;
  INT386( VIDEO, &regs, &regs);
  } */

  /* success! */
  return(1);
}

void LCDBiosPageWait( void )
/*
Description:
Waits until the current page is actually advanced (vertical retrace).  If
3D mode is not on, or if 3D mode is in its calibration period
(pageAdvanced = 2), then this function returns immediately.
*/
{
  if (mode == Off) return;		/* return if 3D mode is Off */

  while (!(LCD_FLAGS->pageAdvanced));	/* wait for page advance */
  return;
} 
