/* File name: mwindisp.c
 *
 * Synopsis:  This program displays a welcoming message to the user
 *            under Windows. It use the MIL VGA system and its extension 
 *            MvgaDispSelectClientArea() to display a message in a user
 *            created client window. Other systems like METEOR, PULSAR, ...
 *            also support MvgaDispSelectClientArea().
 *
 *            MvgaDispSelectClientArea() acts as the regular MdispSelect()
 *            of the MIL system but uses the specified client Window 
 *            to display instead of creating its own.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <windows.h>
#include <mil.h>
#include <mwinmil.h>

#define BUFFERSIZEX 512
#define BUFFERSIZEY 480
#define MAX_PATH_NAME_LEN  256

/* Prototypes */
void MilApplication(HWND UserWindowHandle);
void MilApplicationPaint(HWND UserWindowHandle);

/***************************************************************************/
/*/
* Name:         MilApplication()
*
* synopsis:     This function is the core of the MIL application that
*               will be executed when the "Start" menu item of this
*               Windows program will be selected. See WinMain() below
*               for the program entry point.
*
*               It will display a welcoming message in the user window
*               using MIL.
*
*/

void MilApplication(HWND UserWindowHandle)
{
  /* MIL variables */
  MIL_ID MilApplication,  /* MIL Application identifier.  */
         MilSystem,       /* MIL System identifier.       */
         MilDisplay,      /* MIL Display identifier       */
         MilImage;        /* MIL Image buffer identifier. */

  /* Allocate a MIL application. */
  MappAlloc(M_DEFAULT, &MilApplication);

  /* Allocate a MIL system. */
  MsysAlloc(M_SYSTEM_VGA, M_DEV0, M_DEFAULT, &MilSystem);

  /* Allocate a MIL display. */
  MdispAlloc(MilSystem, M_DEV0, "M_DEFAULT", M_DEFAULT, &MilDisplay);

  /* Allocate a MIL buffer. */
  MbufAlloc2d(MilSystem,BUFFERSIZEX,BUFFERSIZEY,
              8+M_UNSIGNED,M_IMAGE+M_DISP,&MilImage);

  /* Clear the buffer */
  MbufClear(MilImage,0);

  /* Select the buffer into it's display object using the given window */
  MvgaDispSelectClientArea(MilDisplay,MilImage,UserWindowHandle);

  /* Print a string in the image buffer using MIL.
     Note: After a MIL command writing in a MIL buffer, the display 
     will automaticly update the window given for the MvgaDispSelectClientArea()
  */
  MgraText(M_DEFAULT, MilImage, 176L, 210L, " ------------------ ");
  MgraText(M_DEFAULT, MilImage, 176L, 235L, " Welcome to MIL !!! ");
  MgraText(M_DEFAULT, MilImage, 176L, 260L, " ------------------ ");


  /* Windows code to open a message box for the user to wait a key. */
  MessageBox(0,"""Welcome to MIL !!!"" was printed",
               "MIL application example",
               MB_APPLMODAL | MB_ICONEXCLAMATION );


  /* Deselect the buffer from it's display object and given window */
  MvgaDispDeselectClientArea(MilDisplay,MilImage,UserWindowHandle);

  /* Free allocated objects */
  MbufFree(MilImage);
  MdispFree(MilDisplay);
  MsysFree(MilSystem);
  MappFree(MilApplication);
}


/***************************************************************************/
/*/
* Name:       MilApplicationPaint()
*
* synopsis:  This function can be used to complete the work done by
*            the MIL paint manager when the client window receives
*            a paint message.
*
*            This is called everytime the WindowProc receives a
*            WM_PAINT message
*/

void MilApplicationPaint(HWND UserWindowHandle)
{
   /* Nothing is done here because we dont draw other things on top
    * or around of our displayed MIL buffer in the window.
    */
    UserWindowHandle = UserWindowHandle; /* line to remove warning */
}

/***************************************************************************/
/* 
 * Synopsis:  The next functions let the user create a Windows program
 *            easily and very quickly. They will call the MilApplication()
 *            function each time the Start menu item is selected. The user
 *            may also use the function called MilApplicationPaint() that
 *            is called each time the window receives a paint message to
 *            complete the work already done by the MIL paint manager.
 */

/* Prototypes for Windows application */
int WINAPI WinMain(HINSTANCE, HINSTANCE, LPSTR, int);
long WINAPI MainWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL InitApplication(HINSTANCE, int);

/***************************************************************************/
/*
 *   Name:     WinMain()
 *
 *   Synopsis: Calls initialization function, processes message loop.
 */

int WINAPI WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPSTR     lpCmdLine,
                   int       nCmdShow)
{
   MSG msg;

   /* Avoid parameter not used warning */
   lpCmdLine = lpCmdLine;

   /* Only allow one instance of program. */
   if (hPrevInstance)
      {
      MessageBox(0, "Sample MIL Host Windows Application already active.\n"
                    "This version sample application does not allow multiple instances.",
                    "Sample MIL Host Application - Error", 
                     MB_OK | MB_ICONSTOP);
      return (FALSE);
      }

   /* Exits if unable to initialize */
   if (!InitApplication(hInstance,nCmdShow)) 
      return (FALSE);                      

   /* Get and dispatch messages until a WM_QUIT message is received. */
   while (GetMessage(&msg,(HWND)NULL,(UINT)NULL,(UINT)NULL))      
      {
      TranslateMessage(&msg);   /* Translates virtual key codes      */
      DispatchMessage(&msg);    /* Dispatches message to window      */
      }

   return (msg.wParam);    /* Returns the value from PostQuitMessage */
}


/***************************************************************************/
/*
 *   Name:      MainWndProc()
 *
 *   Synopsis:  Processes messages:
 *              WM_COMMAND     - application menu
 *              WM_ENDSESSION  - destroy window
 *              WM_CLOSE       - destroy window
 *              WM_DESTROY     - destroy window
 */

long WINAPI MainWndProc(HWND   hWnd,
                        UINT   message,
                        WPARAM wParam,
                        LPARAM lParam)
{
   HMENU hAppMenu;

   switch (message)
      {
    case WM_COMMAND:     /* message: command from application menu */
        switch(wParam)
            {
            case IDM_STARTMAIN:
               hAppMenu = GetMenu(hWnd);
               EnableMenuItem(hAppMenu,MF_BYCOMMAND | IDM_STARTMAIN,MF_GRAYED);
               DrawMenuBar(hWnd);
               MilApplication(hWnd);
               EnableMenuItem(hAppMenu,MF_BYCOMMAND | IDM_STARTMAIN,MF_ENABLED);
               DrawMenuBar(hWnd);
               break;
            default:
               return (DefWindowProc(hWnd, message, wParam, lParam));
            }
       break;
      case WM_PAINT:
         MilApplicationPaint(hWnd);
         return (DefWindowProc(hWnd, message, wParam, lParam));
      case WM_ENDSESSION:
      case WM_CLOSE:
         PostMessage(hWnd, WM_DESTROY, (WORD)0, (LONG)0);
         break;
      case WM_DESTROY:
         /* Free the reserved extra window LPSTR to szAppName */
         free((void *)GetWindowLong(hWnd,sizeof(LPVOID)));
         PostQuitMessage(0);

      default:
         return (DefWindowProc(hWnd, message, wParam, lParam));
      }
   return(0L);
}


/***************************************************************************/
/*
 *   Name:     InitApplication()
 *             
 *   Synopsis: Initializes window data, registers window class and 
 *             creates main window.
 */

BOOL InitApplication(HINSTANCE hInstance,
                     int       nCmdShow)

{
   WNDCLASS        wc;
   HWND            hwnd;
   char            TmpName[MAX_PATH_NAME_LEN];
   char *          szAppName;

   /* Allocation and storage of application name */
   szAppName = (LPSTR)malloc((40+MAX_PATH_NAME_LEN) * sizeof(char));
   strcpy(szAppName,MILAPPLNAME);

   wc.style          = CS_VREDRAW | CS_HREDRAW;
   wc.lpfnWndProc    = (WNDPROC)MainWndProc;
   wc.cbClsExtra     = 0;
   wc.cbWndExtra     = sizeof(LPVOID)+sizeof(LPSTR);
   wc.hInstance      = hInstance;
   wc.hIcon          = LoadIcon(0,IDI_APPLICATION);
   wc.hCursor        = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground  = CreateSolidBrush(RGB(BACKCOLORRED,
                                            BACKCOLORGREEN,
                                            BACKCOLORBLUE));
   wc.lpszMenuName   = "MILAPPLMENU";
   wc.lpszClassName  = szAppName;

   /* Register the class */
   if (!RegisterClass(&wc))
      return (FALSE);

   /* Create the window */
   hwnd = CreateWindow (szAppName,
                        szAppName,
                        WS_OVERLAPPEDWINDOW,
                        CW_USEDEFAULT, CW_USEDEFAULT,
                        CW_USEDEFAULT, CW_USEDEFAULT,
                        0, 0, hInstance, 0) ;
   if (!hwnd)
      return (FALSE);

   lstrcat(szAppName," - ");
   GetModuleFileName(hInstance,TmpName,MAX_PATH_NAME_LEN);
   lstrcat(szAppName,TmpName);
   SetWindowText(hwnd,szAppName);

   /* Set the reserved extra window LPVOID to NULL */
   SetWindowLong(hwnd,0,0L);

   /* Set the reserved extra window LPSTR to szAppName */
   SetWindowLong(hwnd,sizeof(LPVOID),(DWORD)szAppName);

   /* Display the window */
   ShowWindow (hwnd, nCmdShow) ;
   UpdateWindow (hwnd) ;

   return (TRUE);
}


