
/*=========================================================================*/
/* Konzept        : DATA BECKERs Sound Blaster Superbuch                   */
/* Modul SBDSP    : Stellt elementare Routinen zur direkten Programmierung */
/*                  (also nicht ber Treiber wie CT-VOICE.DRV) des DSP     */
/*                  (Digital Sound Processor) zur Verfgung.               */
/*=========================================================================*/
/* Autor          : Arthur Burda                                           */
/* Dateiname      : SBDSP.C                                                */
/* entwickelt am  : 04.07.1993                                             */
/* letztes Update : 01.09.1993                                             */
/* Version        : 1.06                                                   */
/* Compiler       : Turbo C, Turbo C++, Borland C++                        */
/*=========================================================================*/

#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include "sbdsp.h"

/*-------------------------------------------------------------------------*/
/* interne Variablen im Zusammenhang mit der Routine PlayDirect            */
/*-------------------------------------------------------------------------*/

static Byte    *BytePtr;                            /* Zeiger auf ein Byte */
static LongInt Count_;                                       /* ein Zhler */
static int     Done;         /* TRUE, wenn die Soundausgabe zu beenden ist */
static Word    BlockSize;          /* Gre des abzuspielenden Datenblocks */

/*=========================================================================*/
/* NewTimerInt: Neue Timer-Interruptroutine. ber sie wird eine Soundaus-  */
/*              gabe im Direktmodus realisiert.                            */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void interrupt NewTimerInt()

{
  WriteDirect(*BytePtr);                        /* Bytewert zum DSP senden */
  if (Count_ == BlockSize)                       /* Datenblock abgespielt? */
    Done = TRUE;                               /* ja, Soundausgabe beenden */
  else                                                             /* nein */
    {
      Count_++;                                          /* Zhler erhhen */
      BytePtr++;                                    /* nchstes Byte holen */
    }
  outportb(0x20, 0x20);                        /* Interruptroutine beenden */
}

/*=========================================================================*/
/* InitDSP: Initialisiert den DSP.                                         */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void InitDSP()

{
  Word Count;                                                /* ein Zhler */
  int  Reading;       /* TRUE, wenn der Bytewert AAh gelesen werden konnte */

  Count = 0;
  Reading = FALSE;
  outportb(SB_BaseAddress+0x06, 0x01);    /* Port 2x6h mit 01h beschreiben */
  delay(1);                                    /* eine Millisekunde warten */
  outportb(SB_BaseAddress+0x06, 0x00);    /* Port 2x6h mit 00h beschreiben */
  while (!Reading)            /* solange wiederholen, bis Wert AAh gelesen */
    {
      Count++;                                           /* Zhler erhhen */
      Reading = (ReadDSP() == 0xAA);                  /* Wert AAh gelesen? */
      if (Count == 1000)   /* 1000 Mikrosekunden (1 Millisek.) abgelaufen? */
        {                                               /* ja, also Fehler */
          printf("\n");
          printf("Fehler: Falsche Basis-I/O-Adresse oder Reset-Fehler\n");
          exit(1);                                     /* Programm beenden */
        }
    }
}

/*=========================================================================*/
/* SetTimerFreq: Stellt eine Frequenz fr den Timer ein. Der Wertebereich  */
/*		 liegt zwischen 19 und 50000 Hz.                           */
/*=========================================================================*/
/* Eingabe: Freq = Timer-Frequenz in Hz (19..50000)                        */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetTimerFreq(Word Freq)

{
  Word Timer;                          /* interner Wert fr Timer-Frequenz */

  /* Prfen, ob die angegebene Frequenz im Wertebereich zwischen */
  /* 19 und 50000 Hz liegt. Falls nicht, korrigieren.            */

  if (Freq < 19)
    Freq = 19;
  if (Freq > 50000)
    Freq = 50000;

  outportb(0x43, 0x36);                                    /* Steuerbefehl */
  Timer = 1193180/Freq;      /* internen Wert fr Timer-Frequenz berechnen */
  outportb(0x40, Timer & 0xFF);      /* Low-Byte fr den Timer-Wert setzen */
  outportb(0x40, (Timer >> 8) & 0xFF);                 /* High-Byte setzen */
}

/*=========================================================================*/
/* SetOrigFreq: Stellt die Original-Frequenz des Timer von 18,2 Hz ein.    */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetOrigFreq()

{
  Word Timer;                          /* interner Wert fr Timer-Frequenz */

  outportb(0x43, 0x36);                                    /* Steuerbefehl */
  Timer = 0xFFFF;             /* Timer-Frequenz auf Originalwert (18,2 Hz) */
  outportb(0x40, Timer & 0xFF);         /* Low-Byte fr die Timer-Frequenz */
  outportb(0x40, (Timer >> 8) & 0xFF); /* High-Byte fr die Timer-Frequenz */
}

/*=========================================================================*/
/* ReadDSP: Fragt den Wert des DSP-Ports SB_BaseAddress+0Ah ab. Die Funk-  */
/*          tion wird von CheckBaseAddress aufgerufen, um zu prfen, ob    */
/*          die Basis-I/O-Adresse richtig gesetzt ist.                     */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: ein Bytewert                                                   */
/*-------------------------------------------------------------------------*/

Byte ReadDSP()

{
  Word Count;                                                /* ein Zhler */
  int  Reading;                 /* TRUE, wenn der DSP zum Lesen bereit ist */

  Count = 0;
  Reading = FALSE;
  while (!Reading)            /* wiederholen, bis DSP zum Lesen bereit ist */
    {
      Count++;                                           /* Zhler erhhen */
      Reading = ((inportb(SB_BaseAddress+0x0E) & 0x80) == 0x80);
      if (Count == 10000)      /* Hat der Zhler den Stand 10000 erreicht? */
	{                                               /* ja, also Fehler */
	  printf("\n");
	  printf("Fehler: Der DSP der Sound Blaster-Karte kann nicht ");
	  printf("gelesen werden.\n");
	  printf("        Vielleicht ist keine Soundkarte vorhanden.\n");
	  exit(1);                                     /* Programm beenden */
	}
    }
  return inportb(SB_BaseAddress+0x0A); /* DSP o.k., Bytewert zurckliefern */
}

/*=========================================================================*/
/* WriteDSP: Sendet einen Bytewert an den Soundprozessor. Ist dieser mit   */
/*           seiner letzten Aktion beschftigt, mu beim Senden des Byte-  */
/*           wertes gewartet werden, bis diese Aktion ordungsgem beendet */
/*           ist.                                                          */
/*=========================================================================*/
/* Eingabe: Value = Bytewert                                               */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void WriteDSP(Byte Value)

{
  Word Count;                                                /* ein Zhler */
  int  Writing;          /* TRUE, wenn der DSP zum Datenempfang bereit ist */

  Count = 0;
  Writing = FALSE;
  while (!Writing)    /* wiederholen, solange der DSP nicht empfangsbereit */
    {
      Count++;                                           /* Zhler erhhen */
      Writing =	((inportb(SB_BaseAddress+0x0C) & 0x80) == 0x00);
      if (Count == 10000)      /* Hat der Zhler den Stand 10000 erreicht? */
	{
	  printf("\n");
	  printf("Fehler: Der DSP kann nicht beschrieben werden.\n");
	  exit(1);                                     /* Programm beenden */
	}
    }
  outportb(SB_BaseAddress+0x0C, Value);               /* schreiben zum DSP */
}

/*=========================================================================*/
/* WriteDirect: Veranlat den DSP zur Ausgabe von ungepackten Sounddaten   */
/*              im Direktmodus.                                            */
/*                                                                         */
/*              HINWEIS: Um eine gute VOICE-Ausgabe mit Hilfe dieser Pro-  */
/*                       zedur zu erreichen, kann man den Timer-Interrupt  */
/*                       verwenden. Dabei mu man aber beachten, da die-  */
/*                       ser umprogrammiert werden mu, um eine hhere     */
/*                       Ausgabe-Frequenz als 18,2 mal in der Sekunde zu   */
/*                       erreichen.                                        */
/*=========================================================================*/
/* Eingabe: Value = Bytewert                                               */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void WriteDirect(Byte Value)

{
  WriteDSP(dsp_WriteDirectUnpack);                  /* DSP-Kommando senden */
  WriteDSP(Value);                                  /* Wert zum DSP senden */
}

/*=========================================================================*/
/* Speaker_On: Schaltet den Lautsprecher an.                               */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void Speaker_On()

{
  WriteDSP(dsp_SpeakerOn);
}

/*=========================================================================*/
/* Speaker_Off: Schaltet den Lautsprecher aus.                             */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void Speaker_Off()

{
  WriteDSP(dsp_SpeakerOff);
}

/*=========================================================================*/
/* PlayDirect: Spielt einen Block von ungepackten Daten im Direktmodus mit */
/*             der angegebenen Ausgabe-Frequenz ab. Die Sounddaten mssen  */
/*             sich im Speicher befinden.                                  */
/*=========================================================================*/
/* Eingabe: DataPtr = Zeiger auf den Blockanfang                           */
/*          Size    = Gre des Datenblocks                                */
/*          Freq    = Ausgabe-Frequenz in Hz (19..50000)                   */
/*          Init    = TRUE, wenn der DSP vor der Soundausgabe initiali-    */
/*                    siert werden soll, sonst FALSE                       */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void PlayDirect(Byte *DataPtr, Word Size, Word Freq, int Init)

{
  void interrupt (*TimerOld)();       /* Zeiger auf die alte Timer-Routine */

  if (Size != 0)                                       /* Blockgre <> 0? */
    {
      if (Init)
	InitDSP();                                   /* DSP initialisieren */
      Speaker_On();                                    /* Lautsprecher ein */
      TimerOld = getvect(0x08);           /* alten Timer-Interrupt sichern */
      SetTimerFreq(Freq);                     /* Timer-Frequenz einstellen */
      BytePtr = DataPtr;
      Count_ = 0;
      Done = FALSE;
      BlockSize = Size;
      setvect(0x08, NewTimerInt);              /* Timer-Interrupt umlenken */

      /* wiederholen, bis der Datenblock abgespielt */
      /* oder irgendeine Taste gedrckt wurde       */

      while (!Done)
	if (kbhit())
	  Done = TRUE;

      SetOrigFreq();         /* Original-Frequenz fr den Timer einstellen */
      setvect(0x08, TimerOld);                /* alte Timer-Routine setzen */
      Speaker_Off();                                   /* Lautsprecher aus */
    }
}

/*=========================================================================*/
/* Mixer: Stellt einen der Modi ein: Stereo oder Mono. Auerdem schaltet   */
/*        die Routine den Filter ein bzw. aus.                             */
/*                                                                         */
/*        ANMERKUNG: Die Funktion ist nur bei Sound Blaster Pro verfgbar. */
/*=========================================================================*/
/* Eingabe: Stereo = TRUE, wenn Stereo gesetzt werden soll, FALSE fr Mono */
/*          Filter = TRUE, wenn ein Filter verwendet werden soll, sonst    */
/*                   FALSE                                                 */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void Mixer(int Stereo, int Filter)

{
  Byte Help;                                              /* Hilfsvariable */

  outportb(SB_BaseAddress+0x04, 0x0E);                  /* Kommando senden */
  Help = (inportb(SB_BaseAddress+0x05) & 0xDD);
  if (Stereo)                                                   /* Stereo? */
    Help |= 0x02;                                                    /* ja */
  if (!Filter)                                                  /* Filter? */
    Help |=  0x20;                                                   /* ja */

  /* Einstellungen setzen */

  outportb(SB_BaseAddress+0x04, 0x0E);
  outportb(SB_BaseAddress+0x05, Help);
}
