
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit DMA       : Stellt Routinen zur Programmierung des DMA-Controllers   }
{                  8237A zur Verfgung. In einem PC-XT ist nur ein DMA-     }
{                  Controller vorhanden, in einem PC-AT zwei. In dieser     }
{                  Unit werden beide Controller untersttzt. Die Kanle 4-7 }
{                  des zweiten DMA-Controllers entsprechen im folgenden den }
{                  Kanlen 0-3 (wie beim ersten Controller).                }
{                                                                           }
{                  HINWEIS: Im Zusammenhang mit der Programmierung der DMA- }
{                           Kanle wird darauf hingewiesen, da der Kanal 2 }
{                           (DMA 1) vom Disketten/Festplatten-Controller    }
{                           standardmig belegt ist. Der Kanal 1 wird hu- }
{                           fig von der Sound-Blaster-Karte belegt.         }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : DMA.PAS                                                  }
{ entwickelt am  : 04.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.05                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT DMA;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

CONST

  {-------------------------------------------------------------------------}
  { Bitmasken zum Setzen des Befehlsregisters (nur Schreibzugriff), das     }
  { die Auswahl verschiedener Arbeitsweisen des DMA-Controllers erlaubt.    }
  {-------------------------------------------------------------------------}

  { Bit 0: gesetzt  = Memory-To-Memory-bertragung eingeschaltet }
  {        gelscht = Memory-To-Memory-bertragung ausgeschaltet }

  dma_CmdReg_EnableMemToMem  = $01;
  dma_CmdReg_DisableMemToMem = $00;

  { Bit 1: gesetzt  = Kanal 0/4 liest immer von derselben Speicheradresse }
  {        gelscht = Kanal 0/4 arbeitet im normalen Modus, d.h. die      }
  {                   Speicheradresse wird je nach Programmierung in-     }
  {                   krementiert (erhht) oder dekrementiert (ernied-    }
  {                   rigt)                                               }
  {                                                                       }
  { ANMERKUNG: Dieses Bit hat nur bei Memory-To-Memory-bertragung Be-    }
  {            deutung.                                                   }

  dma_CmdReg_SameAddrHold0   = $02;
  dma_CmdReg_ChangeAddrHold0 = $00;

  { Bit 2: gesetzt  = Controller gesperrt    }
  {        gelscht = Controller freigegeben }

  dma_CmdReg_DisableController = $04;
  dma_CmdReg_EnableController  = $00;

  { Bit 3: gesetzt  = Controller arbeitet im komprimierten Timing-Modus }
  {        gelscht = normales Timing                                   }

  dma_CmdReg_CompressedTiming = $08;
  dma_CmdReg_NormalTiming     = $00;

  { Bit 4: gesetzt  = rotierende Prioritt                          }
  {        gelscht = normale Prioritt nach den Nummern der Kanle }

  dma_CmdReg_RotatePriority = $10;
  dma_CmdReg_NormalPriority = $00;

  { Bit 5: gesetzt  = Controller arbeitet im Extended Write Modus }
  {                   (erweitertes Schreiben)                     }
  {        gelscht = Late Write Modus                            }

  dma_CmdReg_ExtendedWrite = $20;
  dma_CmdReg_LateWrite     = $00;

  { Bit 6: gesetzt  = DMA-Anforderung auf low-aktiv eingestellt  }
  {        gelscht = DMA-Anforderung auf high-aktiv eingestellt }
  {                   (DRQ sensing as high active)               }
  {                                                              }
  { ANMERKUNG: Dieses Bit hat nur Bedeutung, wenn der Controller }
  {            im normalen Timing-Modus arbeitet.                }

  dma_CmdReg_DRQSensingLo = $40;
  dma_CmdReg_DRQSensingHi = $00;

  { Bit 7: gesetzt  = DMA-Besttigung auf high-aktiv eingestellt }
  {                   (DACK sensing as high active)              }
  {        gelscht = DMA-Besttigung auf low-aktiv eingestellt  }
  {                                                              }
  { ANMERKUNG: Dieses Bit hat nur Bedeutung, wenn der Controller }
  {            im normalen Timing-Modus arbeitet.                }

  dma_CmdReg_DACKSensingHi = $80;
  dma_CmdReg_DACKSensingLo = $00;

  {-------------------------------------------------------------------------}
  { Bitmasken zum Lesen des Statusregisters (nur Lesezugruff), das Auf-     }
  { schlu ber den aktuellen Zustand des DMA-Controllers gibt.             }
  {-------------------------------------------------------------------------}

  { Bits 0-7 }

  dma_StatReg_TC0          = $01;   { Terminal Count bei Kanal 0/4 erreicht }
  dma_StatReg_TC1          = $02;   { Terminal Count bei Kanal 1/5 erreicht }
  dma_StatReg_TC2          = $04;   { Terminal Count bei Kanal 2/6 erreicht }
  dma_StatReg_TC3          = $08;   { Terminal Count bei Kanal 3/7 erreicht }
  dma_StatReg_RequestPend0 = $10;                   { Kanal 0/4 angefordert }
  dma_StatReg_RequestPend1 = $20;                   { Kanal 1/5 angefordert }
  dma_StatReg_RequestPend2 = $40;                   { Kanal 2/6 angefordert }
  dma_StatReg_RequestPend3 = $80;                   { Kanal 3/7 angefordert }

  {-------------------------------------------------------------------------}
  { Bitmasken zum Setzen des Startregisters (auch Anforderungsregister ge-  }
  { nannt, nur Schreibzugriff), ber das dem DMA-Controller mitgeteilt      }
  { wird, da er mit der Arbeit beginnen soll.                              }
  {-------------------------------------------------------------------------}

  { Bits 0-1 }

  dma_ReqReg_Select0 = $00;                 { DMA-Anforderung fr Kanal 0/4 }
  dma_ReqReg_Select1 = $01;                 { DMA-Anforderung fr Kanal 1/5 }
  dma_ReqReg_Select2 = $02;                 { DMA-Anforderung fr Kanal 2/6 }
  dma_ReqReg_Select3 = $03;                 { DMA-Anforderung fr Kanal 3/7 }

  { Bit 2: gesetzt  = Bit fr DMA-Anforderung setzen  }
  {        gelscht = Bit fr DMA-Anforderung lschen }

  dma_ReqReg_SetRequestBit   = $04;
  dma_ReqReg_ClearRequestBit = $00;

  { Bits 3-7 haben keine Bedeutung. }

  {-------------------------------------------------------------------------}
  { Bitmasken zum Setzen des Einzelmaskierungsregisters (sog. Single Mask   }
  { Register, nur Schreibzugriff). Das Maskierungsbit eines bestimmten Ka-  }
  { nals mu gelscht werden, damit die DMA-Anforderung fr diesen Kanal    }
  { erfolgen kann.                                                          }
  {-------------------------------------------------------------------------}

  { Bits 0-1 }

  dma_SingleMaskReg_Select0 = $00;                      { Auswahl Kanal 0/4 }
  dma_SingleMaskReg_Select1 = $01;                      { Auswahl Kanal 1/5 }
  dma_SingleMaskReg_Select2 = $02;                      { Auswahl Kanal 2/6 }
  dma_SingleMaskReg_Select3 = $03;                      { Auswahl Kanal 3/7 }

  { Bit 2: gesetzt  = Maskierungsbit fr den gewhlten Kanal setzen }
  {        gelscht = Maskierungsbit lschen                        }

  dma_SingleMaskReg_SetMaskBit   = $04;
  dma_SingleMaskReg_ClearMaskBit = $00;

  { Bits 3-7 haben keine Bedeutung. }

  {-------------------------------------------------------------------------}
  { Bitmasken zum Setzen des Gesamtmaskierungsregisters (nur Schreibzu-     }
  { griff), ber das die Maskierungsbits fr jeden Kanal einzeln gesetzt    }
  { bzw. gelscht werden.                                                   }
  {-------------------------------------------------------------------------}

  { Bits 0-3: jeweils gesetzt  = Maskierungsbit fr den entspr. Kanal }
  {                              setzen                               }
  {           jeweils gelscht = Maskierungsbit lschen               }

  dma_MultiMaskReg_SetMaskBit0 = $01;
  dma_MultiMaskReg_SetMaskBit1 = $02;
  dma_MultiMaskReg_SetMaskBit2 = $04;
  dma_MultiMaskReg_SetMaskBit3 = $08;

  dma_MultiMaskReg_SetAll   = $0F;           { Setzen aller Maskierungsbits }
  dma_MultiMaskReg_ClearAll = $00;          { Lschen aller Maskierungsbits }

  { Bits 4-7 haben keine Bedeutung. }

  {-------------------------------------------------------------------------}
  { Bitmasken zum Setzen des Modusregisters (nur Schreibzugriff), ber das  }
  { verschiedene Modi wie Schreiben oder Lesen der Daten gesetzt werden.    }
  {-------------------------------------------------------------------------}

  { Bits 0-1 }

  dma_ModeReg_Select0 = $00;                            { Auswahl Kanal 0/4 }
  dma_ModeReg_Select1 = $01;                            { Auswahl Kanal 1/5 }
  dma_ModeReg_Select2 = $02;                            { Auswahl Kanal 2/6 }
  dma_ModeReg_Select3 = $03;                            { Auswahl Kanal 3/7 }

  { Bits 2-3 }

  dma_ModeReg_Verify = $00;    { berprfen (es erfolgen keine Operationen) }
  dma_ModeReg_Write  = $04;                                     { Schreiben }
  dma_ModeReg_Read   = $08;                                         { Lesen }

  { Bit 4: gesetzt  = automatische Initialisierung einschalten }
  {        gelscht = automatische Initialisierung ausschalten }

  dma_ModeReg_AutoInitOn  = $10;
  dma_ModeReg_AutoInitOff = $00;

  { Bit 5: gesetzt  = Speicheradresse inkrementieren (erhhen)     }
  {        gelscht = Speicheradresse dekrementieren (erniedrigen) }

  dma_ModeReg_IncAddr = $20;
  dma_ModeReg_DecAddr = $00;

  { Bits 6-7 }

  dma_ModeReg_DemandMode  = $00;                        { Anforderungsmodus }
  dma_ModeReg_SingleMode  = $40;                              { Einzelmodus }
                                         { (Datenbertragung Byte fr Byte) }
  dma_ModeReg_BlockMode   = $80;      { Blockmodus (blockweise bertragung) }
  dma_ModeReg_CascadeMode = $C0;                       { Kaskadierungsmodus }
                                   { (Der zweite DMA-Controller ist mit dem }
                               { Kanal 0 des ersten Controllers verbunden.) }

TYPE

  {=========================================================================}
  { TComputer: Rechnertyp                                                   }
  {=========================================================================}

  TComputer = (XT, AT);

VAR
  Computer : TComputer;

{===========================================================================}
{ Prozedur SetCommandReg: Setzt das Befehlsregister. Dadurch werden ver-    }
{                         schiedene Arbeitsweisen des DMA-Controllers ge-   }
{                         whlt.                                            }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          BitMask = Bitmaske                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetCommandReg(DMA, BitMask : Byte);

{===========================================================================}
{ Funktion GetStatusReg: Liefert den Inhalt des Statusregisters zurck.     }
{===========================================================================}
{ Eingabe: DMA = Nummer des DMA-Controllers                                 }
{ Ausgabe: Bitmaske                                                         }
{---------------------------------------------------------------------------}

FUNCTION GetStatusReg(DMA : Byte) : Byte;

{===========================================================================}
{ Prozedur SetRequestReg: Setzt das Start-, auch Anforderungsregister des   }
{                         DMA-Controllers. Dadurch wird die Anforderung fr }
{                         einen bestimmten Kanal gesetzt (DMA-Start) oder   }
{                         gelscht.                                         }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          BitMask = Bitmaske                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetRequestReg(DMA, BitMask : Byte);

{===========================================================================}
{ Prozedur SetSingleMaskReg: Setzt das Einzelmaskierungsregister des DMA-   }
{                            Controllers.                                   }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          BitMask = Bitmaske                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetSingleMaskReg(DMA, BitMask : Byte);

{===========================================================================}
{ Prozedur SetMultiMaskReg: Setzt das Gesamtmaskierungsregister.            }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          BitMask = Bitmaske                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetMultiMaskReg(DMA, BitMask : Byte);

{===========================================================================}
{ Prozedur SetModeReg: Setzt das Modusregister des DMA-Controllers.         }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          BitMask = Bitmaske                                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetModeReg(DMA, BitMask : Byte);

{===========================================================================}
{ Prozedur StartDMA: Fordert den DMA-Controller auf, seine Arbeit zu begin- }
{                    nen. Dazu wird die Routine SetRequestReg aufgerufen.   }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Nummer des Kanals (0-3)                                }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE StartDMA(DMA, Channel : Byte);

{===========================================================================}
{ Prozedur SetMaskBit: Setzt das Maskierungsbit fr den angegebenen Kanal.  }
{                      Die DMA-Anforderung fr diesen Kanal wird damit be-  }
{                      endet. Intern wird die Routine SetSingleMaskReg auf- }
{                      gerufen.                                             }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Kanalnummer (0-3)                                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetMaskBit(DMA, Channel : Byte);

{===========================================================================}
{ Prozedur ClearMaskBit: Lscht das Maskierungsbit fr das angegebene Ka-   }
{                        nal. Dies mu gemacht werden, damit die DMA-Anfor- }
{                        derung fr diesen Kanal erfolgen kann. Intern wird }
{                        die Routine SetSingleMaskReg aufgerufen.           }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Kanalnummer (0-3)                                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ClearMaskBit(DMA, Channel : Byte);

{===========================================================================}
{ Prozedur SetAllMaskBits: Setzt die Maskierungsbits fr alle vier Kanle   }
{                          auf einmal. Dadurch wird das Setzen der Maskie-  }
{                          rungsbits einzeln gespart. Intern wird die Rou-  }
{                          tine SetMultiMaskReg aufgerufen.                 }
{===========================================================================}
{ Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetAllMaskBits(DMA : Byte);

{===========================================================================}
{ Prozedur ClearAllMaskBits: Lscht die Maskierungsbits fr alle vier Ka-   }
{                            nle auf einmal. Dadurch wird das mehrmalige   }
{                            Lschen gespart. Intern wird SetMultiMaskReg   }
{                            aufgerufen.                                    }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ClearAllMaskBits(DMA : Byte);

{===========================================================================}
{ Prozedur SetDMAMode: Setzt einen Modus oder mehrere Modi fr den angege-  }
{                      benen DMA-Kanal. Dazu wird die Routine SetModeReg    }
{                      aufgerufen.                                          }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Nummer des Kanals (0-3)                                }
{          Mode    = Modus bzw. Modi                                        }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetDMAMode(DMA, Channel, Mode : Byte);

{===========================================================================}
{ Prozedur MasterClear: Lscht das Befehlsregister.                         }
{===========================================================================}
{ Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE MasterClear(DMA : Byte);

{===========================================================================}
{ Prozedur ClearFlipFlop: Lscht den First/Last-FlipFlop, um dadurch dem    }
{                         DMA-Controller mitzuteilen, da zunchst Low-Byte }
{                         und danach High-Byte gesetzt wird, wenn ein Kanal }
{                         programmiert wird.                                }
{===========================================================================}
{ Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ClearFlipFlop(DMA : Byte);

{===========================================================================}
{ Funktion SetAddrAndSideReg: Setzt das Adre- (untere 16 Bits einer 20 Bit }
{                             langen Speicheradresse) und Seitenregister    }
{                             (obere 4 Bits der Speicheradresse) eines be-  }
{                             stimmten Kanals. Die Adresse mu 20 Bit lang  }
{                             sein, wenn ein Adreraum von 1 MB adressiert  }
{                             werden soll.                                  }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Kanalnummer (0-3)                                      }
{          AddrPtr = Speicheradresse in Form eines Zeigers                  }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetAddrAndSideReg(DMA, Channel : Byte; AddrPtr : Pointer);

{===========================================================================}
{ Prozedur SetBaseCountReg: Setzt das Basiszhlregister eines bestimmten    }
{                           Kanals. Dadurch wird dem DMA-Controller mitge-  }
{                           teilt, wie viele Datenbytes bertragen werden.  }
{                           Als Parameter wird die wirkliche Anzahl der     }
{                           Bytes angegeben, gesetzt wird Bytes-1.          }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Nummer des DMA-Kanals (0-3)                            }
{          Bytes   = Anzahl der zu bertragenden Bytes                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetBaseCountReg(DMA, Channel : Byte; Bytes : Word);

{===========================================================================}
{ Funktion GetCurrCountReg: Liefert den Inhalt des aktuellen Zhlregisters  }
{                           zurck, also die Anzahl der noch zu bertragen- }
{                           den Bytes.                                      }
{===========================================================================}
{ Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                  }
{          Channel = Nummer des DMA-Kanals (0-3)                            }
{ Ausgabe: Anzahl der noch zu bertragenden Bytes                           }
{---------------------------------------------------------------------------}

FUNCTION GetCurrCountReg(DMA, Channel : Byte) : Word;

IMPLEMENTATION

{===========================================================================}
{ Prozedur CheckComputer: Stellt den Rechnertyp fest und speichert das Er-  }
{                         gebnis in der Variablen Computer.                 }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE CheckComputer;

BEGIN
  IF Byte(Ptr($F000, $FFFE)^) = $FC THEN                           { PC-AT? }
    Computer := AT                                                     { ja }
  ELSE                                                               { nein }
    Computer := XT;                                                 { PC-XT }
END;

{===========================================================================}
{ Funktion CheckChannel: Setzt die Kanalnummer auf 0, falls eine andere Ka- }
{                        nalnummer als 0, 1, 2 oder 3 gewhlt wurde.        }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Kanals                                          }
{ Ausgabe: Kanalnummer, falls 0, 1, 2 oder 3 gewhlt wurde, sonst 0         }
{---------------------------------------------------------------------------}

FUNCTION CheckChannel(Nmb : Byte) : Byte;

BEGIN
  IF Nmb > 3 THEN                                        { Kanalnummer > 3? }
    CheckChannel := 0                               { ja, Kanalnummer ist 0 }
  ELSE                                                               { nein }
    CheckChannel := Nmb;  { Kanalnummer ist gleich der angegebenen Kanalnr. }
END;

{===========================================================================}
{ Funktion PtrToLongInt: Wandelt einen Zeiger-Ausdruck in einen LongInt-    }
{                        Wert um. Diese Funktion wird benutzt, wenn Spei-   }
{                        cheradressen in Form von Zeigern in LongInt-Zahlen }
{                        umgewandelt werden sollen.                         }
{===========================================================================}
{ Eingabe: P = Zeiger-Ausdruck                                              }
{ Ausgabe: LongInt-Zahl (32 Bit)                                            }
{---------------------------------------------------------------------------}

FUNCTION PtrToLongInt(P : Pointer) : LongInt;

BEGIN
  PtrToLongInt := (LongInt(Seg(P^)) SHL 4)+Ofs(P^);
END;

{===========================================================================}
{ Funktion LowWord: Liefert das niederwertige Wort einer LongInt-Zahl zu-   }
{                   rck.                                                   }
{===========================================================================}
{ Eingabe: LI = LongInt-Wert                                                }
{ Ausgabe: das niederwertige Wort von LI                                    }
{---------------------------------------------------------------------------}

FUNCTION LowWord(LI : LongInt) : Word;

BEGIN
  LowWord := (LI AND $FFFF);
END;

{===========================================================================}
{ Funktion HighWord: Liefert das hherwertige Wort einer LongInt-Zahl zu-   }
{                    rck.                                                  }
{===========================================================================}
{ Eingabe: LI = LongInt-Wert                                                }
{ Ausgabe: das hherwertige Wort von LI                                     }
{---------------------------------------------------------------------------}

FUNCTION HighWord(LI : LongInt) : Word;

BEGIN
  HighWord := (LI SHR 16);
END;

{---------------------------------------------------------------------------}
{ Implementation von modulexternen Routinen                                 }
{---------------------------------------------------------------------------}

PROCEDURE SetCommandReg(DMA, BitMask : Byte);

BEGIN
  IF DMA = 1 THEN                                  { erster DMA-Controller? }
    Port[$08] := BitMask                              { ja, Bitmaske setzen }
  ELSE                                       { nein, zweiter DMA-Controller }
    IF Computer = AT THEN                 { Handelt es sich um einen PC-AT? }
      Port[$D0] := BitMask;                           { ja, Bitmaske setzen }
END;

FUNCTION GetStatusReg(DMA : Byte) : Byte;

BEGIN
  GetStatusReg := 0;
  IF DMA = 1 THEN
    GetStatusReg := Port[$08]
  ELSE
    IF Computer = AT THEN
      GetStatusReg := Port[$D0];
END;

PROCEDURE SetRequestReg(DMA, BitMask : Byte);

BEGIN
  IF DMA = 1 THEN
    Port[$09] := BitMask
  ELSE
    IF Computer = AT THEN
      Port[$D2] := BitMask;
END;

PROCEDURE SetSingleMaskReg(DMA, BitMask : Byte);

BEGIN
  IF DMA = 1 THEN
    Port[$0A] := BitMask
  ELSE
    IF Computer = AT THEN
      Port[$D4] := BitMask;
END;

PROCEDURE SetMultiMaskReg(DMA, BitMask : Byte);

BEGIN
  IF DMA = 1 THEN
    Port[$0F] := BitMask
  ELSE
    IF Computer = AT THEN
      Port[$DE] := BitMask;
END;

PROCEDURE SetModeReg(DMA, BitMask : Byte);

BEGIN
  IF DMA = 1 THEN
    Port[$0B] := BitMask
  ELSE
    IF Computer = AT THEN
      Port[$D6] := BitMask;
END;

PROCEDURE StartDMA(DMA, Channel : Byte);

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  SetRequestReg(DMA, Channel+dma_ReqReg_SetRequestBit);
END;

PROCEDURE SetMaskBit(DMA, Channel : Byte);

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  SetSingleMaskReg(DMA, Channel+dma_SingleMaskReg_SetMaskBit);
END;

PROCEDURE ClearMaskBit(DMA, Channel : Byte);

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  SetSingleMaskReg(DMA, Channel+dma_SingleMaskReg_ClearMaskBit);
END;

PROCEDURE SetAllMaskBits(DMA : Byte);

BEGIN
  SetMultiMaskReg(DMA, dma_MultiMaskReg_SetAll);
END;

PROCEDURE ClearAllMaskBits(DMA : Byte);

BEGIN
  SetMultiMaskReg(DMA, dma_MultiMaskReg_ClearAll);
END;

PROCEDURE SetDMAMode(DMA, Channel, Mode : Byte);

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  Mode := Mode AND $FC;                { Fr Modus sind Bits 2-7 zustndig. }
  IF (Mode AND $0C) = $00 THEN     { Kein Schreiben und Lesen gleichzeitig? }
    SetModeReg(DMA, Channel+Mode);                                   { nein }
END;

PROCEDURE MasterClear(DMA : Byte);

BEGIN
  IF DMA = 1 THEN                                  { erster DMA-Controller? }
    Port[$0D] := $00                          { ja, Befehlsregister lschen }
  ELSE                               { nein, zweiter DMA-Controller gewhlt }
    IF Computer = AT THEN                 { Handelt es sich um einen PC-AT? }
      Port[$DA] := $00;                        { ja, Befehlsregister setzen }
END;

PROCEDURE ClearFlipFlop(DMA : Byte);

BEGIN
  IF DMA = 1 THEN
    Port[$0C] := $00
  ELSE
    IF Computer = AT THEN
      Port[$D8] := $00;
END;

PROCEDURE SetAddrAndSideReg(DMA, Channel : Byte; AddrPtr : Pointer);

VAR
  Side : Byte;

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  Side := Lo(HighWord(PtrToLongInt(AddrPtr)));    { Seitenadresse bestimmen }

  { Seitenadresse setzen }

  IF DMA = 1 THEN
    CASE Channel OF
      0 : Port[$87] := Side;
      1 : Port[$83] := Side;
      2 : Port[$81] := Side;
      3 : Port[$82] := Side;
    END
  ELSE
    IF Computer = AT THEN
      CASE Channel OF
        0 : Port[$8F] := Side;
        1 : Port[$8B] := Side;
        2 : Port[$89] := Side;
        3 : Port[$8A] := Side;
      END;

  ClearFlipFlop(DMA);                                    { FlipFlop lschen }

  { Adreregister setzen }

  IF DMA = 1 THEN
    BEGIN
      Port[2*Channel] := Lo(LowWord(PtrToLongInt(AddrPtr)));
      Port[2*Channel] := Hi(LowWord(PtrToLongInt(AddrPtr)));
    END
  ELSE
    IF Computer = AT THEN
      BEGIN
        Port[2*Channel+$C0] := Lo(LowWord(PtrToLongInt(AddrPtr)));
        Port[2*Channel+$C0] := Hi(LowWord(PtrToLongInt(AddrPtr)));
      END;
END;

PROCEDURE SetBaseCountReg(DMA, Channel : Byte; Bytes : Word);

BEGIN
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  ClearFlipFlop(DMA);                                    { FlipFlop lschen }
  IF DMA = 1 THEN
    BEGIN
      Port[2*Channel+1] := Lo(Bytes-1);                   { Low-Byte setzen }
      Port[2*Channel+1] := Hi(Bytes-1);                  { High-Byte setzen }
    END
  ELSE
    IF Computer = AT THEN
      BEGIN
        Port[2*Channel+$C1] := Lo(Bytes-1);
        Port[2*Channel+$C1] := Hi(Bytes-1);
      END;
END;

FUNCTION GetCurrCountReg(DMA, Channel : Byte) : Word;

VAR
  LowByte, HighByte : Byte;                            { Low- und High-Byte }

BEGIN
  GetCurrCountReg := 0;
  Channel := CheckChannel(Channel);  { Kanalnummer prfen, ggf. korrigieren }
  ClearFlipFlop(DMA);                                    { FlipFlop lschen }
  IF DMA = 1 THEN
    BEGIN
      LowByte := Port[2*Channel+1];                        { Low-Byte lesen }
      HighByte := Port[2*Channel+1];                      { High-Byte lesen }
      GetCurrCountReg := LowByte+(256*HighByte)+1; { Anzahl Bytes insgesamt }
    END
  ELSE
    IF Computer = AT THEN
      BEGIN
        LowByte := Port[2*Channel+$C1];
        HighByte := Port[2*Channel+$C1];
        GetCurrCountReg := LowByte+(256*HighByte)+1;
      END;
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

BEGIN
  CheckComputer;                                   { Rechnertyp feststellen }
END.
